<?php
/**
 * Plugin Name: AegisShield Security
 * Plugin URI:  https://aegisify.com/wordpress-aegis-shield-security-product/
 * Description: AegisShield Security safeguards WordPress sites with enterprise-grade malware detection, intrusion prevention, system hardening, database security, and continuous monitoring.
 * Version:     6.4
 * Author:      Chue Moua | Aegisify
 * Author URI:  https://aegisify.com/
 * Text Domain: aegisshield
 * Domain Path: /languages
 */

defined( 'ABSPATH' ) || exit;

define( 'AEGISSHIELD_VERSION', '6.4' );
define( 'AEGISSHIELD_PLUGIN_FILE', __FILE__ );
define( 'AEGISSHIELD_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'AEGISSHIELD_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

/**
 * Load plugin textdomain.
 *
 * Required for WordPress 6.7+ to avoid early JIT translation notices.
 */
function aegisshield_security_load_textdomain() {
    load_plugin_textdomain(
        'aegisshield-security',
        false,
        dirname( plugin_basename( __FILE__ ) ) . '/languages'
    );
}
add_action( 'init', 'aegisshield_security_load_textdomain' );

// Autoloader.
require_once AEGISSHIELD_PLUGIN_DIR . 'includes/class-as-autoloader.php';
AegisShield\AS_Autoloader::init();

/**
 * Main plugin bootstrap on plugins_loaded.
 */
function aegisshield_security_bootstrap() {
    return AegisShield\AS_Plugin::instance();
}
add_action( 'plugins_loaded', 'aegisshield_security_bootstrap' );

/**
 * Activation hook.
 *
 * Only prepares database tables and schedules cron events.
 */
function aegisshield_security_activate() {
    require_once AEGISSHIELD_PLUGIN_DIR . 'includes/class-as-plugin.php';
    require_once AEGISSHIELD_PLUGIN_DIR . 'includes/class-as-cron.php';

    AegisShield\AS_Plugin::activate_static();
}
register_activation_hook( __FILE__, 'aegisshield_security_activate' );

/**
 * Deactivation hook.
 *
 * Only clears scheduled cron events.
 */
function aegisshield_security_deactivate() {
    require_once AEGISSHIELD_PLUGIN_DIR . 'includes/class-as-cron.php';
    $cron = new AegisShield\AS_Cron();
    $cron->clear_events();
}
register_deactivation_hook( __FILE__, 'aegisshield_security_deactivate' );

/**
 * Check whether AegisShield Pro features are active.
 *
 * This provides a single helper for modules and admin pages to gate Pro-only features.
 *
 * @return bool
 */
if ( ! function_exists( 'aegisshield_is_pro_active' ) ) {
    function aegisshield_is_pro_active() {
        if ( ! class_exists( '\\AegisShield\\AS_Plugin' ) ) {
            return false;
        }

        $plugin = \AegisShield\AS_Plugin::instance();

        if ( method_exists( $plugin, 'get_license_manager' ) && $plugin->get_license_manager() ) {
            /**
             * Filter whether Pro is active.
             *
             * This allows future integrations or overrides.
             */
            return (bool) apply_filters( 'aegisshield_is_pro_active', $plugin->get_license_manager()->is_pro_active() );
        }

        return false;
    }
}

/**
 * Show an upgrade-to-Pro notice under the AegisShield plugin row
 * on the Plugins screen. This keeps the message visually separate
 * from core update notices.
 */
if ( ! function_exists( 'aegisshield_show_pro_upgrade_notice' ) ) {
    add_action(
        'after_plugin_row_aegisshield-security/aegisshield-security.php',
        'aegisshield_show_pro_upgrade_notice',
        10,
        2
    );

    /**
     * Render the Pro upgrade notice on the Plugins list.
     *
     * @param string $plugin_file Plugin file path.
     * @param array  $plugin_data Plugin header data.
     */
    function aegisshield_show_pro_upgrade_notice( $plugin_file, $plugin_data ) {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // Do not show if Pro is active.
        if ( function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active() ) {
            return;
        }

        echo '<tr class="plugin-update-tr"><td colspan="3" class="plugin-update colspanchange"><div class="update-message notice inline notice-warning notice-alt"><p>';

        $message = sprintf(
            /* translators: 1: opening strong tag, 2: closing strong tag, 3: license settings URL. */
            __( '%1$sAegisShield Pro gives you our strongest security shield.%2$s Unlock advanced malware detection, automated deep scanning, database hardening automation, and extended activity logs to keep your site locked down on shared hosting. <a href="%3$s">Upgrade now to enable full protection.</a>', 'aegisshield-security' ),
            '<strong>',
            '</strong>',
            esc_url( admin_url( 'admin.php?page=aegisshield-license' ) )
        );

        echo wp_kses_post( $message );
        echo '</p></div></td></tr>';
    }
}

/**
 * Hook into WordPress plugin update checks using the Chue License Manager
 * as the secure update server for AegisShield.
 *
 * @param object $transient Update transient.
 * @return object
 */
function aegisshield_security_check_for_updates( $transient ) {
    if ( ! is_object( $transient ) || empty( $transient->checked ) ) {
        return $transient;
    }

    if ( ! class_exists( '\\AegisShield\\AS_Plugin' ) ) {
        return $transient;
    }

    $plugin = \AegisShield\AS_Plugin::instance();

    if ( ! method_exists( $plugin, 'get_license_manager' ) ) {
        return $transient;
    }

    $license_manager = $plugin->get_license_manager();
    if ( ! $license_manager || ! method_exists( $license_manager, 'check_for_update' ) ) {
        return $transient;
    }

    $data = $license_manager->check_for_update();
    if ( ! is_array( $data ) || empty( $data['success'] ) || empty( $data['update_available'] ) ) {
        return $transient;
    }

    if ( empty( $data['update_available'] ) || empty( $data['version'] ) || empty( $data['download_url'] ) ) {
        return $transient;
    }

    $plugin_file     = 'aegisshield-security/aegisshield-security.php';
       $current_version = isset( $transient->checked[ $plugin_file ] )
        ? $transient->checked[ $plugin_file ]
        : ( defined( 'AEGISSHIELD_VERSION' ) ? AEGISSHIELD_VERSION : '' );

    if ( $current_version && version_compare( $data['version'], $current_version, '<=' ) ) {
        return $transient;
    }

    $update = (object) array(
        'slug'        => 'aegisshield',
        'plugin'      => $plugin_file,
        'new_version' => $data['version'],
        'requires'    => isset( $data['requires'] ) ? $data['requires'] : '',
        'package'     => $data['download_url'],
        'url'         => isset( $data['homepage'] ) ? $data['homepage'] : 'https://aegisify.com',
    );

    $transient->response[ $plugin_file ] = $update;

    return $transient;
}
add_filter( 'pre_set_site_transient_update_plugins', 'aegisshield_security_check_for_updates' );