<?php
namespace AegisShield\Admin_Pages;

use AegisShield\AS_Plugin;
use AegisShield\Modules\AS_Module_DB_Tools;

defined( 'ABSPATH' ) || exit;

/**
 * Dashboard page.
 */
class AS_Page_Dashboard {

    /**
     * Plugin instance.
     *
     * @var AS_Plugin
     */
    protected $plugin;

    /**
     * Constructor.
     *
     * @param AS_Plugin $plugin Plugin instance.
     */
    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Render dashboard.
     *
     * @return void
     */
    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'aegisshield-security' ) );
        }

        // Core metrics and DB health.
        $db_widget      = $this->build_db_health_widget();
        $health_metrics = $this->build_core_health_metrics();

        // License / Pro status.
        $license_manager = ( $this->plugin && method_exists( $this->plugin, 'get_license_manager' ) )
            ? $this->plugin->get_license_manager()
            : null;

        $is_pro = ( $license_manager && method_exists( $license_manager, 'is_pro_active' ) )
            ? (bool) $license_manager->is_pro_active()
            : false;

        // ---------------------------------------------------------------------
        // Security score calculation (high‑level approximate risk indicator).
        // ---------------------------------------------------------------------
        $score        = 100;
        $score_reason = array();

        if ( isset( $db_widget['prefix_status'] ) && false !== strpos( $db_widget['prefix_status'], 'default' ) ) {
            $score -= 15;
            $score_reason[] = __( 'Database is still using the default table prefix.', 'aegisshield-security' );
        }

        if ( isset( $db_widget['spike_count'] ) && (int) $db_widget['spike_count'] > 0 ) {
            $score -= 10;
            $score_reason[] = __( 'One or more tables show significant growth since the last snapshot.', 'aegisshield-security' );
        }

        if ( ! empty( $health_metrics['plugin_updates'] ) ) {
            $score -= min( 20, (int) $health_metrics['plugin_updates'] * 2 );
            $score_reason[] = sprintf(
                /* translators: %d: number of plugins with updates. */
                _n( '%d plugin has a pending update.', '%d plugins have pending updates.', (int) $health_metrics['plugin_updates'], 'aegisshield-security' ),
                (int) $health_metrics['plugin_updates']
            );
        }

        if ( ! empty( $health_metrics['theme_updates'] ) ) {
            $score -= min( 10, (int) $health_metrics['theme_updates'] * 3 );
            $score_reason[] = sprintf(
                /* translators: %d: number of themes with updates. */
                _n( '%d theme has a pending update.', '%d themes have pending updates.', (int) $health_metrics['theme_updates'], 'aegisshield-security' ),
                (int) $health_metrics['theme_updates']
            );
        }

        if ( isset( $health_metrics['admin_users'] ) && $health_metrics['admin_users'] > 3 ) {
            $score -= 10;
            $score_reason[] = __( 'There are more than three administrator accounts.', 'aegisshield-security' );
        }

        if ( isset( $health_metrics['php_version'] ) && version_compare( $health_metrics['php_version'], '8.0', '<' ) ) {
            $score -= 10;
            $score_reason[] = __( 'The site is running an older PHP version. Consider upgrading.', 'aegisshield-security' );
        }

        if ( $score < 0 ) {
            $score = 0;
        } elseif ( $score > 100 ) {
            $score = 100;
        }

        if ( $score >= 90 ) {
            $score_label = __( 'Secure', 'aegisshield-security' );
            $score_class = 'aegisshield-score-good';
        } elseif ( $score >= 70 ) {
            $score_label = __( 'Needs attention', 'aegisshield-security' );
            $score_class = 'aegisshield-score-warn';
        } else {
            $score_label = __( 'High risk', 'aegisshield-security' );
            $score_class = 'aegisshield-score-bad';
        }

        // ---------------------------------------------------------------------
        // Phase 2: Deep scan summaries (File Integrity + Malware / Quick Scan).
        // ---------------------------------------------------------------------
        $settings = ( $this->plugin && method_exists( $this->plugin, 'get_settings' ) )
            ? $this->plugin->get_settings()
            : null;

        $fi_summary_line   = '';
        $mw_summary_line   = '';
        $attack_story_line = __( 'Attack Story correlates malware hits, file changes, and admin activity into a single timeline so you can see how an intrusion unfolded.', 'aegisshield-security' );

        if ( $settings ) {
            // File Integrity.
            $fi_report = $settings->get( 'file_integrity', 'last_scan_report', array() );
            if ( ! is_array( $fi_report ) ) {
                $fi_report = array();
            }

            $fi_mode   = isset( $fi_report['mode'] ) ? (string) $fi_report['mode'] : '';
            $fi_stats  = ( isset( $fi_report['stats'] ) && is_array( $fi_report['stats'] ) ) ? $fi_report['stats'] : array();
            $fi_high   = isset( $fi_report['high'] ) ? (int) $fi_report['high'] : 0;
            $fi_medium = isset( $fi_report['medium'] ) ? (int) $fi_report['medium'] : 0;
            $fi_low    = isset( $fi_report['low'] ) ? (int) $fi_report['low'] : 0;
            $fi_time   = isset( $fi_report['timestamp'] ) ? (string) $fi_report['timestamp'] : '';

            $fi_scanned  = isset( $fi_stats['scanned'] ) ? (int) $fi_stats['scanned'] : 0;
            $fi_modified = isset( $fi_stats['modified'] ) ? (int) $fi_stats['modified'] : 0;
            $fi_new      = isset( $fi_stats['new'] ) ? (int) $fi_stats['new'] : 0;

            $fi_mode_label = '';
            if ( 'light' === $fi_mode ) {
                $fi_mode_label = __( 'Light', 'aegisshield-security' );
            } elseif ( 'hybrid' === $fi_mode ) {
                $fi_mode_label = __( 'Hybrid', 'aegisshield-security' );
            } elseif ( 'full' === $fi_mode ) {
                $fi_mode_label = __( 'Full', 'aegisshield-security' );
            }

            $fi_time_label = '';
            if ( ! empty( $fi_time ) ) {
                $timestamp = strtotime( $fi_time );
                if ( $timestamp ) {
                    $fi_time_label = date_i18n(
                        get_option( 'date_format' ) . ' ' . get_option( 'time_format' ),
                        $timestamp
                    );
                } else {
                    $fi_time_label = $fi_time;
                }
            }

            if ( ! empty( $fi_report ) && $fi_scanned > 0 ) {
                $fi_summary_line = sprintf(
                    /* translators: 1: scan mode, 2: scanned, 3: modified, 4: new, 5: high, 6: medium, 7: low, 8: time. */
                    __( 'File Integrity: %1$s scan, %2$d files scanned, %3$d modified, %4$d new. Findings: %5$d high / %6$d medium / %7$d low. Last run: %8$s.', 'aegisshield-security' ),
                    $fi_mode_label ? $fi_mode_label : __( 'Unknown mode', 'aegisshield-security' ),
                    $fi_scanned,
                    $fi_modified,
                    $fi_new,
                    $fi_high,
                    $fi_medium,
                    $fi_low,
                    $fi_time_label ? $fi_time_label : __( 'n/a', 'aegisshield-security' )
                );
            } else {
                $fi_summary_line = __( 'File Integrity: no scans have been recorded yet.', 'aegisshield-security' );
            }

            // Malware / Quick Scan.
            $malware_meta = $settings->get( 'malware', 'last_results_meta', array() );
            if ( ! is_array( $malware_meta ) ) {
                $malware_meta = array();
            }

            $mw_type       = isset( $malware_meta['scan_type'] ) ? (string) $malware_meta['scan_type'] : '';
            $mw_started    = isset( $malware_meta['started_at'] ) ? (int) $malware_meta['started_at'] : 0;
            $mw_completed  = isset( $malware_meta['completed_at'] ) ? (int) $malware_meta['completed_at'] : 0;
            $mw_files      = isset( $malware_meta['file_count'] ) ? (int) $malware_meta['file_count'] : 0;
            $mw_suspects   = isset( $malware_meta['suspect_count'] ) ? (int) $malware_meta['suspect_count'] : 0;
            $mw_is_partial = ! empty( $malware_meta['partial'] );

            $mw_label_type = '';
            if ( false !== strpos( $mw_type, 'quick' ) ) {
                $mw_label_type = __( 'Quick malware scan (changed files only)', 'aegisshield-security' );
            } elseif ( 'manual' === $mw_type ) {
                $mw_label_type = __( 'Manual full malware scan', 'aegisshield-security' );
            } elseif ( 'auto' === $mw_type ) {
                $mw_label_type = __( 'Automatic full malware scan', 'aegisshield-security' );
            } elseif ( ! empty( $mw_type ) ) {
                $mw_label_type = $mw_type;
            }

            $mw_time_label = '';
            if ( $mw_completed ) {
                $mw_time_label = date_i18n(
                    get_option( 'date_format' ) . ' ' . get_option( 'time_format' ),
                    $mw_completed
                );
            }

            if ( ! empty( $malware_meta ) && $mw_files > 0 ) {
                $mw_summary_line = sprintf(
                    /* translators: 1: scan label, 2: files, 3: suspects, 4: completed time, 5: partial note. */
                    __( 'Malware / Quick Scan: %1$s scanned %2$d files, %3$d suspects. Completed: %4$s%5$s', 'aegisshield-security' ),
                    $mw_label_type ? $mw_label_type : __( 'Latest scan', 'aegisshield-security' ),
                    $mw_files,
                    $mw_suspects,
                    $mw_time_label ? $mw_time_label : __( 'n/a', 'aegisshield-security' ),
                    $mw_is_partial ? ' ' . __( '(partial quick scan – remaining files will be scanned later)', 'aegisshield-security' ) : ''
                );
            } else {
                $mw_summary_line = __( 'Malware / Quick Scan: no recent scan metadata found.', 'aegisshield-security' );
            }
        } else {
            $fi_summary_line = __( 'File Integrity: settings are not available.', 'aegisshield-security' );
            $mw_summary_line = __( 'Malware / Quick Scan: settings are not available.', 'aegisshield-security' );
        }

        // ---------------------------------------------------------------------
        // Phase 3: Activity stats + Alert Center preview.
        // ---------------------------------------------------------------------
        $activity_stats = $this->get_recent_event_stats();
        $recent_alerts  = $this->get_recent_alert_events( 10 );

        ?>
        <div class="wrap aegisshield-dashboard-wrap">
            <h1><?php esc_html_e( 'AegisShield Security Dashboard', 'aegisshield-security' ); ?></h1>

            <p class="aegisshield-dashboard-intro">
                <?php esc_html_e( 'The AegisShield Dashboard provides a centralized, high-level view of your WordPress site’s security posture by combining system health, active protections, recent events, and detected risks into a single, easy-to-understand interface. It helps administrators quickly assess overall security status, identify issues that need attention, and confidently navigate deeper security tools when action is required—without needing to sift through complex logs or technical data.' ); ?>
            </p>

            <style>
                .aegisshield-dashboard-wrap .aegisshield-section {
                    margin-top: 20px;
                }
                .aegisshield-dashboard-wrap .aegisshield-flex-row {
                    display: flex;
                    flex-wrap: wrap;
                    gap: 16px;
                }
                .aegisshield-dashboard-wrap .aegisshield-panel {
                    background: #ffffff;
                    border: 1px solid #ccd0d4;
                    box-shadow: 0 1px 1px rgba(0, 0, 0, 0.04);
                    padding: 16px 18px;
                    box-sizing: border-box;
                }
                .aegisshield-dashboard-wrap .aegisshield-panel-half {
                    flex: 1 1 48%;
                    min-width: 260px;
                }
                .aegisshield-dashboard-wrap .aegisshield-panel-full {
                    flex: 1 1 100%;
                }
                .aegisshield-dashboard-wrap .aegisshield-panel-header {
                    display: flex;
                    align-items: center;
                    margin-bottom: 8px;
                }
                .aegisshield-dashboard-wrap .aegisshield-panel-title {
                    font-size: 15px;
                    font-weight: 600;
                    margin: 0;
                    color: #1d2327;
                }
                .aegisshield-dashboard-wrap .aegisshield-panel-title-icon {
                    margin-right: 6px;
                }
                .aegisshield-dashboard-wrap .aegisshield-panel-subtitle {
                    font-size: 12px;
                    color: #555d66;
                    margin: 0 0 8px 0;
                }
                .aegisshield-dashboard-wrap .aegisshield-score-value {
                    font-size: 26px;
                    font-weight: 700;
                }
                .aegisshield-dashboard-wrap .aegisshield-score-label {
                    font-size: 13px;
                    margin-top: 4px;
                }
                .aegisshield-dashboard-wrap .aegisshield-score-good .aegisshield-score-value {
                    color: #008a20;
                }
                .aegisshield-dashboard-wrap .aegisshield-score-warn .aegisshield-score-value {
                    color: #d98300;
                }
                .aegisshield-dashboard-wrap .aegisshield-score-bad .aegisshield-score-value {
                    color: #d63638;
                }
                .aegisshield-dashboard-wrap .aegisshield-score-reasons {
                    margin: 8px 0 0 0;
                    padding-left: 18px;
                    font-size: 12px;
                    color: #555d66;
                }
                .aegisshield-dashboard-wrap .aegisshield-kv-list {
                    margin: 0;
                    padding: 0;
                    list-style: none;
                    font-size: 12px;
                }
                .aegisshield-dashboard-wrap .aegisshield-kv-list li {
                    display: flex;
                    justify-content: space-between;
                    padding: 2px 0;
                }
                .aegisshield-dashboard-wrap .aegisshield-kv-label {
                    color: #555d66;
                }
                .aegisshield-dashboard-wrap .aegisshield-kv-value {
                    font-weight: 600;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-panel {
                    border-left: 4px solid #2271b1;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-header-badge {
                    display: inline-block;
                    margin-left: 8px;
                    padding: 2px 8px;
                    border-radius: 20px;
                    font-size: 10px;
                    font-weight: 600;
                    text-transform: uppercase;
                    background: #2271b1;
                    color: #ffffff;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-subpanel-grid {
                    display: flex;
                    flex-wrap: wrap;
                    gap: 16px;
                    margin-top: 10px;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-subpanel {
                    flex: 1 1 32%;
                    min-width: 230px;
                    background: #f8fafc;
                    border-radius: 4px;
                    padding: 10px 12px;
                    box-sizing: border-box;
                    border: 1px solid #e2e8f0;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-subpanel-title {
                    font-size: 13px;
                    font-weight: 600;
                    margin: 0 0 4px 0;
                    color: #1d2327;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-subpanel-list {
                    margin: 0;
                    padding-left: 18px;
                    font-size: 12px;
                    color: #555d66;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-dimmed {
                    opacity: 0.6;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-footer {
                    margin-top: 10px;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                    font-size: 12px;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-footer .button-primary {
                    background: #2271b1;
                    border-color: #2271b1;
                }
                .aegisshield-dashboard-wrap .aegisshield-events-list {
                    margin: 0;
                    padding-left: 18px;
                    font-size: 12px;
                    color: #555d66;
                }
                .aegisshield-dashboard-wrap .aegisshield-alert-list {
                    margin: 0;
                    padding-left: 18px;
                    font-size: 12px;
                    color: #555d66;
                }
                .aegisshield-dashboard-wrap .aegisshield-alert-list li span {
                    font-size: 11px;
                    color: #6c757d;
                }
                .aegisshield-dashboard-wrap .aegisshield-modules-grid {
                    display: flex;
                    flex-wrap: wrap;
                    gap: 16px;
                    margin-top: 10px;
                }
                .aegisshield-dashboard-wrap .aegisshield-module-card {
                    flex: 1 1 32%;
                    min-width: 230px;
                    background: #ffffff;
                    border: 1px solid #ccd0d4;
                    box-shadow: 0 1px 1px rgba(0, 0, 0, 0.04);
                    padding: 12px 14px;
                    box-sizing: border-box;
                }
                .aegisshield-dashboard-wrap .aegisshield-module-header {
                    display: flex;
                    align-items: center;
                    margin-bottom: 4px;
                }
                .aegisshield-dashboard-wrap .aegisshield-module-icon {
                    margin-right: 6px;
                }
                .aegisshield-dashboard-wrap .aegisshield-module-title {
                    font-weight: 600;
                    margin: 0;
                    font-size: 13px;
                }
                .aegisshield-dashboard-wrap .aegisshield-module-status {
                    margin-left: auto;
                    font-size: 11px;
                    padding: 2px 6px;
                    border-radius: 3px;
                }
                .aegisshield-dashboard-wrap .aegisshield-status-ok {
                    background: #e7f7ec;
                    color: #008a20;
                }
                .aegisshield-dashboard-wrap .aegisshield-status-warn {
                    background: #fff4e5;
                    color: #d98300;
                }
                .aegisshield-dashboard-wrap .aegisshield-status-bad {
                    background: #f5c2c7;
                    color: #842029;
                }
                .aegisshield-dashboard-wrap .aegisshield-module-body {
                    font-size: 12px;
                    color: #555d66;
                    margin-bottom: 8px;
                }
                .aegisshield-dashboard-wrap .aegisshield-module-actions .button.button-small {
                    font-size: 11px;
                    padding: 2px 8px;
                    height: auto;
                    line-height: 18px;
                }
                @media (max-width: 960px) {
                    .aegisshield-dashboard-wrap .aegisshield-panel-half {
                        flex: 1 1 100%;
                    }
                    .aegisshield-dashboard-wrap .aegisshield-pro-subpanel {
                        flex: 1 1 100%;
                    }
                    .aegisshield-dashboard-wrap .aegisshield-module-card {
                        flex: 1 1 100%;
                    }
                }
            </style>

            <!-- Section 1: Score + Core Health -->
            <div class="aegisshield-section">
                <div class="aegisshield-flex-row">
                    <div class="aegisshield-panel aegisshield-panel-half <?php echo esc_attr( $score_class ); ?>">
                        <div class="aegisshield-panel-header">
                            <span class="aegisshield-panel-title-icon">🛡️</span>
                            <h2 class="aegisshield-panel-title"><?php esc_html_e( 'Overall Security Score', 'aegisshield-security' ); ?></h2>
                        </div>
                        <div class="aegisshield-panel-body">
                            <div class="aegisshield-score-value">
                                <?php
                                printf(
                                    /* translators: %d: security score. */
                                    esc_html__( '%d/100', 'aegisshield-security' ),
                                    (int) $score
                                );
                                ?>
                            </div>
                            <div class="aegisshield-score-label">
                                <?php echo esc_html( $score_label ); ?>
                            </div>
                            <?php if ( ! empty( $score_reason ) ) : ?>
                                <ul class="aegisshield-score-reasons">
                                    <?php foreach ( $score_reason as $reason ) : ?>
                                        <li><?php echo esc_html( $reason ); ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            <?php else : ?>
                                <p class="aegisshield-panel-subtitle">
                                    <?php esc_html_e( 'All high‑level checks look good. Keep monitoring modules below for changes.', 'aegisshield-security' ); ?>
                                </p>
                            <?php endif; ?>
                        </div>
                    </div>

                    <div class="aegisshield-panel aegisshield-panel-half">
                        <div class="aegisshield-panel-header">
                            <span class="aegisshield-panel-title-icon">🧩</span>
                            <h2 class="aegisshield-panel-title"><?php esc_html_e( 'Core Health Metrics', 'aegisshield-security' ); ?></h2>
                        </div>
                        <p class="aegisshield-panel-subtitle">
                            <?php esc_html_e( 'These metrics summarize your WordPress, PHP, and database state at a glance.', 'aegisshield-security' ); ?>
                        </p>
                        <ul class="aegisshield-kv-list">
                            <li>
                                <span class="aegisshield-kv-label"><?php esc_html_e( 'WordPress version', 'aegisshield-security' ); ?></span>
                                <span class="aegisshield-kv-value">
                                    <?php echo esc_html( $health_metrics['wp_version'] ); ?>
                                </span>
                            </li>
                            <li>
                                <span class="aegisshield-kv-label"><?php esc_html_e( 'PHP version', 'aegisshield-security' ); ?></span>
                                <span class="aegisshield-kv-value">
                                    <?php echo esc_html( $health_metrics['php_version'] ); ?>
                                </span>
                            </li>
                            <li>
                                <span class="aegisshield-kv-label"><?php esc_html_e( 'Plugins with updates', 'aegisshield-security' ); ?></span>
                                <span class="aegisshield-kv-value">
                                    <?php echo esc_html( (int) $health_metrics['plugin_updates'] ); ?>
                                </span>
                            </li>
                            <li>
                                <span class="aegisshield-kv-label"><?php esc_html_e( 'Themes with updates', 'aegisshield-security' ); ?></span>
                                <span class="aegisshield-kv-value">
                                    <?php echo esc_html( (int) $health_metrics['theme_updates'] ); ?>
                                </span>
                            </li>
                            <li>
                                <span class="aegisshield-kv-label"><?php esc_html_e( 'Administrator accounts', 'aegisshield-security' ); ?></span>
                                <span class="aegisshield-kv-value">
                                    <?php echo esc_html( (int) $health_metrics['admin_users'] ); ?>
                                </span>
                            </li>
                            <?php if ( ! empty( $db_widget['total_size_human'] ) ) : ?>
                                <li>
                                    <span class="aegisshield-kv-label"><?php esc_html_e( 'Database size', 'aegisshield-security' ); ?></span>
                                    <span class="aegisshield-kv-value"><?php echo esc_html( $db_widget['total_size_human'] ); ?></span>
                                </li>
                            <?php endif; ?>
                            <?php if ( ! empty( $db_widget['prefix_status'] ) ) : ?>
                                <li>
                                    <span class="aegisshield-kv-label"><?php esc_html_e( 'DB prefix', 'aegisshield-security' ); ?></span>
                                    <span class="aegisshield-kv-value"><?php echo esc_html( $db_widget['prefix_status'] ); ?></span>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </div>
                </div>
            </div>

            <!-- Section 2: AegisShield Pro Intelligence Panel -->
            <div class="aegisshield-section">
                <div class="aegisshield-panel aegisshield-panel-full aegisshield-pro-panel<?php echo esc_attr( $is_pro ? '' : ' aegisshield-pro-dimmed' ); ?>">
                    <div class="aegisshield-panel-header">
                        <span class="aegisshield-panel-title-icon">🤖</span>
                        <h2 class="aegisshield-panel-title">
                            <?php esc_html_e( 'AegisShield Pro Intelligence', 'aegisshield-security' ); ?>
                            <span class="aegisshield-pro-header-badge"><?php esc_html_e( 'Pro', 'aegisshield-security' ); ?></span>
                        </h2>
                    </div>
                    <p class="aegisshield-panel-subtitle">
                        <?php esc_html_e( 'Aggregated intelligence from Activity Log, Login Guard, File Integrity, Malware scanning, and Security Headers into one premium overview.', 'aegisshield-security' ); ?>
                    </p>

                    <div class="aegisshield-pro-subpanel-grid">
                        <!-- Enhanced system health metrics -->
                        <div class="aegisshield-pro-subpanel">
                            <p class="aegisshield-pro-subpanel-title">
                                <?php esc_html_e( 'Enhanced system health metrics', 'aegisshield-security' ); ?>
                            </p>
                            <ul class="aegisshield-pro-subpanel-list">
                                <li>
                                    <?php
                                    printf(
                                        /* translators: 1: admin user count. */
                                        esc_html__( 'Admin accounts: %1$d total (keep this as low as practical).', 'aegisshield-security' ),
                                        (int) $health_metrics['admin_users']
                                    );
                                    ?>
                                </li>
                                <li>
                                    <?php
                                    printf(
                                        /* translators: 1: plugin updates, 2: theme updates. */
                                        esc_html__( 'Updates: %1$d plugins and %2$d themes have pending updates.', 'aegisshield-security' ),
                                        (int) $health_metrics['plugin_updates'],
                                        (int) $health_metrics['theme_updates']
                                    );
                                    ?>
                                </li>
                                <?php if ( ! empty( $db_widget['largest_table'] ) ) : ?>
                                    <li>
                                        <?php
                                        printf(
                                            /* translators: 1: largest table name, 2: largest table size. */
                                            esc_html__( 'Largest DB table: %1$s (%2$s).', 'aegisshield-security' ),
                                            esc_html( $db_widget['largest_table'] ),
                                            ! empty( $db_widget['largest_table_size_human'] ) ? esc_html( $db_widget['largest_table_size_human'] ) : esc_html__( 'size unknown', 'aegisshield-security' )
                                        );
                                        ?>
                                    </li>
                                <?php endif; ?>
                                <li>
                                    <?php
                                    printf(
                                        esc_html__( 'Database growth spikes detected: %d table(s).', 'aegisshield-security' ),
                                        isset( $db_widget['spike_count'] ) ? (int) $db_widget['spike_count'] : 0
                                    );
                                    ?>
                                </li>
                                <li>
                                    <?php esc_html_e( 'Hardening and Security Headers modules help reduce attack surface and enforce browser‑side protections.', 'aegisshield-security' ); ?>
                                </li>
                            </ul>
                        </div>

                        <!-- Deep scan summaries -->
                        <div class="aegisshield-pro-subpanel">
                            <p class="aegisshield-pro-subpanel-title">
                                <?php esc_html_e( 'Deep scan summaries (Integrity, Quick Scan, Attack Story)', 'aegisshield-security' ); ?>
                            </p>
                            <ul class="aegisshield-pro-subpanel-list">
                                <li><?php echo esc_html( $fi_summary_line ); ?></li>
                                <li><?php echo esc_html( $mw_summary_line ); ?></li>
                                <li><?php echo esc_html( $attack_story_line ); ?></li>
                            </ul>
                        </div>

                        <!-- Pro module indicators -->
                        <div class="aegisshield-pro-subpanel">
                            <p class="aegisshield-pro-subpanel-title">
                                <?php esc_html_e( 'Pro module indicators (CSP Builder, Attack Story, etc.)', 'aegisshield-security' ); ?>
                            </p>
                            <ul class="aegisshield-pro-subpanel-list">
                                <li>
                                    <?php esc_html_e( 'CSP Builder Pro & header profiles for advanced Security Headers management.', 'aegisshield-security' ); ?>
                                </li>
                                <li>
                                    <?php esc_html_e( 'Attack Story view on Malware page to reconstruct multi‑step attacks (file changes + malware hits).', 'aegisshield-security' ); ?>
                                </li>
                                <li>
                                    <?php esc_html_e( 'Incremental Quick Scan that focuses on changed files using File Monitor baselines.', 'aegisshield-security' ); ?>
                                </li>
                                <li>
                                    <?php esc_html_e( 'Alert rules on Activity Log to send notifications when critical events occur.', 'aegisshield-security' ); ?>
                                </li>
                                <li>
                                    <?php esc_html_e( 'Email alert logic for file changes and high‑risk events.', 'aegisshield-security' ); ?>
                                </li>
                            </ul>
                        </div>
                    </div>

                    <div class="aegisshield-pro-footer">
                        <?php if ( $is_pro ) : ?>
                            <span><?php esc_html_e( 'Your AegisShield Pro license is active. All Pro intelligence modules are available.', 'aegisshield-security' ); ?></span>
                        <?php else : ?>
                            <span><?php esc_html_e( 'Activate AegisShield Pro to unlock deep scan summaries, Attack Story, and advanced header / alert controls.', 'aegisshield-security' ); ?></span>
                            <a class="button button-primary" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ); ?>">
                                <?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Section 3: Security Events + Alert Center -->
            <div class="aegisshield-section">
                <div class="aegisshield-flex-row">
                    <!-- Security events (last 24 hours) -->
                    <div class="aegisshield-panel aegisshield-panel-half">
                        <div class="aegisshield-panel-header">
                            <span class="aegisshield-panel-title-icon">📈</span>
                            <h2 class="aegisshield-panel-title"><?php esc_html_e( 'Security events (last 24 hours)', 'aegisshield-security' ); ?></h2>
                        </div>
                        <?php
                        $failed        = isset( $activity_stats['login_failed'] ) ? (int) $activity_stats['login_failed'] : 0;
                        $success       = isset( $activity_stats['login_success'] ) ? (int) $activity_stats['login_success'] : 0;
                        $lockouts      = isset( $activity_stats['login_lockout'] ) ? (int) $activity_stats['login_lockout'] : 0;
                        $admin_created = isset( $activity_stats['admin_created'] ) ? (int) $activity_stats['admin_created'] : 0;
                        $admin_deleted = isset( $activity_stats['admin_deleted'] ) ? (int) $activity_stats['admin_deleted'] : 0;
                        $admin_changes = $admin_created + $admin_deleted;
                        ?>
                        <ul class="aegisshield-events-list">
                            <li><?php esc_html_e( 'Failed logins:', 'aegisshield-security' ); ?> <?php echo esc_html( $failed ); ?></li>
                            <li><?php esc_html_e( 'Login lockouts:', 'aegisshield-security' ); ?> <?php echo esc_html( $lockouts ); ?></li>
                            <li><?php esc_html_e( 'Successful logins:', 'aegisshield-security' ); ?> <?php echo esc_html( $success ); ?></li>
                            <li><?php esc_html_e( 'Admin account changes:', 'aegisshield-security' ); ?> <?php echo esc_html( $admin_changes ); ?></li>
                        </ul>
                    </div>

                    <!-- Alert Center preview -->
                    <div class="aegisshield-panel aegisshield-panel-half">
                        <div class="aegisshield-panel-header">
                            <span class="aegisshield-panel-title-icon">🚨</span>
                            <h2 class="aegisshield-panel-title">
                                <?php esc_html_e( 'Alert Center preview', 'aegisshield-security' ); ?>
                            </h2>
                        </div>

                        <?php if ( $is_pro ) : ?>
                            <?php if ( ! empty( $recent_alerts ) ) : ?>
                                <ul class="aegisshield-alert-list">
                                    <?php foreach ( $recent_alerts as $event ) : ?>
                                        <?php
                                        $user_label = '';
                                        if ( ! empty( $event->user_id ) ) {
                                            $user = get_userdata( (int) $event->user_id );
                                            if ( $user && ! is_wp_error( $user ) ) {
                                                $user_label = $user->user_login;
                                            }
                                        }

                                        $time_label = $event->event_time;
                                        if ( ! empty( $event->event_time ) ) {
                                            $ts = strtotime( $event->event_time );
                                            if ( $ts ) {
                                                $time_label = date_i18n(
                                                    get_option( 'date_format' ) . ' ' . get_option( 'time_format' ),
                                                    $ts
                                                );
                                            }
                                        }
                                        ?>
                                        <li>
                                            <strong><?php echo esc_html( $event->event_type ); ?></strong>
                                            &mdash;
                                            <?php echo esc_html( wp_trim_words( $event->message, 12, '…' ) ); ?>
                                            <?php if ( $user_label ) : ?>
                                                (<?php echo esc_html( $user_label ); ?>)
                                            <?php endif; ?>
                                            <?php if ( ! empty( $event->ip_address ) ) : ?>
                                                &middot; <?php echo esc_html( $event->ip_address ); ?>
                                            <?php endif; ?>
                                            &middot; <span><?php echo esc_html( $time_label ); ?></span>
                                        </li>
                                    <?php endforeach; ?>
                                </ul>
                            <?php else : ?>
                                <p class="aegisshield-panel-subtitle">
                                    <?php esc_html_e( 'No recent alert‑level events have been logged yet.', 'aegisshield-security' ); ?>
                                </p>
                            <?php endif; ?>
                        <?php else : ?>
                            <p class="aegisshield-panel-subtitle">
                                <?php esc_html_e( 'Alert Center is part of AegisShield Pro. It surfaces high‑impact security events such as malware detections, lockouts, and admin changes in one feed.', 'aegisshield-security' ); ?>
                            </p>
                            <a class="button button-primary" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ); ?>">
                                <?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Section 4: Modules overview -->
            <div class="aegisshield-section">
                <div class="aegisshield-panel aegisshield-panel-full">
                    <div class="aegisshield-panel-header">
                        <span class="aegisshield-panel-title-icon">🧭</span>
                        <h2 class="aegisshield-panel-title"><?php esc_html_e( 'Modules overview', 'aegisshield-security' ); ?></h2>
                    </div>
                    <p class="aegisshield-panel-subtitle">
                        <?php esc_html_e( 'Jump directly into key security modules below.', 'aegisshield-security' ); ?>
                    </p>

                    <div class="aegisshield-modules-grid">
                        <!-- Activity Log -->
                        <div class="aegisshield-module-card">
                            <div class="aegisshield-module-header">
                                <span class="dashicons dashicons-list-view aegisshield-module-icon" aria-hidden="true"></span>
                                <h3 class="aegisshield-module-title"><?php esc_html_e( 'Activity Log', 'aegisshield-security' ); ?></h3>
                                <span class="aegisshield-module-status aegisshield-status-ok">
                                    <?php esc_html_e( 'OK', 'aegisshield-security' ); ?>
                                </span>
                            </div>
                            <div class="aegisshield-module-body">
                                <?php esc_html_e( 'Track important security‑related events such as logins, changes, and administrative actions.', 'aegisshield-security' ); ?>
                            </div>
                            <div class="aegisshield-module-actions">
                                <a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-activity-log' ) ); ?>">
                                    <?php esc_html_e( 'View Activity Log', 'aegisshield-security' ); ?>
                                </a>
                            </div>
                        </div>

                        <!-- Login Guard -->
                        <div class="aegisshield-module-card">
                            <div class="aegisshield-module-header">
                                <span class="dashicons dashicons-shield-alt aegisshield-module-icon" aria-hidden="true"></span>
                                <h3 class="aegisshield-module-title"><?php esc_html_e( 'Login Guard', 'aegisshield-security' ); ?></h3>
                                <span class="aegisshield-module-status aegisshield-status-ok">
                                    <?php esc_html_e( 'OK', 'aegisshield-security' ); ?>
                                </span>
                            </div>
                            <div class="aegisshield-module-body">
                                <?php esc_html_e( 'Protects your login form against brute‑force attempts by tracking failures and blocking abusive IPs.', 'aegisshield-security' ); ?>
                            </div>
                            <div class="aegisshield-module-actions">
                                <a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-login-guard' ) ); ?>">
                                    <?php esc_html_e( 'View Login Guard', 'aegisshield-security' ); ?>
                                </a>
                            </div>
                        </div>

                        <!-- File Integrity -->
                        <div class="aegisshield-module-card">
                            <div class="aegisshield-module-header">
                                <span class="dashicons dashicons-media-code aegisshield-module-icon" aria-hidden="true"></span>
                                <h3 class="aegisshield-module-title"><?php esc_html_e( 'File Integrity', 'aegisshield-security' ); ?></h3>
                                <span class="aegisshield-module-status aegisshield-status-ok">
                                    <?php esc_html_e( 'OK', 'aegisshield-security' ); ?>
                                </span>
                            </div>
                            <div class="aegisshield-module-body">
                                <?php esc_html_e( 'Monitors key files for unexpected changes, additions, or removals that may indicate tampering.', 'aegisshield-security' ); ?>
                            </div>
                            <div class="aegisshield-module-actions">
                                <a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-file-integrity' ) ); ?>">
                                    <?php esc_html_e( 'Open File Integrity', 'aegisshield-security' ); ?>
                                </a>
                            </div>
                        </div>

                        <!-- Hardening -->
                        <div class="aegisshield-module-card">
                            <div class="aegisshield-module-header">
                                <span class="dashicons dashicons-lock aegisshield-module-icon" aria-hidden="true"></span>
                                <h3 class="aegisshield-module-title"><?php esc_html_e( 'Hardening', 'aegisshield-security' ); ?></h3>
                                <span class="aegisshield-module-status aegisshield-status-ok">
                                    <?php esc_html_e( 'OK', 'aegisshield-security' ); ?>
                                </span>
                            </div>
                            <div class="aegisshield-module-body">
                                <?php esc_html_e( 'Applies recommended WordPress hardening such as disabling file editors and tightening entry points.', 'aegisshield-security' ); ?>
                            </div>
                            <div class="aegisshield-module-actions">
                                <a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-hardening' ) ); ?>">
                                    <?php esc_html_e( 'Open Hardening', 'aegisshield-security' ); ?>
                                </a>
                            </div>
                        </div>

                        <!-- Security Headers -->
                        <div class="aegisshield-module-card">
                            <div class="aegisshield-module-header">
                                <span class="dashicons dashicons-admin-network aegisshield-module-icon" aria-hidden="true"></span>
                                <h3 class="aegisshield-module-title"><?php esc_html_e( 'Security Headers', 'aegisshield-security' ); ?></h3>
                                <span class="aegisshield-module-status aegisshield-status-ok">
                                    <?php esc_html_e( 'OK', 'aegisshield-security' ); ?>
                                </span>
                            </div>
                            <div class="aegisshield-module-body">
                                <?php esc_html_e( 'Helps send modern browser security headers such as X‑Frame‑Options, Referrer‑Policy, and HSTS.', 'aegisshield-security' ); ?>
                            </div>
                            <div class="aegisshield-module-actions">
                                <a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-sec-headers' ) ); ?>">
                                    <?php esc_html_e( 'Open Security Headers', 'aegisshield-security' ); ?>
                                </a>
                            </div>
                        </div>

                        <!-- Malware -->
                        <div class="aegisshield-module-card">
                            <div class="aegisshield-module-header">
                                <span class="dashicons dashicons-warning aegisshield-module-icon" aria-hidden="true"></span>
                                <h3 class="aegisshield-module-title"><?php esc_html_e( 'Malware tools', 'aegisshield-security' ); ?></h3>
                                <span class="aegisshield-module-status aegisshield-status-ok">
                                    <?php esc_html_e( 'OK', 'aegisshield-security' ); ?>
                                </span>
                            </div>
                            <div class="aegisshield-module-body">
                                <?php esc_html_e( 'Scans for suspicious code patterns and known malware signatures.', 'aegisshield-security' ); ?>
                            </div>
                            <div class="aegisshield-module-actions">
                                <a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-malware' ) ); ?>">
                                    <?php esc_html_e( 'Open Malware Tools', 'aegisshield-security' ); ?>
                                </a>
                            </div>
                        </div>

                        <!-- Database Tools -->
                        <div class="aegisshield-module-card">
                            <div class="aegisshield-module-header">
                                <span class="dashicons dashicons-database aegisshield-module-icon" aria-hidden="true"></span>
                                <h3 class="aegisshield-module-title"><?php esc_html_e( 'Database Tools', 'aegisshield-security' ); ?></h3>
                                <span class="aegisshield-module-status <?php echo ( ! empty( $db_widget['spike_count'] ) ) ? 'aegisshield-status-warn' : 'aegisshield-status-ok'; ?>">
                                    <?php
                                    if ( ! empty( $db_widget['spike_count'] ) ) {
                                        esc_html_e( 'Check growth', 'aegisshield-security' );
                                    } else {
                                        esc_html_e( 'OK', 'aegisshield-security' );
                                    }
                                    ?>
                                </span>
                            </div>
                            <div class="aegisshield-module-body">
                                <?php esc_html_e( 'Review table sizes, growth, and perform cleanup or maintenance operations.', 'aegisshield-security' ); ?>
                            </div>
                            <div class="aegisshield-module-actions">
                                <a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-db-tools' ) ); ?>">
                                    <?php esc_html_e( 'Open DB Tools', 'aegisshield-security' ); ?>
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Build core health metrics.
     *
     * @return array
     */
    protected function build_core_health_metrics() {
        $wp_version  = get_bloginfo( 'version' );
        $php_version = PHP_VERSION;

        // Plugin update count.
        $plugin_updates      = 0;
        $plugins_update_data = get_site_transient( 'update_plugins' );
        if ( is_object( $plugins_update_data ) && ! empty( $plugins_update_data->response ) && is_array( $plugins_update_data->response ) ) {
            $plugin_updates = count( $plugins_update_data->response );
        }

        // Theme update count.
        $theme_updates     = 0;
        $theme_update_data = get_site_transient( 'update_themes' );
        if ( is_object( $theme_update_data ) && ! empty( $theme_update_data->response ) && is_array( $theme_update_data->response ) ) {
            $theme_updates = count( $theme_update_data->response );
        }

        // Administrator accounts.
        $admin_users = 0;
        if ( function_exists( 'get_users' ) ) {
            $admins = get_users(
                array(
                    'role'   => 'administrator',
                    'fields' => 'ID',
                )
            );
            if ( is_array( $admins ) ) {
                $admin_users = count( $admins );
            }
        }

        return array(
            'wp_version'     => $wp_version,
            'php_version'    => $php_version,
            'plugin_updates' => (int) $plugin_updates,
            'theme_updates'  => (int) $theme_updates,
            'admin_users'    => (int) $admin_users,
        );
    }

    /**
     * Get the activity log table name.
     *
     * @return string
     */
    protected function get_activity_log_table_name() {
        global $wpdb;

        if ( ! $this->plugin || ! method_exists( $this->plugin, 'get_logger' ) ) {
            return '';
        }

        $logger = $this->plugin->get_logger();
        if ( $logger && method_exists( $logger, 'get_table_name' ) ) {
            return $logger->get_table_name();
        }

        return $wpdb->prefix . 'aegisshield_activity_log';
    }

    /**
     * Get counts for key security events within a recent time window.
     *
     * @param int $seconds Lookback window in seconds.
     * @return array
     */
    protected function get_recent_event_stats( $seconds = 0 ) {
        global $wpdb;

        $table = $this->get_activity_log_table_name();
        if ( empty( $table ) ) {
            return array();
        }

        $default_window = defined( 'DAY_IN_SECONDS' ) ? DAY_IN_SECONDS : 86400;
        $seconds        = absint( $seconds );
        if ( $seconds <= 0 ) {
            $seconds = $default_window;
        }

        $since = gmdate( 'Y-m-d H:i:s', time() - $seconds );

        $types = array(
            'login_failed',
            'login_success',
            'login_lockout',
            'plugin_activated',
            'plugin_deactivated',
            'admin_created',
            'admin_deleted',
        );

        $placeholders = implode( ',', array_fill( 0, count( $types ), '%s' ) );
        $params       = array_merge( array( $since ), $types );

        $sql = $wpdb->prepare(
            "SELECT event_type, COUNT(*) AS cnt
             FROM {$table}
             WHERE event_time >= %s
               AND event_type IN ({$placeholders})
             GROUP BY event_type",
            $params
        );

        $rows  = $wpdb->get_results( $sql );
        $stats = array_fill_keys( $types, 0 );

        if ( $rows ) {
            foreach ( $rows as $row ) {
                if ( isset( $stats[ $row->event_type ] ) ) {
                    $stats[ $row->event_type ] = (int) $row->cnt;
                }
            }
        }

        return $stats;
    }

    /**
     * Get recent alert-like events for the Alert Center preview.
     *
     * @param int $limit Number of events.
     * @return array
     */
    protected function get_recent_alert_events( $limit = 10 ) {
        global $wpdb;

        $table = $this->get_activity_log_table_name();
        if ( empty( $table ) ) {
            return array();
        }

        $limit = absint( $limit );
        if ( $limit <= 0 ) {
            $limit = 10;
        } elseif ( $limit > 50 ) {
            $limit = 50;
        }

        $alert_types = array(
            'malware_detected',
            'file_integrity_changed',
            'login_failed',
            'login_lockout',
            'admin_created',
            'admin_deleted',
            'role_changed',
            'plugin_activated',
            'plugin_deactivated',
            'settings_changed',
        );

        $placeholders = implode( ',', array_fill( 0, count( $alert_types ), '%s' ) );
        $args         = $alert_types;
        $args[]       = $limit;

        $sql = $wpdb->prepare(
            "SELECT event_time, event_type, user_id, ip_address, message
             FROM {$table}
             WHERE event_type IN ({$placeholders})
             ORDER BY event_time DESC
             LIMIT %d",
            $args
        );

        $rows = $wpdb->get_results( $sql );
        if ( ! $rows ) {
            return array();
        }

        return $rows;
    }

    /**
     * Build DB health widget data.
     *
     * @return array
     */
    protected function build_db_health_widget() {
        global $wpdb;

        $data = array(
            'total_size_human'          => '',
            'largest_table'             => '',
            'largest_table_size_human'  => '',
            'spike_count'               => 0,
            'prefix_status'             => '',
        );

        try {
            $module  = new AS_Module_DB_Tools( $this->plugin );
            $summary = $module->get_table_summary();
        } catch ( \Exception $e ) { // phpcs:ignore Generic.CodeAnalysis.EmptyStatement.DetectedCatch
            return $data;
        }

        $tables = isset( $summary['tables'] ) ? $summary['tables'] : array();

        if ( empty( $tables ) ) {
            return $data;
        }

        $total_bytes   = 0;
        $largest_name  = '';
        $largest_bytes = 0;
        $spike_count   = 0;

        foreach ( $tables as $table ) {
            $name        = isset( $table['name'] ) ? $table['name'] : '';
            $total       = isset( $table['total_bytes'] ) ? (int) $table['total_bytes'] : 0;
            $delta_bytes = isset( $table['delta_bytes'] ) ? (int) $table['delta_bytes'] : 0;

            $total_bytes += $total;

            if ( $total > $largest_bytes ) {
                $largest_bytes = $total;
                $largest_name  = $name;
            }

            if ( $delta_bytes > 0 ) {
                $spike_count++;
            }
        }

        $data['total_size_human']         = size_format( $total_bytes );
        $data['largest_table']            = $largest_name;
        $data['largest_table_size_human'] = $largest_bytes ? size_format( $largest_bytes ) : '';
        $data['spike_count']              = $spike_count;

        if ( 'wp_' === $wpdb->prefix ) {
            $data['prefix_status'] = __( 'Using default \"wp_\" prefix (consider changing).', 'aegisshield-security' );
        } else {
            $data['prefix_status'] = __( 'Using a custom table prefix.', 'aegisshield-security' );
        }

        return $data;
    }
}
