<?php
namespace AegisShield\Admin_Pages;

use AegisShield\AS_Plugin;
use AegisShield\Modules\AS_Module_File_Integrity;

defined( 'ABSPATH' ) || exit;

/**
 * File Integrity admin page.
 */
class AS_Page_File_Integrity {

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

	/**
	 * Check if Pro is active.
	 */
	protected function is_pro_active() {
		return function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();
	}

	/**
	 * Standard Pro-locked upsell notice.
	 */
	protected function render_pro_locked_notice() {
		?>
		<div class="notice notice-warning inline">
			<p><strong><?php esc_html_e( 'This feature is available in AegisShield Pro.', 'aegisshield-security' ); ?></strong></p>
			<p><?php esc_html_e( 'Upgrade to unlock full File Integrity features including Scan History and File Change Monitor advanced controls.', 'aegisshield-security' ); ?></p>
			<p>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield_license' ) ); ?>"
				   class="button button-primary">
					<?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
				</a>
			</p>
		</div>
		<?php
	}

    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $settings = $this->plugin->get_settings();
        $section  = 'file_integrity';
        $updated  = false;
        $scan_ran = false;
        $scan_summary = '';
        $scan_items   = array();
        $monitor_section  = 'file_monitor';
        $file_monitor_interval = (string) $settings->get( $monitor_section, 'interval', '15' );
        if ( ! in_array( $file_monitor_interval, array( '5', '10', '15', '30', '60' ), true ) ) {
            $file_monitor_interval = '15';
        }


        $current_mode = (string) $settings->get( $section, 'scan_mode', 'light' );
        if ( ! in_array( $current_mode, array( 'light', 'hybrid', 'full' ), true ) ) {
            $current_mode = 'light';
        }

        
        // Handle settings save / scan.
        $auto_frequency  = (string) $settings->get( $section, 'auto_scan_frequency', 'weekly' );
        $alert_threshold = (string) $settings->get( $section, 'email_alert_threshold', 'none' );
        $history_limit   = (string) $settings->get( $section, 'scan_history_limit', '10' );

        if ( isset( $_POST['aegisshield_file_integrity_nonce'] )
            && check_admin_referer( 'aegisshield_file_integrity_save', 'aegisshield_file_integrity_nonce' )
        ) {
            $mode = isset( $_POST['scan_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['scan_mode'] ) ) : 'light';
            if ( ! in_array( $mode, array( 'light', 'hybrid', 'full' ), true ) ) {
                $mode = 'light';
            }

            $auto_frequency = isset( $_POST['auto_scan_frequency'] ) ? sanitize_text_field( wp_unslash( $_POST['auto_scan_frequency'] ) ) : $auto_frequency;
            if ( ! in_array( $auto_frequency, array( 'daily', 'weekly', 'monthly' ), true ) ) {
                $auto_frequency = 'weekly';
            }

            $alert_threshold = isset( $_POST['email_alert_threshold'] ) ? sanitize_text_field( wp_unslash( $_POST['email_alert_threshold'] ) ) : $alert_threshold;
            if ( ! in_array( $alert_threshold, array( 'none', 'all', 'med_high', 'high_only' ), true ) ) {
                $alert_threshold = 'none';
            }

            $history_limit = isset( $_POST['scan_history_limit'] ) ? sanitize_text_field( wp_unslash( $_POST['scan_history_limit'] ) ) : $history_limit;
            if ( ! in_array( $history_limit, array( '10', '25', '50', '100', 'unlimited' ), true ) ) {
                $history_limit = '10';
            }

            $settings->set( $section, 'scan_mode', $mode );
            $settings->set( $section, 'auto_scan_frequency', $auto_frequency );
            $settings->set( $section, 'email_alert_threshold', $alert_threshold );
            $settings->set( $section, 'scan_history_limit', $history_limit );
            $settings->save();

            $current_mode   = $mode;
            $updated        = true;

            if ( isset( $_POST['run_scan_now'] ) ) {
                // Run scan immediately using the selected mode.
                $module       = new AS_Module_File_Integrity( $this->plugin );
                $result       = $module->run_manual_scan();
                $scan_ran     = true;
                $scan_summary = isset( $result['summary'] ) ? $result['summary'] : '';
                $scan_items   = isset( $result['items'] ) ? $result['items'] : array();
            }
        } else {
            // Refresh current values when not saving.
            $auto_frequency  = (string) $settings->get( $section, 'auto_scan_frequency', 'weekly' );
            $alert_threshold = (string) $settings->get( $section, 'email_alert_threshold', 'none' );
            $history_limit   = (string) $settings->get( $section, 'scan_history_limit', '10' );
        }

        $description  = $this->get_mode_description( $current_mode );
        $last_report  = $settings->get( $section, 'last_scan_report', array() );

        // Messages and selected entry for scan history actions.
        $history_message    = '';
        $history_view_entry = null;

        // Handle scan history actions (View / Delete).
        if ( isset( $_POST['aegisshield_file_integrity_history_action'] ) ) {
            $raw_action = sanitize_text_field( wp_unslash( $_POST['aegisshield_file_integrity_history_action'] ) );
            $action     = $raw_action;

            if ( isset( $_POST['aegisshield_file_integrity_history_nonce'] )
                && wp_verify_nonce(
                    sanitize_text_field( wp_unslash( $_POST['aegisshield_file_integrity_history_nonce'] ) ),
                    'aegisshield_file_integrity_history_action'
                )
            ) {
                $scan_history = $settings->get( $section, 'scan_history', array() );
                if ( ! is_array( $scan_history ) ) {
                    $scan_history = array();
                }

                $index = isset( $_POST['history_index'] ) ? (int) $_POST['history_index'] : -1;

                if ( $index >= 0 && $index < count( $scan_history ) ) {
                    if ( 'view' === $action ) {
                        $history_view_entry = $scan_history[ $index ];
                    } elseif ( 'delete' === $action ) {
                        array_splice( $scan_history, $index, 1 );
                        $settings->set( $section, 'scan_history', $scan_history );
                        $settings->save();
                        $history_message = __( 'Selected scan history entry deleted.', 'aegisshield-security' );
                    }
                } else {
                    $history_message = __( 'Invalid scan history entry selected.', 'aegisshield-security' );
                }
            } else {
                $history_message = __( 'Security check failed for scan history action.', 'aegisshield-security' );
            }
        }

        
        // Save File Change Monitor settings (interval) if requested.
        if ( isset( $_POST['aegisshield_file_monitor_nonce'] )
            && check_admin_referer( 'aegisshield_file_monitor_save', 'aegisshield_file_monitor_nonce' )
        ) {
            $interval = isset( $_POST['file_monitor_interval'] ) ? sanitize_text_field( wp_unslash( $_POST['file_monitor_interval'] ) ) : $file_monitor_interval;
            if ( ! in_array( $interval, array( '5', '10', '15', '30', '60' ), true ) ) {
                $interval = '15';
            }

            $settings->set( $monitor_section, 'interval', $interval );
            $settings->save();

            $file_monitor_interval = $interval;
            $updated               = true;
        }

		// Save File Monitor Email Alerts (PRO)
		if (
			isset( $_POST['aegisshield_file_monitor_email_nonce'] )
			&& check_admin_referer(
				'aegisshield_file_monitor_email_save',
				'aegisshield_file_monitor_email_nonce'
			)
			&& $this->is_pro_active()
		) {
			$email_mode = isset( $_POST['file_monitor_email_mode'] )
				? sanitize_text_field( wp_unslash( $_POST['file_monitor_email_mode'] ) )
				: 'instant';

			if ( ! in_array( $email_mode, array( 'instant', 'summary' ), true ) ) {
				$email_mode = 'instant';
			}

			$email_events = isset( $_POST['file_monitor_email_events'] ) && is_array( $_POST['file_monitor_email_events'] )
				? array_map( 'sanitize_key', wp_unslash( $_POST['file_monitor_email_events'] ) )
				: array();
				
			$allowed_events = array( 'php', 'plugin_theme', 'high_risk', 'new', 'modified', 'deleted' );

			$email_recipients = isset( $_POST['file_monitor_email_recipients'] )
				? sanitize_text_field( wp_unslash( $_POST['file_monitor_email_recipients'] ) )
				: '';

			$settings->set( 'file_monitor', 'email_mode', $email_mode );
			$settings->set( 'file_monitor', 'email_events', $email_events );
			$settings->set( 'file_monitor', 'email_recipients', $email_recipients );
			$settings->save();

			$updated = true;
		}
		if ( $updated && isset( $_POST['aegisshield_file_monitor_email_nonce'] ) ) {
			echo '<div class="notice notice-success is-dismissible">';
			echo '<p>' . esc_html__( 'Email alert settings saved.', 'aegisshield-security' ) . '</p>';
			echo '</div>';
		}

// Final scan history array for display.
        $scan_history = $settings->get( $section, 'scan_history', array() );
        if ( ! is_array( $scan_history ) ) {
            $scan_history = array();
        }

        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'File Integrity Monitoring', 'aegisshield-security' ); ?></h1>
            <p><?php esc_html_e( 'AegisShield File Integrity Monitoring protects your WordPress site by continuously scanning and tracking changes to core files, plugins, and themes, helping administrators quickly detect unauthorized or unexpected modifications. With flexible scan modes, scheduled monitoring, and detailed audit logs, it provides clear visibility into file activity while integrating seamlessly with alerts and enforcement rules to support proactive, policy-driven security management.', 'aegisshield-security' ); ?></p>

            <?php if ( $updated && ! $scan_ran ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'File Integrity settings saved.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( $scan_ran ) : ?>
                <div class="notice notice-info">
                    <p><?php echo esc_html( $scan_summary ); ?></p>
                </div>
            <?php endif; ?>

            
            <h2 class="nav-tab-wrapper aegisshield-fi-tabs">
                <a href="#" class="nav-tab nav-tab-active" data-as-fi-tab="scan"><?php esc_html_e( 'Integrity Scan', 'aegisshield-security' ); ?></a>
                <a href="#" class="nav-tab" data-as-fi-tab="history"><?php esc_html_e( 'Scan History', 'aegisshield-security' ); ?>
					<?php if ( ! $this->is_pro_active() ) : ?>
						<span class="aegisshield-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
							<?php esc_html_e( 'Pro', 'aegisshield-security' ); ?>
						</span>
					<?php endif; ?>
				</a>
                <a href="#" class="nav-tab" data-as-fi-tab="monitor"><?php esc_html_e( 'File Change Monitor', 'aegisshield-security' ); ?>
					<?php if ( ! $this->is_pro_active() ) : ?>
						<span class="aegisshield-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
							<?php esc_html_e( 'Pro', 'aegisshield-security' ); ?>
						</span>
					<?php endif; ?>
				</a>
            </h2>

            <div class="aegisshield-fi-tab aegisshield-fi-tab-scan" data-as-fi-tab-content="scan">
<form method="post">
                <?php wp_nonce_field( 'aegisshield_file_integrity_save', 'aegisshield_file_integrity_nonce' ); ?>

                
                <h2><?php esc_html_e( 'Scan Settings', 'aegisshield-security' ); ?></h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="scan_mode"><?php esc_html_e( 'Scan Mode', 'aegisshield-security' ); ?></label>
                        </th>
                        <td>
                            <select name="scan_mode" id="aegisshield_scan_mode">
                                <option value="light" <?php selected( $current_mode, 'light' ); ?>><?php esc_html_e( 'Light Scan', 'aegisshield-security' ); ?></option>
                                <option value="hybrid" <?php selected( $current_mode, 'hybrid' ); ?>><?php esc_html_e( 'Hybrid Scan', 'aegisshield-security' ); ?></option>
                                <option value="full" <?php selected( $current_mode, 'full' ); ?>><?php esc_html_e( 'Full Scan', 'aegisshield-security' ); ?></option>
                            </select>
                            <div id="aegisshield-scan-description" class="description">
                                <?php echo wp_kses_post( $description ); ?>
                            </div>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="auto_scan_frequency"><?php esc_html_e( 'Automatic Scan Frequency', 'aegisshield-security' ); ?></label>
                        </th>
                        <td>
                            <select name="auto_scan_frequency" id="auto_scan_frequency">
                                <option value="daily" <?php selected( $auto_frequency, 'daily' ); ?>><?php esc_html_e( 'Daily', 'aegisshield-security' ); ?></option>
                                <option value="weekly" <?php selected( $auto_frequency, 'weekly' ); ?>><?php esc_html_e( 'Weekly (recommended)', 'aegisshield-security' ); ?></option>
                                <option value="monthly" <?php selected( $auto_frequency, 'monthly' ); ?>><?php esc_html_e( 'Monthly', 'aegisshield-security' ); ?></option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="email_alert_threshold"><?php esc_html_e( 'Email Alert Threshold', 'aegisshield-security' ); ?></label>
                        </th>
                        <td>
                            <select name="email_alert_threshold" id="email_alert_threshold">
                                <option value="none" <?php selected( $alert_threshold, 'none' ); ?>><?php esc_html_e( 'NONE (no email alerts)', 'aegisshield-security' ); ?></option>
                                <option value="all" <?php selected( $alert_threshold, 'all' ); ?>><?php esc_html_e( 'All Warnings', 'aegisshield-security' ); ?></option>
                                <option value="med_high" <?php selected( $alert_threshold, 'med_high' ); ?>><?php esc_html_e( 'Medium & High only', 'aegisshield-security' ); ?></option>
                                <option value="high_only" <?php selected( $alert_threshold, 'high_only' ); ?>><?php esc_html_e( 'High only', 'aegisshield-security' ); ?></option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="scan_history_limit"><?php esc_html_e( 'Scan History to Keep', 'aegisshield-security' ); ?></label>
                        </th>
                        <td>
                            <select name="scan_history_limit" id="scan_history_limit">
                                <option value="10" <?php selected( $history_limit, '10' ); ?>><?php esc_html_e( '10 (recommended)', 'aegisshield-security' ); ?></option>
                                <option value="25" <?php selected( $history_limit, '25' ); ?>><?php esc_html_e( '25', 'aegisshield-security' ); ?></option>
                                <option value="50" <?php selected( $history_limit, '50' ); ?>><?php esc_html_e( '50', 'aegisshield-security' ); ?></option>
                                <option value="100" <?php selected( $history_limit, '100' ); ?>><?php esc_html_e( '100', 'aegisshield-security' ); ?></option>
                                <option value="unlimited" <?php selected( $history_limit, 'unlimited' ); ?>><?php esc_html_e( 'Unlimited', 'aegisshield-security' ); ?></option>
                            </select>
                        </td>
                    </tr>
                </table>


                <p>
                    <?php submit_button( __( 'Save Settings', 'aegisshield-security' ), 'secondary', 'submit', false ); ?>
                    <button type="submit" name="run_scan_now" class="button button-primary">
                        <?php esc_html_e( 'Run Scan Now', 'aegisshield-security' ); ?>
                    </button>
                
            </p>
            </form>

            </div>


            <hr />
            

			<div class="aegisshield-fi-tab aegisshield-fi-tab-monitor" data-as-fi-tab-content="monitor" style="display:none;">

				<?php if ( ! $this->is_pro_active() ) : ?>

					<?php $this->render_pro_locked_notice(); ?>

				</div>
				<?php else : ?>	
                <?php
                $is_pro_monitor = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();
                ?>

                <h2><?php esc_html_e( 'File Change Monitor', 'aegisshield-security' ); ?></h2>

                <p><?php esc_html_e( 'The File Change Monitor watches your WordPress files for new, modified, or deleted files using a lightweight background scan.', 'aegisshield-security' ); ?></p>

                <div class="aegisshield-file-monitor-status">
                    <form method="post">
                        <?php wp_nonce_field( 'aegisshield_file_monitor_save', 'aegisshield_file_monitor_nonce' ); ?>
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="file_monitor_interval"><?php esc_html_e( 'Scan Interval', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <select name="file_monitor_interval" id="file_monitor_interval">
                                        <option value="5" <?php selected( $file_monitor_interval, '5' ); ?>><?php esc_html_e( 'Every 5 minutes', 'aegisshield-security' ); ?></option>
                                        <option value="10" <?php selected( $file_monitor_interval, '10' ); ?>><?php esc_html_e( 'Every 10 minutes', 'aegisshield-security' ); ?></option>
                                        <option value="15" <?php selected( $file_monitor_interval, '15' ); ?>><?php esc_html_e( 'Every 15 minutes (recommended)', 'aegisshield-security' ); ?></option>
                                        <option value="30" <?php selected( $file_monitor_interval, '30' ); ?>><?php esc_html_e( 'Every 30 minutes', 'aegisshield-security' ); ?></option>
                                        <option value="60" <?php selected( $file_monitor_interval, '60' ); ?>><?php esc_html_e( 'Every 60 minutes', 'aegisshield-security' ); ?></option>
                                    </select>
                                    <p class="description">
                                        <?php esc_html_e( 'Adjust how often the File Change Monitor runs. Shorter intervals give faster detection but use slightly more resources.', 'aegisshield-security' ); ?>
                                    </p>
                                </td>
                            </tr>
                        </table>

                        <?php submit_button( __( 'Save Monitor Settings', 'aegisshield-security' ), 'secondary', 'submit', false ); ?>
                    </form>
                </div>

                <hr />

                <div class="aegisshield-file-monitor-log">
                    <?php
                    global $wpdb;
                    $monitor_table = $wpdb->prefix . 'aegisshield_file_monitor';
                    $file_changes  = array();

                    // Attempt to load the most recent file changes, if the table exists.
                    $monitor_table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $monitor_table ) );
                    if ( $monitor_table_exists === $monitor_table ) {
                        $file_changes = $wpdb->get_results(
                            "SELECT file_path, status, size, mtime, first_seen, last_seen FROM {$monitor_table} ORDER BY last_seen DESC LIMIT 100",
                            ARRAY_A
                        );
                    }
                    ?>
                    <h3><?php esc_html_e( 'Recent File Changes', 'aegisshield-security' ); ?></h3>
                    <p class="description">
                        <?php esc_html_e( 'Below are the most recent file changes seen by the File Change Monitor. These entries help you spot unexpected or suspicious modifications in your WordPress installation.', 'aegisshield-security' ); ?>
                    </p>

                    <?php if ( empty( $file_changes ) ) : ?>
                        <p><?php esc_html_e( 'No file changes have been recorded yet. Monitoring will begin after the next scheduled scan.', 'aegisshield-security' ); ?></p>
                    <?php else : ?>
                        <table class="widefat fixed striped">
                            <thead>
                                <tr>
                                    <th><?php esc_html_e( 'File', 'aegisshield-security' ); ?></th>
                                    <th><?php esc_html_e( 'Status', 'aegisshield-security' ); ?></th>
                                    <th><?php esc_html_e( 'Size (bytes)', 'aegisshield-security' ); ?></th>
                                    <th><?php esc_html_e( 'Last Modified', 'aegisshield-security' ); ?></th>
                                    <th><?php esc_html_e( 'First Seen', 'aegisshield-security' ); ?></th>
                                    <th><?php esc_html_e( 'Last Seen', 'aegisshield-security' ); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ( $file_changes as $change ) : ?>
                                    <?php
                                    $file_path  = isset( $change['file_path'] ) ? $change['file_path'] : '';
                                    $status     = isset( $change['status'] ) ? $change['status'] : '';
                                    $size       = isset( $change['size'] ) ? (int) $change['size'] : 0;
                                    $mtime      = isset( $change['mtime'] ) ? (int) $change['mtime'] : 0;
                                    $first_seen = isset( $change['first_seen'] ) ? $change['first_seen'] : '';
                                    $last_seen  = isset( $change['last_seen'] ) ? $change['last_seen'] : '';

                                    if ( $mtime > 0 ) {
                                        $mtime_str = date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $mtime );
                                    } else {
                                        $mtime_str = '';
                                    }

                                    if ( 'new' === $status ) {
                                        $status_label = __( 'New', 'aegisshield-security' );
                                    } elseif ( 'modified' === $status ) {
                                        $status_label = __( 'Modified', 'aegisshield-security' );
                                    } elseif ( 'deleted' === $status ) {
                                        $status_label = __( 'Deleted', 'aegisshield-security' );
                                    } else {
                                        $status_label = __( 'Unchanged', 'aegisshield-security' );
                                    }
                                    ?>
                                    <tr>
                                        <td><code><?php echo esc_html( $file_path ); ?></code></td>
                                        <td><?php echo esc_html( $status_label ); ?></td>
                                        <td><?php echo esc_html( $size ); ?></td>
                                        <td><?php echo esc_html( $mtime_str ); ?></td>
                                        <td><?php echo esc_html( $first_seen ); ?></td>
                                        <td><?php echo esc_html( $last_seen ); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>

                <hr />
				
				<div class="<?php echo esc_attr( 'aegisshield-file-monitor-pro-settings' . ( $is_pro_monitor ? '' : ' aegisshield-pro-locked' ) ); ?>">
                    <h3><?php esc_html_e( 'Advanced Monitoring (PRO)', 'aegisshield-security' ); ?></h3>
                    <p class="description">
                        <?php esc_html_e( 'Upgrade to AegisShield Pro to unlock custom paths, directory exclusions, and email alerts when critical files change.', 'aegisshield-security' ); ?>
                    </p>

                    <?php if ( ! $is_pro_monitor ) : ?>
                        <div class="notice notice-warning inline">
                            <p>
                                <?php esc_html_e( 'These settings are available in AegisShield Pro. Free installs can still see how the feature works, but changes require a Pro license.', 'aegisshield-security' ); ?>
                            </p>
                            <p>
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield_license' ) ); ?>" class="button button-primary">
                                    <?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
                                </a>
                            </p>
                        </div>
                    <?php endif; ?>

					<?php
					$email_mode       = $settings->get( 'file_monitor', 'email_mode', 'instant' );
					$email_events     = (array) $settings->get( 'file_monitor', 'email_events', array() );
					$email_recipients = $settings->get(
						'file_monitor',
						'email_recipients',
						get_option( 'admin_email' )
					);
					?>

                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label><?php esc_html_e( 'Custom Paths (PRO)', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <textarea rows="4" cols="60" <?php disabled( ! $is_pro_monitor ); ?>></textarea>
                                <p class="description">
                                    <?php esc_html_e( 'In Pro you can add additional directories to monitor, such as custom plugin folders or external libraries.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label><?php esc_html_e( 'Exclude Directories (PRO)', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <textarea rows="3" cols="60"<?php disabled( ! $is_pro_monitor ); ?>></textarea>
                                <p class="description">
                                    <?php esc_html_e( 'Exclude cache folders, backup directories or other noisy paths from monitoring.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                                                <tr>
                            <th scope="row">
                                <label><?php esc_html_e( 'Email Alerts (PRO)', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <?php if ( ! $is_pro_monitor ) : ?>
                                    <div class="notice notice-warning inline">
                                        <p>
                                            <strong><?php esc_html_e( 'Email Alerts is a PRO feature. Upgrade to activate.', 'aegisshield-security' ); ?></strong>
                                        </p>
                                        <p>
                                            <a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield_license' ) ); ?>" class="button button-primary">
                                                <?php esc_html_e( 'License & Upgrades', 'aegisshield-security' ); ?>
                                            </a>
                                        </p>
                                    </div>
                                <?php endif; ?>
								
								<form method="post">
									<?php wp_nonce_field(
										'aegisshield_file_monitor_email_save',
										'aegisshield_file_monitor_email_nonce'
									); ?>
									
								<input type="hidden" name="aegisshield_active_tab" value="monitor" />
								
                                <fieldset class="<?php echo esc_attr( 'aegisshield-email-alerts-settings' . ( $is_pro_monitor ? '' : ' aegisshield-pro-locked' ) ); ?>">
                                    <legend class="screen-reader-text">
                                        <span><?php esc_html_e( 'Email Alerts (PRO)', 'aegisshield-security' ); ?></span>
                                    </legend>

                                    <p>
                                        <strong><?php esc_html_e( 'Delivery mode', 'aegisshield-security' ); ?></strong><br />
										<label>
											<input type="radio"
												   name="file_monitor_email_mode"
												   value="instant"
												   <?php checked( $email_mode, 'instant' ); ?>
												   <?php disabled( ! $is_pro_monitor ); ?> />
											<?php esc_html_e( 'Instantly send one email per file change.', 'aegisshield-security' ); ?>
										</label><br />

										<label>
											<input type="radio"
												   name="file_monitor_email_mode"
												   value="summary"
												   <?php checked( $email_mode, 'summary' ); ?>
												   <?php disabled( ! $is_pro_monitor ); ?> />
											<?php esc_html_e( 'Summary and group file changes into periodic email summaries.', 'aegisshield-security' ); ?>
										</label>
                                    </p>

									<p>
										<strong><?php esc_html_e( 'Notify when', 'aegisshield-security' ); ?></strong><br />

										<label>
											<input type="checkbox"
												   name="file_monitor_email_events[]"
												   value="php"
												   <?php checked( in_array( 'php', $email_events, true ) ); ?>
												   <?php disabled( ! $is_pro_monitor ); ?> />
											<?php esc_html_e( 'PHP files change', 'aegisshield-security' ); ?>
										</label><br />

										<label>
											<input type="checkbox"
												   name="file_monitor_email_events[]"
												   value="plugin_theme"
												   <?php checked( in_array( 'plugin_theme', $email_events, true ) ); ?>
												   <?php disabled( ! $is_pro_monitor ); ?> />
											<?php esc_html_e( 'Plugin or theme files change', 'aegisshield-security' ); ?>
										</label><br />

										<label>
											<input type="checkbox"
												   name="file_monitor_email_events[]"
												   value="high_risk"
												   <?php checked( in_array( 'high_risk', $email_events, true ) ); ?>
												   <?php disabled( ! $is_pro_monitor ); ?> />
											<?php esc_html_e( 'Files change in high-risk directories (wp-admin, wp-includes)', 'aegisshield-security' ); ?>
										</label><br />

										<label>
											<input type="checkbox"
												   name="file_monitor_email_events[]"
												   value="new"
												   <?php checked( in_array( 'new', $email_events, true ) ); ?>
												   <?php disabled( ! $is_pro_monitor ); ?> />
											<?php esc_html_e( 'Any new file is created', 'aegisshield-security' ); ?>
										</label><br />

										<label>
											<input type="checkbox"
												   name="file_monitor_email_events[]"
												   value="modified"
												   <?php checked( in_array( 'modified', $email_events, true ) ); ?>
												   <?php disabled( ! $is_pro_monitor ); ?> />
											<?php esc_html_e( 'Existing files are modified (hash changed)', 'aegisshield-security' ); ?>
										</label><br />

										<label>
											<input type="checkbox"
												   name="file_monitor_email_events[]"
												   value="deleted"
												   <?php checked( in_array( 'deleted', $email_events, true ) ); ?>
												   <?php disabled( ! $is_pro_monitor ); ?> />
											<?php esc_html_e( 'Files are deleted', 'aegisshield-security' ); ?>
										</label>
									</p>

                                    <p>
                                        <strong><?php esc_html_e( 'Send alerts to', 'aegisshield-security' ); ?></strong><br />
										<input type="text"
											   name="file_monitor_email_recipients"
											   id="file_monitor_email_recipients"
											   class="regular-text"
											   value="<?php echo esc_attr( $email_recipients ); ?>"
											   <?php disabled( ! $is_pro_monitor ); ?> />
                                        <span class="description"><?php esc_html_e( 'Comma-separated list of email addresses to receive alerts.', 'aegisshield-security' ); ?></span>
                                    </p>

                                    <p class="description">
                                        <?php esc_html_e( 'In AegisShield Pro, you can enable real-time intrusion awareness and email alerts are sent when sensitive files change so you can respond quickly.', 'aegisshield-security' ); ?>
                                    </p>
                                </fieldset>
								<?php if ( $is_pro_monitor ) : ?>
									<?php submit_button(
										__( 'Save Email Alert Settings', 'aegisshield-security' ),
										'secondary',
										'submit',
										false
									); ?>
								<?php endif; ?>
								</form>
								
                            </td>
                        </tr>
				<?php endif; ?> <!-- closes Pro gate for File Change Monitor -->
                    </table>
                </div>
            </div>

            <style>
                .aegisshield-file-monitor-pro-settings.aegisshield-pro-locked {
                    opacity: 0.6;
                }
            </style>

			<?php
			// Determine which tab should be active, safely.
			$active_tab = 'scan';

			if ( isset( $_POST['aegisshield_active_tab'] ) ) {
				$active_tab = sanitize_key( wp_unslash( $_POST['aegisshield_active_tab'] ) );
			}

			// Allowlist only the known tabs used by this page.
			if ( ! in_array( $active_tab, array( 'scan', 'history', 'monitor' ), true ) ) {
				$active_tab = 'scan';
			}
			?>
			
            <script type="text/javascript">
            (function($){
                function aegisshieldUpdateScanDescription() {
                    var mode = $('#aegisshield_scan_mode').val();
                    var html = "";
                    if (mode === 'light') {
                        html = '<div class="aegisshield-scan-box">' +
                               '<h3>Light Scan (recommended)</h3>' +
                               '<ul>' +
                               '<li>Scans WordPress core, active theme, and active plugins.</li>' +
                               '<li>Avoids scanning large uploads directories.</li>' +
                               '<li>Fastest and safest option for shared hosting.</li>' +
                               '<li>Lowest impact on CPU and disk usage.</li>' +
                               '<li>Ideal for Hostinger, Bluehost, and other shared servers.</li>' +
                               '</ul>' +
                               '</div>';
                    } else if (mode === 'hybrid') {
                        html = '<div class="aegisshield-scan-box">' +
                               '<h3>Hybrid Scan</h3>' +
                               '<ul>' +
                               '<li>Scans core, active theme, and plugins deeply.</li>' +
                               '<li>Scans /uploads only for suspicious files (PHP, JS, executable or unusual extensions).</li>' +
                               '<li>Detects common upload-based web shells and backdoors.</li>' +
                               '<li>More coverage than Light Scan with moderate performance impact.</li>' +
                               '</ul>' +
                               '</div>';
                    } else if (mode === 'full') {
                        html = '<div class="aegisshield-scan-box">' +
                               '<h3>Full Scan</h3>' +
                               '<ul>' +
                               '<li>Scans all files under your WordPress root directory (e.g. /public_html/).</li>' +
                               '<li>Maximum visibility into unexpected files or changes.</li>' +
                               '<li>Recommended only for smaller sites or during off-peak hours.</li>' +
                               '<li>May be slow or hit timeouts on large sites or shared hosting.</li>' +
                               '</ul>' +
                               '</div>';
                    }
                    $('#aegisshield-scan-description').html(html);
                }
                $(document).ready(function(){
                    aegisshieldUpdateScanDescription();
                    $('#aegisshield_scan_mode').on('change', aegisshieldUpdateScanDescription);
                });
            
                // File Integrity page tabs.
                $('.aegisshield-fi-tabs .nav-tab').on('click', function(e){
                    e.preventDefault();
                    var tab = $(this).data('as-fi-tab');
                    $('.aegisshield-fi-tabs .nav-tab').removeClass('nav-tab-active');
                    $(this).addClass('nav-tab-active');
                    $('.aegisshield-fi-tab').hide();
                    $('.aegisshield-fi-tab[data-as-fi-tab-content="' + tab + '"]').show();
                });

                // Ensure initial tab visibility.
				var activeTab = '<?php echo esc_js( $active_tab ); ?>';

				$('.aegisshield-fi-tab').hide();
				$('.aegisshield-fi-tab[data-as-fi-tab-content="' + activeTab + '"]').show();

				$('.aegisshield-fi-tabs .nav-tab').removeClass('nav-tab-active');
				$('.aegisshield-fi-tabs .nav-tab[data-as-fi-tab="' + activeTab + '"]').addClass('nav-tab-active');


                // PRO monitor upsell behavior on File Change Monitor tab.
                var isProMonitor = <?php echo $is_pro_monitor ? 'true' : 'false'; ?>;
                if (!isProMonitor) {
                    // Redirect any interaction with PRO monitor settings to License & Upgrades page.
                    $('.aegisshield-file-monitor-pro-settings').on('click', 'input, textarea, select, label', function(e){
                        // Allow Upgrade button in the notice to work normally.
                        var $btn = $(this).closest('a.button-primary');
                        if ($btn.length) {
                            return;
                        }
                        e.preventDefault();
                        window.location.href = 'admin.php?page=aegisshield_license';
                    });
                }

})(jQuery);
            </script>
            <script type="text/javascript">
            (function($){
                $(document).on('click', '.aegisshield-download-report', function(e){
                    e.preventDefault();
                    var $btn   = $(this);
                    var data   = $btn.attr('data-report');
                    if (!data) {
                        return;
                    }
                    var report;
                    try {
                        report = JSON.parse(data);
                    } catch (err) {
                        return;
                    }

                    var lines = [];
                    if (report.summary) {
                        lines.push('Summary: ' + report.summary);
                    }
                    if (report.timestamp) {
                        lines.push('Timestamp: ' + report.timestamp);
                    }
                    if (report.mode) {
                        lines.push('Mode: ' + report.mode);
                    }

                    var hasSeverity = (typeof report.high !== 'undefined') ||
                        (typeof report.medium !== 'undefined') ||
                        (typeof report.low !== 'undefined');

                    if (hasSeverity) {
                        lines.push('');
                        lines.push('Severity counts:');
                        lines.push('  High: ' + (report.high || 0));
                        lines.push('  Medium: ' + (report.medium || 0));
                        lines.push('  Low: ' + (report.low || 0));
                    }

                    if (report.stats) {
                        lines.push('');
                        lines.push('Stats:');
                        if (typeof report.stats.scanned !== 'undefined') {
                            lines.push('  Scanned: ' + report.stats.scanned);
                        }
                        if (typeof report.stats.modified !== 'undefined') {
                            lines.push('  Modified: ' + report.stats.modified);
                        }
                        if (typeof report.stats.new !== 'undefined') {
                            lines.push('  New: ' + report.stats.new);
                        }
                        if (typeof report.stats.errors !== 'undefined') {
                            lines.push('  Errors: ' + report.stats.errors);
                        }
                    }

                    if (!lines.length) {
                        return;
                    }

                    var blob = new Blob([lines.join('\n')], { type: 'text/plain' });
                    var url  = window.URL.createObjectURL(blob);
                    var a    = document.createElement('a');
                    var ts   = (report.timestamp || '').replace(/[^0-9T]/g, '');
                    a.href   = url;
                    a.download = 'aegisshield-file-integrity-' + (ts || 'scan') + '.txt';
                    document.body.appendChild(a);
                    a.click();
                    document.body.removeChild(a);
                    window.URL.revokeObjectURL(url);
                });
            })(jQuery);
            </script>

            <?php if ( $scan_ran && ! empty( $scan_items ) ) : ?>
                <h2 style="margin-top:30px;"><?php esc_html_e( 'Scan Results', 'aegisshield-security' ); ?></h2>
                <table class="widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e( 'Path', 'aegisshield-security' ); ?></th>
                            <th><?php esc_html_e( 'Status', 'aegisshield-security' ); ?></th>
                            <th><?php esc_html_e( 'Last Scanned', 'aegisshield-security' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ( $scan_items as $item ) : ?>
                            <tr>
                                <td><?php echo esc_html( $item['path'] ); ?></td>
                                <td><?php echo esc_html( ucfirst( $item['status'] ) ); ?></td>
                                <td><?php echo esc_html( $item['last_scanned'] ); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php elseif ( $scan_ran ) : ?>
                <h2 style="margin-top:30px;"><?php esc_html_e( 'Scan Results', 'aegisshield-security' ); ?></h2>
                <p><?php esc_html_e( 'No files were scanned or no changes were detected.', 'aegisshield-security' ); ?></p>
            <?php endif; ?>

            <?php if ( ! empty( $last_report ) ) : ?>
                <hr />
                <h2><?php esc_html_e( 'Last Scan Report', 'aegisshield-security' ); ?></h2>
                <p>
                    <strong><?php esc_html_e( 'Timestamp:', 'aegisshield-security' ); ?></strong>
                    <?php echo esc_html( isset( $last_report['timestamp'] ) ? $last_report['timestamp'] : '' ); ?><br />
                    <strong><?php esc_html_e( 'Mode:', 'aegisshield-security' ); ?></strong>
                    <?php echo isset( $last_report['mode'] ) ? esc_html( ucfirst( $last_report['mode'] ) ) : ''; ?><br />
                </p>
                <?php if ( ! empty( $last_report['stats'] ) ) : ?>
                    <p>
                        <strong><?php esc_html_e( 'Files scanned:', 'aegisshield-security' ); ?></strong>
                        <?php echo isset( $last_report['stats']['scanned'] ) ? intval( $last_report['stats']['scanned'] ) : 0; ?><br />
                        <strong><?php esc_html_e( 'Modified:', 'aegisshield-security' ); ?></strong>
                        <?php echo isset( $last_report['stats']['modified'] ) ? intval( $last_report['stats']['modified'] ) : 0; ?><br />
                        <strong><?php esc_html_e( 'New:', 'aegisshield-security' ); ?></strong>
                        <?php echo isset( $last_report['stats']['new'] ) ? intval( $last_report['stats']['new'] ) : 0; ?><br />
                        <strong><?php esc_html_e( 'Errors:', 'aegisshield-security' ); ?></strong>
                        <?php echo isset( $last_report['stats']['errors'] ) ? intval( $last_report['stats']['errors'] ) : 0; ?><br />
                    </p>
                <?php endif; ?>

                <?php if ( ! empty( $last_report['items'] ) && is_array( $last_report['items'] ) ) : ?>
                    <h3><?php esc_html_e( 'Top 10 Findings', 'aegisshield-security' ); ?></h3>
                    <table class="widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Severity', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Path', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Status', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Reason', 'aegisshield-security' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $items_for_table = array_slice( $last_report['items'], 0, 10 );
                            foreach ( $items_for_table as $item ) :
                                $sev    = isset( $item['severity'] ) ? $item['severity'] : 'low';
                                $reason = isset( $item['reason'] ) ? $item['reason'] : '';
                                ?>
                                <tr>
                                    <td><?php echo esc_html( ucfirst( $sev ) ); ?></td>
                                    <td><?php echo esc_html( $item['path'] ); ?></td>
                                    <td><?php echo esc_html( ucfirst( $item['status'] ) ); ?></td>
                                    <td><?php echo esc_html( $reason ); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            <?php endif; ?>

            <?php if ( ! empty( $history_message ) ) : ?>
                <div class="notice notice-info">
                    <p><?php echo esc_html( $history_message ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( ! empty( $history_view_entry ) && is_array( $history_view_entry ) ) : ?>
                <h2><?php esc_html_e( 'Scan History - Selected Report', 'aegisshield-security' ); ?></h2>
                <table class="widefat fixed striped">
                    <tbody>
                        <tr>
                            <th><?php esc_html_e( 'Timestamp', 'aegisshield-security' ); ?></th>
                            <td><?php echo esc_html( isset( $history_view_entry['timestamp'] ) ? $history_view_entry['timestamp'] : '' ); ?></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Mode', 'aegisshield-security' ); ?></th>
                            <td><?php echo esc_html( isset( $history_view_entry['mode'] ) ? ucfirst( $history_view_entry['mode'] ) : '' ); ?></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Summary', 'aegisshield-security' ); ?></th>
                            <td><?php echo esc_html( isset( $history_view_entry['summary'] ) ? $history_view_entry['summary'] : '' ); ?></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Severity counts', 'aegisshield-security' ); ?></th>
                            <td>
                                <?php
                                $high   = isset( $history_view_entry['high'] ) ? (int) $history_view_entry['high'] : 0;
                                $medium = isset( $history_view_entry['medium'] ) ? (int) $history_view_entry['medium'] : 0;
                                $low    = isset( $history_view_entry['low'] ) ? (int) $history_view_entry['low'] : 0;
                                printf(
                                    /* translators: 1: high count, 2: medium count, 3: low count. */
                                    esc_html__( 'High: %1$d, Medium: %2$d, Low: %3$d', 'aegisshield-security' ),
                                    $high,
                                    $medium,
                                    $low
                                );
                                ?>
                            </td>
                        </tr>
                        <?php if ( ! empty( $history_view_entry['stats'] ) && is_array( $history_view_entry['stats'] ) ) : ?>
                            <tr>
                                <th><?php esc_html_e( 'Stats', 'aegisshield-security' ); ?></th>
                                <td>
                                    <?php
                                    $stats = $history_view_entry['stats'];
                                    $parts = array();
                                    if ( isset( $stats['scanned'] ) ) {
                                        $parts[] = sprintf(
                                            /* translators: %d: number of scanned files. */
                                            esc_html__( 'Scanned: %d', 'aegisshield-security' ),
                                            (int) $stats['scanned']
                                        );
                                    }
                                    if ( isset( $stats['modified'] ) ) {
                                        $parts[] = sprintf(
                                            /* translators: %d: number of modified files. */
                                            esc_html__( 'Modified: %d', 'aegisshield-security' ),
                                            (int) $stats['modified']
                                        );
                                    }
                                    if ( isset( $stats['new'] ) ) {
                                        $parts[] = sprintf(
                                            /* translators: %d: number of new files. */
                                            esc_html__( 'New: %d', 'aegisshield-security' ),
                                            (int) $stats['new']
                                        );
                                    }
                                    if ( isset( $stats['errors'] ) ) {
                                        $parts[] = sprintf(
                                            /* translators: %d: number of scan errors. */
                                            esc_html__( 'Errors: %d', 'aegisshield-security' ),
                                            (int) $stats['errors']
                                        );
                                    }
                                    echo esc_html( implode( ' | ', $parts ) );
                                    ?>
                                </td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            <?php endif; ?>

			<?php if ( ! empty( $scan_history ) && is_array( $scan_history ) ) : ?>
				<hr />
				<div class="aegisshield-fi-tab aegisshield-fi-tab-history" data-as-fi-tab-content="history" style="display:none;">
					<?php if ( ! $this->is_pro_active() ) : ?>
						<?php $this->render_pro_locked_notice(); ?>
					<?php else : ?>
				</div>
                <h2><?php esc_html_e( 'Scan History', 'aegisshield-security' ); ?></h2>
                <table class="widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e( 'Timestamp', 'aegisshield-security' ); ?></th>
                            <th><?php esc_html_e( 'Mode', 'aegisshield-security' ); ?></th>
                            <th><?php esc_html_e( 'Summary', 'aegisshield-security' ); ?></th>
                            <th><?php esc_html_e( 'Actions', 'aegisshield-security' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ( $scan_history as $index => $entry ) : ?>
                            <tr>
                                <td><?php echo esc_html( isset( $entry['timestamp'] ) ? $entry['timestamp'] : '' ); ?></td>
                                <td><?php echo esc_html( isset( $entry['mode'] ) ? ucfirst( $entry['mode'] ) : '' ); ?></td>
                                <td><?php echo esc_html( isset( $entry['summary'] ) ? $entry['summary'] : '' ); ?></td>
                                <td>
                                    <form method="post" style="display:inline;">
                                        <?php wp_nonce_field( 'aegisshield_file_integrity_history_action', 'aegisshield_file_integrity_history_nonce' ); ?>
                                        <input type="hidden" name="history_index" value="<?php echo esc_attr( $index ); ?>" />
                                        <button type="submit" name="aegisshield_file_integrity_history_action" value="view" class="button-link">
                                            <?php esc_html_e( 'View', 'aegisshield-security' ); ?>
                                        </button>
                                        |
                                        <button type="button"
                                            class="button-link aegisshield-download-report"
                                            data-report="<?php echo esc_attr( wp_json_encode( $entry ) ); ?>">
                                            <?php esc_html_e( 'Download', 'aegisshield-security' ); ?>
                                        </button>
                                        |
                                        <button type="submit" name="aegisshield_file_integrity_history_action" value="delete" class="button-link delete">
                                            <?php esc_html_e( 'Delete', 'aegisshield-security' ); ?>
                                        </button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
				<?php endif; ?>

            <?php endif; ?>


        </div>
        <?php
    }

    /**
     * Get human-readable description for scan mode.
     *
     * @param string $mode Scan mode.
     * @return string
     */
    protected function get_mode_description( $mode ) {
    // Fallback server-side description (initial render). JavaScript enhances this dynamically.
    switch ( $mode ) {
        case 'hybrid':
            return '<div class="aegisshield-scan-box">'
                . '<h3>Hybrid Scan</h3>'
                . '<ul>'
                . '<li>Scans core, active theme, and plugins deeply.</li>'
                . '<li>Scans /uploads only for suspicious files (PHP, JS, executable or unusual extensions).</li>'
                . '<li>Detects common upload-based web shells and backdoors.</li>'
                . '<li>More coverage than Light Scan with moderate performance impact.</li>'
                . '</ul>'
                . '</div>';
        case 'full':
            return '<div class="aegisshield-scan-box">'
                . '<h3>Full Scan</h3>'
                . '<ul>'
                . '<li>Scans all files under your WordPress root directory (e.g. /public_html/).</li>'
                . '<li>Maximum visibility into unexpected files or changes.</li>'
                . '<li>Recommended only for smaller sites or during off-peak hours.</li>'
                . '<li>May be slow or hit timeouts on large sites or shared hosting.</li>'
                . '</ul>'
                . '</div>';
        case 'light':
        default:
            return '<div class="aegisshield-scan-box">'
                . '<h3>Light Scan (recommended)</h3>'
                . '<ul>'
                . '<li>Scans WordPress core, active theme, and active plugins.</li>'
                . '<li>Avoids scanning large uploads directories.</li>'
                . '<li>Fastest and safest option for shared hosting.</li>'
                . '<li>Lowest impact on CPU and disk usage.</li>'
                . '<li>Ideal for Hostinger, Bluehost, and other shared servers.</li>'
                . '</ul>'
                . '</div>';
    }
}
}