<?php
namespace AegisShield\Admin_Pages;

use AegisShield\AS_Plugin;
use AegisShield\AS_Logger;

defined( 'ABSPATH' ) || exit;

/**
 * Hardening admin page.
 */
class AS_Page_Hardening {

    /**
     * Plugin instance.
     *
     * @var AS_Plugin
     */
    protected $plugin;

    /**
     * Constructor.
     *
     * @param AS_Plugin $plugin Plugin instance.
     */
    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Check if Pro is active (same pattern as Alerts Pro, Login Guard Pro).
     *
     * @return bool
     */
    protected function is_pro_active() {
        return function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();
    }

    /**
     * Shared Pro-locked notice for Vulnerability Scan & Role Risk Analyzer.
     */
    protected function render_pro_locked_notice() {
        ?>
        <div class="notice notice-warning inline">
            <p>
                <strong><?php esc_html_e( 'This feature is available in AegisShield Pro.', 'aegisshield-security' ); ?></strong>
            </p>
            <p>
                <?php esc_html_e( 'Upgrade to unlock vulnerability scanning and role risk analytics as part of your security hardening toolkit.', 'aegisshield-security' ); ?>
            </p>
            <p>
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ); ?>" class="button button-primary">
                    <?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
                </a>
            </p>
        </div>
        <?php
    }

    /**
     * Render the page.
     */
    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // Pro state for tab badges and gating.
        $is_pro = $this->is_pro_active();

        $settings = $this->plugin->get_settings();
        $section  = 'hardening';

        // Current tab.
        $current_tab = isset( $_GET['tab'] )
            ? sanitize_key( wp_unslash( $_GET['tab'] ) )
            : 'basic';

        if ( ! in_array( $current_tab, array( 'basic', 'vuln', 'roles' ), true ) ) {
            $current_tab = 'basic';
        }

        $page_url = admin_url( 'admin.php?page=aegisshield-hardening' );

        // Values for Basic tab.
        $disable_file_edit      = (string) $settings->get( $section, 'disable_file_edit', 'off' );
        $xmlrpc_behavior        = (string) $settings->get( $section, 'disable_xmlrpc_behavior', 'c' );
        $block_user_enum        = (string) $settings->get( $section, 'block_user_enum_behavior', 'a' );
        $hide_wp_version        = (string) $settings->get( $section, 'hide_wp_version', 'off' );
        $disable_editor_screens = (string) $settings->get( $section, 'disable_editor_screens', 'off' );
        $enforce_passwords      = (string) $settings->get( $section, 'enforce_strong_passwords', 'off' );
        $password_min_length    = (int) $settings->get( $section, 'password_min_length', 12 );
        $require_upper          = (string) $settings->get( $section, 'password_require_uppercase', 'on' );
        $require_lower          = (string) $settings->get( $section, 'password_require_lowercase', 'on' );
        $require_num            = (string) $settings->get( $section, 'password_require_number', 'on' );
        $require_sym            = (string) $settings->get( $section, 'password_require_symbol', 'on' );

        $updated = false;

        // Phase 2: capture old hardening values for change logging.
        $old_values = array(
            'disable_file_edit'          => $disable_file_edit,
            'disable_xmlrpc_behavior'    => $xmlrpc_behavior,
            'block_user_enum_behavior'   => $block_user_enum,
            'hide_wp_version'            => $hide_wp_version,
            'disable_editor_screens'     => $disable_editor_screens,
            'enforce_strong_passwords'   => $enforce_passwords,
            'password_min_length'        => $password_min_length,
            'password_require_uppercase' => $require_upper,
            'password_require_lowercase' => $require_lower,
            'password_require_number'    => $require_num,
            'password_require_symbol'    => $require_sym,
        );

        // Handle POST only for Basic tab.
        if (
            'basic' === $current_tab
            && isset( $_POST['aegisshield_hardening_nonce'] )
            && wp_verify_nonce(
                sanitize_text_field( wp_unslash( $_POST['aegisshield_hardening_nonce'] ) ),
                'aegisshield_hardening_save'
            )
        ) {
            $disable_file_edit      = isset( $_POST['disable_file_edit'] ) ? 'on' : 'off';
            $xmlrpc_behavior        = isset( $_POST['disable_xmlrpc_behavior'] )
                ? sanitize_text_field( wp_unslash( $_POST['disable_xmlrpc_behavior'] ) )
                : 'c';
            $block_user_enum        = isset( $_POST['block_user_enum_behavior'] )
                ? sanitize_text_field( wp_unslash( $_POST['block_user_enum_behavior'] ) )
                : 'a';
            $hide_wp_version        = isset( $_POST['hide_wp_version'] ) ? 'on' : 'off';
            $disable_editor_screens = isset( $_POST['disable_editor_screens'] ) ? 'on' : 'off';
            $enforce_passwords      = isset( $_POST['enforce_strong_passwords'] ) ? 'on' : 'off';
            $password_min_length    = isset( $_POST['password_min_length'] )
                ? absint( wp_unslash( $_POST['password_min_length'] ) )
                : 12;

            if ( $password_min_length < 1 ) {
                $password_min_length = 1;
            }
			
			// Get logger instance (AS_Logger::log is non-static in this build).
			$logger = $this->plugin->get_logger();

            // Phase 2: log invalid/minimum value coercion.
            $logger->log(
                'hardening_rule_failed',
                __( 'Hardening input corrected: password_min_length was below 1 and was coerced to 1.', 'aegisshield-security' ),
                'low',
                array(
                    'setting_key' => 'password_min_length',
                    'original'    => isset( $_POST['password_min_length'] ) ? wp_unslash( $_POST['password_min_length'] ) : null,
                    'new'         => 1,
                    'user_id'     => get_current_user_id(),
                )
            );


            $require_upper = isset( $_POST['password_require_uppercase'] ) ? 'on' : 'off';
            $require_lower = isset( $_POST['password_require_lowercase'] ) ? 'on' : 'off';
            $require_num   = isset( $_POST['password_require_number'] ) ? 'on' : 'off';
            $require_sym   = isset( $_POST['password_require_symbol'] ) ? 'on' : 'off';

            // Phase 2: log hardening state changes (always logged; alerts controlled elsewhere).
            foreach ( array(
                'disable_file_edit'          => $disable_file_edit,
                'disable_xmlrpc_behavior'    => $xmlrpc_behavior,
                'block_user_enum_behavior'   => $block_user_enum,
                'hide_wp_version'            => $hide_wp_version,
                'disable_editor_screens'     => $disable_editor_screens,
                'enforce_strong_passwords'   => $enforce_passwords,
                'password_min_length'        => $password_min_length,
                'password_require_uppercase' => $require_upper,
                'password_require_lowercase' => $require_lower,
                'password_require_number'    => $require_num,
                'password_require_symbol'    => $require_sym,
            ) as $k => $new_val ) {
                $old_val = isset( $old_values[ $k ] ) ? $old_values[ $k ] : null;
                if ( (string) $old_val === (string) $new_val ) {
                    continue;
                }

                $event_type = 'hardening_policy_changed';
                $severity   = 'medium';
                if ( in_array( $k, array( 'disable_file_edit', 'hide_wp_version', 'disable_editor_screens', 'enforce_strong_passwords' ), true ) ) {
                    if ( (string) $new_val === 'on' ) {
                        $event_type = 'hardening_enabled';
                        $severity   = 'low';
                    } else {
                        $event_type = 'hardening_disabled';
                        $severity   = 'high';
                    }
                }

                $logger->log(
                    $event_type,
                    sprintf( __( 'Hardening setting changed: %s', 'aegisshield-security' ), $k ),
                    $severity,
                    array(
                        'setting_key' => $k,
                        'old'         => $old_val,
                        'new'         => $new_val,
                        'user_id'     => get_current_user_id(),
                    )
                );
            }

            if ( ! in_array( $xmlrpc_behavior, array( 'a', 'b', 'c' ), true ) ) {
                // Phase 2: log invalid coercion.
                $logger->log(
                    'hardening_rule_failed',
                    __( 'Hardening input corrected: invalid XML-RPC behavior value was coerced to safe default.', 'aegisshield-security' ),
                    'low',
                    array(
                        'setting_key' => 'disable_xmlrpc_behavior',
                        'original'    => isset( $_POST['disable_xmlrpc_behavior'] ) ? wp_unslash( $_POST['disable_xmlrpc_behavior'] ) : null,
                        'new'         => 'c',
                        'user_id'     => get_current_user_id(),
                    )
                );
                $xmlrpc_behavior = 'c';
            }
            if ( ! in_array( $block_user_enum, array( 'a', 'b' ), true ) ) {
                // Phase 2: log invalid coercion.
                $logger->log(
                    'hardening_rule_failed',
                    __( 'Hardening input corrected: invalid user enumeration behavior value was coerced to safe default.', 'aegisshield-security' ),
                    'low',
                    array(
                        'setting_key' => 'block_user_enum_behavior',
                        'original'    => isset( $_POST['block_user_enum_behavior'] ) ? wp_unslash( $_POST['block_user_enum_behavior'] ) : null,
                        'new'         => 'a',
                        'user_id'     => get_current_user_id(),
                    )
                );
                $block_user_enum = 'a';
            }

            $settings->set( $section, 'disable_file_edit', $disable_file_edit );
            $settings->set( $section, 'disable_xmlrpc_behavior', $xmlrpc_behavior );
            $settings->set( $section, 'block_user_enum_behavior', $block_user_enum );
            $settings->set( $section, 'hide_wp_version', $hide_wp_version );
            $settings->set( $section, 'disable_editor_screens', $disable_editor_screens );
            $settings->set( $section, 'enforce_strong_passwords', $enforce_passwords );
            $settings->set( $section, 'password_min_length', $password_min_length );
            $settings->set( $section, 'password_require_uppercase', $require_upper );
            $settings->set( $section, 'password_require_lowercase', $require_lower );
            $settings->set( $section, 'password_require_number', $require_num );
            $settings->set( $section, 'password_require_symbol', $require_sym );

            $settings->save();
            $updated = true;
        }

        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Hardening', 'aegisshield-security' ); ?></h1>

            <p class="description">
                <?php esc_html_e(
                    'AegisShield Hardening strengthens WordPress by closing common security gaps while giving administrators full visibility into how their site is protected. Instead of silently blocking threats, it enforces proven security controls and records every violation with clear context, allowing admins to monitor, alert, and respond based on their own security policies. The result is a hardened WordPress environment that is proactive, transparent, and fully under the administrator’s control.',
                    'aegisshield-security'
                ); ?>
            </p>

            <?php if ( $updated && 'basic' === $current_tab ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'Hardening settings saved.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <?php
            $basic_url = add_query_arg( 'tab', 'basic', $page_url );
            $vuln_url  = add_query_arg( 'tab', 'vuln',  $page_url );
            $roles_url = add_query_arg( 'tab', 'roles', $page_url );
            ?>

            <h2 class="nav-tab-wrapper">
                <a href="<?php echo esc_url( $basic_url ); ?>"
                   class="nav-tab <?php echo esc_attr( ( 'basic' === $current_tab ) ? 'nav-tab-active' : '' ); ?>">
                    <?php esc_html_e( 'Basic Hardening', 'aegisshield-security' ); ?>
                </a>
                <a href="<?php echo esc_url( $vuln_url ); ?>"
                   class="nav-tab <?php echo ( 'vuln' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Vulnerability Scan', 'aegisshield-security' ); ?>
                    <?php if ( ! $is_pro ) : ?>
                        <span class="aegis-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
                            <?php esc_html_e( 'PRO', 'aegisshield-security' ); ?>
                        </span>
                    <?php endif; ?>
                </a>
                <a href="<?php echo esc_url( $roles_url ); ?>"
                   class="nav-tab <?php echo ( 'roles' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Role Risk Analyzer', 'aegisshield-security' ); ?>
                    <?php if ( ! $is_pro ) : ?>
                        <span class="aegis-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
                            <?php esc_html_e( 'PRO', 'aegisshield-security' ); ?>
                        </span>
                    <?php endif; ?>
                </a>
            </h2>

            <div class="aegisshield-tab-panel">
                <?php if ( 'basic' === $current_tab ) : ?>

                    <h2><?php esc_html_e( 'Core Hardening Options', 'aegisshield-security' ); ?></h2>

                    <form method="post">
                        <?php
                        wp_nonce_field(
                            'aegisshield_hardening_save',
                            'aegisshield_hardening_nonce'
                        );
                        ?>

                        <table class="form-table" role="presentation">
                            <tbody>
                                <tr>
                                    <th scope="row">
                                        <label for="disable_file_edit">
                                            <?php esc_html_e( 'Disable file editor', 'aegisshield-security' ); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <label>
                                            <input type="checkbox"
                                                   id="disable_file_edit"
                                                   name="disable_file_edit"
                                                <?php checked( $disable_file_edit, 'on' ); ?> />
                                            <?php esc_html_e(
                                                'Prevent plugins & themes from being edited via the WordPress dashboard.',
                                                'aegisshield-security'
                                            ); ?>
                                        </label>
                                        <p class="description">
                                            <?php esc_html_e(
                                                'Recommended for most sites to reduce the impact if an admin account is compromised.',
                                                'aegisshield-security'
                                            ); ?>
                                        </p>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row">
                                        <label for="disable_xmlrpc_behavior">
                                            <?php esc_html_e( 'XML-RPC behavior', 'aegisshield-security' ); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <select id="disable_xmlrpc_behavior"
                                                name="disable_xmlrpc_behavior">
                                            <option value="c" <?php selected( $xmlrpc_behavior, 'c' ); ?>>
                                                <?php esc_html_e(
                                                    'Allow core XML-RPC but restrict dangerous methods (recommended)',
                                                    'aegisshield-security'
                                                ); ?>
                                            </option>
                                            <option value="a" <?php selected( $xmlrpc_behavior, 'a' ); ?>>
                                                <?php esc_html_e(
                                                    'Deny XML-RPC requests with error page',
                                                    'aegisshield-security'
                                                ); ?>
                                            </option>
                                            <option value="b" <?php selected( $xmlrpc_behavior, 'b' ); ?>>
                                                <?php esc_html_e(
                                                    'Return HTTP 403 for XML-RPC (silent block)',
                                                    'aegisshield-security'
                                                ); ?>
                                            </option>
                                        </select>
                                        <p class="description">
                                            <?php esc_html_e(
                                                'XML-RPC is used by some apps and Jetpack. If you are unsure, keep the default safe option.',
                                                'aegisshield-security'
                                            ); ?>
                                        </p>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row">
                                        <label for="block_user_enum_behavior">
                                            <?php esc_html_e( 'Block user enumeration', 'aegisshield-security' ); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <select id="block_user_enum_behavior"
                                                name="block_user_enum_behavior">
                                            <option value="a" <?php selected( $block_user_enum, 'a' ); ?>>
                                                <?php esc_html_e(
                                                    'Redirect ?author=N URLs to homepage',
                                                    'aegisshield-security'
                                                ); ?>
                                            </option>
                                            <option value="b" <?php selected( $block_user_enum, 'b' ); ?>>
                                                <?php esc_html_e(
                                                    'Return 404 for user enumeration attempts',
                                                    'aegisshield-security'
                                                ); ?>
                                            </option>
                                        </select>
                                        <p class="description">
                                            <?php esc_html_e(
                                                'Prevents attackers from discovering usernames via author archives.',
                                                'aegisshield-security'
                                            ); ?>
                                        </p>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row">
                                        <label for="hide_wp_version">
                                            <?php esc_html_e( 'Hide WordPress version', 'aegisshield-security' ); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <label>
                                            <input type="checkbox"
                                                   id="hide_wp_version"
                                                   name="hide_wp_version"
                                                <?php checked( $hide_wp_version, 'on' ); ?> />
                                            <?php esc_html_e(
                                                'Remove WordPress version from page output & assets.',
                                                'aegisshield-security'
                                            ); ?>
                                        </label>
                                        <p class="description">
                                            <?php esc_html_e(
                                                'This makes it slightly harder for automated tools to fingerprint your site.',
                                                'aegisshield-security'
                                            ); ?>
                                        </p>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row">
                                        <label for="disable_editor_screens">
                                            <?php esc_html_e( 'Disable plugin & theme editor screens', 'aegisshield-security' ); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <label>
                                            <input type="checkbox"
                                                   id="disable_editor_screens"
                                                   name="disable_editor_screens"
                                                <?php checked( $disable_editor_screens, 'on' ); ?> />
                                            <?php esc_html_e(
                                                'Hide the built-in plugin and theme code editor screens.',
                                                'aegisshield-security'
                                            ); ?>
                                        </label>
                                        <p class="description">
                                            <?php esc_html_e(
                                                'Use SFTP or version control instead of editing PHP files directly from the dashboard.',
                                                'aegisshield-security'
                                            ); ?>
                                        </p>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row">
                                        <?php esc_html_e( 'Enforce strong passwords', 'aegisshield-security' ); ?>
                                    </th>
                                    <td>
                                        <fieldset>
                                            <label>
                                                <input type="checkbox"
                                                       name="enforce_strong_passwords"
                                                    <?php checked( $enforce_passwords, 'on' ); ?> />
                                                <?php esc_html_e(
                                                    'Require strong passwords for new and updated accounts.',
                                                    'aegisshield-security'
                                                ); ?>
                                            </label>

                                            <p class="description">
                                                <?php esc_html_e(
                                                    'When enabled, password strength checks will be enforced on the server.',
                                                    'aegisshield-security'
                                                ); ?>
                                            </p>

                                            <table>
                                                <tr>
                                                    <td>
                                                        <label for="password_min_length">
                                                            <?php esc_html_e(
                                                                'Minimum length',
                                                                'aegisshield-security'
                                                            ); ?>
                                                        </label>
                                                    </td>
                                                    <td>
                                                        <input type="number"
                                                               id="password_min_length"
                                                               name="password_min_length"
                                                               min="1"
                                                               value="<?php echo esc_attr( $password_min_length ); ?>" />
                                                    </td>
                                                </tr>
                                                <tr>
                                                    <td colspan="2">
                                                        <label>
                                                            <input type="checkbox"
                                                                   name="password_require_uppercase"
                                                                <?php checked( $require_upper, 'on' ); ?> />
                                                            <?php esc_html_e(
                                                                'Require at least one uppercase letter',
                                                                'aegisshield-security'
                                                            ); ?>
                                                        </label>
                                                    </td>
                                                </tr>
                                                <tr>
                                                    <td colspan="2">
                                                        <label>
                                                            <input type="checkbox"
                                                                   name="password_require_lowercase"
                                                                <?php checked( $require_lower, 'on' ); ?> />
                                                            <?php esc_html_e(
                                                                'Require at least one lowercase letter',
                                                                'aegisshield-security'
                                                            ); ?>
                                                        </label>
                                                    </td>
                                                </tr>
                                                <tr>
                                                    <td colspan="2">
                                                        <label>
                                                            <input type="checkbox"
                                                                   name="password_require_number"
                                                                <?php checked( $require_num, 'on' ); ?> />
                                                            <?php esc_html_e(
                                                                'Require at least one number',
                                                                'aegisshield-security'
                                                            ); ?>
                                                        </label>
                                                    </td>
                                                </tr>
                                                <tr>
                                                    <td colspan="2">
                                                        <label>
                                                            <input type="checkbox"
                                                                   name="password_require_symbol"
                                                                <?php checked( $require_sym, 'on' ); ?> />
                                                            <?php esc_html_e(
                                                                'Require at least one symbol (e.g. !, @, #)',
                                                                'aegisshield-security'
                                                            ); ?>
                                                        </label>
                                                    </td>
                                                </tr>
                                            </table>
                                        </fieldset>
                                    </td>
                                </tr>
                            </tbody>
                        </table>

                        <?php submit_button( __( 'Save Changes', 'aegisshield-security' ) ); ?>
                    </form>

                <?php elseif ( 'vuln' === $current_tab ) : ?>

                    <?php
                    // PRO gating – same pattern as Login Guard PRO tabs.
                    if ( ! $is_pro ) {
                        $this->render_pro_locked_notice();
                    } else {
                        if ( class_exists( '\AegisShield\Admin_Pages\Hardening\AS_Page_Hardening_Tab_Vuln' ) ) {
                            \AegisShield\Admin_Pages\Hardening\AS_Page_Hardening_Tab_Vuln::render_tab( $this->plugin );
                        } else {
                            echo '<p>' . esc_html__(
                                'Vulnerability Scan tab will be available in a future release.',
                                'aegisshield-security'
                            ) . '</p>';
                        }
                    }
                    ?>

                <?php elseif ( 'roles' === $current_tab ) : ?>

                    <?php
                    // PRO gating – same pattern as Login Guard PRO tabs.
                    if ( ! $is_pro ) {
                        $this->render_pro_locked_notice();
                    } else {
                        if ( class_exists( '\AegisShield\Admin_Pages\Hardening\AS_Page_Hardening_Tab_Roles' ) ) {
                            \AegisShield\Admin_Pages\Hardening\AS_Page_Hardening_Tab_Roles::render_tab( $this->plugin );
                        } else {
                            echo '<p>' . esc_html__(
                                'Role Risk Analyzer tab will be available in a future release.',
                                'aegisshield-security'
                            ) . '</p>';
                        }
                    }
                    ?>

                <?php endif; ?>
            </div>
        </div>
        <?php
    }
}