<?php
namespace AegisShield\Admin_Pages;

use AegisShield\AS_Plugin;
use AegisShield\Modules\Malware\AS_Malware_Incidents;
use AegisShield\Reports\AS_Incident_Report;

defined( 'ABSPATH' ) || exit;

/**
 * Incident report admin page.
 *
 * This is a hidden admin page used to render a printable HTML report
 * for a single malware incident.
 */
class AS_Page_Incident_Report {

    /**
     * Plugin instance.
     *
     * @var AS_Plugin
     */
    protected $plugin;

    /**
     * Constructor.
     *
     * @param AS_Plugin $plugin Plugin instance.
     */
    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Render the report page.
     *
     * @return void
     */
    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to view this report.', 'aegisshield-security' ) );
        }

        $incident_id = '';
        $file_param  = '';

        if ( isset( $_GET['incident'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $incident_id = sanitize_text_field( wp_unslash( $_GET['incident'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        }

        if ( isset( $_GET['file'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $file_param = wp_unslash( $_GET['file'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $file_param = sanitize_text_field( $file_param );
        }

        $incidents_helper = new AS_Malware_Incidents( $this->plugin );
        $incident         = null;

        if ( $incident_id ) {
            $incident = $incidents_helper->get_incident_by_id( $incident_id );
        }

        // Fallback: resolve by file path from the most recent incidents.
        if ( null === $incident && $file_param ) {
            $recent = $incidents_helper->get_recent_incidents( 25 );
            foreach ( $recent as $candidate ) {
                if ( isset( $candidate['file'] ) && (string) $candidate['file'] === $file_param ) {
                    $incident = $candidate;
                    break;
                }
            }
        }

        if ( null === $incident ) {
            ?>
            <div class="wrap">
                <h1><?php esc_html_e( 'Incident Report', 'aegisshield-security' ); ?></h1>
                <p><?php esc_html_e( 'The requested incident could not be found. It may have been purged from the log or the link was invalid.', 'aegisshield-security' ); ?></p>
            </div>
            <?php
            return;
        }

        $timeline = $incidents_helper->build_timeline_for_incident( $incident );

        $reporter = new AS_Incident_Report( $this->plugin );
        $html     = $reporter->render_html( $incident, $timeline );

        // Output the report HTML directly. We intentionally do not add extra
        // admin chrome here so that printing to PDF remains clean.
        echo wp_kses_post( $html );
    }
}
