<?php
namespace AegisShield\Admin_Pages;

use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

/**
 * Security Headers settings page.
 */
class AS_Page_Sec_Headers {

    /**
     * Plugin instance.
     *
     * @var AS_Plugin
     */
    protected $plugin;

    /**
     * Constructor.
     *
     * @param AS_Plugin $plugin Plugin instance.
     */
    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Check if Pro is active.
     *
     * @return bool
     */
    protected function is_pro_active() {
        return function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();
    }

    /**
     * Shared Pro-locked notice for Security Headers PRO-only tabs.
     *
     * @return void
     */
    protected function render_pro_locked_notice() {
        ?>
        <div class="notice notice-warning inline">
            <p>
                <strong><?php esc_html_e( 'This feature is available in AegisShield Pro.', 'aegisshield-security' ); ?></strong>
            </p>
            <p>
                <?php esc_html_e( 'Upgrade to unlock advanced header protection features including the Visual CSP Builder, per-area header profiles, and header health scoring.', 'aegisshield-security' ); ?>
            </p>
            <p>
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ); ?>" class="button button-primary">
                    <?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
                </a>
            </p>
        </div>
        <?php
    }

    /**
     * Render the settings page.
     *
     * @return void
     */
    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'aegisshield-security' ) );
        }

        $settings = $this->plugin->get_settings();
        $logger   = $this->plugin->get_logger();
        $section  = 'sec_headers';

        $enabled      = (string) $settings->get( $section, 'enabled', 'on' );
        $csp_level    = (string) $settings->get( $section, 'csp_level', 'none' );
        $enable_hsts  = (string) $settings->get( $section, 'enable_hsts', 'off' );

        // Pro gating.
        $is_pro = $this->is_pro_active();

        // Tabs: Standard / CSP Builder / Profiles & Health.
        $current_tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'standard';
        if ( ! in_array( $current_tab, array( 'standard', 'csp_builder', 'profiles' ), true ) ) {
            $current_tab = 'standard';
        }

        // CSP Builder settings (Pro only).
        $builder_mode            = (string) $settings->get( $section, 'builder_mode', 'off' );
        $builder_script_src      = (string) $settings->get( $section, 'builder_script_src', '' );
        $builder_style_src       = (string) $settings->get( $section, 'builder_style_src', '' );
		// Added - chue
		$builder_script_mode   = (string) $settings->get( $section, 'builder_script_mode', 'off' );
		$builder_style_mode    = (string) $settings->get( $section, 'builder_style_mode', 'off' );
		$builder_img_mode      = (string) $settings->get( $section, 'builder_img_mode', 'off' );
		$builder_font_mode     = (string) $settings->get( $section, 'builder_font_mode', 'off' );
		$builder_connect_mode  = (string) $settings->get( $section, 'builder_connect_mode', 'off' );
		$builder_frame_mode    = (string) $settings->get( $section, 'builder_frame_mode', 'off' );
		$builder_extra_mode    = (string) $settings->get( $section, 'builder_extra_mode', 'off' );
        // Fixed typo here: removed extra "$".
        $builder_img_src         = (string) $settings->get( $section, 'builder_img_src', '' );
        $builder_font_src        = (string) $settings->get( $section, 'builder_font_src', '' );
        $builder_connect_src     = (string) $settings->get( $section, 'builder_connect_src', '' );
        $builder_frame_ancestors = (string) $settings->get( $section, 'builder_frame_ancestors', '' );
        $builder_extra           = (string) $settings->get( $section, 'builder_extra_directives', '' );
        $builder_allow_inline_js = (string) $settings->get( $section, 'builder_allow_inline_js', '' );
        $builder_allow_inline_css= (string) $settings->get( $section, 'builder_allow_inline_css', '' );

        // Per-area profiles.
        $csp_frontend       = (string) $settings->get( $section, 'csp_level_frontend', '' );
        $csp_admin          = (string) $settings->get( $section, 'csp_level_admin', '' );
        $csp_custom         = (string) $settings->get( $section, 'csp_level_custom', '' );
        $profiles_paths_raw = (string) $settings->get( $section, 'profiles_custom_paths', '' );

        $updated_standard = false;
        $updated_builder  = false;
        $updated_profiles = false;

        // Handle saves per tab.
        if (
            'standard' === $current_tab
            && isset( $_POST['aegisshield_sec_headers_nonce'] )
            && check_admin_referer( 'aegisshield_sec_headers_save', 'aegisshield_sec_headers_nonce' )
        ) {
            $enabled     = isset( $_POST['sec_headers_enabled'] ) ? 'on' : 'off';
            $csp_level   = isset( $_POST['csp_level'] ) ? sanitize_text_field( wp_unslash( $_POST['csp_level'] ) ) : $csp_level;
            $enable_hsts = isset( $_POST['enable_hsts'] ) ? 'on' : 'off';

            if ( ! in_array( $csp_level, array( 'none', 'basic', 'strict', 'builder_enforce', 'builder_report' ), true ) ) {
                $csp_level = 'none';
            }

            $settings->set( $section, 'enabled', $enabled );
            $settings->set( $section, 'csp_level', $csp_level );
            $settings->set( $section, 'enable_hsts', $enable_hsts );
            $settings->save();

            if ( $logger ) {
                $logger->log(
                    'sec_headers_settings_updated',
                    __( 'Security Headers settings were updated.', 'aegisshield-security' )
                );
            }

            $updated_standard = true;
        }

        if (
            'csp_builder' === $current_tab
            && $is_pro
            && isset( $_POST['aegisshield_sec_headers_builder_nonce'] )
            && check_admin_referer( 'aegisshield_sec_headers_builder_save', 'aegisshield_sec_headers_builder_nonce' )
        ) {
            $builder_mode = isset( $_POST['builder_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['builder_mode'] ) ) : $builder_mode;
            if ( ! in_array( $builder_mode, array( 'off', 'report_only', 'enforce' ), true ) ) {
                $builder_mode = 'off';
            }

            $builder_script_src      = isset( $_POST['builder_script_src'] ) ? sanitize_textarea_field( wp_unslash( $_POST['builder_script_src'] ) ) : '';
            $builder_style_src       = isset( $_POST['builder_style_src'] ) ? sanitize_textarea_field( wp_unslash( $_POST['builder_style_src'] ) ) : '';
            $builder_img_src         = isset( $_POST['builder_img_src'] ) ? sanitize_textarea_field( wp_unslash( $_POST['builder_img_src'] ) ) : '';
            $builder_font_src        = isset( $_POST['builder_font_src'] ) ? sanitize_textarea_field( wp_unslash( $_POST['builder_font_src'] ) ) : '';
            $builder_connect_src     = isset( $_POST['builder_connect_src'] ) ? sanitize_textarea_field( wp_unslash( $_POST['builder_connect_src'] ) ) : '';
            $builder_frame_ancestors = isset( $_POST['builder_frame_ancestors'] ) ? sanitize_textarea_field( wp_unslash( $_POST['builder_frame_ancestors'] ) ) : '';
            $builder_extra           = isset( $_POST['builder_extra_directives'] ) ? sanitize_textarea_field( wp_unslash( $_POST['builder_extra_directives'] ) ) : '';
            $builder_allow_inline_js = isset( $_POST['builder_allow_inline_js'] ) ? 'on' : '';
			$builder_script_mode  = isset( $_POST['builder_script_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['builder_script_mode'] ) ) : 'off';
			$builder_style_mode   = isset( $_POST['builder_style_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['builder_style_mode'] ) ) : 'off';
			$builder_img_mode     = isset( $_POST['builder_img_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['builder_img_mode'] ) ) : 'off';
			$builder_font_mode    = isset( $_POST['builder_font_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['builder_font_mode'] ) ) : 'off';
			$builder_connect_mode = isset( $_POST['builder_connect_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['builder_connect_mode'] ) ) : 'off';
			$builder_frame_mode   = isset( $_POST['builder_frame_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['builder_frame_mode'] ) ) : 'off';
			$builder_extra_mode   = isset( $_POST['builder_extra_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['builder_extra_mode'] ) ) : 'off';
            $builder_allow_inline_css= isset( $_POST['builder_allow_inline_css'] ) ? 'on' : '';

            $settings->set( $section, 'builder_mode', $builder_mode );
            $settings->set( $section, 'builder_script_src', $builder_script_src );
            $settings->set( $section, 'builder_style_src', $builder_style_src );
            $settings->set( $section, 'builder_img_src', $builder_img_src );
            $settings->set( $section, 'builder_font_src', $builder_font_src );
            $settings->set( $section, 'builder_connect_src', $builder_connect_src );
            $settings->set( $section, 'builder_frame_ancestors', $builder_frame_ancestors );
            $settings->set( $section, 'builder_extra_directives', $builder_extra );
            $settings->set( $section, 'builder_allow_inline_js', $builder_allow_inline_js );
            $settings->set( $section, 'builder_allow_inline_css', $builder_allow_inline_css );
			$settings->set( $section, 'builder_script_mode', $builder_script_mode );
			$settings->set( $section, 'builder_style_mode', $builder_style_mode );
			$settings->set( $section, 'builder_img_mode', $builder_img_mode );
			$settings->set( $section, 'builder_font_mode', $builder_font_mode );
			$settings->set( $section, 'builder_connect_mode', $builder_connect_mode );
			$settings->set( $section, 'builder_frame_mode', $builder_frame_mode );
			$settings->set( $section, 'builder_extra_mode', $builder_extra_mode );
            $settings->save();

            if ( $logger ) {
                $logger->log(
                    'sec_headers_csp_builder_updated',
                    __( 'CSP builder settings were updated.', 'aegisshield-security' )
                );
            }

            $updated_builder = true;
        }

        if (
            'profiles' === $current_tab
            && $is_pro
            && isset( $_POST['aegisshield_sec_headers_profiles_nonce'] )
            && check_admin_referer( 'aegisshield_sec_headers_profiles_save', 'aegisshield_sec_headers_profiles_nonce' )
        ) {
            $apply_profile = isset( $_POST['apply_profile'] ) ? sanitize_key( wp_unslash( $_POST['apply_profile'] ) ) : '';

            if ( $apply_profile ) {
                // Click-to-fix: apply predefined profiles.
                $enabled     = 'on';
                $enable_hsts = is_ssl() ? 'on' : $enable_hsts;

                switch ( $apply_profile ) {
                    case 'blog':
                        $csp_level      = 'basic';
                        $csp_frontend   = 'basic';
                        $csp_admin      = 'strict';
                        $csp_custom     = '';
                        break;
                    case 'woo':
                        $csp_level      = 'basic';
                        $csp_frontend   = 'basic';
                        $csp_admin      = 'strict';
                        $csp_custom     = 'basic';
                        break;
                    case 'membership':
                        $csp_level      = 'strict';
                        $csp_frontend   = 'basic';
                        $csp_admin      = 'strict';
                        $csp_custom     = 'strict';
                        break;
                }
            } else {
                $valid_levels = array( '', 'inherit', 'none', 'basic', 'strict', 'builder_enforce', 'builder_report' );

                $csp_frontend = isset( $_POST['csp_level_frontend'] ) ? sanitize_text_field( wp_unslash( $_POST['csp_level_frontend'] ) ) : $csp_frontend;
                $csp_admin    = isset( $_POST['csp_level_admin'] ) ? sanitize_text_field( wp_unslash( $_POST['csp_level_admin'] ) ) : $csp_admin;
                $csp_custom   = isset( $_POST['csp_level_custom'] ) ? sanitize_text_field( wp_unslash( $_POST['csp_level_custom'] ) ) : $csp_custom;

                if ( ! in_array( $csp_frontend, $valid_levels, true ) ) {
                    $csp_frontend = '';
                }
                if ( ! in_array( $csp_admin, $valid_levels, true ) ) {
                    $csp_admin = '';
                }
                if ( ! in_array( $csp_custom, $valid_levels, true ) ) {
                    $csp_custom = '';
                }
            }

            $profiles_paths_raw = isset( $_POST['profiles_custom_paths'] )
                ? sanitize_textarea_field( wp_unslash( $_POST['profiles_custom_paths'] ) )
                : $profiles_paths_raw;

            $settings->set( $section, 'enabled', $enabled );
            $settings->set( $section, 'enable_hsts', $enable_hsts );
            $settings->set( $section, 'csp_level', $csp_level );
            $settings->set( $section, 'csp_level_frontend', $csp_frontend );
            $settings->set( $section, 'csp_level_admin', $csp_admin );
            $settings->set( $section, 'csp_level_custom', $csp_custom );
            $settings->set( $section, 'profiles_custom_paths', $profiles_paths_raw );
            $settings->save();

            if ( $logger ) {
                $logger->log(
                    'sec_headers_profiles_updated',
                    __( 'Security header profiles were updated.', 'aegisshield-security' )
                );
            }

            $updated_profiles = true;
        }

        $is_https = is_ssl();

        // Header health (Pro only, used on Profiles tab).
        $health = $this->calculate_header_health( $enabled, $csp_level, $enable_hsts, $is_https );

        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Security Headers', 'aegisshield-security' ); ?></h1>
            <p>
                <?php esc_html_e( 'AegisShield Security Headers provides a practical, browser-level hardening layer that helps protect WordPress sites against common client-side attacks such as clickjacking, unsafe resource loading, and protocol downgrade risks. It applies modern HTTP security headers intelligently—enforcing them when possible, logging when enforcement is blocked, and recording real-world policy violations—so administrators gain both protection and visibility without breaking site functionality.', 'aegisshield-security' ); ?>
            </p>

            <h2 class="nav-tab-wrapper">
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'standard' ) ); ?>" class="nav-tab <?php echo ( 'standard' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Standard headers', 'aegisshield-security' ); ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'csp_builder' ) ); ?>" class="nav-tab <?php echo ( 'csp_builder' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'CSP Builder', 'aegisshield-security' ); ?>
                    <?php if ( ! $is_pro ) : ?>
                        <span class="aegisshield-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
                            <?php esc_html_e( 'Pro', 'aegisshield-security' ); ?>
                        </span>
                    <?php endif; ?>

                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'profiles' ) ); ?>" class="nav-tab <?php echo ( 'profiles' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Profiles &amp; Health', 'aegisshield-security' ); ?>
                    <?php if ( ! $is_pro ) : ?>
                        <span class="aegisshield-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
                            <?php esc_html_e( 'Pro', 'aegisshield-security' ); ?>
                        </span>
                    <?php endif; ?>
                </a>
            </h2>

            <?php if ( $updated_standard && 'standard' === $current_tab ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'Security Headers settings saved.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( $updated_builder && 'csp_builder' === $current_tab ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'CSP builder settings saved.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( $updated_profiles && 'profiles' === $current_tab ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'Header profiles and health settings saved.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( ! $is_https ) : ?>
                <div class="notice notice-warning">
                    <p>
                        <?php esc_html_e( 'Your current request is not served over HTTPS. Some headers (like HSTS) rely on HTTPS to be effective. Make sure your site is served over HTTPS.', 'aegisshield-security' ); ?>
                    </p>
                </div>
            <?php endif; ?>

            <?php if ( 'standard' === $current_tab ) : ?>

                <form method="post">
                    <?php wp_nonce_field( 'aegisshield_sec_headers_save', 'aegisshield_sec_headers_nonce' ); ?>

                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="sec_headers_enabled"><?php esc_html_e( 'Enable security headers', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <label for="sec_headers_enabled">
                                    <input type="checkbox" id="sec_headers_enabled" name="sec_headers_enabled" <?php checked( $enabled, 'on' ); ?> />
                                    <?php esc_html_e( 'Send recommended security headers on all front-end and admin requests.', 'aegisshield-security' ); ?>
                                </label>
                                <p class="description">
                                    <?php esc_html_e( 'If disabled, AegisShield will not send any of the HTTP security headers configured on this page.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                    </table>

                    <h2><?php esc_html_e( 'Content-Security-Policy (CSP)', 'aegisshield-security' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="csp_level"><?php esc_html_e( 'CSP level', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <select name="csp_level" id="csp_level">
                                    <option value="none" <?php selected( $csp_level, 'none' ); ?>>
                                        <?php esc_html_e( 'None (do not send CSP header)', 'aegisshield-security' ); ?>
                                    </option>
                                    <option value="basic" <?php selected( $csp_level, 'basic' ); ?>>
                                        <?php esc_html_e( 'Basic – safe defaults for most sites (allows common CDNs over HTTPS)', 'aegisshield-security' ); ?>
                                    </option>
                                    <option value="strict" <?php selected( $csp_level, 'strict' ); ?>>
                                        <?php esc_html_e( 'Strict – locked-down policy (may require adjustments for some plugins/themes)', 'aegisshield-security' ); ?>
                                    </option>
                                    <?php if ( $is_pro ) : ?>
                                        <option value="builder_report" <?php selected( $csp_level, 'builder_report' ); ?>>
                                            <?php esc_html_e( 'Pro: Use Visual CSP Builder (Report-Only)', 'aegisshield-security' ); ?>
                                        </option>
                                        <option value="builder_enforce" <?php selected( $csp_level, 'builder_enforce' ); ?>>
                                            <?php esc_html_e( 'Pro: Use Visual CSP Builder (Enforced)', 'aegisshield-security' ); ?>
                                        </option>
                                    <?php endif; ?>
                                </select>
                                <p class="description">
                                    <?php esc_html_e( 'Content-Security-Policy helps control which scripts, styles, images, and other resources can load on your site. Start with Basic, test, then move towards Strict or the Visual CSP Builder for advanced control.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                    </table>

                    <h2><?php esc_html_e( 'HSTS (HTTP Strict Transport Security)', 'aegisshield-security' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="enable_hsts"><?php esc_html_e( 'Enable HSTS', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <label for="enable_hsts">
                                    <input type="checkbox" id="enable_hsts" name="enable_hsts" <?php checked( $enable_hsts, 'on' ); ?> />
                                    <?php esc_html_e( 'Send Strict-Transport-Security header on HTTPS requests.', 'aegisshield-security' ); ?>
                                </label>
                                <p class="description">
                                    <?php esc_html_e( 'Only enable HSTS after you confirm your site is always reachable over HTTPS. Once browsers see this header, they will refuse to connect over HTTP and may cache this behavior for a long time.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                    </table>

                    <?php submit_button( __( 'Save Changes', 'aegisshield-security' ) ); ?>
                </form>

            <?php elseif ( 'csp_builder' === $current_tab ) : ?>

                <?php if ( ! $is_pro ) : ?>
                    <?php $this->render_pro_locked_notice(); ?>
                <?php else : ?>

                <?php endif; ?>

                    <form method="post">
                        <?php wp_nonce_field( 'aegisshield_sec_headers_builder_save', 'aegisshield_sec_headers_builder_nonce' ); ?>

                        <h2><?php esc_html_e( 'Visual CSP Builder', 'aegisshield-security' ); ?></h2>
						<?php
						// Phase 3: Per-directive CSP mode snapshot (UI only).
						$directive_modes = array(
							'Script sources'   => $builder_script_mode,
							'Style sources'    => $builder_style_mode,
							'Image sources'    => $builder_img_mode,
							'Font sources'     => $builder_font_mode,
							'XHR / Fetch'      => $builder_connect_mode,
							'Frame ancestors'  => $builder_frame_mode,
							'Extra directives' => $builder_extra_mode,
						);
						?>

						<div class="aegisshield-csp-overview" style="margin:15px 0;padding:12px;border:1px solid #ccd0d4;background:#fff;">
							<strong><?php esc_html_e( 'CSP Directive Status Overview', 'aegisshield-security' ); ?></strong>
							<ul style="margin:8px 0 0 18px;list-style:disc;">
								<?php foreach ( $directive_modes as $label => $mode ) : ?>
									<li>
										<?php echo esc_html( $label ); ?> —
										<?php
										if ( 'enforce' === $mode ) {
											echo '<span style="color:#2e7d32;font-weight:600;">Enforced</span>';
										} elseif ( 'report_only' === $mode ) {
											echo '<span style="color:#ed6c02;font-weight:600;">Report-Only</span>';
										} else {
											echo '<span style="color:#b71c1c;font-weight:600;">Disabled</span>';
										}
										?>
									</li>
								<?php endforeach; ?>
							</ul>
						</div>
                        <p>
                            <?php esc_html_e( 'Build a Content-Security-Policy visually by choosing allowed sources for different resource types. Start in Report-Only mode to collect violations without breaking your site, then switch to Enforced when you are confident.', 'aegisshield-security' ); ?>
                        </p>
						<?php
						// Phase 3: CSP configuration warnings (UI only).
						$warnings = array();

						// Enforced CSP without report-only anywhere.
						if (
							'enforce' === $builder_script_mode ||
							'enforce' === $builder_style_mode ||
							'enforce' === $builder_img_mode ||
							'enforce' === $builder_font_mode ||
							'enforce' === $builder_connect_mode ||
							'enforce' === $builder_frame_mode
						) {
							$has_report = in_array(
								'report_only',
								array(
									$builder_script_mode,
									$builder_style_mode,
									$builder_img_mode,
									$builder_font_mode,
									$builder_connect_mode,
									$builder_frame_mode,
								),
								true
							);

							if ( ! $has_report ) {
								$warnings[] = __( 'You are enforcing CSP directives without any Report-Only directives enabled. It is recommended to monitor violations before full enforcement.', 'aegisshield-security' );
							}
						}

						// Inline allowed while enforced.
						if ( 'on' === $builder_allow_inline_js && 'enforce' === $builder_script_mode ) {
							$warnings[] = __( 'Inline scripts are allowed while Script Sources are enforced. This weakens CSP protection.', 'aegisshield-security' );
						}

						if ( 'on' === $builder_allow_inline_css && 'enforce' === $builder_style_mode ) {
							$warnings[] = __( 'Inline styles are allowed while Style Sources are enforced. This weakens CSP protection.', 'aegisshield-security' );
						}
						?>

						<?php if ( ! empty( $warnings ) ) : ?>
							<div class="notice notice-warning">
								<p><strong><?php esc_html_e( 'CSP Configuration Warnings', 'aegisshield-security' ); ?></strong></p>
								<ul class="ul-disc">
									<?php foreach ( $warnings as $warning ) : ?>
										<li><?php echo esc_html( $warning ); ?></li>
									<?php endforeach; ?>
								</ul>
							</div>
						<?php endif; ?>

                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <?php esc_html_e( 'Mode', 'aegisshield-security' ); ?>
                                </th>
                                <td>
                                    <fieldset>
                                        <p class="description">
                                            <?php esc_html_e( 'When your CSP Level (on the Standard tab or via profiles) is set to "Pro: Use Visual CSP Builder", this mode controls whether the policy is enforced or only logged.', 'aegisshield-security' ); ?>
                                        </p>
                                    </fieldset>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="builder_script_src"><?php esc_html_e( 'Script sources', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <textarea name="builder_script_src" id="builder_script_src" rows="3" cols="60"><?php echo esc_textarea( $builder_script_src ); ?></textarea>
									<fieldset style="margin-top:6px;">
										<strong><?php esc_html_e( 'Mode', 'aegisshield-security' ); ?></strong><br />
										<label>
											<input type="radio" name="builder_script_mode" value="off" <?php checked( $builder_script_mode, 'off' ); ?> />
											<?php esc_html_e( 'Disabled', 'aegisshield-security' ); ?>
										</label><br />
										<label>
											<input type="radio" name="builder_script_mode" value="report_only" <?php checked( $builder_script_mode, 'report_only' ); ?> />
											<?php esc_html_e( 'Report-Only', 'aegisshield-security' ); ?>
										</label><br />
										<label>
											<input type="radio" name="builder_script_mode" value="enforce" <?php checked( $builder_script_mode, 'enforce' ); ?> />
											<?php esc_html_e( 'Enforced', 'aegisshield-security' ); ?>
										</label>
									</fieldset>									
                                    <p class="description">
                                        <?php esc_html_e( 'One source per line (e.g. https://cdn.example.com). The policy will always include \'self\'.', 'aegisshield-security' ); ?>
                                    </p>
                                    <label>
                                        <input type="checkbox" name="builder_allow_inline_js" <?php checked( $builder_allow_inline_js, 'on' ); ?> />
                                        <?php esc_html_e( 'Allow inline scripts (adds \'unsafe-inline\') – not recommended unless required.', 'aegisshield-security' ); ?>
                                    </label>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="builder_style_src"><?php esc_html_e( 'Style sources', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <textarea name="builder_style_src" id="builder_style_src" rows="3" cols="60"><?php echo esc_textarea( $builder_style_src ); ?></textarea>
									<fieldset style="margin-top:6px;">
										<strong><?php esc_html_e( 'Mode', 'aegisshield-security' ); ?></strong><br />
										<label>
											<input type="radio" name="builder_style_mode" value="off" <?php checked( $builder_style_mode, 'off' ); ?> />
											<?php esc_html_e( 'Disabled', 'aegisshield-security' ); ?>
										</label><br />
										<label>
											<input type="radio" name="builder_style_mode" value="report_only" <?php checked( $builder_style_mode, 'report_only' ); ?> />
											<?php esc_html_e( 'Report-Only', 'aegisshield-security' ); ?>
										</label><br />
										<label>
											<input type="radio" name="builder_style_mode" value="enforce" <?php checked( $builder_style_mode, 'enforce' ); ?> />
											<?php esc_html_e( 'Enforced', 'aegisshield-security' ); ?>
										</label>
									</fieldset>	
                                    <p class="description">
                                        <?php esc_html_e( 'One source per line (e.g. https://fonts.googleapis.com). The policy will always include \'self\'.', 'aegisshield-security' ); ?>
                                    </p>
                                    <label>
                                        <input type="checkbox" name="builder_allow_inline_css" <?php checked( $builder_allow_inline_css, 'on' ); ?> />
                                        <?php esc_html_e( 'Allow inline styles (adds \'unsafe-inline\') – use only if your theme requires it.', 'aegisshield-security' ); ?>
                                    </label>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="builder_img_src"><?php esc_html_e( 'Image sources', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <textarea name="builder_img_src" id="builder_img_src" rows="3" cols="60"><?php echo esc_textarea( $builder_img_src ); ?></textarea>
									<fieldset style="margin-top:6px;">
										<strong><?php esc_html_e( 'Mode', 'aegisshield-security' ); ?></strong><br />
										<label>
											<input type="radio" name="builder_img_mode" value="off" <?php checked( $builder_img_mode, 'off' ); ?> />
											<?php esc_html_e( 'Disabled', 'aegisshield-security' ); ?>
										</label><br />
										<label>
											<input type="radio" name="builder_img_mode" value="report_only" <?php checked( $builder_img_mode, 'report_only' ); ?> />
											<?php esc_html_e( 'Report-Only', 'aegisshield-security' ); ?>
										</label><br />
										<label>
											<input type="radio" name="builder_img_mode" value="enforce" <?php checked( $builder_img_mode, 'enforce' ); ?> />
											<?php esc_html_e( 'Enforced', 'aegisshield-security' ); ?>
										</label>
									</fieldset>	
                                    <p class="description">
                                        <?php esc_html_e( 'One source per line (e.g. https://images.example.com). The policy will include \'self\' and data: for inline images.', 'aegisshield-security' ); ?>
                                    </p>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="builder_font_src"><?php esc_html_e( 'Font sources', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <textarea name="builder_font_src" id="builder_font_src" rows="3" cols="60"><?php echo esc_textarea( $builder_font_src ); ?></textarea>
									<fieldset style="margin-top:6px;">
										<strong><?php esc_html_e( 'Mode', 'aegisshield-security' ); ?></strong><br />
										<label>
											<input type="radio" name="builder_font_mode" value="off" <?php checked( $builder_font_mode, 'off' ); ?> />
											<?php esc_html_e( 'Disabled', 'aegisshield-security' ); ?>
										</label><br />
										<label>
											<input type="radio" name="builder_font_mode" value="report_only" <?php checked( $builder_font_mode, 'report_only' ); ?> />
											<?php esc_html_e( 'Report-Only', 'aegisshield-security' ); ?>
										</label><br />
										<label>
											<input type="radio" name="builder_font_mode" value="enforce" <?php checked( $builder_font_mode, 'enforce' ); ?> />
											<?php esc_html_e( 'Enforced', 'aegisshield-security' ); ?>
										</label>
									</fieldset>	
                                    <p class="description">
                                        <?php esc_html_e( 'One source per line (e.g. https://fonts.gstatic.com). The policy will always include \'self\' and data:.', 'aegisshield-security' ); ?>
                                    </p>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="builder_connect_src"><?php esc_html_e( 'XHR / fetch / API endpoints', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <textarea name="builder_connect_src" id="builder_connect_src" rows="3" cols="60"><?php echo esc_textarea( $builder_connect_src ); ?></textarea>
									<fieldset style="margin-top:6px;">
										<strong><?php esc_html_e( 'Mode', 'aegisshield-security' ); ?></strong><br />
										<label>
											<input type="radio" name="builder_connect_mode" value="off" <?php checked( $builder_connect_mode, 'off' ); ?> />
											<?php esc_html_e( 'Disabled', 'aegisshield-security' ); ?>
										</label><br />
										<label>
											<input type="radio" name="builder_connect_mode" value="report_only" <?php checked( $builder_connect_mode, 'report_only' ); ?> />
											<?php esc_html_e( 'Report-Only', 'aegisshield-security' ); ?>
										</label><br />
										<label>
											<input type="radio" name="builder_connect_mode" value="enforce" <?php checked( $builder_connect_mode, 'enforce' ); ?> />
											<?php esc_html_e( 'Enforced', 'aegisshield-security' ); ?>
										</label>
									</fieldset>	
                                    <p class="description">
                                        <?php esc_html_e( 'One source per line (e.g. https://api.example.com). These are used for AJAX/fetch/WebSocket requests.', 'aegisshield-security' ); ?>
                                    </p>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="builder_frame_ancestors"><?php esc_html_e( 'Where your site may be embedded (frame-ancestors)', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <textarea name="builder_frame_ancestors" id="builder_frame_ancestors" rows="3" cols="60"><?php echo esc_textarea( $builder_frame_ancestors ); ?></textarea>
									<fieldset style="margin-top:6px;">
										<strong><?php esc_html_e( 'Mode', 'aegisshield-security' ); ?></strong><br />
										<label>
											<input type="radio" name="builder_frame_mode" value="off" <?php checked( $builder_frame_mode, 'off' ); ?> />
											<?php esc_html_e( 'Disabled', 'aegisshield-security' ); ?>
										</label><br />
										<label>
											<input type="radio" name="builder_frame_mode" value="report_only" <?php checked( $builder_frame_mode, 'report_only' ); ?> />
											<?php esc_html_e( 'Report-Only', 'aegisshield-security' ); ?>
										</label><br />
										<label>
											<input type="radio" name="builder_frame_mode" value="enforce" <?php checked( $builder_frame_mode, 'enforce' ); ?> />
											<?php esc_html_e( 'Enforced', 'aegisshield-security' ); ?>
										</label>
									</fieldset>	
                                    <p class="description">
                                        <?php esc_html_e( 'One origin per line. By default the policy will restrict framing to \'self\'.', 'aegisshield-security' ); ?>
                                    </p>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="builder_extra_directives"><?php esc_html_e( 'Extra directives (advanced)', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <textarea name="builder_extra_directives" id="builder_extra_directives" rows="3" cols="60"><?php echo esc_textarea( $builder_extra ); ?></textarea>
									<fieldset style="margin-top:6px;">
										<strong><?php esc_html_e( 'Mode', 'aegisshield-security' ); ?></strong><br />
										<label>
											<input type="radio" name="builder_extra_mode" value="off" <?php checked( $builder_extra_mode, 'off' ); ?> />
											<?php esc_html_e( 'Disabled', 'aegisshield-security' ); ?>
										</label><br />
										<label>
											<input type="radio" name="builder_extra_mode" value="report_only" <?php checked( $builder_extra_mode, 'report_only' ); ?> />
											<?php esc_html_e( 'Report-Only', 'aegisshield-security' ); ?>
										</label><br />
										<label>
											<input type="radio" name="builder_extra_mode" value="enforce" <?php checked( $builder_extra_mode, 'enforce' ); ?> />
											<?php esc_html_e( 'Enforced', 'aegisshield-security' ); ?>
										</label>
									</fieldset>	
                                    <p class="description">
                                        <?php esc_html_e( 'Optional extra directives, one per line (for example: upgrade-insecure-requests).', 'aegisshield-security' ); ?>
                                    </p>
                                </td>
                            </tr>
                        </table>

                        <?php submit_button( __( 'Save CSP Builder Settings', 'aegisshield-security' ) ); ?>
                    </form>

                    <?php
                    // Recent CSP violation reports (if any).
                    global $wpdb;
                    $reports = array();
                    if ( $logger && method_exists( $logger, 'get_table_name' ) ) {
                        $table = $logger->get_table_name();
                        if ( $table ) {
                            $reports = $wpdb->get_results(
                                $wpdb->prepare(
                                    "SELECT event_time, ip_address, message, context FROM {$table} WHERE event_type = %s ORDER BY event_time DESC LIMIT 50",
                                    'csp_violation'
                                ),
                                ARRAY_A
                            );
                        }
                    }
                    ?>

                    <h2><?php esc_html_e( 'CSP Reports (last 50)', 'aegisshield-security' ); ?></h2>
					<?php
					// Phase 3: CSP recommendations based on logged violations.
					$recommend = array();

					if ( ! empty( $reports ) ) {
						foreach ( $reports as $row ) {
							if ( empty( $row['context'] ) ) {
								continue;
							}

							$ctx = json_decode( $row['context'], true );
							if ( ! is_array( $ctx ) ) {
								continue;
							}

							if ( ! empty( $ctx['effective_directive'] ) ) {
								$recommend[ $ctx['effective_directive'] ] = true;
							}
						}
					}
					?>

					<?php if ( ! empty( $recommend ) ) : ?>
						<div class="notice notice-info">
							<p><strong><?php esc_html_e( 'CSP Optimization Suggestions', 'aegisshield-security' ); ?></strong></p>
							<p><?php esc_html_e( 'Recent CSP violations indicate that the following directives may need adjustment or monitoring:', 'aegisshield-security' ); ?></p>
							<ul class="ul-disc">
								<?php foreach ( array_keys( $recommend ) as $directive ) : ?>
									<li>
										<?php echo esc_html( $directive ); ?> —
										<?php esc_html_e( 'consider adding sources or keeping this directive in Report-Only mode while testing.', 'aegisshield-security' ); ?>
									</li>
								<?php endforeach; ?>
							</ul>
						</div>
					<?php endif; ?>

                    <p class="description">
                        <?php esc_html_e( 'When your CSP is in Report-Only mode, browsers send violation reports here. Use them to refine your policy before enforcing it.', 'aegisshield-security' ); ?>
                    </p>

                    <?php if ( empty( $reports ) ) : ?>
                        <p><?php esc_html_e( 'No CSP violation reports have been logged yet.', 'aegisshield-security' ); ?></p>
                    <?php else : ?>
                        <table class="widefat striped">
                            <thead>
                                <tr>
                                    <th><?php esc_html_e( 'Time', 'aegisshield-security' ); ?></th>
                                    <th><?php esc_html_e( 'IP Address', 'aegisshield-security' ); ?></th>
                                    <th><?php esc_html_e( 'Directive', 'aegisshield-security' ); ?></th>
                                    <th><?php esc_html_e( 'Blocked URI', 'aegisshield-security' ); ?></th>
                                    <th><?php esc_html_e( 'Page', 'aegisshield-security' ); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ( $reports as $row ) : ?>
                                    <?php
                                    $ctx         = array();
                                    $directive   = '';
                                    $blocked_uri = '';
                                    $document    = '';

                                    if ( ! empty( $row['context'] ) ) {
                                        $decoded = json_decode( $row['context'], true );
                                        if ( is_array( $decoded ) ) {
                                            $ctx = $decoded;
                                        }
                                    }

                                    if ( isset( $ctx['effective_directive'] ) ) {
                                        $directive = $ctx['effective_directive'];
                                    } elseif ( isset( $ctx['violated_directive'] ) ) {
                                        $directive = $ctx['violated_directive'];
                                    }

                                    if ( isset( $ctx['blocked_uri'] ) ) {
                                        $blocked_uri = $ctx['blocked_uri'];
                                    }

                                    if ( isset( $ctx['document_uri'] ) ) {
                                        $document = $ctx['document_uri'];
                                    }
                                    ?>
                                    <tr>
                                        <td><?php echo esc_html( $row['event_time'] ); ?></td>
                                        <td><?php echo esc_html( $row['ip_address'] ); ?></td>
                                        <td><?php echo esc_html( $directive ); ?></td>
                                        <td><?php echo esc_html( $blocked_uri ); ?></td>
                                        <td><?php echo esc_html( $document ); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>

                <?php if ( ! $is_pro ) : ?>
                    </div>
                <?php endif; ?>

            <?php elseif ( 'profiles' === $current_tab ) : ?>

                <?php if ( ! $is_pro ) : ?>
                    <?php $this->render_pro_locked_notice(); ?>
                <?php else : ?>

                    <form method="post">
                        <?php wp_nonce_field( 'aegisshield_sec_headers_profiles_save', 'aegisshield_sec_headers_profiles_nonce' ); ?>

                        <!-- KEEP ALL EXISTING PROFILES + HEALTH + PRESET BUTTONS MARKUP HERE UNCHANGED -->

                    </form>

                <?php endif; ?>
				
                    <form method="post">
                        <?php wp_nonce_field( 'aegisshield_sec_headers_profiles_save', 'aegisshield_sec_headers_profiles_nonce' ); ?>

                        <h2><?php esc_html_e( 'Header profiles by area', 'aegisshield-security' ); ?></h2>
                        <p>
                            <?php esc_html_e( 'Use different header profiles for the public site, the WordPress admin, and sensitive custom paths (for example your checkout or course area).', 'aegisshield-security' ); ?>
                        </p>

                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="csp_level_frontend"><?php esc_html_e( 'Front-end', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <select name="csp_level_frontend" id="csp_level_frontend">
                                        <option value="" <?php selected( $csp_frontend, '' ); ?>><?php esc_html_e( 'Inherit global CSP level', 'aegisshield-security' ); ?></option>
                                        <option value="none" <?php selected( $csp_frontend, 'none' ); ?>><?php esc_html_e( 'None (no CSP header)', 'aegisshield-security' ); ?></option>
                                        <option value="basic" <?php selected( $csp_frontend, 'basic' ); ?>><?php esc_html_e( 'Basic', 'aegisshield-security' ); ?></option>
                                        <option value="strict" <?php selected( $csp_frontend, 'strict' ); ?>><?php esc_html_e( 'Strict', 'aegisshield-security' ); ?></option>
                                        <option value="builder_report" <?php selected( $csp_frontend, 'builder_report' ); ?>><?php esc_html_e( 'Pro builder (Report-Only)', 'aegisshield-security' ); ?></option>
                                        <option value="builder_enforce" <?php selected( $csp_frontend, 'builder_enforce' ); ?>><?php esc_html_e( 'Pro builder (Enforced)', 'aegisshield-security' ); ?></option>
                                    </select>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="csp_level_admin"><?php esc_html_e( 'wp-admin', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <select name="csp_level_admin" id="csp_level_admin">
                                        <option value="" <?php selected( $csp_admin, '' ); ?>><?php esc_html_e( 'Inherit global CSP level', 'aegisshield-security' ); ?></option>
                                        <option value="none" <?php selected( $csp_admin, 'none' ); ?>><?php esc_html_e( 'None (no CSP header)', 'aegisshield-security' ); ?></option>
                                        <option value="basic" <?php selected( $csp_admin, 'basic' ); ?>><?php esc_html_e( 'Basic', 'aegisshield-security' ); ?></option>
                                        <option value="strict" <?php selected( $csp_admin, 'strict' ); ?>><?php esc_html_e( 'Strict', 'aegisshield-security' ); ?></option>
                                        <option value="builder_report" <?php selected( $csp_admin, 'builder_report' ); ?>><?php esc_html_e( 'Pro builder (Report-Only)', 'aegisshield-security' ); ?></option>
                                        <option value="builder_enforce" <?php selected( $csp_admin, 'builder_enforce' ); ?>><?php esc_html_e( 'Pro builder (Enforced)', 'aegisshield-security' ); ?></option>
                                    </select>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="csp_level_custom"><?php esc_html_e( 'Custom paths', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <select name="csp_level_custom" id="csp_level_custom">
                                        <option value="" <?php selected( $csp_custom, '' ); ?>><?php esc_html_e( 'Inherit global CSP level', 'aegisshield-security' ); ?></option>
                                        <option value="none" <?php selected( $csp_custom, 'none' ); ?>><?php esc_html_e( 'None (no CSP header)', 'aegisshield-security' ); ?></option>
                                        <option value="basic" <?php selected( $csp_custom, 'basic' ); ?>><?php esc_html_e( 'Basic', 'aegisshield-security' ); ?></option>
                                        <option value="strict" <?php selected( $csp_custom, 'strict' ); ?>><?php esc_html_e( 'Strict', 'aegisshield-security' ); ?></option>
                                        <option value="builder_report" <?php selected( $csp_custom, 'builder_report' ); ?>><?php esc_html_e( 'Pro builder (Report-Only)', 'aegisshield-security' ); ?></option>
                                        <option value="builder_enforce" <?php selected( $csp_custom, 'builder_enforce' ); ?>><?php esc_html_e( 'Pro builder (Enforced)', 'aegisshield-security' ); ?></option>
                                    </select>
                                    <p class="description">
                                        <?php esc_html_e( 'Custom paths below will use this profile. If they are empty or do not match, the request falls back to the front-end or wp-admin profile.', 'aegisshield-security' ); ?>
                                    </p>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="profiles_custom_paths"><?php esc_html_e( 'Custom paths', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <textarea name="profiles_custom_paths" id="profiles_custom_paths" rows="4" cols="60"><?php echo esc_textarea( $profiles_paths_raw ); ?></textarea>
                                    <p class="description">
                                        <?php esc_html_e( 'One path per line, starting with /. For example: /checkout, /courses/, /membership/. Requests whose URL begins with one of these paths will use the "Custom paths" profile.', 'aegisshield-security' ); ?>
                                    </p>
                                </td>
                            </tr>
                        </table>

                        <h2><?php esc_html_e( 'Header Health score', 'aegisshield-security' ); ?></h2>
                        <p>
                            <?php esc_html_e( 'A quick, opinionated score to help you see whether your current header configuration is missing important protections or could be improved.', 'aegisshield-security' ); ?>
                        </p>

                        <p>
                            <strong><?php esc_html_e( 'Score:', 'aegisshield-security' ); ?></strong>
                            <?php echo esc_html( (string) (int) $health['score'] ); ?>/100
                        </p>

                        <?php if ( ! empty( $health['issues'] ) ) : ?>
                            <ul class="ul-disc">
                                <?php foreach ( $health['issues'] as $issue ) : ?>
                                    <li><?php echo esc_html( $issue ); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>

                        <h3><?php esc_html_e( 'Click-to-fix presets', 'aegisshield-security' ); ?></h3>
                        <p>
                            <?php esc_html_e( 'Choose a starting profile that matches your site and let AegisShield apply a safe baseline. You can still fine-tune the details above afterwards.', 'aegisshield-security' ); ?>
                        </p>

                        <p>
                            <button type="submit" name="apply_profile" value="blog" class="button">
                                <?php esc_html_e( 'Apply Basic Blog profile', 'aegisshield-security' ); ?>
                            </button>
                            <button type="submit" name="apply_profile" value="woo" class="button">
                                <?php esc_html_e( 'Apply WooCommerce profile', 'aegisshield-security' ); ?>
                            </button>
                            <button type="submit" name="apply_profile" value="membership" class="button">
                                <?php esc_html_e( 'Apply Membership/LMS profile', 'aegisshield-security' ); ?>
                            </button>
                        </p>

                        <?php submit_button( __( 'Save Profiles & Health Settings', 'aegisshield-security' ) ); ?>
                    </form>

                <?php if ( ! $is_pro ) : ?>
                    </div>
                <?php endif; ?>

            <?php endif; // Tab switch. ?>
        </div>
        <?php
    }

    /**
     * Calculate a simple header health score and list of issues.
     *
     * @param string $enabled     Whether headers are enabled.
     * @param string $csp_level   Current global CSP level.
     * @param string $enable_hsts Whether HSTS is enabled.
     * @param bool   $is_https    Whether current request is HTTPS.
     *
     * @return array{score:int,issues:array}
     */
    protected function calculate_header_health( $enabled, $csp_level, $enable_hsts, $is_https ) {
        $score  = 0;
        $issues = array();

        if ( 'on' === $enabled ) {
            $score += 35;
        } else {
            $issues[] = __( 'Security headers are disabled. Turn them on to protect against common attacks.', 'aegisshield-security' );
        }

        if ( $is_https ) {
            if ( 'on' === $enable_hsts ) {
                $score += 25;
            } else {
                $issues[] = __( 'HSTS is not enabled. Enabling it helps prevent protocol downgrade attacks.', 'aegisshield-security' );
            }
        } else {
            if ( 'on' === $enable_hsts ) {
                $issues[] = __( 'HSTS is enabled but the site is not being served over HTTPS. This can lead to confusing behavior in browsers.', 'aegisshield-security' );
            }
        }

        if ( 'none' === $csp_level || '' === $csp_level ) {
            $issues[] = __( 'No Content-Security-Policy is configured. Adding at least a basic CSP significantly hardens your site.', 'aegisshield-security' );
        } elseif ( in_array( $csp_level, array( 'basic', 'builder_report', 'builder_enforce', 'builder_enforce' ), true ) ) {
            // Note: builder_report / builder_enforce treated similar to basic for scoring.
            $score += 20;
        } elseif ( 'strict' === $csp_level ) {
            $score += 30;
        }

        // Rough normalization.
        if ( $score > 100 ) {
            $score = 100;
        }
        if ( $score < 0 ) {
            $score = 0;
        }

        return array(
            'score'  => (int) $score,
            'issues' => $issues,
        );
    }
}