<?php
namespace AegisShield\Admin_Pages\Hardening;

use AegisShield\AS_Plugin;
use AegisShield\Modules\AS_Module_Hardening;

defined( 'ABSPATH' ) || exit;

/**
 * Hardening – Role Risk Analyzer tab.
 */
class AS_Page_Hardening_Tab_Roles {

    /**
     * Render the Role Risk Analyzer tab.
     *
     * @param AS_Plugin $plugin Plugin instance.
     * @return void
     */
    public static function render_tab( AS_Plugin $plugin ) {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $settings = $plugin->get_settings();
        $section  = 'hardening';

        $is_pro = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();

        $mode    = (string) $settings->get( $section, 'hardening_role_risk_mode', 'off' );
        $results = $settings->get( $section, 'hardening_role_risk_results', array() );
        if ( ! is_array( $results ) ) {
            $results = array();
        }

        $scan_ran = false;

		if (
			isset( $_POST['aegisshield_hardening_roles_nonce'] )
			&& wp_verify_nonce(
				sanitize_text_field( wp_unslash( $_POST['aegisshield_hardening_roles_nonce'] ) ),
				'aegisshield_hardening_roles_save'
			)
		) {
			if ( ! current_user_can( 'manage_options' ) ) {
				wp_die( esc_html__( 'You do not have permission to perform this action.', 'aegisshield-security' ) );
			}
            $mode = isset( $_POST['hardening_role_risk_mode'] ) ? 'on' : 'off';
            $settings->set( $section, 'hardening_role_risk_mode', $mode );
            $settings->save();

            if ( $is_pro && isset( $_POST['run_role_risk_scan_now'] ) ) {
                $module  = new AS_Module_Hardening( $plugin );
                $results = $module->run_role_risk_scan();
                $scan_ran = true;
            }

            $results = $settings->get( $section, 'hardening_role_risk_results', $results );
            if ( ! is_array( $results ) ) {
                $results = array();
            }
        }

        ?>
        <div class="as-hardening-tab as-hardening-tab-roles">
            <h2><?php esc_html_e( 'Role & Capability Risk Analyzer', 'aegisshield-security' ); ?></h2>

            <p class="description">
                <?php esc_html_e(
                    'Review administrator and high-privilege accounts, dangerous capabilities, and dormant admins.',
                    'aegisshield-security'
                ); ?>
            </p>

            <?php if ( ! $is_pro ) : ?>
                <div class="notice notice-info">
                    <p>
                        <?php esc_html_e(
                            'The Role Risk Analyzer runs as a Pro feature. You can configure it here, but scans require an active Pro license.',
                            'aegisshield-security'
                        ); ?>
                    </p>
                </div>
            <?php endif; ?>

            <?php if ( $scan_ran ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'Role risk scan completed. See the report below.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <form method="post">
                <?php
                wp_nonce_field(
                    'aegisshield_hardening_roles_save',
                    'aegisshield_hardening_roles_nonce'
                );
                ?>

                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th scope="row">
                                <?php esc_html_e( 'Enable analyzer', 'aegisshield-security' ); ?>
                            </th>
                            <td>
                                <label>
                                    <input type="checkbox"
                                           name="hardening_role_risk_mode"
                                        <?php checked( $mode, 'on' ); ?>
                                        <?php disabled( ! $is_pro ); ?> />
                                    <?php esc_html_e(
                                        'Track and flag high-privilege accounts and risky roles.',
                                        'aegisshield-security'
                                    ); ?>
                                </label>
                                <p class="description">
                                    <?php esc_html_e(
                                        'No changes are made automatically. AegisShield only highlights accounts for your review.',
                                        'aegisshield-security'
                                    ); ?>
                                </p>
                            </td>
                        </tr>
                    </tbody>
                </table>

                <p>
                    <?php submit_button(
                        __( 'Save Settings', 'aegisshield-security' ),
                        'secondary',
                        'submit',
                        false
                    ); ?>

                    <button type="submit"
                            name="run_role_risk_scan_now"
                            class="button button-primary"
                        <?php disabled( ! $is_pro ); ?>>
                        <?php esc_html_e( 'Run Role Risk Analyzer Now', 'aegisshield-security' ); ?>
                    </button>
                </p>
            </form>

            <hr />

            <h3><?php esc_html_e( 'Role Risk Report', 'aegisshield-security' ); ?></h3>

            <?php if ( empty( $results ) ) : ?>
                <p class="description">
                    <?php esc_html_e(
                        'No risky accounts have been flagged yet. Run a scan to populate this report.',
                        'aegisshield-security'
                    ); ?>
                </p>
                <?php
                return;
            endif;
            ?>

            <table class="widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php esc_html_e( 'User', 'aegisshield-security' ); ?></th>
                        <th><?php esc_html_e( 'Role(s)', 'aegisshield-security' ); ?></th>
                        <th><?php esc_html_e( 'Dangerous capabilities', 'aegisshield-security' ); ?></th>
                        <th><?php esc_html_e( 'Last login / activity', 'aegisshield-security' ); ?></th>
                        <th><?php esc_html_e( 'Risk', 'aegisshield-security' ); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ( $results as $row ) : ?>
                        <?php
                        $username  = isset( $row['username'] ) ? $row['username'] : '';
                        $role      = isset( $row['role'] ) ? $row['role'] : '';
                        $caps      = isset( $row['dangerous_caps'] ) && is_array( $row['dangerous_caps'] )
                            ? $row['dangerous_caps']
                            : array();
                        $last_login = isset( $row['last_login'] ) ? (int) $row['last_login'] : 0;
                        $risk       = isset( $row['risk'] ) ? $row['risk'] : 'low';
                        ?>
                        <tr>
                            <td>
                                <?php echo esc_html( $username ); ?>
                            </td>
                            <td>
                                <?php echo esc_html( $role ); ?>
                            </td>
                            <td>
                                <?php
                                if ( ! empty( $caps ) ) {
                                    echo esc_html( implode( ', ', $caps ) );
                                } else {
                                    esc_html_e( '—', 'aegisshield-security' );
                                }
                                ?>
                            </td>
                            <td>
                                <?php
                                if ( $last_login > 0 ) {
                                    echo esc_html(
                                        wp_date(
                                            get_option( 'date_format' ) . ' ' . get_option( 'time_format' ),
                                            $last_login
                                        )
                                    );
                                } else {
                                    esc_html_e( 'Unknown', 'aegisshield-security' );
                                }
                                ?>
                            </td>
                            <td>
                                <?php echo esc_html( ucfirst( $risk ) ); ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php
    }
}