<?php
namespace AegisShield\Admin_Pages\Malware;

use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

/**
 * Malware Scan – Scheduled Scans (Pro) tab.
 *
 * Lets Pro users enable lightweight daily quick scans and weekly full scans.
 */
class AS_Page_Malware_Tab_Scheduler {

    /**
     * Render the tab content.
     *
     * @param AS_Plugin $plugin Plugin instance.
     * @return void
     */
    public static function render_tab( AS_Plugin $plugin ) {
        $settings = $plugin->get_settings();
        $section  = 'malware';

        $is_pro = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();

        // Handle Pro-only settings save for scheduled scans.
        if (
            $is_pro
            && isset( $_POST['aegisshield_malware_schedule_nonce'] )
            && wp_verify_nonce(
                sanitize_text_field( wp_unslash( $_POST['aegisshield_malware_schedule_nonce'] ) ),
                'aegisshield_malware_schedule_save'
            )
        ) {
            $quick_enabled  = isset( $_POST['malware_schedule_quick_enabled'] ) ? 'on' : '';
            $weekly_enabled = isset( $_POST['malware_schedule_weekly_enabled'] ) ? 'on' : '';

            $settings->set( $section, 'schedule_quick_enabled', $quick_enabled );
            $settings->set( $section, 'schedule_weekly_enabled', $weekly_enabled );
            $settings->save();

            ?>
            <div class="notice notice-success is-dismissible">
                <p><?php esc_html_e( 'Scheduled malware scan settings saved.', 'aegisshield-security' ); ?></p>
            </div>
            <?php
        }

        $quick_enabled  = (string) $settings->get( $section, 'schedule_quick_enabled', '' );
        $weekly_enabled = (string) $settings->get( $section, 'schedule_weekly_enabled', '' );

        $last_quick = (int) $settings->get( $section, 'last_quick_scan', 0 );
        $last_full  = (int) $settings->get( $section, 'last_full_scan', 0 );
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Malware Scan – Scheduled Scans', 'aegisshield-security' ); ?></h1>

            <p>
                <?php esc_html_e( 'Configure automatic malware scans so AegisShield can run in the background and alert you to new threats.', 'aegisshield-security' ); ?>
            </p>

            <?php if ( ! $is_pro ) : ?>
                <p class="description">
                    <?php
                    printf(
                        /* translators: %s: License & Upgrades page link. */
                        esc_html__( 'Scheduled malware scans are available in AegisShield Pro. Activate your license on the %s page to unlock this feature.', 'aegisshield-security' ),
                        '<a href="' . esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ) . '">' . esc_html__( 'License &amp; Upgrades', 'aegisshield-security' ) . '</a>'
                    );
                    ?>
                </p>
            <?php else : ?>
                <form method="post">
                    <?php wp_nonce_field( 'aegisshield_malware_schedule_save', 'aegisshield_malware_schedule_nonce' ); ?>

                    <table class="form-table" role="presentation">
                        <tr>
                            <th scope="row">
                                <label for="malware_schedule_quick_enabled">
                                    <?php esc_html_e( 'Daily quick scan (incremental)', 'aegisshield-security' ); ?>
                                </label>
                            </th>
                            <td>
                                <label>
                                    <input type="checkbox" id="malware_schedule_quick_enabled" name="malware_schedule_quick_enabled" <?php checked( $quick_enabled, 'on' ); ?> />
                                    <?php esc_html_e( 'Run a lightweight quick scan approximately once per day, focused on recently changed files.', 'aegisshield-security' ); ?>
                                </label>
                                <p class="description">
                                    <?php esc_html_e( 'Quick scans use the File Monitor baseline so only new and modified files are rescanned, keeping resource usage low on shared hosting.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="malware_schedule_weekly_enabled">
                                    <?php esc_html_e( 'Weekly full scan', 'aegisshield-security' ); ?>
                                </label>
                            </th>
                            <td>
                                <label>
                                    <input type="checkbox" id="malware_schedule_weekly_enabled" name="malware_schedule_weekly_enabled" <?php checked( $weekly_enabled, 'on' ); ?> />
                                    <?php esc_html_e( 'Run a full malware scan roughly once per week.', 'aegisshield-security' ); ?>
                                </label>
                                <p class="description">
                                    <?php esc_html_e( 'Weekly scans walk all selected directories and may take longer, but provide deeper coverage.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                    </table>

                    <?php submit_button( __( 'Save scheduled scan settings', 'aegisshield-security' ) ); ?>
                </form>

                <h2><?php esc_html_e( 'Last automatic scans', 'aegisshield-security' ); ?></h2>
                <ul>
                    <li>
                        <?php
                        esc_html_e( 'Last quick scan: ', 'aegisshield-security' );
                        echo $last_quick ? esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $last_quick ) ) : esc_html__( 'never', 'aegisshield-security' );
                        ?>
                    </li>
                    <li>
                        <?php
                        esc_html_e( 'Last weekly full scan: ', 'aegisshield-security' );
                        echo $last_full ? esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $last_full ) ) : esc_html__( 'never', 'aegisshield-security' );
                        ?>
                    </li>
                </ul>
            <?php endif; ?>
        </div>
        <?php
    }
}
