<?php
namespace AegisShield;

defined( 'ABSPATH' ) || exit;

/**
 * Notification handler for AegisShield.
 */
class AS_Notifier {

    public function __construct() {
        // Attach SMTP configuration if enabled
        add_action( 'phpmailer_init', array( $this, 'configure_smtp' ) );
    }

    /**
     * Configure PHPMailer using AegisShield SMTP settings.
     *
     * @param \PHPMailer\PHPMailer\PHPMailer $phpmailer
     */
    public function configure_smtp( $phpmailer ) {

        $settings = get_option( 'aegisshield_mail_settings', array() );

        if ( empty( $settings['smtp_enabled'] ) ) {
            return;
        }

        if ( empty( $settings['smtp_host'] ) || empty( $settings['smtp_port'] ) ) {
            return;
        }

        $phpmailer->isSMTP();
        $phpmailer->Host = $settings['smtp_host'];
        $phpmailer->Port = (int) $settings['smtp_port'];

        // Authentication
        if ( ! empty( $settings['smtp_username'] ) && ! empty( $settings['smtp_password'] ) ) {
            $phpmailer->SMTPAuth = true;
            $phpmailer->Username = $settings['smtp_username'];
            $phpmailer->Password = $settings['smtp_password'];
        } else {
            $phpmailer->SMTPAuth = false;
        }

        // Encryption
        if ( ! empty( $settings['smtp_encryption'] ) ) {
            if ( in_array( $settings['smtp_encryption'], array( 'ssl', 'tls' ), true ) ) {
                $phpmailer->SMTPSecure = $settings['smtp_encryption'];
            }
        }

        // From address
        if ( ! empty( $settings['smtp_from_email'] ) && is_email( $settings['smtp_from_email'] ) ) {
            $from_name = ! empty( $settings['smtp_from_name'] )
                ? $settings['smtp_from_name']
                : get_bloginfo( 'name' );

            $phpmailer->setFrom( $settings['smtp_from_email'], $from_name, false );
        }

        // Hardening / compatibility
        $phpmailer->CharSet  = 'UTF-8';
        $phpmailer->Encoding = 'base64';
    }

    /**
     * Backward-compatible wrapper.
     */
    public function send_email( $subject, $message, $headers = array() ) {
        return $this->send_email_to(
            array( get_option( 'admin_email' ) ),
            $subject,
            $message,
            $headers
        );
    }

    /**
     * Send an email to one or more recipients.
     *
     * @param array|string $emails
     * @param string       $subject
     * @param string       $message
     * @param array        $headers
     *
     * @return bool
     */
	public function send_email_to( $emails, $subject, $message, $headers = array() ) {

		// -----------------------------------------
		// Prefer Global Alert Recipients (Central)
		// -----------------------------------------
		$mail_settings = get_option( 'aegisshield_mail_settings', array() );

		if ( ! empty( $mail_settings['alert_recipients'] ) ) {
			$emails = array_map(
				'trim',
				explode( ',', $mail_settings['alert_recipients'] )
			);
		}

		// -----------------------------------------
		// Existing sanitization logic (unchanged)
		// -----------------------------------------
		$emails = is_array( $emails ) ? $emails : array( $emails );
		$emails = array_values(
			array_filter(
				array_map( 'sanitize_email', $emails ),
				'is_email'
			)
		);

		if ( empty( $emails ) ) {
			return false;
		}

		$defaults = array( 'Content-Type: text/html; charset=UTF-8' );
		$headers  = array_merge( $defaults, (array) $headers );

		$ok = true;

		foreach ( $emails as $email ) {
			$sent = wp_mail( $email, $subject, $message, $headers );
			if ( ! $sent ) {
				$ok = false;
			}
		}

		return $ok;
	}

    /**
     * Standardized email formatting.
     */
    public function format_email( $title, $body ) {

        $style = 'font-family:-apple-system,BlinkMacSystemFont,Segoe UI,Roboto,Oxygen,Ubuntu,Cantarell,Open Sans,Helvetica Neue,sans-serif; font-size:14px; color:#222;';
        $title = esc_html( $title );
        $body  = wp_kses_post( $body );

        return '<div style="' . esc_attr( $style ) . '">'
            . '<h2>' . $title . '</h2>'
            . '<p>' . $body . '</p>'
            . '<p style="font-size:12px;color:#888;">'
            . esc_html__( 'This alert was generated by AegisShield Security.', 'aegisshield-security' )
            . '</p>'
            . '</div>';
    }
}
