<?php
namespace AegisShield;

use AegisShield\Modules\AS_Module_Interface;
use AegisShield\Modules\AS_Module_Activity_Log;
use AegisShield\Modules\AS_Module_Login_Guard;
use AegisShield\Modules\AS_Module_File_Integrity;
use AegisShield\Modules\AS_Module_Hardening;
use AegisShield\Modules\AS_Module_Sec_Headers;
use AegisShield\Modules\AS_Module_Malware;
use AegisShield\Modules\AS_Module_DB_Tools;
use AegisShield\Modules\AS_Module_File_Monitor;
use AegisShield\AS_License_Manager;

defined( 'ABSPATH' ) || exit;

/**
 * Main plugin controller.
 */
class AS_Plugin {

    protected static $instance;

    protected $loader;
    protected $settings;
    protected $logger;
    protected $cron;
    protected $notifier;
    protected $license_manager;
    protected $modules = array();

    public static function instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
            self::$instance->init();
        }
        return self::$instance;
    }

    protected function init() {
        $this->loader          = new AS_Loader();
        $this->settings        = new AS_Settings();
        $this->logger          = new AS_Logger();
        $this->cron            = new AS_Cron();
        $this->notifier        = new AS_Notifier();
        $this->license_manager = new AS_License_Manager( $this );
		// Safety: ensure required tables exist (handles migrations / failed activation).
		add_action( 'admin_init', array( $this, 'ensure_tables_exist' ) );

        $admin_ui = new AS_Admin_UI( $this );
        $admin_ui->init();

        // -------------------------------------------------
        // Register core modules
        // -------------------------------------------------
        $this->register_modules();

        // -------------------------------------------------
        // Load QR library (biohzrdmx/qrc-php) defensively
        // -------------------------------------------------
        if ( defined( 'AEGISSHIELD_PLUGIN_PATH' ) ) {
            $qr_autoload = AEGISSHIELD_PLUGIN_PATH . 'includes/lib/qrc-php/autoload.php';
        } else {
            // Fallback safety (should never be needed, but prevents fatal)
            $qr_autoload = plugin_dir_path( dirname( __FILE__ ) ) . 'includes/lib/qrc-php/autoload.php';
        }

        if ( file_exists( $qr_autoload ) ) {
            require_once $qr_autoload;
        }

        // -------------------------------------------------
        // Register MFA-related shortcodes (thin adapters)
        // -------------------------------------------------
        \AegisShield\Helpers\AS_Login_MFA_Shortcode::register();
        \AegisShield\Helpers\AS_Register_MFA_Shortcode::register();

        $this->init_modules();

        $this->cron->init();
        $this->loader->run();
    }

    protected function register_modules() {
        $activity_log   = new AS_Module_Activity_Log( $this );
        $login_guard    = new AS_Module_Login_Guard( $this );
        $file_integrity = new AS_Module_File_Integrity( $this );
        $hardening      = new AS_Module_Hardening( $this );
        $sec_headers    = new AS_Module_Sec_Headers( $this );
        $malware        = new AS_Module_Malware( $this );
        $db_tools       = new AS_Module_DB_Tools( $this );
        $file_monitor   = new AS_Module_File_Monitor( $this );

        $modules = array(
            $activity_log,
            $login_guard,
            $file_integrity,
            $hardening,
            $sec_headers,
            $malware,
            $db_tools,
            $file_monitor,
        );

        foreach ( $modules as $module ) {
            if ( $module instanceof AS_Module_Interface ) {
                $this->modules[ $module->get_slug() ] = $module;
                $module->register_settings();
                $module->init();
            }
        }
    }

    public function get_settings() {
        return $this->settings;
    }

    public function get_logger() {
        return $this->logger;
    }

    public function get_notifier() {
        return $this->notifier;
    }

    public function get_license_manager() {
        return $this->license_manager;
    }

    public function get_loader() {
        return $this->loader;
    }

    public static function activate_static() {
        $instance = new self();
        $instance->create_tables();

        $cron = new AS_Cron();
        $cron->schedule_events();
    }

    public function init_modules() {
        // existing module inits…
        \AegisShield\Helpers\AS_MFA_Test_Form::init();
    }

    protected function create_tables() {
        global $wpdb;
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $charset_collate = $wpdb->get_charset_collate();

        $activity_table = $wpdb->prefix . 'aegisshield_activity_log';
        $login_table    = $wpdb->prefix . 'aegisshield_login_attempts';
        $file_table     = $wpdb->prefix . 'aegisshield_file_hashes';
		$file_monitor_table = $wpdb->prefix . 'aegisshield_file_monitor';

		$sql_file_monitor = "CREATE TABLE {$file_monitor_table} (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			file_path VARCHAR(255) NOT NULL,
			hash VARCHAR(64) NOT NULL,
			size BIGINT UNSIGNED NOT NULL DEFAULT 0,
			mtime BIGINT UNSIGNED NOT NULL DEFAULT 0,
			first_seen DATETIME NOT NULL,
			last_seen DATETIME NOT NULL,
			status VARCHAR(20) DEFAULT 'unchanged',
			PRIMARY KEY  (id),
			KEY status (status),
			KEY mtime (mtime)
		) {$charset_collate};";
		
        $sql_activity = "CREATE TABLE {$activity_table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            event_time DATETIME NOT NULL,
            user_id BIGINT UNSIGNED DEFAULT 0,
            ip_address VARCHAR(45) DEFAULT '',
            event_type VARCHAR(50) DEFAULT '',
            message TEXT,
            context LONGTEXT,
            PRIMARY KEY  (id),
            KEY event_time (event_time),
            KEY event_type (event_type)
        ) {$charset_collate};";

        $sql_login = "CREATE TABLE {$login_table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            ip_address VARCHAR(45) NOT NULL,
            attempts INT UNSIGNED NOT NULL DEFAULT 0,
            last_attempt DATETIME NOT NULL,
            locked_until DATETIME DEFAULT NULL,
            PRIMARY KEY  (id),
            UNIQUE KEY ip_address (ip_address),
            KEY locked_until (locked_until)
        ) {$charset_collate};";

        $sql_file = "CREATE TABLE {$file_table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            path VARCHAR(255) NOT NULL,
            hash VARCHAR(64) NOT NULL,
            last_scanned DATETIME NOT NULL,
            status VARCHAR(20) DEFAULT 'clean',
            PRIMARY KEY  (id),
            UNIQUE KEY path (path),
            KEY status (status)
        ) {$charset_collate};";

        dbDelta( $sql_activity );
        dbDelta( $sql_login );
        dbDelta( $sql_file );
		dbDelta( $sql_file_monitor );
		}
		
		public function ensure_tables_exist() {
			if ( ! current_user_can( 'manage_options' ) ) {
				return;
			}
			$this->create_tables(); // dbDelta is safe to run repeatedly.
		}
}
