<?php
namespace AegisShield\Modules;

use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

/**
 * Activity Log module.
 */
class AS_Module_Activity_Log implements AS_Module_Interface {

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function get_slug() {
        return 'activity_log';
    }

    public function register_settings() {
        $settings = $this->plugin->get_settings();
        if ( null === $settings->get( 'activity_log', 'retention_days', null ) ) {
            $settings->set( 'activity_log', 'retention_days', 30 );
            $settings->save();
        }
    }

    public function init() {
        add_action( 'wp_login', array( $this, 'log_login_success' ), 10, 2 );
        add_action( 'wp_logout', array( $this, 'log_logout' ) );
        add_action( 'wp_login_failed', array( $this, 'log_login_failed' ) );
        add_action( 'activated_plugin', array( $this, 'log_plugin_activated' ) );
        add_action( 'deactivated_plugin', array( $this, 'log_plugin_deactivated' ) );
        add_action( 'upgrader_process_complete', array( $this, 'log_plugin_install_update' ), 10, 2 );
        add_action( 'user_register', array( $this, 'log_user_created' ) );
        add_action( 'set_user_role', array( $this, 'log_user_role_changed' ), 10, 3 );
    }

    public function log_login_success( $user_login, $user ) {
        $logger = $this->plugin->get_logger();

        if ( ! $logger || ! is_object( $logger ) || ! method_exists( $logger, 'log' ) ) {
            self::log_violation(
                $this->plugin,
                'activity_log',
                'logger_unavailable',
                'Logger unavailable; login_success could not be recorded.',
                array(
                    'method'     => __FUNCTION__,
                    'user_login' => (string) $user_login,
                    'user_id'    => is_object( $user ) && isset( $user->ID ) ? (int) $user->ID : 0,
                ),
                false,
                'failed'
            );
            return;
        }

        if ( ! is_object( $user ) || ! isset( $user->ID ) ) {
            self::log_violation(
                $this->plugin,
                'activity_log',
                'invalid_user_object',
                'Invalid user object; login_success skipped.',
                array(
                    'method'     => __FUNCTION__,
                    'user_login' => (string) $user_login,
                ),
                false,
                'failed'
            );
            return;
        }

        // Phase 1 Trust Event: login from a new IP address.
        $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';


        // Phase 1 Trust Event: login from a new IP address.
        $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        $prev_ip = (string) get_user_meta( $user->ID, 'aegisshield_last_login_ip', true );
        if ( $ip && $prev_ip && $prev_ip !== $ip ) {
            $logger->log(
                'login_new_ip',
                sprintf(
                    /* translators: 1: username, 2: old IP, 3: new IP */
                    __( 'User %1$s logged in from a new IP (%2$s → %3$s).', 'aegisshield-security' ),
                    $user_login,
                    $prev_ip,
                    $ip
                ),
                array( 'user_id' => $user->ID, 'old_ip' => $prev_ip, 'new_ip' => $ip )
            );
        }
        if ( $ip ) {
            update_user_meta( $user->ID, 'aegisshield_last_login_ip', $ip );
        }
        $logger->log(
            'login_success',
            sprintf(
                /* translators: %s: username */
                __( 'User %s logged in successfully.', 'aegisshield-security' ),
                $user_login
            ),
            array( 'user_id' => $user->ID )
        );
    }

    public function log_logout() {
        $logger  = $this->plugin->get_logger();
        $user_id = get_current_user_id();
        if ( $user_id ) {
            $user = get_userdata( $user_id );
            $logger->log(
                'logout',
                sprintf(
                    /* translators: %s: username */
                    __( 'User %s logged out.', 'aegisshield-security' ),
                    $user ? $user->user_login : '#' . $user_id
                ),
                array( 'user_id' => $user_id )
            );
        }
    }

    public function log_login_failed( $username ) {
        $logger = $this->plugin->get_logger();
        $logger->log(
            'login_failed',
            sprintf(
                /* translators: %s: username */
                __( 'Failed login attempt for username %s.', 'aegisshield-security' ),
                $username
            ),
            array( 'username' => $username )
        );
    }

    public function log_plugin_activated( $plugin ) {
        $logger = $this->plugin->get_logger();
        $logger->log(
            'plugin_activated',
            sprintf(
                /* translators: %s: plugin path */
                __( 'Plugin activated: %s', 'aegisshield-security' ),
                $plugin
            )
        );
    }

    public function log_plugin_deactivated( $plugin ) {
        $logger = $this->plugin->get_logger();
        $logger->log(
            'plugin_deactivated',
            sprintf(
                /* translators: %s: plugin path */
                __( 'Plugin deactivated: %s', 'aegisshield-security' ),
                $plugin
            )
        );
    }

    /**
     * Phase 1 Trust Event: User account created (admin or otherwise).
     */
    public function log_user_created( $user_id ) {
        $user = get_userdata( $user_id );
        if ( ! $user ) {
            return;
        }

        $logger = $this->plugin->get_logger();
        $roles  = is_array( $user->roles ) ? $user->roles : array();

        $logger->log(
            'user_created',
            sprintf(
                /* translators: %s: username */
                __( 'New user account created: %s.', 'aegisshield-security' ),
                $user->user_login
            ),
            array(
                'user_id'   => $user_id,
                'username'  => $user->user_login,
                'email'     => $user->user_email,
                'roles'     => $roles,
                'is_admin'  => in_array( 'administrator', $roles, true ),
            )
        );
    }

    /**
     * Phase 1 Trust Event: User role changed (privilege escalation / reduction).
     *
     * @param int    $user_id
     * @param string $role
     * @param array  $old_roles
     */
    public function log_user_role_changed( $user_id, $role, $old_roles ) {
        $user = get_userdata( $user_id );
        if ( ! $user ) {
            return;
        }

        $logger    = $this->plugin->get_logger();
        $old_roles = is_array( $old_roles ) ? $old_roles : array();
        $new_roles = is_array( $user->roles ) ? $user->roles : array();

        $logger->log(
            'user_role_changed',
            sprintf(
                /* translators: 1: username, 2: role */
                __( 'User %1$s role updated to %2$s.', 'aegisshield-security' ),
                $user->user_login,
                $role
            ),
            array(
                'user_id'     => $user_id,
                'username'    => $user->user_login,
                'old_roles'   => $old_roles,
                'new_roles'   => $new_roles,
                'changed_to'  => $role,
                'is_admin_now'=> in_array( 'administrator', $new_roles, true ),
            )
        );
    }

    /**
     * Phase 1 Trust Event: Plugin installed/updated via WordPress upgrader.
     *
     * @param \WP_Upgrader $upgrader
     * @param array        $hook_extra
     */
    public function log_plugin_install_update( $upgrader, $hook_extra ) {
        if ( empty( $hook_extra['type'] ) || 'plugin' !== $hook_extra['type'] ) {
            return;
        }

        $action = isset( $hook_extra['action'] ) ? (string) $hook_extra['action'] : '';
        if ( 'install' !== $action && 'update' !== $action ) {
            return;
        }

        $logger = $this->plugin->get_logger();

        $plugins = array();
        if ( ! empty( $hook_extra['plugins'] ) && is_array( $hook_extra['plugins'] ) ) {
            $plugins = $hook_extra['plugins'];
        } elseif ( ! empty( $hook_extra['plugin'] ) && is_string( $hook_extra['plugin'] ) ) {
            $plugins = array( $hook_extra['plugin'] );
        }

        foreach ( $plugins as $plugin_file ) {
            $plugin_data = function_exists( 'get_plugin_data' ) && file_exists( WP_PLUGIN_DIR . '/' . $plugin_file )
                ? get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin_file, false, false )
                : array();

            $name = isset( $plugin_data['Name'] ) && $plugin_data['Name'] ? $plugin_data['Name'] : $plugin_file;

            if ( 'install' === $action ) {
                $logger->log(
                    'plugin_installed',
                    sprintf(
                        /* translators: %s: plugin name */
                        __( 'Plugin installed: %s.', 'aegisshield-security' ),
                        $name
                    ),
                    array( 'plugin' => $plugin_file, 'plugin_name' => $name )
                );
            } elseif ( 'update' === $action ) {
                $logger->log(
                    'plugin_updated',
                    sprintf(
                        /* translators: %s: plugin name */
                        __( 'Plugin updated: %s.', 'aegisshield-security' ),
                        $name
                    ),
                    array( 'plugin' => $plugin_file, 'plugin_name' => $name )
                );
            }
        }
    }
    /**
     * Shared violation logger used by both module + admin page.
     *
     * This logs:
     * - when enforcement occurs (ex: Free clamp, Pro-locked action blocked)
     * - when something "should have logged" but couldn't (logger missing, invalid inputs)
     *
     * It never fatal-errors; it falls back to error_log if the logger isn't available.
     *
     * @param AS_Plugin $plugin
     * @param string    $feature   High-level feature area (ex: activity_log, alerts, sessions)
     * @param string    $rule      Specific rule/key (ex: pro_locked_export_csv, logger_unavailable)
     * @param string    $details   Human-readable message
     * @param array     $context   Extra structured context for investigation
     * @param bool      $enforced  True if AegisShield actively enforced/blocked/clamped something
     * @param string    $status    One of: violation|enforced|failed|allowed
     * @return void
     */
    public static function log_violation( AS_Plugin $plugin, $feature, $rule, $details, array $context = array(), $enforced = false, $status = 'violation' ) {

        $ip        = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        $request   = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
        $user_id   = function_exists( 'get_current_user_id' ) ? (int) get_current_user_id() : 0;

        $payload = array_merge(
            array(
                'feature'   => (string) $feature,
                'rule'      => (string) $rule,
                'status'    => (string) $status,
                'enforced'  => (bool) $enforced,
                'user_id'   => $user_id,
                'ip'        => $ip,
                'request'   => $request,
            ),
            $context
        );

        // Prefer the internal logger (writes into Activity Log storage).
        $logger = is_object( $plugin ) ? $plugin->get_logger() : null;

        if ( $logger && is_object( $logger ) && method_exists( $logger, 'log' ) ) {
            $logger->log(
                'security_violation',
                sprintf(
                    /* translators: 1: feature, 2: details */
                    __( '[%1$s] %2$s', 'aegisshield-security' ),
                    (string) $feature,
                    (string) $details
                ),
                $payload
            );
            return;
        }

        // Hard fallback: never silently fail.
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log( '[AegisShield] security_violation: ' . wp_json_encode( $payload ) );
    }


}