<?php
namespace AegisShield\Modules;

use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

/**
 * DB Tools module.
 *
 * Provides lightweight database hygiene helpers, warnings, and optional monitoring.
 */
class AS_Module_DB_Tools implements AS_Module_Interface {

    /**
     * Plugin instance.
     *
     * @var AS_Plugin
     */
    protected $plugin;

    /**
     * Constructor.
     *
     * @param AS_Plugin $plugin Plugin instance.
     */
    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Module slug.
     *
     * @return string
     */
    public function get_slug() {
        return 'db_tools';
    }

    /**
     * Register settings for DB tools.
     *
     * Intentionally minimal. Admin must configure options explicitly.
     *
     * @return void
     */
    public function register_settings() {
        // No hard defaults; options are driven by the admin UI.
    }

    /**
     * Initialize hooks.
     *
     * @return void
     */
    public function init() {
        // Weekly safe optimization (if enabled).
        add_action( 'aegisshield_hourly_maintenance', array( $this, 'maybe_run_weekly_optimization' ) );
        // Growth monitoring and optional email notifications (if enabled).
        add_action( 'aegisshield_hourly_maintenance', array( $this, 'maybe_check_growth' ) );
    }

	/**
	 * Single shared logger for DB Tools enforcement + audit.
	 *
	 * IMPORTANT:
	 * - Logs when a feature runs successfully
	 * - Logs when a feature is skipped/disabled
	 * - Logs when a violation is detected OR when an operation is blocked for safety
	 * - Logs errors with context (wpdb errors, user_id, table name, etc.)
	 *
	 * @param string            $event  Event key.
	 * @param string|array      $message_or_context Message string OR context array.
	 * @param array             $context Additional context.
	 * @param string            $level  info|warning|error|critical|debug (stored in context).
	 * @return void
	 */
	public function log_violation( $event, $message_or_context = '', $context = array(), $level = 'info' ) {
		$logger = $this->plugin ? $this->plugin->get_logger() : null;
		if ( ! $logger ) {
			return;
		}

		// Allow signature: log_violation( 'event', array( ...context... ) )
		if ( is_array( $message_or_context ) && empty( $context ) ) {
			$context            = $message_or_context;
			$message_or_context = '';
		}

		if ( ! is_array( $context ) ) {
			$context = array();
		}

		// Always tag this module and severity.
		$context = array_merge(
			array(
				'module' => 'db_tools',
				'level'  => (string) $level,
			),
			$context
		);

		// Attach user_id when available.
		if ( function_exists( 'get_current_user_id' ) && empty( $context['user_id'] ) ) {
			$context['user_id'] = (int) get_current_user_id();
		}

		// Use whatever logger signature your system supports (your code already uses both forms).
		if ( is_string( $message_or_context ) && '' !== $message_or_context ) {
			// (event, message, context)
			$logger->log( $event, $message_or_context, $context );
			return;
		}

		// (event, context)
		$logger->log( $event, $context );
	}

    /**
     * Possibly run weekly optimization on WordPress tables.
     *
     * @return void
     */
    public function maybe_run_weekly_optimization() {
        $settings = $this->plugin->get_settings();
        $section  = 'db_tools';
		
		// Threshold used by DB Tools UI (for logging/metadata and future alignment).
		$threshold_mb = (int) $settings->get( $section, 'growth_threshold_mb', 100 );
		
		$growth_email           = trim( (string) $settings->get( $section, 'growth_email', '' ) );
		$growth_email_addresses = trim( (string) $settings->get( $section, 'growth_email_addresses', '' ) );

		// Build recipients:
		// - If DB Tools overrides exist, use them (do NOT mix in global).
		// - Else use Global recipients.
		// - Else fallback to admin_email as last resort.
		$recipients = array();

		if ( '' !== $growth_email ) {
			$recipients[] = $growth_email;
		}

		if ( '' !== $growth_email_addresses ) {
			$parts = preg_split( '/[\s,;]+/', $growth_email_addresses );
			if ( is_array( $parts ) ) {
				foreach ( $parts as $p ) {
					$p = sanitize_email( $p );
					if ( $p && is_email( $p ) ) {
						$recipients[] = $p;
					}
				}
			}
		}

		$recipients = array_values( array_unique( array_filter( $recipients ) ) );

		if ( empty( $recipients ) ) {
			$notifier = $this->plugin->get_notifier();
			if ( $notifier && method_exists( $notifier, 'get_global_recipients' ) ) {
				$recipients = (array) $notifier->get_global_recipients();
				$recipients = array_values( array_unique( array_filter( $recipients ) ) );
			}
		}

		if ( empty( $recipients ) ) {
			$recipients = array( get_option( 'admin_email' ) );
		}

		$weekly = (string) $settings->get( $section, 'weekly_optimize', '' );
		if ( 'on' !== $weekly ) {
			$this->log_violation(
				'db_tools_weekly_optimize_skipped',
				__( 'Weekly optimization is disabled.', 'aegisshield-security' ),
				array( 'context' => 'auto' ),
				'debug'
			);
			return;
		}

		$this->log_violation(
			'db_tools_weekly_optimize_enabled',
			__( 'Weekly optimization is enabled; evaluating schedule window.', 'aegisshield-security' ),
			array( 'context' => 'auto' ),
			'debug'
		);

        $last = (int) $settings->get( $section, 'last_optimize', 0 );
        $now  = time();

        if ( $last && ( $now - $last ) < WEEK_IN_SECONDS ) {
            return;
        }

        $this->run_optimization( 'auto' );

        $settings->set( $section, 'last_optimize', $now );
        $settings->save();
    }

    /**
     * Run optimization manually from the admin UI.
     *
     * @return void
     */
	public function run_manual_optimization() {
		if ( function_exists( 'current_user_can' ) && ! current_user_can( 'manage_options' ) ) {
			$this->log_violation(
				'db_tools_manual_optimize_blocked',
				__( 'Manual optimization blocked: insufficient permissions.', 'aegisshield-security' ),
				array( 'context' => 'manual', 'reason' => 'permissions' ),
				'critical'
			);
			return;
		}

		$this->log_violation(
			'db_tools_manual_optimize_requested',
			__( 'Manual optimization requested.', 'aegisshield-security' ),
			array( 'context' => 'manual' ),
			'info'
		);

		$this->run_optimization( 'manual' );
	}

    /**
     * Execute OPTIMIZE TABLE on WordPress tables.
     *
     * @param string $context 'manual' or 'auto'.
     * @return void
     */
    protected function run_optimization( $context ) {
        global $wpdb;

        $logger = $this->plugin->get_logger();
        $prefix = $wpdb->prefix;
        $like   = $wpdb->esc_like( $prefix ) . '%';

        $tables = $wpdb->get_col(
            $wpdb->prepare( 'SHOW TABLES LIKE %s', $like )
        );

        if ( empty( $tables ) ) {
            return;
        }

        foreach ( $tables as $table ) {
            // OPTIMIZE TABLE is safe for InnoDB/MyISAM and is non-destructive.
            $result = $wpdb->query( "OPTIMIZE TABLE `{$table}`" ); // phpcs:ignore WordPress.DB

			if ( false === $result ) {
				$this->log_violation(
					'db_tools_optimize_table_failed',
					__( 'OPTIMIZE TABLE failed.', 'aegisshield-security' ),
					array(
						'context'     => $context,
						'table'       => $table,
						'wpdb_error'  => isset( $wpdb->last_error ) ? (string) $wpdb->last_error : '',
					),
					'error'
				);
			} else {
				$this->log_violation(
					'db_tools_optimize_table_ok',
					__( 'OPTIMIZE TABLE succeeded.', 'aegisshield-security' ),
					array(
						'context' => $context,
						'table'   => $table,
					),
					'debug'
				);
			}
        }

		$this->log_violation(
			'db_tools_optimized',
			sprintf(
				/* translators: %s: context (manual/auto). */
				__( 'Database tables optimized (%s).', 'aegisshield-security' ),
				$context
			),
			array(
				'context' => $context,
				'tables'  => $tables,
			),
			'info'
		);
    }

    /**
     * Possibly check table growth and send notifications.
     *
     * @return void
     */
    public function maybe_check_growth() {
        $settings = $this->plugin->get_settings();
        $section  = 'db_tools';

		$monitoring = (string) $settings->get( $section, 'growth_monitoring', '' );
		if ( 'on' !== $monitoring ) {
			$this->log_violation(
				'db_tools_growth_monitoring_skipped',
				__( 'Growth monitoring is disabled.', 'aegisshield-security' ),
				array(),
				'debug'
			);
			return;
		}

		$this->log_violation(
			'db_tools_growth_monitoring_running',
			__( 'Growth monitoring check started.', 'aegisshield-security' ),
			array(),
			'debug'
		);

        $summary = $this->get_table_summary();
        $warnings = isset( $summary['warnings'] ) ? $summary['warnings'] : array();
		if ( empty( $warnings ) ) {
			$this->log_violation(
				'db_tools_growth_monitoring_no_warnings',
				__( 'No unusual database growth detected.', 'aegisshield-security' ),
				array(),
				'debug'
			);
			return;
		}

		// DB Tools overrides (email recipients)
		$growth_email           = trim( (string) $settings->get( $section, 'growth_email', '' ) );
		$growth_email_addresses = trim( (string) $settings->get( $section, 'growth_email_addresses', '' ) );

		// If DB Tools recipients are empty, we will fall back to Global recipients later.
		// But if BOTH are empty AND no global recipients exist, there is nobody to notify → return early.
		if ( '' === $growth_email && '' === $growth_email_addresses ) {
			$notifier = $this->plugin->get_notifier();
			$global   = ( $notifier && method_exists( $notifier, 'get_global_recipients' ) )
				? (array) $notifier->get_global_recipients()
				: array();

			if ( empty( $global ) ) {
				return;
			}
		}

        $last_email = (int) $settings->get( $section, 'last_growth_email', 0 );
        $now        = time();

        // Avoid spamming; at most one email every 12 hours.
        if ( $last_email && ( $now - $last_email ) < 12 * HOUR_IN_SECONDS ) {
            return;
        }

        $tables = isset( $summary['tables'] ) ? $summary['tables'] : array();
		$this->log_violation(
			'db_table_growth_detected',
			__( 'Database table growth detected.', 'aegisshield-security' ),
			array(
				'warning_count' => is_array( $warnings ) ? count( $warnings ) : 0,
			),
			'warning'
		);
        $this->send_growth_email( $warnings, $tables );
		$this->log_violation(
			'db_tools_growth_email_sent',
			__( 'Growth alert email sent.', 'aegisshield-security' ),
			array(),
			'info'
		);

        $settings->set( $section, 'last_growth_email', $now );
        $settings->save();
    }

    /**
     * Send a growth / DB warning email.
     *
     * @param array $warnings Warning messages.
     * @param array $tables   Table summary.
     * @return void
     */
    protected function send_growth_email( array $warnings, array $tables ) {
        $settings = $this->plugin->get_settings();
        $section  = 'db_tools';

        $to_raw = (string) $settings->get( $section, 'growth_email_addresses', '' );
        $recipients = array();

        if ( '' !== $to_raw ) {
            $parts = array_map( 'trim', explode( ',', $to_raw ) );
            foreach ( $parts as $addr ) {
                if ( is_email( $addr ) ) {
                    $recipients[] = $addr;
                }
            }
        }

        if ( empty( $recipients ) ) {
            $admin_email = get_option( 'admin_email' );
            if ( $admin_email && is_email( $admin_email ) ) {
                $recipients[] = $admin_email;
            }
        }

        if ( empty( $recipients ) ) {
            return;
        }

        $site_name = wp_specialchars_decode( get_bloginfo( 'name' ), ENT_QUOTES );
        $subject   = sprintf(
            /* translators: %s: site name. */
            __( '[AegisShield] Database growth warning on %s', 'aegisshield-security' ),
            $site_name
        );

        // Build a brief body with warnings and top heavy tables.
        $body_lines   = array();
        $body_lines[] = __( 'AegisShield DB Tools has detected significant database table growth or other warnings:', 'aegisshield-security' );
        $body_lines[] = '';
        foreach ( $warnings as $warning ) {
            $body_lines[] = ' - ' . wp_strip_all_tags( $warning );
        }

        $body_lines[] = '';
        $body_lines[] = __( 'Largest tables (by total size):', 'aegisshield-security' );

        // Sort tables by total size descending.
        usort(
            $tables,
            function ( $a, $b ) {
                $at = isset( $a['total_bytes'] ) ? (int) $a['total_bytes'] : 0;
                $bt = isset( $b['total_bytes'] ) ? (int) $b['total_bytes'] : 0;
                if ( $at === $bt ) {
                    return 0;
                }
                return ( $at > $bt ) ? -1 : 1;
            }
        );

        $max_tables = 5;
        $count      = 0;
        foreach ( $tables as $table ) {
            $name  = isset( $table['name'] ) ? $table['name'] : '';
            $total = isset( $table['total_bytes'] ) ? (int) $table['total_bytes'] : 0;
            $size  = size_format( $total );
            $body_lines[] = sprintf( ' - %s (%s)', $name, $size );
            $count++;
            if ( $count >= $max_tables ) {
                break;
            }
        }

        $body_lines[] = '';
        $body_lines[] = __( 'You can review detailed DB statistics in AegisShield → DB Tools.', 'aegisshield-security' );

		$body = implode( "\n", $body_lines );

        // Phase 2: Always log DB table growth findings; notification handled via Alerts Pro rules.
		// Send the email (prefer notifier if available).
		$headers  = array( 'Content-Type: text/plain; charset=UTF-8' );
		$notifier = $this->plugin->get_notifier();

		if ( $notifier && method_exists( $notifier, 'send_email_to' ) ) {
			$notifier->send_email_to( $recipients, $subject, $body, $headers );
		} else {
			wp_mail( implode( ',', $recipients ), $subject, $body, $headers );
		}
		
        $this->plugin->get_logger()->log(
            'db_table_growth_detected',
            __( 'Database table growth detected.', 'aegisshield-security' ),
            'medium',
			array(
				'top_tables'    => $body_lines,
				'threshold_mb'  => $threshold_mb,
				'recipients'    => count( $recipients ),
			)
        );
    }

    /**
     * Get a summary of WordPress tables including size and engine.
     *
     * Also records a snapshot for growth comparisons and warnings.
     *
     * @return array
     */
    public function get_table_summary() {
        global $wpdb;

        $settings = $this->plugin->get_settings();
        $section  = 'db_tools';

        $prefix = $wpdb->prefix;
        $like   = $wpdb->esc_like( $prefix ) . '%';

        $status_rows = $wpdb->get_results(
            $wpdb->prepare( 'SHOW TABLE STATUS LIKE %s', $like ),
            ARRAY_A
        );

        $previous_snapshot = $settings->get( $section, 'last_table_snapshot', array() );
        if ( ! is_array( $previous_snapshot ) ) {
            $previous_snapshot = array();
        }

        $snapshot = array();
        $tables   = array();
        $spikes   = array();

        if ( $status_rows ) {
            foreach ( $status_rows as $row ) {
                $name   = isset( $row['Name'] ) ? $row['Name'] : '';
                $engine = isset( $row['Engine'] ) ? $row['Engine'] : '';
                $rows   = isset( $row['Rows'] ) ? (int) $row['Rows'] : 0;
                $data   = isset( $row['Data_length'] ) ? (int) $row['Data_length'] : 0;
                $index  = isset( $row['Index_length'] ) ? (int) $row['Index_length'] : 0;

                $total = $data + $index;

                $prev_size = 0;
                if ( isset( $previous_snapshot[ $name ]['size'] ) ) {
                    $prev_size = (int) $previous_snapshot[ $name ]['size'];
                }

                $delta     = $total - $prev_size;
                $delta_pct = ( $prev_size > 0 && $total > 0 ) ? ( ( $total / $prev_size ) - 1 ) : 0.0;

                if ( $prev_size > 0 && $delta > 5 * 1024 * 1024 && $delta_pct >= 1.0 ) {
                    // Grew by more than ~100% and at least 5MB.
                    $spikes[] = $name;
                }

                $tables[] = array(
                    'name'        => $name,
                    'engine'      => $engine,
                    'rows'        => $rows,
                    'data_bytes'  => $data,
                    'index_bytes' => $index,
                    'total_bytes' => $total,
                    'delta_bytes' => $delta,
                    'delta_pct'   => $delta_pct,
                );

                $snapshot[ $name ] = array(
                    'size' => $total,
                    'rows' => $rows,
                );
            }
        }

        // Save new snapshot for future comparisons.
        $settings->set( $section, 'last_table_snapshot', $snapshot );
        $settings->save();

        $warnings = array();

        // Warn if prefix is still default wp_.
        if ( 'wp_' === $prefix ) {
            $warnings[] = __( 'Your database table prefix is still the default "wp_". Consider using a custom prefix for a small hardening benefit.', 'aegisshield-security' );
        }

        // Warn on significant growth spikes.
        if ( ! empty( $spikes ) ) {
            /* translators: %s: comma-separated table names. */
            $warnings[] = sprintf(
                __( 'The following tables have grown significantly since the last snapshot: %s. This may indicate log or cache bloat.', 'aegisshield-security' ),
                implode( ', ', $spikes )
            );
        }

        return array(
            'tables'   => $tables,
            'warnings' => $warnings,
        );
    }


    /**
     * Phase 1: Create a timestamped SQL backup snapshot (file export).
     * This does NOT modify database tables or prefixes.
     *
     * @return array {success:bool, message:string, file?:string}
     */
    public function create_db_backup_snapshot() {
        global $wpdb;

        $dir = $this->get_db_tools_backup_dir();
        if ( empty( $dir ) || ! is_dir( $dir ) || ! is_writable( $dir ) ) {
            return array(
                'success' => false,
                'message' => __( 'Backup directory is not writable.', 'aegisshield-security' ),
            );
        }

        $timestamp = gmdate( 'Y-m-d_H-i-s' );
        $db_name   = defined( 'DB_NAME' ) ? (string) DB_NAME : 'database';
        $prefix    = isset( $wpdb->prefix ) ? (string) $wpdb->prefix : '';

        $safe_db   = preg_replace( '/[^A-Za-z0-9_\-]/', '_', $db_name );
        $safe_pref = preg_replace( '/[^A-Za-z0-9_\-]/', '_', $prefix );

        $filename = 'aegisshield-db-backup-' . $safe_db . '-' . $safe_pref . '-' . $timestamp . '.sql';
        $path     = trailingslashit( $dir ) . $filename;

        $fh = @fopen( $path, 'w' );
        if ( ! $fh ) {
            return array(
                'success' => false,
                'message' => __( 'Unable to create backup file.', 'aegisshield-security' ),
            );
        }

        @fwrite( $fh, "-- AegisShield DB Tools Backup\n" );
        @fwrite( $fh, '-- Created (UTC): ' . gmdate( 'c' ) . "\n" );
        @fwrite( $fh, '-- Database: ' . $safe_db . "\n" );
        @fwrite( $fh, '-- Prefix: ' . $safe_pref . "\n\n" );

        // Export tables.
        $tables = $wpdb->get_col( 'SHOW TABLES' );
        if ( empty( $tables ) ) {
            @fclose( $fh );
            @unlink( $path );
            return array(
                'success' => false,
                'message' => __( 'No tables found to back up.', 'aegisshield-security' ),
            );
        }

        foreach ( $tables as $table ) {
            $table = (string) $table;
            if ( '' === $table ) {
                continue;
            }

            // Structure.
            $create_row = $wpdb->get_row( 'SHOW CREATE TABLE `' . esc_sql( $table ) . '`', ARRAY_N );
            if ( empty( $create_row ) || empty( $create_row[1] ) ) {
                continue;
            }

            @fwrite( $fh, "\n-- ----------------------------------------\n" );
            @fwrite( $fh, '-- Table: ' . $table . "\n" );
            @fwrite( $fh, "DROP TABLE IF EXISTS `{$table}`;\n" );
            @fwrite( $fh, $create_row[1] . ";\n\n" );

            // Data in chunks to reduce memory.
            $row_count = (int) $wpdb->get_var( 'SELECT COUNT(*) FROM `' . esc_sql( $table ) . '`' );
            if ( $row_count <= 0 ) {
                continue;
            }

            $limit  = 500;
            $offset = 0;

            while ( $offset < $row_count ) {
                $results = $wpdb->get_results(
                    'SELECT * FROM `' . esc_sql( $table ) . '` LIMIT ' . (int) $offset . ',' . (int) $limit,
                    ARRAY_A
                );

                if ( empty( $results ) ) {
                    break;
                }

                foreach ( $results as $row ) {
                    $columns = array();
                    $values  = array();

                    foreach ( $row as $col => $val ) {
                        $columns[] = '`' . str_replace( '`', '``', (string) $col ) . '`';
                        $values[]  = $this->sql_escape_value( $val );
                    }

                    $sql = 'INSERT INTO `' . $table . '` (' . implode( ',', $columns ) . ') VALUES (' . implode( ',', $values ) . ");\n";
                    @fwrite( $fh, $sql );
                }

                $offset += $limit;
            }
        }

        @fclose( $fh );

        $size = @filesize( $path );
        $size_h = $size ? size_format( (int) $size, 2 ) : '';

                // Phase 2: Log backup creation (always logged; alerts controlled elsewhere).
		$this->log_violation(
			'db_backup_created',
			__( 'Database backup snapshot created.', 'aegisshield-security' ),
			array( /* keep your existing context array here */ ),
			'info'
		);

return array(
            'success' => true,
            'message' => sprintf(
                /* translators: 1: file name, 2: size */
                __( 'Backup created: %1$s %2$s', 'aegisshield-security' ),
                $filename,
                $size_h ? '(' . $size_h . ')' : ''
            ),
            'file'    => $path,
        );
    }

    /**
     * Phase 1: List backup files for UI display.
     *
     * @return array[] List of backups.
     */
    public function list_db_backups() {
        $dir = $this->get_db_tools_backup_dir();
        if ( empty( $dir ) || ! is_dir( $dir ) ) {
            return array();
        }

        $files = glob( trailingslashit( $dir ) . '*.sql' );
        if ( empty( $files ) ) {
            return array();
        }

        rsort( $files );

        $out = array();
        foreach ( $files as $path ) {
            $path = (string) $path;
            if ( ! file_exists( $path ) ) {
                continue;
            }

            $out[] = array(
                'date' => gmdate( 'Y-m-d H:i:s', (int) filemtime( $path ) ) . ' UTC',
                'file' => basename( $path ),
                'size' => size_format( (int) filesize( $path ), 2 ),
                'path' => $path,
            );
        }

        return $out;
    }

    /**
     * Get the backup directory path for DB Tools.
     *
     * @return string Absolute directory path.
     */
    protected function get_db_tools_backup_dir() {
        $upload = wp_upload_dir();
        $base   = isset( $upload['basedir'] ) ? (string) $upload['basedir'] : '';
        if ( empty( $base ) ) {
            return '';
        }

        $dir = trailingslashit( $base ) . 'aegisshield-backups/db-tools';

        if ( ! file_exists( $dir ) ) {
            wp_mkdir_p( $dir );
        }

        // Best-effort hardening: prevent browsing.
        $index = trailingslashit( $dir ) . 'index.html';
        if ( ! file_exists( $index ) ) {
            @file_put_contents( $index, '' );
        }

        $htaccess = trailingslashit( $dir ) . '.htaccess';
        if ( ! file_exists( $htaccess ) ) {
            @file_put_contents( $htaccess, "Deny from all\n" );
        }

        return $dir;
    }

    /**
     * Escape a value for SQL insert statements.
     *
     * @param mixed $value Raw value.
     * @return string Escaped SQL literal.
     */
    protected function sql_escape_value( $value ) {
        global $wpdb;

        if ( is_null( $value ) ) {
            return 'NULL';
        }

        if ( is_bool( $value ) ) {
            return $value ? '1' : '0';
        }

        if ( is_int( $value ) || is_float( $value ) ) {
            return (string) $value;
        }

        $str = (string) $value;

        // Use the underlying DB connection when available.
        if ( isset( $wpdb->dbh ) && $wpdb->dbh ) {
            $escaped = mysqli_real_escape_string( $wpdb->dbh, $str );
        } else {
            $escaped = addslashes( $str );
        }

        return "'" . $escaped . "'";
    }


    /**
     * Phase 2 (APPLY): Change WordPress DB prefix safely with rollback.
     *
     * IMPORTANT:
     * - This method performs DB writes (rename tables + update keys).
     * - It attempts rollback on failure, but DB backups are still REQUIRED.
     * - Multisite is intentionally blocked in this Phase 2 implementation.
     *
     * @param array  $tables         Preview table list (ignored for safety; we re-query live DB).
     * @param string $prefix_current Current prefix (e.g., wp_).
     * @param string $prefix_new     New prefix (e.g., cmoua_).
     * @param string $prefix_mode    'core' or 'all'
     * @param string $exec_mode      reserved for future (standard/atomic)
     * @param string $backup_file    backup file selected in UI (must exist)
     *
     * @return array Result {success:bool, message:string, details?:array, manual_wpconfig?:string}
     */
    public function apply_prefix_change( $tables, $prefix_current, $prefix_new, $prefix_mode, $exec_mode, $backup_file = '' ) {
        global $wpdb;

		

        // Phase 2: Normalize legacy variable name.
        $mode = (string) $prefix_mode;
		if ( $mode !== 'all' ) {
			$this->log_violation(
				'db_prefix_change_blocked',
				__( 'Unsafe prefix mode blocked. Only full-prefix migrations are allowed.', 'aegisshield-security' ),
				array(
					'prefix_mode' => (string) $prefix_mode,
					'exec_mode'   => (string) $exec_mode,
				),
				'critical'
			);

			return array(
				'status'  => 'error',
				'message' => 'Unsafe prefix mode blocked. Only full-prefix migrations are allowed.',
			);
		}
		
        // Capability guard (double-check even if page already checks).
        if ( function_exists( 'current_user_can' ) && ! current_user_can( 'manage_options' ) ) {
            $this->plugin->get_logger()->log(
                'db_prefix_change_failed',
                __( 'DB prefix change failed: insufficient permissions.', 'aegisshield-security' ),
                'critical',
                array( 'old_prefix' => (string) $prefix_current, 'new_prefix' => (string) $prefix_new, 'reason' => 'permissions', 'user_id' => get_current_user_id() )
            );
            return array(
                'success' => false,
                'message' => __( 'Insufficient permissions.', 'aegisshield-security' ),
            );
        }

        // Phase 2 safety: block multisite until explicitly implemented/tested.
        if ( function_exists( 'is_multisite' ) && is_multisite() ) {
            return array(
                'success' => false,
                'message' => __( 'DB Prefix change is currently disabled on Multisite installations. (Safety block)', 'aegisshield-security' ),
            );
        }

        // Phase 2: Log prefix change start (always logged; alerts controlled via Alerts Pro).
        $this->plugin->get_logger()->log(
            'db_prefix_change_started',
            __( 'DB prefix change started.', 'aegisshield-security' ),
            'high',
            array(
                'old_prefix' => (string) $prefix_current,
                'new_prefix' => (string) $prefix_new,
                'mode'       => (string) $prefix_mode,
                'exec_mode'  => (string) $exec_mode,
                'backup'     => (string) $backup_file,
                'tables'     => is_array( $tables ) ? count( $tables ) : 0,
                'user_id'    => get_current_user_id(),
            )
        );

        $prefix_current = (string) $prefix_current;
        $prefix_new     = (string) $prefix_new;
        $prefix_mode    = (string) $prefix_mode;

        // Safety: applying a prefix change requires renaming ALL tables using the current prefix.
        // Partial renames ("core" only) can break WordPress and plugins (example: Action Scheduler tables).
        if ( 'all' !== $prefix_mode ) {
            return array(
                'success' => false,
                'message' => __( 'Safety block: Apply is only allowed when Scope is set to "All tables". Preview-only is allowed for Core scope.', 'aegisshield-security' ),
            );
        }

        $validation_error = $this->validate_new_prefix_value( $prefix_new, $prefix_current );
        if ( $validation_error ) {
            return array(
                'success' => false,
                'message' => $validation_error,
            );
        }

        // Require an existing backup file selection (created in Phase 1).
        $backup_file = (string) $backup_file;
        if ( empty( $backup_file ) ) {
            return array(
                'success' => false,
                'message' => __( 'Backup selection is required before applying a prefix change.', 'aegisshield-security' ),
            );
        }

        $backup_dir = $this->get_db_tools_backup_dir();
        $backup_abs = trailingslashit( $backup_dir ) . basename( $backup_file );
        if ( empty( $backup_dir ) || ! file_exists( $backup_abs ) ) {
            return array(
                'success' => false,
                'message' => __( 'Selected backup file was not found. Create a new backup snapshot and try again.', 'aegisshield-security' ),
            );
        }

        // Build table rename map (re-query live DB for safety).
        $like = $wpdb->esc_like( $prefix_current ) . '%';
        $sql  = $wpdb->prepare( 'SHOW TABLES LIKE %s', $like );
        $rows = $wpdb->get_col( $sql );

        if ( empty( $rows ) ) {
            return array(
                'success' => false,
                'message' => __( 'No tables matched the current prefix. Aborting.', 'aegisshield-security' ),
            );
        }

        $core_suffixes = $this->get_wp_core_table_suffixes();
        $rename_map    = array();

        foreach ( $rows as $tbl ) {
            $tbl = (string) $tbl;

            // Only rename tables that truly start with current prefix.
            if ( 0 !== strpos( $tbl, $prefix_current ) ) {
                continue;
            }

            $suffix = substr( $tbl, strlen( $prefix_current ) );

            if ( 'core' === $prefix_mode ) {
                // Only core tables (single-site).
                if ( ! in_array( $suffix, $core_suffixes, true ) ) {
                    continue;
                }
            }

            $rename_map[ $tbl ] = $prefix_new . $suffix;
        }

        if ( empty( $rename_map ) ) {
            return array(
                'success' => false,
                'message' => __( 'No tables qualified for renaming under the selected scope.', 'aegisshield-security' ),
            );
        }

        // Pre-flight: ensure destination tables do not already exist.
        foreach ( $rename_map as $old => $new ) {
            $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $new ) );
            if ( ! empty( $exists ) ) {
                return array(
                    'success' => false,
                    'message' => sprintf(
                        /* translators: 1: destination table */
                        __( 'Destination table already exists: %s. Aborting to avoid collisions.', 'aegisshield-security' ),
                        esc_html( $new )
                    ),
                );
            }
        }

        $results = array(
            'renamed_tables' => array(),
            'options_rows'   => 0,
            'usermeta_rows'  => 0,
            'wpconfig'       => array( 'updated' => false, 'writable' => false, 'path' => '' ),
            'backup_used'    => basename( $backup_abs ),
            'prefix_current' => $prefix_current,
            'prefix_new'     => $prefix_new,
            'prefix_mode'    => $prefix_mode,
        );

        // Rename tables with rollback on error.
        // Preflight: ensure no destination tables already exist.
        $dest_collisions = array();
        foreach ( $rename_map as $old => $new ) {
            $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $new ) );
            if ( ! empty( $exists ) ) {
                $dest_collisions[] = $new;
            }
        }

        if ( ! empty( $dest_collisions ) ) {
            return array(
                'success' => false,
                'message' => sprintf(
                    /* translators: %s: comma-separated table names. */
                    __( 'Safety block: one or more destination tables already exist. This usually means a previous migration partially completed. Existing destination tables: %s', 'aegisshield-security' ),
                    implode( ', ', array_slice( $dest_collisions, 0, 10 ) )
                ),
                'details' => array(
                    'collisions' => $dest_collisions,
                ),
            );
        }

        $renamed_new_to_old = array();

        foreach ( $rename_map as $old => $new ) {
            $q  = 'RENAME TABLE `' . str_replace( '`', '``', $old ) . '` TO `' . str_replace( '`', '``', $new ) . '`';
            $ok = $wpdb->query( $q );

            if ( false === $ok ) {
                // Rollback already renamed tables (best-effort).
                foreach ( array_reverse( $renamed_new_to_old, true ) as $rolled_new => $rolled_old ) {
                    $rq = 'RENAME TABLE `' . str_replace( '`', '``', $rolled_new ) . '` TO `' . str_replace( '`', '``', $rolled_old ) . '`';
                    $wpdb->query( $rq );
                }

                return array(
                    'success' => false,
                    'message' => sprintf(
                        /* translators: 1: old table 2: new table */
                        __( 'Failed renaming table %1$s → %2$s. Rollback attempted. No further changes were applied.', 'aegisshield-security' ),
                        esc_html( $old ),
                        esc_html( $new )
                    ),
                    'details' => $results,
                );
            }

            $renamed_new_to_old[ $new ] = $old;
            $results['renamed_tables'][] = array( 'from' => $old, 'to' => $new );
        }

                // Use explicit table names for updates (do not rely on $wpdb->prefix yet).
        $new_options  = $prefix_new . 'options';
        $new_usermeta = $prefix_new . 'usermeta';

        // Phase 3 requirement: update prefix-bound keys inside options/usermeta (collision-safe).
        // We DO NOT run a single REPLACE/UPDATE statement, because that can create duplicate key errors
        // (example: both chue_force_deactivated_plugins and wp_force_deactivated_plugins exist).
        $opt_table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $new_options ) );
        if ( ! empty( $opt_table_exists ) ) {
            $opt_like = $prefix_current . '%';
            $opt_rows = $wpdb->get_col(
                $wpdb->prepare(
                    'SELECT option_name FROM `' . str_replace( '`', '``', $new_options ) . '` WHERE option_name LIKE %s',
                    $opt_like
                )
            );

            if ( is_array( $opt_rows ) && ! empty( $opt_rows ) ) {
                $updated = 0;
                $skipped = 0;

                foreach ( $opt_rows as $old_key ) {
                    $old_key = (string) $old_key;
                    $suffix  = substr( $old_key, strlen( $prefix_current ) ); // includes everything after old prefix
                    $new_key = $prefix_new . $suffix;

                    if ( $new_key === $old_key ) {
                        continue;
                    }

                    $exists = (int) $wpdb->get_var(
                        $wpdb->prepare(
                            'SELECT COUNT(*) FROM `' . str_replace( '`', '``', $new_options ) . '` WHERE option_name = %s',
                            $new_key
                        )
                    );

                    if ( $exists > 0 ) {
                        // Collision: keep the existing target key, remove the old-prefixed one to avoid duplicates.
                        $wpdb->delete( $new_options, array( 'option_name' => $old_key ), array( '%s' ) );
                        $skipped++;
                        continue;
                    }

                    $ok = $wpdb->update(
                        $new_options,
                        array( 'option_name' => $new_key ),
                        array( 'option_name' => $old_key ),
                        array( '%s' ),
                        array( '%s' )
                    );

                    if ( false === $ok ) {
                        // Rollback already renamed tables (best-effort) before returning.
                        foreach ( array_reverse( $renamed_new_to_old, true ) as $rolled_new => $rolled_old ) {
                            $rq = 'RENAME TABLE `' . str_replace( '`', '``', $rolled_new ) . '` TO `' . str_replace( '`', '``', $rolled_old ) . '`';
                            $wpdb->query( $rq );
                        }
                        return array(
                            'success' => false,
                            'message' => __( 'Failed while updating options prefix-bound keys (collision-safe step).', 'aegisshield-security' ),
                        );
                    }

                    $updated++;
                }

                $results['options_rows'] = (int) $updated;
                if ( $skipped > 0 ) {
                    $results['options_collisions_skipped'] = (int) $skipped;
                }
            }
        }

        $um_table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $new_usermeta ) );
        if ( ! empty( $um_table_exists ) ) {
            $um_like = $prefix_current . '%';
            $um_rows = $wpdb->get_col(
                $wpdb->prepare(
                    'SELECT DISTINCT meta_key FROM `' . str_replace( '`', '``', $new_usermeta ) . '` WHERE meta_key LIKE %s',
                    $um_like
                )
            );

            if ( is_array( $um_rows ) && ! empty( $um_rows ) ) {
                $updated = 0;
                $skipped = 0;

                foreach ( $um_rows as $old_key ) {
                    $old_key = (string) $old_key;
                    $suffix  = substr( $old_key, strlen( $prefix_current ) );
                    $new_key = $prefix_new . $suffix;

                    if ( $new_key === $old_key ) {
                        continue;
                    }

                    $exists = (int) $wpdb->get_var(
                        $wpdb->prepare(
                            'SELECT COUNT(*) FROM `' . str_replace( '`', '``', $new_usermeta ) . '` WHERE meta_key = %s',
                            $new_key
                        )
                    );

                    if ( $exists > 0 ) {
                        // Collision: keep target key; delete old-prefixed rows to avoid duplicated capabilities keys.
                        $wpdb->delete( $new_usermeta, array( 'meta_key' => $old_key ), array( '%s' ) );
                        $skipped++;
                        continue;
                    }

                    $ok = $wpdb->update(
                        $new_usermeta,
                        array( 'meta_key' => $new_key ),
                        array( 'meta_key' => $old_key ),
                        array( '%s' ),
                        array( '%s' )
                    );

                    if ( false === $ok ) {
                        foreach ( array_reverse( $renamed_new_to_old, true ) as $rolled_new => $rolled_old ) {
                            $rq = 'RENAME TABLE `' . str_replace( '`', '``', $rolled_new ) . '` TO `' . str_replace( '`', '``', $rolled_old ) . '`';
                            $wpdb->query( $rq );
                        }
                        return array(
                            'success' => false,
                            'message' => __( 'Failed while updating usermeta prefix-bound keys (collision-safe step).', 'aegisshield-security' ),
                        );
                    }

                    $updated++;
                }

                $results['usermeta_rows'] = (int) $updated;
                if ( $skipped > 0 ) {
                    $results['usermeta_collisions_skipped'] = (int) $skipped;
                }
            }
        }

        // Store last migration info for Phase 3A verification and for safer support workflows.
        update_option(
            'aegisshield_dbtools_last_prefix_migration',
            array(
                'old_prefix'  => $prefix_current,
                'new_prefix'  => $prefix_new,
                'timestamp'   => time(),
                'backup_file' => basename( $backup_abs ),
                'tables_renamed' => isset( $results['renamed_tables'] ) ? count( $results['renamed_tables'] ) : 0,
            ),
            false
        );

        // wp-config.php update (best-effort).
        $wpconfig_result = $this->try_update_wpconfig_table_prefix( $prefix_new );
        $results['wpconfig'] = $wpconfig_result;

        // Set a pending notice if wp-config is not updated.
        if ( empty( $wpconfig_result['updated'] ) ) {
            update_option( 'aegisshield_dbtools_pending_table_prefix', $prefix_new, false );
        } else {
            delete_option( 'aegisshield_dbtools_pending_table_prefix' );
        }

        $manual_line = '$table_prefix = \''
            . $prefix_new
            . '\';';

                // Phase 2: Log successful completion.
        $this->plugin->get_logger()->log(
            'db_prefix_change_completed',
            __( 'DB prefix change completed successfully.', 'aegisshield-security' ),
            'medium',
            array(
                'old_prefix' => (string) $prefix_current,
                'new_prefix' => (string) $prefix_new,
                'mode'       => (string) $prefix_mode,
                'exec_mode'  => (string) $exec_mode,
                'backup'     => isset( $results['backup_used'] ) ? $results['backup_used'] : (string) $backup_file,
                'details'    => $results,
                'user_id'    => get_current_user_id(),
            )
        );

return array(
            'success'         => true,
            'message'         => __( 'Prefix change applied. IMPORTANT: Ensure wp-config.php $table_prefix matches the NEW prefix before leaving this page.', 'aegisshield-security' ),
            'details'         => $results,
            'manual_wpconfig' => $manual_line,
        );
    }

    /**
     * Phase 3A (safe): Verification scan after a prefix change.
     *
     * This performs READ-ONLY checks:
     * - Confirms required core tables exist under the expected prefix
     * - Detects any leftover tables still using the old prefix (if provided)
     * - Detects leftover prefix-bound keys inside options/usermeta (old prefix keys)
     * - Highlights Action Scheduler/AegisShield tables if they exist under a different prefix
     *
     * @param string $expected_prefix The prefix that WordPress should currently be using (usually $wpdb->prefix).
     * @param string $old_prefix      Optional old prefix (e.g., "wp_") to detect leftovers.
     * @return array {success:bool, issues:array, warnings:array, counts:array, samples:array}
     */
    public function run_prefix_verification_scan( $expected_prefix, $old_prefix = '' ) {
        global $wpdb;

        $expected_prefix = (string) $expected_prefix;
        $old_prefix      = (string) $old_prefix;

        $issues   = array();
        $warnings = array();
        $counts   = array(
            'missing_core_tables'     => 0,
            'leftover_old_tables'     => 0,
            'leftover_old_option_keys'=> 0,
            'leftover_old_usermeta'   => 0,
        );
        $samples = array(
            'missing_core_tables' => array(),
            'leftover_old_tables' => array(),
            'leftover_old_option_keys' => array(),
            'leftover_old_usermeta' => array(),
        );

        // 1) Core table existence under expected prefix.
        $core_suffixes = array(
            'options',
            'users',
            'usermeta',
            'posts',
            'postmeta',
            'terms',
            'termmeta',
            'term_taxonomy',
            'term_relationships',
            'comments',
            'commentmeta',
            'links',
        );

        foreach ( $core_suffixes as $suffix ) {
            $t = $expected_prefix . $suffix;
            $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $t ) );
            if ( empty( $exists ) ) {
                $counts['missing_core_tables']++;
                $samples['missing_core_tables'][] = $t;
            }
        }

        if ( $counts['missing_core_tables'] > 0 ) {
            $issues[] = __( 'One or more required WordPress core tables are missing under the expected prefix. Do NOT proceed with any further changes until this is resolved.', 'aegisshield-security' );
        }

        // 2) Leftover old-prefixed tables (if old prefix provided).
        if ( '' !== $old_prefix && $old_prefix !== $expected_prefix ) {
            $old_like = $wpdb->esc_like( $old_prefix ) . '%';
            $leftover_tables = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', $old_like ) );
            if ( is_array( $leftover_tables ) && ! empty( $leftover_tables ) ) {
                $counts['leftover_old_tables'] = count( $leftover_tables );
                $samples['leftover_old_tables'] = array_slice( $leftover_tables, 0, 25 );
                $warnings[] = __( 'Some tables still exist with the old prefix. This can be normal if you intentionally excluded certain plugin tables, but it becomes dangerous if wp-config.php has been updated to the new prefix while essential plugin tables remain on the old prefix.', 'aegisshield-security' );
            }

            // 3) Leftover old-prefix keys in options/usermeta (stored inside the expected-prefix tables).
            $opt_table = $expected_prefix . 'options';
            $um_table  = $expected_prefix . 'usermeta';

            $opt_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $opt_table ) );
            if ( ! empty( $opt_exists ) ) {
                $opt_like = $old_prefix . '%';
                $opt_count = (int) $wpdb->get_var(
                    $wpdb->prepare(
                        'SELECT COUNT(*) FROM `' . str_replace( '`', '``', $opt_table ) . '` WHERE option_name LIKE %s',
                        $opt_like
                    )
                );
                $counts['leftover_old_option_keys'] = $opt_count;

                if ( $opt_count > 0 ) {
                    $rows = $wpdb->get_col(
                        $wpdb->prepare(
                            'SELECT option_name FROM `' . str_replace( '`', '``', $opt_table ) . '` WHERE option_name LIKE %s LIMIT 25',
                            $opt_like
                        )
                    );
                    $samples['leftover_old_option_keys'] = is_array( $rows ) ? $rows : array();
                    $warnings[] = __( 'Some option_name values still begin with the old prefix. This usually includes keys like {prefix}user_roles and should be updated to match the new prefix.', 'aegisshield-security' );
                }
            }

            $um_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $um_table ) );
            if ( ! empty( $um_exists ) ) {
                $um_like = $old_prefix . '%';
                $um_count = (int) $wpdb->get_var(
                    $wpdb->prepare(
                        'SELECT COUNT(*) FROM `' . str_replace( '`', '``', $um_table ) . '` WHERE meta_key LIKE %s',
                        $um_like
                    )
                );
                $counts['leftover_old_usermeta'] = $um_count;

                if ( $um_count > 0 ) {
                    $rows = $wpdb->get_col(
                        $wpdb->prepare(
                            'SELECT DISTINCT meta_key FROM `' . str_replace( '`', '``', $um_table ) . '` WHERE meta_key LIKE %s LIMIT 25',
                            $um_like
                        )
                    );
                    $samples['leftover_old_usermeta'] = is_array( $rows ) ? $rows : array();
                    $warnings[] = __( 'Some usermeta meta_key values still begin with the old prefix (example: wp_capabilities). These must be updated to the new prefix or user roles/capabilities may break.', 'aegisshield-security' );
                }
            }
        }

        // 4) Action Scheduler / AegisShield table prefix mismatch hints.
        $as_any = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', '%actionscheduler_actions' ) );
        if ( is_array( $as_any ) && ! empty( $as_any ) ) {
            $expected_as = $expected_prefix . 'actionscheduler_actions';
            $expected_present = in_array( $expected_as, $as_any, true );
            if ( ! $expected_present ) {
                $warnings[] = __( 'Action Scheduler tables were detected, but not under the expected prefix. If wp-config.php is set to the expected prefix, Action Scheduler-based cron jobs may error until those tables are aligned.', 'aegisshield-security' );
            }
        }

        $aegis_any = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', '%aegisshield_activity_log' ) );
        if ( is_array( $aegis_any ) && ! empty( $aegis_any ) ) {
            $expected_aegis = $expected_prefix . 'aegisshield_activity_log';
            $expected_present = in_array( $expected_aegis, $aegis_any, true );
            if ( ! $expected_present ) {
                $warnings[] = __( 'AegisShield custom tables were detected, but not under the expected prefix. This can cause AegisShield cron maintenance (log pruning, etc.) to error until those tables are aligned.', 'aegisshield-security' );
            }
        }

        return array(
            'success'  => empty( $issues ),
            'issues'   => $issues,
            'warnings' => $warnings,
            'counts'   => $counts,
            'samples'  => $samples,
        );
    }


/**
     * Validate the new prefix value.
     *
     * @param string $new_prefix
     * @param string $current_prefix
     * @return string Empty string when valid; otherwise translated error message.
     */
    private function validate_new_prefix_value( $new_prefix, $current_prefix ) {
        $new_prefix     = (string) $new_prefix;
        $current_prefix = (string) $current_prefix;

        if ( empty( $new_prefix ) ) {
            return __( 'New prefix is required.', 'aegisshield-security' );
        }

        // Only letters, numbers, underscores. Must end with underscore.
        if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $new_prefix ) ) {
            return __( 'Prefix contains invalid characters. Use only letters, numbers, and underscores.', 'aegisshield-security' );
        }

        if ( '_' !== substr( $new_prefix, -1 ) ) {
            return __( 'Prefix must end with an underscore ( _ ).', 'aegisshield-security' );
        }

        if ( $new_prefix === $current_prefix ) {
            return __( 'New prefix must be different from the current prefix.', 'aegisshield-security' );
        }

        if ( strlen( $new_prefix ) > 20 ) {
            return __( 'Prefix is too long. Keep it under 20 characters.', 'aegisshield-security' );
        }

        return '';
    }

    /**
     * Get core WordPress table suffixes for single-site installs.
     *
     * @return array
     */
    private function get_wp_core_table_suffixes() {
        return array(
            'commentmeta',
            'comments',
            'links',
            'options',
            'postmeta',
            'posts',
            'termmeta',
            'terms',
            'term_relationships',
            'term_taxonomy',
            'usermeta',
            'users',
        );
    }

    /**
     * Attempt to update wp-config.php $table_prefix when possible (best-effort).
     *
     * @param string $new_prefix
     * @return array {updated:bool, writable:bool, path:string}
     */
    private function try_update_wpconfig_table_prefix( $new_prefix ) {
        $path = defined( 'ABSPATH' ) ? trailingslashit( ABSPATH ) . 'wp-config.php' : '';
        $writable = ( ! empty( $path ) && file_exists( $path ) && is_writable( $path ) );

        // Some hosts keep wp-config.php one directory above ABSPATH.
        if ( ! $writable && defined( 'ABSPATH' ) ) {
            $alt = dirname( ABSPATH ) . '/wp-config.php';
            if ( file_exists( $alt ) ) {
                $path = $alt;
                $writable = is_writable( $path );
            }
        }

        $result = array(
            'updated'  => false,
            'writable' => (bool) $writable,
            'path'     => (string) $path,
        );

        if ( ! $writable || empty( $path ) ) {
            return $result;
        }

        $contents = file_get_contents( $path );
        if ( false === $contents || empty( $contents ) ) {
            return $result;
        }

        $pattern = '/\$table_prefix\s*=\s*[\'\"][A-Za-z0-9_]*[\'\"]\s*;/';

        if ( ! preg_match( $pattern, $contents ) ) {
            // Do not attempt to write if we cannot find the line safely.
            return $result;
        }

        $replacement = '$table_prefix = \''
            . $new_prefix
            . '\';';

        $new_contents = preg_replace( $pattern, $replacement, $contents, 1 );
        if ( empty( $new_contents ) || $new_contents === $contents ) {
            return $result;
        }

        $written = file_put_contents( $path, $new_contents );
        if ( false === $written ) {
            return $result;
        }

        $result['updated'] = true;
        return $result;
    }


}