<?php
namespace AegisShield\Modules;

use AegisShield\AS_Plugin;
use AegisShield\AS_Logger;

defined( 'ABSPATH' ) || exit;

/**
 * File Integrity module.
 */
class AS_Module_File_Integrity implements AS_Module_Interface {

    /**
     * Plugin instance.
     *
     * @var AS_Plugin
     */
    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function get_slug() {
        return 'file_integrity';
    }

    public function register_settings() {
        $settings = $this->plugin->get_settings();
        $section  = 'file_integrity';

        if ( null === $settings->get( $section, 'scan_mode', null ) ) {
            $settings->set( $section, 'scan_mode', 'light' );
        }
        if ( null === $settings->get( $section, 'last_auto_scan', null ) ) {
            $settings->set( $section, 'last_auto_scan', 0 );
        }
        if ( null === $settings->get( $section, 'auto_scan_frequency', null ) ) {
            $settings->set( $section, 'auto_scan_frequency', 'weekly' );
        }
        if ( null === $settings->get( $section, 'email_alert_threshold', null ) ) {
            $settings->set( $section, 'email_alert_threshold', 'none' );
        }
        if ( null === $settings->get( $section, 'scan_history_limit', null ) ) {
            $settings->set( $section, 'scan_history_limit', '10' );
        }
        if ( null === $settings->get( $section, 'last_scan_report', null ) ) {
            $settings->set( $section, 'last_scan_report', array() );
        }
        if ( null === $settings->get( $section, 'scan_history', null ) ) {
			$settings->set( $section, 'scan_history', array() );
		}

		/**
		 * File Change Monitor (section: file_monitor) defaults
		 * These are referenced later in handle_scan_result() for rule decisions.
		 */
		$mon_section = 'file_monitor';

		if ( null === $settings->get( $mon_section, 'interval', null ) ) {
			$settings->set( $mon_section, 'interval', '15' ); // minutes
		}
		if ( null === $settings->get( $mon_section, 'email_mode', null ) ) {
			$settings->set( $mon_section, 'email_mode', 'instant' ); // instant|summary
		}
		if ( null === $settings->get( $mon_section, 'email_events', null ) ) {
			$settings->set( $mon_section, 'email_events', array() ); // e.g. ['php','modified','new','deleted','high_risk','plugin_theme']
		}
		if ( null === $settings->get( $mon_section, 'email_recipients', null ) ) {
			$settings->set( $mon_section, 'email_recipients', '' );
		}

		$settings->save();

    }

    public function init() {
        // Hook into hourly maintenance to perform at most weekly scans.
        add_action( 'aegisshield_hourly_maintenance', array( $this, 'maybe_run_auto_scan' ) );
    }

	/**
	 * Centralized FIM violation/event logger.
	 *
	 * This is the ONLY place we emit FIM security events so:
	 * - Activity Log captures them consistently
	 * - Alerts Pro can alert by event type/rule/severity
	 *
	 * @param string $rule     Short rule key (matches checkbox intent), e.g. 'php', 'modified', 'deleted', 'high_risk', etc.
	 * @param string $message  Human readable message.
	 * @param string $severity 'low'|'medium'|'high'|'critical'
	 * @param array  $meta     Extra context (counts, sample paths, mode, etc.)
	 * @param string $event    Event slug written to Activity Log (default 'fim_violation')
	 */
	protected function log_fim_violation( $rule, $message, $severity = 'medium', $meta = array(), $event = 'fim_violation' ) {
		$logger = $this->plugin ? $this->plugin->get_logger() : null;
		if ( ! $logger ) {
			return;
		}

		if ( ! is_array( $meta ) ) {
			$meta = array();
		}

		// Keep payload safe for shared hosting logs.
		if ( isset( $meta['sample_paths'] ) && is_array( $meta['sample_paths'] ) ) {
			$meta['sample_paths'] = array_slice( $meta['sample_paths'], 0, 20 );
		}

		$meta = array_merge(
			array(
				'module' => 'file_integrity',
				'rule'   => (string) $rule,
			),
			$meta
		);

		// Use the 4-argument form already used elsewhere in this file.
		$logger->log( (string) $event, (string) $message, (string) $severity, $meta );
	}

	/**
	 * Helper: is a file-monitor rule enabled (checkbox) in settings?
	 *
	 * @param string $rule
	 * @return bool
	 */
	protected function fim_rule_enabled( $rule ) {
		$settings     = $this->plugin->get_settings();
		$email_events = (array) $settings->get( 'file_monitor', 'email_events', array() );

		return in_array( (string) $rule, $email_events, true );
	}

    /**
     * Possibly run an automatic scan (no more than once per week).
     */
    public 
    function maybe_run_auto_scan() {
        $settings = $this->plugin->get_settings();
        $section  = 'file_integrity';

        $last_auto = (int) $settings->get( $section, 'last_auto_scan', 0 );
        $now       = time();

        $frequency = (string) $settings->get( $section, 'auto_scan_frequency', 'weekly' );
        if ( ! in_array( $frequency, array( 'daily', 'weekly', 'monthly' ), true ) ) {
            $frequency = 'weekly';
        }

        switch ( $frequency ) {
            case 'daily':
                $interval = DAY_IN_SECONDS;
                break;
            case 'monthly':
                $interval = MONTH_IN_SECONDS;
                break;
            case 'weekly':
            default:
                $interval = WEEK_IN_SECONDS;
                break;
        }

        if ( $last_auto && ( $now - $last_auto ) < $interval ) {
            return;
        }

        $mode = (string) $settings->get( $section, 'scan_mode', 'light' );
        if ( ! in_array( $mode, array( 'light', 'hybrid', 'full' ), true ) ) {
            $mode = 'light';
        }

        $result = $this->run_scan( $mode, false );
        $this->handle_scan_result( $mode, $result, false );

        $settings->set( $section, 'last_auto_scan', $now );
        $settings->save();
    }


    /**
     * Run manual scan triggered from admin page.
     *
     * @return array Summary + items.
     */
    public 
    function run_manual_scan() {
        $settings = $this->plugin->get_settings();
        $mode     = (string) $settings->get( 'file_integrity', 'scan_mode', 'light' );
        if ( ! in_array( $mode, array( 'light', 'hybrid', 'full' ), true ) ) {
            $mode = 'light';
        }

        $result = $this->run_scan( $mode, true );
        $this->handle_scan_result( $mode, $result, true );

        return $result;
    }


    /**
     * Core scan routine.
     *
     * @param string $mode      Scan mode (light|hybrid|full).
     * @param bool   $is_manual Whether this is a manual scan.
     * @return array
     */
    protected function run_scan( $mode, $is_manual ) {
        global $wpdb;

        $logger     = $this->plugin->get_logger();
        $table      = $wpdb->prefix . 'aegisshield_file_hashes';
        $now_mysql  = current_time( 'mysql' );
        $base_paths = $this->get_base_paths_for_mode( $mode );

        $scanned   = 0;
        $modified  = 0;
        $added     = 0;
        $errors    = 0;
        $items_out = array();
        $seen_paths = array();

        foreach ( $base_paths as $base_info ) {
            $root    = $base_info['path'];
            $filter  = $base_info['filter'];
            if ( ! is_dir( $root ) ) {
                continue;
            }

            try {
                $iterator = new \RecursiveIteratorIterator(
                    new \RecursiveDirectoryIterator(
                        $root,
                        \FilesystemIterator::SKIP_DOTS | \FilesystemIterator::FOLLOW_SYMLINKS
                    ),
                    \RecursiveIteratorIterator::SELF_FIRST
                );
            } catch ( \Exception $e ) {
                $errors++;
                continue;
            }

            foreach ( $iterator as $file ) {
                if ( ! $file->isFile() ) {
                    continue;
                }

                $path = $file->getPathname();

                if ( ! $this->file_matches_filter( $path, $filter ) ) {
                    continue;
                }

                $rel_path = ltrim( str_replace( ABSPATH, '', $path ), '/' );

                $seen_paths[] = $rel_path;
                $scanned++;

                $hash = @hash_file( 'sha256', $path );
                if ( ! $hash ) {
                    $errors++;
                    continue;
                }

                $row = $wpdb->get_row(
                    $wpdb->prepare(
                        "SELECT * FROM {$table} WHERE path = %s",
                        $rel_path
                    )
                );

                if ( $row ) {
                    $status = ( $row->hash === $hash ) ? 'clean' : 'modified';

                    if ( 'modified' === $status ) {
                        $modified++;
                        $logger->log(
                            'file_modified',
                            sprintf(
                                /* translators: %s: file path */
                                __( 'File modified: %s', 'aegisshield-security' ),
                                $rel_path
                            ),
                            array( 'path' => $rel_path )
                        );
                    }

                    $wpdb->update(
                        $table,
                        array(
                            'hash'         => $hash,
                            'last_scanned' => $now_mysql,
                            'status'       => $status,
                        ),
                        array( 'id' => $row->id )
                    );
                } else {
                    $added++;
                    $status = 'new';

                    $logger->log(
                        'file_new',
                        sprintf(
                            /* translators: %s: file path */
                            __( 'New file detected: %s', 'aegisshield-security' ),
                            $rel_path
                        ),
                        array( 'path' => $rel_path )
                    );

                    $wpdb->insert(
                        $table,
                        array(
                            'path'         => $rel_path,
                            'hash'         => $hash,
                            'last_scanned' => $now_mysql,
                            'status'       => $status,
                        )
                    );
                }

                // For manual scan, collect limited number of items to display.
                if ( $is_manual && count( $items_out ) < 200 ) {
                    $items_out[] = array(
                        'path'         => $rel_path,
                        'status'       => isset( $status ) ? $status : 'clean',
                        'last_scanned' => $now_mysql,
                    );
                }
            }
        }

        // Optionally mark missing files (those in DB but not seen in this scan).
        if ( ! empty( $seen_paths ) ) {
            $placeholders = implode( ',', array_fill( 0, count( $seen_paths ), '%s' ) );
            $wpdb->query(
                $wpdb->prepare(
                    "UPDATE {$table} SET status = 'missing' WHERE path NOT IN ($placeholders)",
                    $seen_paths
                )
            );
        }

        $summary = sprintf(
            /* translators: 1: scanned, 2: modified, 3: new, 4: errors */
            __( 'Scan complete. Files scanned: %1$d, modified: %2$d, new: %3$d, errors: %4$d.', 'aegisshield-security' ),
            $scanned,
            $modified,
            $added,
            $errors
        );

        if ( $is_manual ) {
            $this->plugin->get_logger()->log(
                'file_scan_manual',
                $summary,
                array(
                    'mode'      => $mode,
                    'scanned'   => $scanned,
                    'modified'  => $modified,
                    'new'       => $added,
                    'errors'    => $errors,
                    'admin_id'  => get_current_user_id(),
                )
            );
        } else {
            $this->plugin->get_logger()->log(
                'file_scan_auto',
                $summary,
                array(
                    'mode'      => $mode,
                    'scanned'   => $scanned,
                    'modified'  => $modified,
                    'new'       => $added,
                    'errors'    => $errors,
                )
            );
        }

        return array(
            'summary' => $summary,
            'items'   => $items_out,
            'stats'   => array(
                'scanned'  => $scanned,
                'modified' => $modified,
                'new'      => $added,
                'errors'   => $errors,
            ),
        );
    }

    /**
     * Build base paths and filters for a given mode.
     *
     * @param string $mode Scan mode.
     * @return array
     */
    protected function get_base_paths_for_mode( $mode ) {
        $paths = array();

        // Common core/theme/plugin paths.
        $paths[] = array(
            'path'   => ABSPATH . 'wp-admin',
            'filter' => 'code',
        );
        $paths[] = array(
            'path'   => ABSPATH . 'wp-includes',
            'filter' => 'code',
        );
        $theme = get_option( 'stylesheet' );
        if ( $theme ) {
            $paths[] = array(
                'path'   => WP_CONTENT_DIR . '/themes/' . $theme,
                'filter' => 'code',
            );
        }
        $paths[] = array(
            'path'   => WP_PLUGIN_DIR,
            'filter' => 'code',
        );

        if ( 'hybrid' === $mode || 'full' === $mode ) {
            // Uploads directory: suspicious files only.
            $upload_dir = wp_get_upload_dir();
            if ( ! empty( $upload_dir['basedir'] ) && is_dir( $upload_dir['basedir'] ) ) {
                $paths[] = array(
                    'path'   => $upload_dir['basedir'],
                    'filter' => 'suspicious',
                );
            }
        }

        if ( 'full' === $mode ) {
            // Entire WordPress root (code + suspicious filters).
            $paths[] = array(
                'path'   => ABSPATH,
                'filter' => 'code',
            );
        }

        return $paths;
    }

    /**
     * Decide if a file should be scanned based on filter type.
     *
     * @param string $path   File path.
     * @param string $filter Filter type.
     * @return bool
     */

    /**
     * Handle scan result: store last report, maintain history, and trigger alerts.
     *
     * @param string $mode      Scan mode.
     * @param array  $result    Scan result array from run_scan().
     * @param bool   $is_manual Whether this was a manual scan.
     */
    protected function handle_scan_result( $mode, $result, $is_manual ) {
        $settings = $this->plugin->get_settings();
        $section  = 'file_integrity';
		$logger   = $this->plugin->get_logger();

        $summary_text = isset( $result['summary'] ) ? (string) $result['summary'] : '';
        $items        = isset( $result['items'] ) && is_array( $result['items'] ) ? $result['items'] : array();
        $stats        = isset( $result['stats'] ) && is_array( $result['stats'] ) ? $result['stats'] : array(
            'scanned'  => 0,
            'modified' => 0,
            'new'      => 0,
            'errors'   => 0,
        );

        $timestamp = current_time( 'mysql' );

        // Classify items by severity and prepare extended item data.
        $extended_items = array();
        $high_count     = 0;
        $medium_count   = 0;
        $low_count      = 0;

        foreach ( $items as $item ) {
            if ( empty( $item['path'] ) ) {
                continue;
            }

            $path   = (string) $item['path'];
            $status = isset( $item['status'] ) ? (string) $item['status'] : 'clean';

            // Only classify non-clean statuses.
            if ( 'clean' === $status ) {
                $severity = 'low';
                $reason   = __( 'File scanned with no issues.', 'aegisshield-security' );
            } else {
                list( $severity, $reason ) = $this->classify_item_severity( $path, $status );
            }

            switch ( $severity ) {
                case 'high':
                    $high_count++;
                    break;
                case 'medium':
                    $medium_count++;
                    break;
                default:
                    $low_count++;
                    break;
            }

            $extended_items[] = array(
                'path'         => $path,
                'status'       => $status,
                'last_scanned' => isset( $item['last_scanned'] ) ? $item['last_scanned'] : $timestamp,
                'severity'     => $severity,
                'reason'       => $reason,
            );
        }

        $report = array(
            'timestamp' => $timestamp,
            'mode'      => $mode,
            'summary'   => $summary_text,
            'stats'     => $stats,
            'high'      => $high_count,
            'medium'    => $medium_count,
            'low'       => $low_count,
            'items'     => $extended_items,
        );

        // Store last report.
        $settings->set( $section, 'last_scan_report', $report );

        // Maintain scan history (summaries only).
        $history       = $settings->get( $section, 'scan_history', array() );
        if ( ! is_array( $history ) ) {
            $history = array();
        }

        $history_entry = array(
            'timestamp' => $timestamp,
            'mode'      => $mode,
            'summary'   => $summary_text,
            'stats'     => $stats,
            'high'      => $high_count,
            'medium'    => $medium_count,
            'low'       => $low_count,
        );

        array_unshift( $history, $history_entry );

        $limit = (string) $settings->get( $section, 'scan_history_limit', '10' );
        if ( 'unlimited' !== $limit ) {
            $limit_int = (int) $limit;
            if ( $limit_int > 0 && count( $history ) > $limit_int ) {
                $history = array_slice( $history, 0, $limit_int );
            }
        }

        $settings->set( $section, 'scan_history', $history );
        $settings->save();
		// ==============================
		// EMAIL ALERTS (File Monitor)
		// ==============================
		$email_mode       = $settings->get( 'file_monitor', 'email_mode', 'instant' );
		$email_events     = (array) $settings->get( 'file_monitor', 'email_events', array() );
		$email_recipients = $settings->get( 'file_monitor', 'email_recipients', '' );

		// Do not proceed if alerts are not configured
		if ( empty( $email_events ) || empty( $email_recipients ) ) {
			return;
		}

        // Phase 2: Always log scan completion and significant integrity findings.
        $logger = $this->plugin->get_logger();
		
		$missing_count  = 0;
        $modified_count = isset( $stats['modified'] ) ? (int) $stats['modified'] : 0;
        $new_count      = isset( $stats['new'] ) ? (int) $stats['new'] : 0;
        $error_count    = isset( $stats['errors'] ) ? (int) $stats['errors'] : 0;
        $sample_paths   = array();

        foreach ( $extended_items as $ei ) {
            if ( empty( $ei['path'] ) || empty( $ei['status'] ) ) {
                continue;
            }
            if ( 'missing' === (string) $ei['status'] ) {
                $missing_count++;
            }
            // Capture a small sample for context (avoid huge log payloads on shared hosting).
            if ( ( 'clean' !== (string) $ei['status'] ) && count( $sample_paths ) < 20 ) {
                $sample_paths[] = (string) $ei['path'];
            }
        }

		if ( $logger ) {
			$logger->log(
				'file_integrity_scan_completed',
				sprintf( __( 'File integrity scan completed (%s).', 'aegisshield-security' ), $mode ),
				'low',
				array(
					'mode'          => $mode,
					'is_manual'     => (bool) $is_manual,
					'stats'         => $stats,
					'high'          => (int) $high_count,
					'medium'        => (int) $medium_count,
					'low_findings'  => (int) $low_count,
					'missing_count' => (int) $missing_count,
					'sample_paths'  => $sample_paths,
				)
			);
		}

		if ( $logger && $missing_count > 0 ) {
			$logger->log(
				'file_integrity_deleted',
				__( 'File integrity detected missing files.', 'aegisshield-security' ),
				'critical',
				array(
					'mode'          => $mode,
					'missing_count' => (int) $missing_count,
					'sample_paths'  => $sample_paths,
				)
			);
		}
		
		if ( $logger && ( $modified_count > 0 || $high_count > 0 || $medium_count > 0 ) ) {
			$logger->log(
				'file_integrity_modified',
				__( 'File integrity detected modified files.', 'aegisshield-security' ),
				( $high_count > 0 ? 'high' : 'medium' ),
				array(
					'mode'           => $mode,
					'modified_count' => (int) $modified_count,
					'new_count'      => (int) $new_count,
					'high'           => (int) $high_count,
					'medium'         => (int) $medium_count,
					'sample_paths'   => $sample_paths,
				)
			);
		}


		if ( $logger && $error_count > 0 ) {
			$logger->log(
				'file_integrity_scan_errors',
				__( 'File integrity scan encountered errors.', 'aegisshield-security' ),
				'medium',
				array(
					'mode'        => $mode,
					'error_count' => (int) $error_count,
				)
			);
		}

		// If the scan had errors, log a FIM violation so Alerts Pro can act on it.
		if ( $error_count > 0 ) {
			$this->log_fim_violation(
				'scan_errors',
				__( 'File Integrity Monitoring encountered scan errors (some files could not be hashed or read).', 'aegisshield-security' ),
				'high',
				array(
					'mode'        => $mode,
					'errors'      => (int) $error_count,
					'sample_paths'=> $sample_paths,
					'is_manual'   => (bool) $is_manual,
				),
				'fim_violation'
			);
		}

        // ===========================
        // Email Alerts (File Monitor)
        // ===========================
        $email_events = (array) $settings->get( 'file_monitor', 'email_events', array() );

        // If nothing selected, don't send anything.
        if ( empty( $email_events ) ) {
            return;
        }

        $should_alert = false;

        // Map scan stats to alert types
        if ( in_array( 'new', $email_events, true ) && ! empty( $stats['new'] ) ) {
            $should_alert = true;
        }

        if ( in_array( 'modified', $email_events, true ) && ! empty( $stats['modified'] ) ) {
            $should_alert = true;
        }

        if ( in_array( 'deleted', $email_events, true ) && ! empty( $missing_count ) ) {
            $should_alert = true;
        }

        if ( in_array( 'high_risk', $email_events, true ) && $high_count > 0 ) {
            $should_alert = true;
        }

		/**
		 * ENFORCEMENT + LOGGING:
		 * If a rule checkbox is enabled and the scan indicates a violation, emit a FIM event.
		 * Alerts Pro can then decide whether/how to notify.
		 */

		// PHP rule (high severity) — triggered when classify_item_severity flags high.
		if ( $this->fim_rule_enabled( 'php' ) && $high_count > 0 ) {
			$this->log_fim_violation(
				'php',
				__( 'FIM rule triggered: PHP file changes detected.', 'aegisshield-security' ),
				'high',
				array(
					'mode'         => $mode,
					'high'         => (int) $high_count,
					'sample_paths' => $sample_paths,
					'is_manual'    => (bool) $is_manual,
				),
				'fim_php_file_violation'
			);
		}

		// Modified files rule
		if ( $this->fim_rule_enabled( 'modified' ) && ! empty( $stats['modified'] ) ) {
			$this->log_fim_violation(
				'modified',
				__( 'FIM rule triggered: modified files detected.', 'aegisshield-security' ),
				( $high_count > 0 ? 'high' : 'medium' ),
				array(
					'mode'          => $mode,
					'modified_count'=> (int) $stats['modified'],
					'sample_paths'  => $sample_paths,
					'is_manual'     => (bool) $is_manual,
				),
				'fim_file_modified'
			);
		}

		// New files rule
		if ( $this->fim_rule_enabled( 'new' ) && ! empty( $stats['new'] ) ) {
			$this->log_fim_violation(
				'new',
				__( 'FIM rule triggered: new files detected.', 'aegisshield-security' ),
				'medium',
				array(
					'mode'        => $mode,
					'new_count'   => (int) $stats['new'],
					'sample_paths'=> $sample_paths,
					'is_manual'   => (bool) $is_manual,
				),
				'fim_file_created'
			);
		}

		// Deleted / Missing files rule
		if ( $this->fim_rule_enabled( 'deleted' ) && ! empty( $missing_count ) ) {
			$this->log_fim_violation(
				'deleted',
				__( 'FIM rule triggered: missing/deleted files detected.', 'aegisshield-security' ),
				'high',
				array(
					'mode'          => $mode,
					'missing_count' => (int) $missing_count,
					'sample_paths'  => $sample_paths,
					'is_manual'     => (bool) $is_manual,
				),
				'fim_file_deleted'
			);
		}

		// High-risk directories rule (wp-admin/wp-includes)
		if ( $this->fim_rule_enabled( 'high_risk' ) && $high_count > 0 ) {
			$this->log_fim_violation(
				'high_risk',
				__( 'FIM rule triggered: changes detected in high-risk WordPress directories.', 'aegisshield-security' ),
				'high',
				array(
					'mode'         => $mode,
					'high'         => (int) $high_count,
					'sample_paths' => $sample_paths,
					'is_manual'    => (bool) $is_manual,
				),
				'fim_high_risk_path_violation'
			);
		}

		// If rules indicate an alert should occur, but recipients are missing, log a "not working" violation.
		if ( $should_alert ) {
			$recipients = (string) $settings->get( 'file_monitor', 'email_recipients', '' );
			if ( '' === trim( $recipients ) ) {
				$this->log_fim_violation(
					'alert_recipients_missing',
					__( 'FIM alert was eligible to send, but no email recipients are configured.', 'aegisshield-security' ),
					'medium',
					array(
						'mode'        => $mode,
						'is_manual'   => (bool) $is_manual,
						'high'        => (int) $high_count,
						'medium'      => (int) $medium_count,
						'missing'     => (int) $missing_count,
						'modified'    => isset( $stats['modified'] ) ? (int) $stats['modified'] : 0,
						'new'         => isset( $stats['new'] ) ? (int) $stats['new'] : 0,
					)
				);
			}
		}

        if ( ! $should_alert ) {
            return;
        }

        // Send alert email using existing formatter
        $this->send_scan_alert_email(
            $report,
            'file_monitor',
            (bool) $is_manual
        );

    }

    /**
     * Classify a file item into severity bucket.
     *
     * @param string $path   File path.
     * @param string $status File status.
     * @return array Array of [severity, reason].
     */
    protected function classify_item_severity( $path, $status ) {
        $path_lower = strtolower( $path );
        $status     = strtolower( $status );

        // Default.
        $severity = 'low';
        $reason   = __( 'File change detected.', 'aegisshield-security' );

        // Missing core or plugin/theme files are at least medium.
        if ( 'missing' === $status ) {
            if ( false !== strpos( $path_lower, 'wp-admin' ) || false !== strpos( $path_lower, 'wp-includes' ) ) {
                return array(
                    'high',
                    __( 'Core file missing from WordPress.', 'aegisshield-security' ),
                );
            }

            return array(
                'medium',
                __( 'File missing since last scan.', 'aegisshield-security' ),
            );
        }

        // Suspicious uploads with executable extensions.
        $is_uploads = ( false !== strpos( $path_lower, 'wp-content/uploads' ) );
        $is_exec    = (bool) preg_match( '/\.(php|php5|php7|phtml|sh|pl|py|exe|bat|cmd)$/', $path_lower );

        if ( $is_uploads && $is_exec ) {
            return array(
                'high',
                __( 'Executable file detected in uploads directory.', 'aegisshield-security' ),
            );
        }

        // Core file modified.
        if ( false !== strpos( $path_lower, 'wp-admin' ) || false !== strpos( $path_lower, 'wp-includes' ) ) {
            return array(
                'high',
                __( 'WordPress core file modified since last scan.', 'aegisshield-security' ),
            );
        }

        // Plugin / theme files modified.
        if ( false !== strpos( $path_lower, 'wp-content/plugins' ) || false !== strpos( $path_lower, 'wp-content/themes' ) ) {
            return array(
                'medium',
                __( 'Plugin or theme file modified or added.', 'aegisshield-security' ),
            );
        }

        // Generic new or modified files.
        if ( 'new' === $status ) {
            return array(
                'medium',
                __( 'New file added since last scan.', 'aegisshield-security' ),
            );
        }

        if ( 'modified' === $status ) {
            return array(
                'medium',
                __( 'File contents have changed since last scan.', 'aegisshield-security' ),
            );
        }

        return array( $severity, $reason );
    }

    /**
     * Send scan alert email to the admin.
     *
     * @param array $report    Full scan report.
     * @param string $threshold Threshold key.
     * @param bool   $is_manual Whether this was a manual scan.
     */
    protected function send_scan_alert_email( $report, $threshold, $is_manual ) {
        $admin_email = get_option( 'admin_email' );
        if ( empty( $admin_email ) || ! is_email( $admin_email ) ) {
            return;
        }

        $blogname = wp_specialchars_decode( get_option( 'blogname' ), ENT_QUOTES );
        $site_url = home_url( '/' );

        $subject_context = $is_manual ? __( 'manual scan', 'aegisshield-security' ) : __( 'automatic scan', 'aegisshield-security' );
        /* translators: 1: site name, 2: context (manual/automatic). */
        $subject = sprintf(
            __( '[AegisShield] File Integrity alert on %1$s (%2$s)', 'aegisshield-security' ),
            $blogname,
            $subject_context
        );

        $lines   = array();
        $lines[] = sprintf( __( 'Site: %s', 'aegisshield-security' ), $site_url );
        $lines[] = sprintf( __( 'Timestamp: %s', 'aegisshield-security' ), isset( $report['timestamp'] ) ? $report['timestamp'] : '' );
        $lines[] = sprintf( __( 'Mode: %s', 'aegisshield-security' ), isset( $report['mode'] ) ? ucfirst( $report['mode'] ) : '' );
        $lines[] = '';

        if ( ! empty( $report['stats'] ) ) {
            $stats = $report['stats'];
            $lines[] = __( 'Scan summary:', 'aegisshield-security' );
            $lines[] = sprintf(
                __( 'Files scanned: %1$d, modified: %2$d, new: %3$d, errors: %4$d.', 'aegisshield-security' ),
                isset( $stats['scanned'] ) ? (int) $stats['scanned'] : 0,
                isset( $stats['modified'] ) ? (int) $stats['modified'] : 0,
                isset( $stats['new'] ) ? (int) $stats['new'] : 0,
                isset( $stats['errors'] ) ? (int) $stats['errors'] : 0
            );
            $lines[] = sprintf(
                __( 'High: %1$d, Medium: %2$d, Low: %3$d findings.', 'aegisshield-security' ),
                isset( $report['high'] ) ? (int) $report['high'] : 0,
                isset( $report['medium'] ) ? (int) $report['medium'] : 0,
                isset( $report['low'] ) ? (int) $report['low'] : 0
            );
            $lines[] = '';
        }

        $lines[] = __( 'Top findings:', 'aegisshield-security' );

        $items = isset( $report['items'] ) && is_array( $report['items'] ) ? $report['items'] : array();

        // Sort by severity (high > medium > low).
        usort(
            $items,
            function ( $a, $b ) {
                $priority = array(
                    'high'   => 3,
                    'medium' => 2,
                    'low'    => 1,
                );

                $sa = isset( $a['severity'] ) ? $a['severity'] : 'low';
                $sb = isset( $b['severity'] ) ? $b['severity'] : 'low';

                $pa = isset( $priority[ $sa ] ) ? $priority[ $sa ] : 1;
                $pb = isset( $priority[ $sb ] ) ? $priority[ $sb ] : 1;

                if ( $pa === $pb ) {
                    return 0;
                }

                return ( $pa > $pb ) ? -1 : 1;
            }
        );

        $top = array_slice( $items, 0, 10 );

        foreach ( $top as $item ) {
            $sev    = isset( $item['severity'] ) ? $item['severity'] : 'low';
            $path   = isset( $item['path'] ) ? $item['path'] : '';
            $status = isset( $item['status'] ) ? $item['status'] : '';
            $reason = isset( $item['reason'] ) ? $item['reason'] : '';

            $lines[] = sprintf(
                ' - [%1$s] %2$s (%3$s) — %4$s',
                strtoupper( $sev ),
                $path,
                $status,
                $reason
            );
        }

        if ( empty( $top ) ) {
            $lines[] = __( 'No individual file findings to report.', 'aegisshield-security' );
        }

        $lines[] = '';
        $lines[] = sprintf(
            __( 'View full report in WordPress admin: %s', 'aegisshield-security' ),
            admin_url( 'admin.php?page=aegisshield-file-integrity' )
        );

        $body = implode( "
", $lines );

        // Build JSON attachment with full report.
        if ( ! function_exists( 'wp_json_encode' ) ) {
            return;
        }

        $json = wp_json_encode(
            array(
                'site'      => $site_url,
                'timestamp' => isset( $report['timestamp'] ) ? $report['timestamp'] : '',
                'mode'      => isset( $report['mode'] ) ? $report['mode'] : '',
                'summary'   => isset( $report['summary'] ) ? $report['summary'] : '',
                'stats'     => isset( $report['stats'] ) ? $report['stats'] : array(),
                'items'     => $items,
            ),
            JSON_PRETTY_PRINT
        );

        $upload_dir = wp_upload_dir();
        if ( empty( $upload_dir['basedir'] ) ) {
            $attachments = array();
        } else {
            $filename    = 'aegisshield-scan-' . gmdate( 'Ymd-His' ) . '.json';
            $filepath    = trailingslashit( $upload_dir['basedir'] ) . $filename;
            file_put_contents( $filepath, $json );
            $attachments = array( $filepath );
        }

        $headers = array( 'Content-Type: text/plain; charset=UTF-8' );

        wp_mail( $admin_email, $subject, $body, $headers, $attachments );
    }

    protected function file_matches_filter( $path, $filter ) {
        $ext = strtolower( pathinfo( $path, PATHINFO_EXTENSION ) );

        $code_exts = array( 'php', 'php5', 'php7', 'phtml', 'js', 'css', 'inc' );
        $suspicious_exts = array( 'php', 'php5', 'php7', 'phtml', 'js', 'jsp', 'sh', 'pl', 'py', 'exe', 'bat', 'cmd' );

        if ( 'code' === $filter ) {
            return in_array( $ext, $code_exts, true );
        }

        if ( 'suspicious' === $filter ) {
            return in_array( $ext, $suspicious_exts, true );
        }

        return false;
    }
}