<?php
namespace AegisShield\Modules;

use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

/**
 * File Change Monitor module (Pro-only).
 *
 * This module tracks new, modified and deleted files under key WordPress
 * directories and records the latest state into a lightweight custom table.
 * It runs on a WP-Cron schedule and never modifies any files.
 */
class AS_Module_File_Monitor implements AS_Module_Interface {

    /**
     * Plugin instance.
     *
     * @var AS_Plugin
     */
    protected $plugin;

    /**
     * Constructor.
     *
     * @param AS_Plugin $plugin Plugin instance.
     */
    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Unique slug for this module.
     *
     * @return string
     */
    public function get_slug() {
        return 'file_monitor';
    }

    /**
     * Register default settings for the module.
     *
     * Currently only stores an enabled flag so the module can be toggled
     * in a future UI without breaking.
     *
     * @return void
     */
    public function register_settings() {
        $settings = $this->plugin->get_settings();
        $section  = 'file_monitor';

        if ( null === $settings->get( $section, 'enabled', null ) ) {
            // Default to enabled for all sites – monitoring itself is safe for Free.
            $settings->set( $section, 'enabled', 'yes' );
        }

        if ( null === $settings->get( $section, 'interval', null ) ) {
            // Default scan interval (minutes).
            $settings->set( $section, 'interval', '15' );
        }

        $settings->save();
    }

    /**
     * Initialise the module.
     *
     * Monitoring itself is available for both Free and Pro installs. Pro adds
     * additional controls (custom paths, exclusions, email alerts) but the core
     * detection runs everywhere so that admins can see real value immediately.
     *
     * @return void
     */
    public function init() {
        // Register custom schedules used by the monitor.
        add_filter( 'cron_schedules', array( $this, 'register_cron_schedules' ) );

        $hook = 'aegisshield_file_monitor_scan';

        // Determine desired schedule based on stored settings.
        $schedule = $this->get_cron_schedule_slug();

		$next = wp_next_scheduled( $hook );

		if ( $next ) {
			// Unschedule existing event and reschedule with the new interval.
			wp_unschedule_event( $next, $hook );
			$next = false; // IMPORTANT: force reschedule below
		}

		if ( ! $next ) {
			// Schedule the first run shortly in the future using the chosen schedule.
			wp_schedule_event( time() + MINUTE_IN_SECONDS, $schedule, $hook );
		}

        add_action( $hook, array( $this, 'run_monitor_scan' ) );
    }

	/**
	 * Centralized File Integrity Monitoring (FIM) violation/event logger.
	 *
	 * - Always logs "not working" conditions (cron/db/scan failures)
	 * - Logs rule violations ONLY when the rule is enabled
	 * - Alerts Pro decides if/when to notify
	 *
	 * @param string $rule     Rule key (e.g. 'new','modified','deleted','php','high_risk','plugin_theme','monitor_disabled', etc.)
	 * @param string $message  Human message
	 * @param string $severity 'low'|'medium'|'high'|'critical'
	 * @param array  $meta     Extra context
	 * @param string $event    Activity Log event slug (default fim_violation)
	 * @return void
	 */
	protected function log_fim_violation( $rule, $message, $severity = 'medium', $meta = array(), $event = 'fim_violation' ) {
		$logger = $this->plugin ? $this->plugin->get_logger() : null;
		if ( ! $logger ) {
			return;
		}

		if ( ! is_array( $meta ) ) {
			$meta = array();
		}

		// keep payload small / safe
		if ( isset( $meta['sample_paths'] ) && is_array( $meta['sample_paths'] ) ) {
			$meta['sample_paths'] = array_slice( $meta['sample_paths'], 0, 20 );
		}

		$meta = array_merge(
			array(
				'module' => 'file_monitor',
				'rule'   => (string) $rule,
			),
			$meta
		);

		// Use the 4-arg signature used across AegisShield modules
		$logger->log( (string) $event, (string) $message, (string) $severity, $meta );
	}

	/**
	 * Helper: is a monitor rule enabled via settings?
	 * Uses the same "Notify when" array used by your File Integrity UI.
	 *
	 * @param string $rule
	 * @return bool
	 */
	protected function fim_rule_enabled( $rule ) {
		$settings     = $this->plugin->get_settings();
		$email_events = (array) $settings->get( 'file_monitor', 'email_events', array() );

		return in_array( (string) $rule, $email_events, true );
	}

	/**
	 * Helper: quick checks for classification.
	 */
	protected function is_high_risk_path( $rel_path ) {
		$rel_path = ltrim( (string) $rel_path, '/\\' );
		return ( 0 === strpos( $rel_path, 'wp-admin/' ) || 0 === strpos( $rel_path, 'wp-includes/' ) );
	}

	protected function is_plugin_or_theme_path( $abs_path ) {
		$abs_path = (string) $abs_path;

		if ( defined( 'WP_PLUGIN_DIR' ) && 0 === strpos( $abs_path, wp_normalize_path( WP_PLUGIN_DIR ) ) ) {
			return true;
		}

		if ( function_exists( 'get_theme_root' ) ) {
			$theme_root = wp_normalize_path( get_theme_root() );
			if ( $theme_root && 0 === strpos( $abs_path, $theme_root ) ) {
				return true;
			}
		}

		return false;
	}

    /**
     * Register cron schedules used by the file monitor.
     *
     * @param array $schedules Existing schedules.
     * @return array
     */
    public function register_cron_schedules( $schedules ) {
        // 5 minutes.
        if ( ! isset( $schedules['aegisshield_file_monitor_5min'] ) ) {
            $schedules['aegisshield_file_monitor_5min'] = array(
                'interval' => 5 * MINUTE_IN_SECONDS,
                'display'  => __( 'Every 5 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
            );
        }

        // 10 minutes.
        if ( ! isset( $schedules['aegisshield_file_monitor_10min'] ) ) {
            $schedules['aegisshield_file_monitor_10min'] = array(
                'interval' => 10 * MINUTE_IN_SECONDS,
                'display'  => __( 'Every 10 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
            );
        }

        // 15 minutes (default).
        if ( ! isset( $schedules['aegisshield_file_monitor_15min'] ) ) {
            $schedules['aegisshield_file_monitor_15min'] = array(
                'interval' => 15 * MINUTE_IN_SECONDS,
                'display'  => __( 'Every 15 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
            );
        }

        // 30 minutes.
        if ( ! isset( $schedules['aegisshield_file_monitor_30min'] ) ) {
            $schedules['aegisshield_file_monitor_30min'] = array(
                'interval' => 30 * MINUTE_IN_SECONDS,
                'display'  => __( 'Every 30 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
            );
        }

        // 60 minutes.
        if ( ! isset( $schedules['aegisshield_file_monitor_60min'] ) ) {
            $schedules['aegisshield_file_monitor_60min'] = array(
                'interval' => 60 * MINUTE_IN_SECONDS,
                'display'  => __( 'Every 60 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
            );
        }

        return $schedules;
    }

    /**
     * Determine the cron schedule slug to use based on stored settings.
     *
     * @return string
     */
    protected function get_cron_schedule_slug() {
        $settings = $this->plugin->get_settings();
        $section  = 'file_monitor';

        $interval = (string) $settings->get( $section, 'interval', '15' );
        if ( ! in_array( $interval, array( '5', '10', '15', '30', '60' ), true ) ) {
            $interval = '15';
        }

        switch ( $interval ) {
            case '5':
                return 'aegisshield_file_monitor_5min';
            case '10':
                return 'aegisshield_file_monitor_10min';
            case '30':
                return 'aegisshield_file_monitor_30min';
            case '60':
                return 'aegisshield_file_monitor_60min';
            case '15':
            default:
                return 'aegisshield_file_monitor_15min';
        }
    }

    /**
     * Return the fully qualified table name for the monitor.
     *
     * @return string
     */
    protected function get_table_name() {
        global $wpdb;

        return $wpdb->prefix . 'aegisshield_file_monitor';
    }

    /**
     * Create the monitor table on demand for existing installations.
     *
     * @return void
     */
    protected function maybe_create_table() {
        global $wpdb;

        $table  = $this->get_table_name();
        $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) );

        if ( $exists === $table ) {
            return;
        }

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            file_path VARCHAR(255) NOT NULL,
            hash VARCHAR(64) NOT NULL,
            size BIGINT UNSIGNED NOT NULL DEFAULT 0,
            mtime BIGINT UNSIGNED NOT NULL DEFAULT 0,
            first_seen DATETIME NOT NULL,
            last_seen DATETIME NOT NULL,
            status VARCHAR(20) DEFAULT 'unchanged',
            PRIMARY KEY  (id),
            KEY status (status),
            KEY mtime (mtime)
        ) {$charset_collate};";

        dbDelta( $sql );
    }

    /**
     * Cron callback – perform a lightweight scan of key directories.
     *
     * @return void
     */
	public function run_monitor_scan() {
		$settings = $this->plugin->get_settings();

		// ENFORCE: if monitor is disabled, do not scan.
		$enabled = (string) $settings->get( 'file_monitor', 'enabled', 'yes' );
		if ( 'yes' !== $enabled ) {
			// LOG: cron fired while disabled (useful to detect mis-scheduling)
			$this->log_fim_violation(
				'monitor_disabled',
				__( 'File Change Monitor is disabled but the scheduled scan hook executed.', 'aegisshield-security' ),
				'medium',
				array(),
				'fim_violation'
			);
			return;
		}

		$this->maybe_create_table();
		
		global $wpdb;
		$table = $this->get_table_name();

		// LOG: if table still doesn't exist, the monitor cannot function.
		$exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table ) );
		if ( $exists !== $table ) {
			$this->log_fim_violation(
				'monitor_table_missing',
				__( 'File Change Monitor table is missing; monitoring cannot run.', 'aegisshield-security' ),
				'high',
				array( 'table' => $table ),
				'fim_violation'
			);
			return;
		}

        global $wpdb;

        $table = $this->get_table_name();

        // Build monitored roots – all are read‑only scans.
        $roots = array();

        if ( defined( 'WP_CONTENT_DIR' ) ) {
            $roots[] = WP_CONTENT_DIR;
        }
        if ( defined( 'WP_PLUGIN_DIR' ) ) {
            $roots[] = WP_PLUGIN_DIR;
        }
        if ( function_exists( 'get_theme_root' ) ) {
            $roots[] = get_theme_root();
        }

        if ( empty( $roots ) ) {
            return;
        }

        // Normalise and de‑duplicate paths.
        $roots = array_values(
            array_unique(
                array_map(
                    'untrailingslashit',
                    array_filter( $roots, 'is_dir' )
                )
            )
        );

		// PRO: custom paths and exclusions (if present in settings).
		$custom_paths = (array) $settings->get( 'file_monitor', 'custom_paths', array() );
		$exclude_dirs = (array) $settings->get( 'file_monitor', 'exclude_dirs', array() );

		// Merge custom roots (enforce)
		foreach ( $custom_paths as $p ) {
			$p = (string) $p;
			if ( '' === trim( $p ) ) {
				continue;
			}
			$p = untrailingslashit( wp_normalize_path( $p ) );
			if ( @is_dir( $p ) ) { // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
				$roots[] = $p;
			} else {
				$this->log_fim_violation(
					'custom_path_invalid',
					__( 'File Change Monitor custom path is not a directory and was skipped.', 'aegisshield-security' ),
					'medium',
					array( 'path' => $p ),
					'fim_violation'
				);
			}
		}

		// Normalize roots again after merge
		$roots = array_values( array_unique( array_map( 'untrailingslashit', $roots ) ) );

		// Normalize exclusions once
		$exclude_dirs = array_values(
			array_unique(
				array_filter(
					array_map(
						function ( $p ) {
							$p = (string) $p;
							if ( '' === trim( $p ) ) {
								return '';
							}
							return untrailingslashit( wp_normalize_path( $p ) );
						},
						$exclude_dirs
					)
				)
			)
		);

        if ( empty( $roots ) ) {
            return;
        }

        // Load existing rows into a map keyed by file_path.
        $existing_rows = array();
        $rows          = $wpdb->get_results(
            "SELECT id, file_path, hash, size, mtime, status FROM {$table}",
            ARRAY_A
        );

        if ( is_array( $rows ) ) {
            foreach ( $rows as $row ) {
                $existing_rows[ $row['file_path'] ] = $row;
            }
        }

        $now_mysql  = current_time( 'mysql' );
        $seen_paths = array();

        $logger = $this->plugin->get_logger();

        foreach ( $roots as $root ) {
            try {
                $iterator = new \RecursiveIteratorIterator(
                    new \RecursiveDirectoryIterator(
                        $root,
                        \FilesystemIterator::SKIP_DOTS | \FilesystemIterator::FOLLOW_SYMLINKS
                    ),
                    \RecursiveIteratorIterator::SELF_FIRST
                );
            } catch ( \Exception $e ) {
                // If a directory cannot be read, skip it gracefully.
                continue;
            }

            foreach ( $iterator as $file ) {
                if ( ! $file->isFile() ) {
                    continue;
                }

                $path = $file->getPathname();

				$norm_path = untrailingslashit( wp_normalize_path( $path ) );

				// ENFORCE exclusions
				$excluded = false;
				foreach ( $exclude_dirs as $ex ) {
					if ( '' !== $ex && 0 === strpos( $norm_path, $ex ) ) {
						$excluded = true;
						break;
					}
				}

				if ( $excluded ) {
					// If admin enabled exclusions but scan is seeing excluded files, log once per run is better,
					// but for now keep it lightweight: no per-file logs.
					continue;
				}

                // Store relative path from ABSPATH for portability.
                $rel_path = ltrim( str_replace( ABSPATH, '', $path ), '/' );

                $seen_paths[ $rel_path ] = true;

                $size  = (int) $file->getSize();
                $mtime = (int) $file->getMTime();

                // Only hash likely code files to keep the scan lightweight.
                $ext      = strtolower( pathinfo( $path, PATHINFO_EXTENSION ) );
                $hashable = in_array(
                    $ext,
                    array( 'php', 'php5', 'php7', 'phtml', 'js', 'css', 'inc', 'html', 'htm' ),
                    true
                );

                $hash = '';
                if ( $hashable ) {
                    $hash_raw = @hash_file( 'sha256', $path ); // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
                    if ( false !== $hash_raw ) {
                        $hash = $hash_raw;
                    }
                }

                if ( ! isset( $existing_rows[ $rel_path ] ) ) {
                    // New file.
                    $wpdb->insert(
                        $table,
                        array(
                            'file_path'  => $rel_path,
                            'hash'       => $hash,
                            'size'       => $size,
                            'mtime'      => $mtime,
                            'first_seen' => $now_mysql,
                            'last_seen'  => $now_mysql,
                            'status'     => 'new',
                        ),
                        array(
                            '%s',
                            '%s',
                            '%d',
                            '%d',
                            '%s',
                            '%s',
                            '%s',
                        )
                    );

					// LOG: rule-based events only when enabled
					if ( $this->fim_rule_enabled( 'new' ) ) {
						$sev = 'medium';

						// PHP changes are higher risk
						if ( in_array( $ext, array( 'php', 'php5', 'php7', 'phtml' ), true ) && $this->fim_rule_enabled( 'php' ) ) {
							$sev = 'high';
							$this->log_fim_violation(
								'php',
								__( 'FIM rule triggered: new PHP file detected.', 'aegisshield-security' ),
								'high',
								array(
									'path' => $rel_path,
									'ext'  => $ext,
								),
								'fim_php_file_violation'
							);
						}

						// High-risk directory
						if ( $this->fim_rule_enabled( 'high_risk' ) && $this->is_high_risk_path( $rel_path ) ) {
							$sev = 'high';
							$this->log_fim_violation(
								'high_risk',
								__( 'FIM rule triggered: new file detected in high-risk WordPress directory.', 'aegisshield-security' ),
								'high',
								array(
									'path' => $rel_path,
									'ext'  => $ext,
								),
								'fim_high_risk_path_violation'
							);
						}

						// Plugin/theme directories
						if ( $this->fim_rule_enabled( 'plugin_theme' ) && $this->is_plugin_or_theme_path( wp_normalize_path( $path ) ) ) {
							$this->log_fim_violation(
								'plugin_theme',
								__( 'FIM rule triggered: new file detected in plugin/theme directory.', 'aegisshield-security' ),
								$sev,
								array(
									'path' => $rel_path,
									'ext'  => $ext,
								),
								'fim_rules_access'
							);
						}

						$this->log_fim_violation(
							'new',
							__( 'FIM rule triggered: new file detected.', 'aegisshield-security' ),
							$sev,
							array(
								'path' => $rel_path,
								'ext'  => $ext,
							),
							'fim_file_created'
						);
					}
                } else {
                    // Existing file – check for changes.
                    $row           = $existing_rows[ $rel_path ];
                    $previous_hash = (string) $row['hash'];
                    $previous_size = (int) $row['size'];
                    $previous_mtime = (int) $row['mtime'];

                    $status = 'unchanged';

                    if ( $previous_hash !== (string) $hash || $previous_size !== $size || $previous_mtime !== $mtime ) {
                        $status = 'modified';

						if ( $this->fim_rule_enabled( 'modified' ) ) {
							$sev = 'medium';

							if ( in_array( $ext, array( 'php', 'php5', 'php7', 'phtml' ), true ) && $this->fim_rule_enabled( 'php' ) ) {
								$sev = 'high';
								$this->log_fim_violation(
									'php',
									__( 'FIM rule triggered: PHP file modified.', 'aegisshield-security' ),
									'high',
									array(
										'path' => $rel_path,
										'ext'  => $ext,
									),
									'fim_php_file_violation'
								);
							}

							if ( $this->fim_rule_enabled( 'high_risk' ) && $this->is_high_risk_path( $rel_path ) ) {
								$sev = 'high';
								$this->log_fim_violation(
									'high_risk',
									__( 'FIM rule triggered: file modified in high-risk WordPress directory.', 'aegisshield-security' ),
									'high',
									array(
										'path' => $rel_path,
										'ext'  => $ext,
									),
									'fim_high_risk_path_violation'
								);
							}

							if ( $this->fim_rule_enabled( 'plugin_theme' ) && $this->is_plugin_or_theme_path( wp_normalize_path( $path ) ) ) {
								$this->log_fim_violation(
									'plugin_theme',
									__( 'FIM rule triggered: plugin/theme file modified.', 'aegisshield-security' ),
									$sev,
									array(
										'path' => $rel_path,
										'ext'  => $ext,
									),
									'fim_rules_access'
								);
							}

							$this->log_fim_violation(
								'modified',
								__( 'FIM rule triggered: file modified.', 'aegisshield-security' ),
								$sev,
								array(
									'path' => $rel_path,
									'ext'  => $ext,
								),
								'fim_file_modified'
							);
						}
                    }

                    $wpdb->update(
                        $table,
                        array(
                            'hash'      => $hash,
                            'size'      => $size,
                            'mtime'     => $mtime,
                            'last_seen' => $now_mysql,
                            'status'    => $status,
                        ),
                        array(
                            'id' => (int) $row['id'],
                        ),
                        array(
                            '%s',
                            '%d',
                            '%d',
                            '%s',
                            '%s',
                        ),
                        array(
                            '%d',
                        )
                    );

                    // Remove from the map so remaining rows at the end
                    // represent files that disappeared.
                    unset( $existing_rows[ $rel_path ] );
                }
            }
        }

        // Any rows still left in $existing_rows correspond to files
        // that previously existed but were not seen in this scan.
        if ( ! empty( $existing_rows ) ) {
            foreach ( $existing_rows as $rel_path => $row ) {
                $wpdb->update(
                    $table,
                    array(
                        'last_seen' => $now_mysql,
                        'status'    => 'deleted',
                    ),
                    array(
                        'id' => (int) $row['id'],
                    ),
                    array(
                        '%s',
                        '%s',
                    ),
                    array(
                        '%d',
                    )
                );

				if ( $this->fim_rule_enabled( 'deleted' ) ) {
					$sev = 'high';

					if ( $this->fim_rule_enabled( 'high_risk' ) && $this->is_high_risk_path( $rel_path ) ) {
						$this->log_fim_violation(
							'high_risk',
							__( 'FIM rule triggered: file deleted in high-risk WordPress directory.', 'aegisshield-security' ),
							'high',
							array( 'path' => $rel_path ),
							'fim_high_risk_path_violation'
						);
					}

					$this->log_fim_violation(
						'deleted',
						__( 'FIM rule triggered: file deleted/missing.', 'aegisshield-security' ),
						$sev,
						array( 'path' => $rel_path ),
						'fim_file_deleted'
					);
				}
            }
        }
    }
}
