<?php
namespace AegisShield\Modules;

use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

/**
 * Hardening module.
 *
 * Provides toggleable safe hardening options.
 */
class AS_Module_Hardening implements AS_Module_Interface {

    /**
     * Plugin instance.
     *
     * @var AS_Plugin
     */
    protected $plugin;

    /**
     * Constructor.
     *
     * @param AS_Plugin $plugin Plugin instance.
     */
    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Get module slug.
     *
     * @return string
     */
    public function get_slug() {
        return 'hardening';
    }

    /**
     * Register settings defaults.
     */
    public function register_settings() {
        $settings = $this->plugin->get_settings();
        $section  = 'hardening';

        /* --- Existing defaults preserved --- */
        $defaults = [
            'disable_file_edit'          => 'off',
            'disable_xmlrpc_behavior'    => 'c',
            'block_user_enum_behavior'   => 'a',
            'hide_wp_version'            => 'off',
            'disable_editor_screens'     => 'off',
            'enforce_strong_passwords'   => 'off',
            'password_min_length'        => 12,
            'password_require_uppercase' => 'on',
            'password_require_lowercase' => 'on',
            'password_require_number'    => 'on',
            'password_require_symbol'    => 'on',

            /* --- PHASE 1 + 2 New Defaults --- */
            'hardening_vuln_source'      => 'local',
            'hardening_vuln_patchstack_key' => '',
            'hardening_vuln_last_run'    => 0,
            'hardening_vuln_results'     => [],
            'hardening_role_risk_mode'   => 'off',
            'hardening_role_risk_results'=> [],
            'hardening_rest_mode'        => 'default',
            'hardening_xmlrpc_mode'      => 'default',
            'hardening_audit_enabled'    => 'off',
            'hardening_audit_last_sent'  => 0,
        ];

        foreach ( $defaults as $key => $value ) {
            if ( null === $settings->get( $section, $key, null ) ) {
                $settings->set( $section, $key, $value );
            }
        }

        $settings->save();
    }

    /**
     * Initialize hooks for the hardening module.
     */
    public function init() {
        $settings = $this->plugin->get_settings();
        $section  = 'hardening';

        /* ---------- Existing Hardening Logic (Unchanged) ---------- */

        $disable_file_edit      = (string) $settings->get( $section, 'disable_file_edit', 'off' );
        $xmlrpc_behavior        = (string) $settings->get( $section, 'disable_xmlrpc_behavior', 'c' );
        $block_user_enum        = (string) $settings->get( $section, 'block_user_enum_behavior', 'a' );
        $hide_wp_version        = (string) $settings->get( $section, 'hide_wp_version', 'off' );
        $disable_editor_screens = (string) $settings->get( $section, 'disable_editor_screens', 'off' );
        $enforce_passwords      = (string) $settings->get( $section, 'enforce_strong_passwords', 'off' );

        /* Disable file editor */
        if ( in_array( $disable_file_edit, [ 'on', '1', 'yes' ], true ) ) {
            if ( ! defined( 'DISALLOW_FILE_EDIT' ) ) {
                define( 'DISALLOW_FILE_EDIT', true );
            }
            add_filter( 'editable_extensions', [ $this, 'disable_editable_extensions' ] );
        }

		add_action( 'load-theme-editor.php', function() use ( $disable_file_edit ) {
			if ( $disable_file_edit === 'on' ) {
				$this->log_hardening_violation(
					'disable_file_edit',
					'high',
					[ 'screen' => 'theme-editor' ]
				);
			}
		});

		add_action( 'load-plugin-editor.php', function() use ( $disable_file_edit ) {
			if ( $disable_file_edit === 'on' ) {
				$this->log_hardening_violation(
					'disable_file_edit',
					'high',
					[ 'screen' => 'plugin-editor' ]
				);
			}
		});

        /* XML-RPC handling */
        if ( $xmlrpc_behavior === 'c' ) {
            add_filter( 'xmlrpc_methods', function ( $methods ) {
                unset( $methods['wp.deletePage'], $methods['wp.deletePost'] );
                unset( $methods['wp.editPage'], $methods['wp.editPost'] );
                unset( $methods['wp.newPage'], $methods['wp.newPost'] );
                return $methods;
            });
        } else {
            add_filter( 'xmlrpc_enabled', '__return_false' );
            add_action( 'init', [ $this, 'handle_xmlrpc_request' ] );
        }

        /* Block user enumeration */
        add_action( 'template_redirect', [ $this, 'block_user_enumeration' ] );

        /* Hide WP version */
        if ( $hide_wp_version === 'on' ) {
            add_action( 'init', [ $this, 'hide_wp_version_meta' ] );
            add_filter( 'the_generator', '__return_empty_string' );
            add_filter( 'style_loader_src', [ $this, 'strip_version_from_assets' ], 9999 );
            add_filter( 'script_loader_src', [ $this, 'strip_version_from_assets' ], 9999 );
        }

        /* Disable editor screens */
        if ( $disable_editor_screens === 'on' ) {
            add_action( 'admin_menu', [ $this, 'remove_editor_submenus' ], 999 );
        }
		
		add_action( 'load-theme-editor.php', function() use ( $disable_file_edit ) {
			if ( $disable_file_edit === 'on' ) {
				$this->log_hardening_violation(
					'disable_file_edit',
					'high',
					[ 'screen' => 'theme-editor' ]
				);
			}
		});

		add_action( 'load-plugin-editor.php', function() use ( $disable_file_edit ) {
			if ( $disable_file_edit === 'on' ) {
				$this->log_hardening_violation(
					'disable_file_edit',
					'high',
					[ 'screen' => 'plugin-editor' ]
				);
			}
		});

        /* Enforce strong passwords */
        if ( $enforce_passwords === 'on' ) {
            add_action( 'user_profile_update_errors', [ $this, 'enforce_password_strength_profile' ], 10, 3 );
            add_action( 'register_post', [ $this, 'enforce_password_strength_registration' ], 10, 3 );
            add_action( 'validate_password_reset', [ $this, 'enforce_password_strength_reset' ], 10, 2 );
        }

        /* ---------------------------------------------------------
         * PHASE 2 — NEW REAL HARDENING LOGIC (Pro)
         * --------------------------------------------------------- */

        $this->init_rest_api_controls();
        $this->init_xmlrpc_controls();
        $this->init_weekly_hardening_audit();

        /* Optional: scheduled vulnerability scanning in future phase */
    }

    /************************************************************
     * PHASE 2 — REAL HARDENING LOGIC
     ************************************************************/

    /**
     * REST API Enforcement Engine
     */
    protected function init_rest_api_controls() {
        $settings = $this->plugin->get_settings();
        $mode = $settings->get( 'hardening', 'hardening_rest_mode', 'default' );

        add_filter( 'rest_authentication_errors', function( $result ) use ( $mode ) {

            if ( is_user_logged_in() ) { return $result; }
            if ( $mode === 'default' ) { return $result; }

            $route = $_SERVER['REQUEST_URI'];

            /* SAFE MODE — block anonymous user enumeration */
            if ( $mode === 'safe' ) {
                if ( strpos( $route, '/wp/v2/users' ) !== false ) {
                    $this->log_hardening_violation(
						'hardening_rest_mode',
						'high',
						[ 'route' => $route ]
					);

					return new \WP_Error( 'rest_blocked', 'REST endpoint blocked for security.', [ 'status' => 403 ] );
                }
                return $result;
            }

            /* HARDENED MODE — only allow basic public content */
            if ( $mode === 'hardened' ) {
                $allowed = [ 'wp/v2/posts', 'wp/v2/categories', 'wp/v2/tags' ];
                foreach ( $allowed as $a ) {
                    if ( strpos( $route, $a ) !== false ) {
                        return $result;
                    }
                }
                return new \WP_Error( 'rest_blocked', 'Anonymous REST API blocked.', [ 'status' => 403 ] );
            }

            return $result;
        });
    }

    /**
     * Advanced XML-RPC Controls
     */
    protected function init_xmlrpc_controls() {
        $settings = $this->plugin->get_settings();
        $mode = $settings->get( 'hardening', 'hardening_xmlrpc_mode', 'default' );

        if ( $mode === 'disabled' ) {
            add_filter( 'xmlrpc_enabled', '__return_false' );
            return;
        }

        if ( $mode === 'jetpack' ) {
            add_filter( 'xmlrpc_methods', function( $methods ) {
                $allowed = [
                    'system.multicall',
                    'system.getCapabilities',
                    'jetpack.verify'
                ];
                return array_intersect_key( $methods, array_flip( $allowed ) );
            });
        }
    }

    /**
     * Schedule weekly hardening audit reports.
     */
    protected function init_weekly_hardening_audit() {

        if ( ! wp_next_scheduled( 'aegisshield_hardening_audit_weekly' ) ) {
            wp_schedule_event( time() + 3600, 'weekly', 'aegisshield_hardening_audit_weekly' );
        }

        add_action( 'aegisshield_hardening_audit_weekly', function() {

            $settings = $this->plugin->get_settings();
            $section  = 'hardening';

            if ( $settings->get( $section, 'hardening_audit_enabled', 'off' ) !== 'on' ) {
                return;
            }

            $vuln = $settings->get( $section, 'hardening_vuln_results', [] );
            $roles = $settings->get( $section, 'hardening_role_risk_results', [] );

            $body = "AegisShield Weekly Hardening Report\n\n";
            $body .= "Vulnerability Findings:\n" . print_r( $vuln, true ) . "\n\n";
            $body .= "Role & Capability Risk Findings:\n" . print_r( $roles, true ) . "\n\n";

            $notifier = $this->plugin->get_notifier();
            $notifier->send_admin_alert( 'Weekly Hardening Report', nl2br( $body ) );

            $settings->set( $section, 'hardening_audit_last_sent', time() );
            $settings->save();
        });
    }

    /**
     * Run vulnerability scan on-demand or via cron.
     */
    public function run_vulnerability_scan() {

        $settings = $this->plugin->get_settings();
        $section  = 'hardening';

        $source  = $settings->get( $section, 'hardening_vuln_source', 'local' );
        $api_key = $settings->get( $section, 'hardening_vuln_patchstack_key', '' );

        $results = [];

        /* Plugins */
        foreach ( get_plugins() as $slug => $data ) {
            $version = $data['Version'] ?? '';
            $risk = $this->lookup_plugin_vulnerability( $source, $slug, $version, $api_key );
            if ( $risk ) {
                $results['plugins'][$slug] = $risk;
            }
        }

        /* Themes */
        foreach ( wp_get_themes() as $slug => $theme ) {
            $version = $theme->get( 'Version' );
            $risk = $this->lookup_theme_vulnerability( $source, $slug, $version, $api_key );
            if ( $risk ) {
                $results['themes'][$slug] = $risk;
            }
        }

        $settings->set( $section, 'hardening_vuln_results', $results );
        $settings->set( $section, 'hardening_vuln_last_run', time() );
        $settings->save();

        return $results;
    }

    /**
     * Run role & capability analyzer.
     */
    public function run_role_risk_scan() {

        $dangerous_caps = [
            'manage_options',
            'edit_plugins',
            'delete_users',
            'unfiltered_html',
            'edit_theme_options'
        ];

        $results = [];

        foreach ( get_users() as $u ) {

            $caps = array_keys( $u->allcaps );
            $hit  = array_intersect( $caps, $dangerous_caps );

            $last_login = get_user_meta( $u->ID, 'last_login', true );
            if ( ! $last_login ) {
                $last_login = strtotime( $u->user_registered );
            }

            $inactive = floor( ( time() - $last_login ) / DAY_IN_SECONDS );

            $risk = 'low';
            if ( ! empty( $hit ) ) { $risk = 'medium'; }
            if ( $inactive > 30 && ! empty( $hit ) ) { $risk = 'high'; }

            if ( $risk !== 'low' ) {
                $results[] = [
                    'user_id'       => $u->ID,
                    'username'      => $u->user_login,
                    'role'          => implode( ',', $u->roles ),
                    'dangerous_caps'=> $hit,
                    'last_login'    => $last_login,
                    'risk'          => $risk
                ];
            }
        }

        $settings = $this->plugin->get_settings();
        $settings->set( 'hardening', 'hardening_role_risk_results', $results );
        $settings->save();

        return $results;
    }

    /**
     * Vulnerability lookup routing
     */
    protected function lookup_plugin_vulnerability( $source, $slug, $version, $api_key ) {

        if ( $source === 'local' ) {
            return $this->local_risk_heuristic( $version );
        }

        if ( $source === 'patchstack' ) {
            return $this->patchstack_lookup( $slug, $version, $api_key );
        }

        if ( $source === 'wpvulndb' ) {
            return $this->wpvulndb_lookup( $slug, $version );
        }

        return null;
    }

    protected function lookup_theme_vulnerability( $source, $slug, $version, $api_key ) {
        return $this->lookup_plugin_vulnerability( $source, $slug, $version, $api_key );
    }

    /**
     * Local heuristic fallback
     */
    protected function local_risk_heuristic( $version ) {
        if ( empty( $version ) ) {
            return [
                'risk'   => 'medium',
                'reason' => 'Unknown version, cannot determine safety.'
            ];
        }

        return [
            'risk'   => 'low',
            'reason' => 'No external vulnerability feed enabled.'
        ];
    }

    /**
     * Patchstack lookup
     */
    protected function patchstack_lookup( $slug, $version, $api_key ) {

        if ( empty( $api_key ) ) {
            return null;
        }

        $url  = "https://api.patchstack.com/v1/plugin/" . urlencode( $slug );
        $args = [
            'headers' => [ 'Authorization' => 'Bearer ' . $api_key ],
            'timeout' => 6,
        ];

        $res = wp_remote_get( $url, $args );
        if ( is_wp_error( $res ) ) { return null; }

        $body = json_decode( wp_remote_retrieve_body( $res ), true );
        if ( ! $body || empty( $body['vulnerabilities'] ) ) { return null; }

        return [
            'risk'   => 'high',
            'reason' => 'Patchstack reports known vulnerabilities.',
            'raw'    => $body['vulnerabilities']
        ];
    }

    /**
     * WPVulnDB lookup
     */
    protected function wpvulndb_lookup( $slug, $version ) {

        $url  = "https://wpvulndb.com/api/v3/plugins/" . urlencode( $slug );
        $args = [ 'timeout' => 6 ];

        $res = wp_remote_get( $url, $args );
        if ( is_wp_error( $res ) ) { return null; }

        $body = json_decode( wp_remote_retrieve_body( $res ), true );
        if ( ! $body || empty( $body[$slug]['vulnerabilities'] ) ) { return null; }

        return [
            'risk'   => 'high',
            'reason' => 'WPVulnDB reports known vulnerabilities.',
            'raw'    => $body[$slug]['vulnerabilities']
        ];
    }

	/************************************************************
	 * HARDENING — CENTRALIZED VIOLATION LOGGER
	 ************************************************************/

	/**
	 * Log a hardening violation event.
	 *
	 * @param string $feature_key  The hardening setting key (e.g. disable_file_edit).
	 * @param string $severity     low | medium | high
	 * @param array  $context      Additional context data.
	 */
	protected function log_hardening_violation( $feature_key, $severity = 'medium', array $context = [] ) {

		// Logger must exist
		if ( ! method_exists( $this->plugin, 'get_logger' ) ) {
			return;
		}

		$logger = $this->plugin->get_logger();
		if ( ! $logger ) {
			return;
		}

		$user_id = get_current_user_id();

		$payload = array_merge(
			[
				'feature_key'  => $feature_key,
				'user_id'      => $user_id ?: 0,
				'requested_url'=> $_SERVER['REQUEST_URI'] ?? '',
				'ip_address'   => $_SERVER['REMOTE_ADDR'] ?? '',
			],
			$context
		);

		$logger->log(
			'hardening_violation',
			$severity,
			$payload
		);
	}

    /************************************************************
     * BELOW THIS — Original helper functions unchanged
     ************************************************************/

    public function disable_editable_extensions( $extensions ) {
        if ( isset( $extensions['php'] ) ) {
            unset( $extensions['php'] );
        }
        return $extensions;
    }

    public function handle_xmlrpc_request() {
        $settings = $this->plugin->get_settings();
        $section  = 'hardening';
        $xmlrpc_behavior = $settings->get( $section, 'disable_xmlrpc_behavior', 'c' );

		if ( $xmlrpc_behavior === 'a' ) {

			$this->log_hardening_violation(
				'disable_xmlrpc_behavior',
				'high',
				[ 'method' => 'xmlrpc' ]
			);

			wp_die( 'XML-RPC requests have been disabled by AegisShield.', '', [ 'response' => 403 ] );
        } elseif ( $xmlrpc_behavior === 'b' ) {

			$this->log_hardening_violation(
				'disable_xmlrpc_behavior',
				'high',
				[ 'method' => 'xmlrpc' ]
			);

			status_header( 403 );
			exit;
		}
    }

    public function block_user_enumeration() {
        if ( ! is_admin() && isset( $_GET['author'] ) ) {
            $settings = $this->plugin->get_settings();
            $behavior = $settings->get( 'hardening', 'block_user_enum_behavior', 'a' );

			if ( $behavior === 'a' ) {

				$this->log_hardening_violation(
					'block_user_enum_behavior',
					'medium',
					[ 'query' => $_SERVER['QUERY_STRING'] ?? '' ]
				);

				wp_safe_redirect( home_url( '/' ) );
				exit;
            } else {
                global $wp_query;
				$this->log_hardening_violation(
					'block_user_enum_behavior',
					'medium',
					[ 'query' => $_SERVER['QUERY_STRING'] ?? '' ]
				);

				$wp_query->set_404();
				status_header( 404 );
				nocache_headers();
            }
        }
    }

    public function hide_wp_version_meta() {
        remove_action( 'wp_head', 'wp_generator' );
    }

    public function strip_version_from_assets( $src ) {
        if ( strpos( $src, 'ver=' ) !== false ) {
            $src = remove_query_arg( 'ver', $src );
        }
        return $src;
    }

    public function remove_editor_submenus() {
        remove_submenu_page( 'themes.php', 'theme-editor.php' );
        remove_submenu_page( 'plugins.php', 'plugin-editor.php' );
    }

    public function enforce_password_strength_profile( $errors, $update, $user ) {
        if ( ! empty( $_POST['pass1'] ) ) {
            $password = (string) $_POST['pass1'];
            $msg = $this->validate_password_strength( $password, $user->ID );
            if ( $msg ) { $errors->add( 'weak_password', $msg ); }
        }
    }

    public function enforce_password_strength_registration( $sanitized_user_login, $user_email, $errors ) {
        $password = '';

        // Support common password field names (core + popular forms).
        if ( ! empty( $_POST['pass1'] ) ) {
            $password = (string) $_POST['pass1'];
        } elseif ( ! empty( $_POST['user_pass'] ) ) {
            $password = (string) $_POST['user_pass'];
        } elseif ( ! empty( $_POST['password'] ) ) {
            $password = (string) $_POST['password'];
        }

        if ( $password !== '' ) {
            $msg = $this->validate_password_strength( $password, 0 );
            if ( $msg ) {
                $errors->add( 'weak_password', $msg );
            }
        }

        return $errors;
    }

    public function enforce_password_strength_reset( $errors, $user ) {
        if ( ! empty( $_POST['pass1'] ) ) {
            $password = (string) $_POST['pass1'];
            $msg = $this->validate_password_strength( $password, $user->ID );
            if ( $msg ) { $errors->add( 'weak_password', $msg ); }
        }
    }

    protected function validate_password_strength( $password, $user_id ) {
        $password = (string) $password;
        $settings = $this->plugin->get_settings();
        $section  = 'hardening';

        $min  = (int) $settings->get( $section, 'password_min_length', 12 );
        $up   = $settings->get( $section, 'password_require_uppercase', 'on' );
        $low  = $settings->get( $section, 'password_require_lowercase', 'on' );
        $num  = $settings->get( $section, 'password_require_number', 'on' );
        $sym  = $settings->get( $section, 'password_require_symbol', 'on' );

		if ( strlen( $password ) < $min ) {
			$this->log_hardening_violation(
				'enforce_strong_passwords',
				'medium',
				[ 'reason' => 'min_length' ]
			);
			return "Password must be at least {$min} characters.";
		}
        if ( $up === 'on'  && ! preg_match( '/[A-Z]/', $password ) ) return "Password must contain uppercase letters.";
        if ( $low === 'on' && ! preg_match( '/[a-z]/', $password ) ) return "Password must contain lowercase letters.";
        if ( $num === 'on' && ! preg_match( '/[0-9]/', $password ) ) return "Password must contain numbers.";
        if ( $sym === 'on' && ! preg_match( '/[^a-zA-Z0-9]/', $password ) ) return "Password must contain symbols.";

        return '';
    }
}
