<?php
namespace AegisShield\Modules;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use AegisShield\AS_Plugin;
use AegisShield\Utils\AS_QR;
use AegisShield\MFA\AS_MFA_Email;
use AegisShield\MFA\AS_MFA_Engine;


/**
 * Login Guard module with PRO MFA (TOTP), trusted devices, and smart lockouts.
 */
class AS_Module_Login_Guard implements AS_Module_Interface {

	/**
	 * Plugin instance.
	 *
	 * @var AS_Plugin
	 */
	protected AS_Plugin $plugin;

    /**
     * One-time, in-memory backup codes generated this request (per user ID).
     *
     * @var array<int, string[]>
     */
    protected $last_generated_backup_codes = array();

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function get_slug() {
        return 'login_guard';
    }

    /**
     * Register settings, including FREE baseline + PRO keys used by UI.
     */
    public function register_settings() {
        $settings = $this->plugin->get_settings();
        $section  = 'login_guard';

        // FREE defaults.
        if ( null === $settings->get( $section, 'max_attempts', null ) ) {
            $settings->set( $section, 'max_attempts', 5 );
        }
        if ( null === $settings->get( $section, 'lockout_minutes', null ) ) {
            $settings->set( $section, 'lockout_minutes', 15 );
        }
        if ( null === $settings->get( $section, 'honeypot_enabled', null ) ) {
            $settings->set( $section, 'honeypot_enabled', 1 );
        }
        if ( null === $settings->get( $section, 'block_unknown_user', null ) ) {
            $settings->set( $section, 'block_unknown_user', 1 );
        }

        // PRO defaults (MFA TOTP, trusted devices, adaptive lockouts, geo, honeypot URL).
        $pro_defaults = array(
            // MFA core toggles (internally still uses pro_2fa_* keys for backward compatibility).
            'pro_2fa_enabled'                     => 'off', // Global switch; admin turns ON in Login Guard Pro panel.
            'pro_2fa_enforced_roles'              => array( 'administrator', 'editor' ), // Per-role enforcement.
            // Trusted devices.
            'pro_trusted_devices_enabled'         => 'off',
            'pro_trusted_device_lifetime_days'    => 30,     // 30 / 60 / 90 days – default 30.
            // Adaptive lockouts (can be extended later without schema changes).
            'pro_lockout_known_threshold'         => 5,
            'pro_lockout_unknown_threshold'       => 3,
            'pro_lockout_admin_threshold'         => 3,
            'pro_lockout_progressive_enabled'     => 'on',
            'pro_username_enum_detection_enabled' => 'on',
            // Geo rules + honeypot URL options (Phase 2/3).
            'pro_geo_mode'                        => 'alert', // "off", "alert", "block", "both".
            'pro_allowed_countries'               => '',
            'pro_honeypot_enabled'                => 'off',
            'pro_honeypot_slug'                   => '',
        );

        foreach ( $pro_defaults as $key => $val ) {
            if ( null === $settings->get( $section, $key, null ) ) {
                $settings->set( $section, $key, $val );
            }
        }

        $settings->save();
    }

    /**
     * Hook runtime behavior.
     */
    public function init() {
        // Login form: honeypot + MFA (TOTP) panel (if PRO MFA is enabled).
        add_action( 'login_form', array( $this, 'output_login_form_extras' ) );
		
		// PRO Honeypot Login URL (decoy endpoint).
        add_action( 'init', array( $this, 'register_honeypot_login_rewrite' ) );
        add_filter( 'query_vars', array( $this, 'add_honeypot_query_vars' ) );
        add_action( 'template_redirect', array( $this, 'handle_honeypot_login_hit' ), 0 );

        // Core login checks: honeypot, lockouts, unknown username, then MFA enforcement.
        add_filter( 'authenticate', array( $this, 'check_login_attempt_and_2fa' ), 30, 3 );

        // Failed / successful login tracking.
        add_action( 'wp_login_failed', array( $this, 'record_failed_login' ), 10, 1 );
        add_action( 'wp_login', array( $this, 'clear_login_attempts' ), 10, 2 );

        // User profile MFA setup UI (enrollment and backup codes).
        add_action( 'show_user_profile', array( $this, 'render_user_2fa_profile_section' ) );
        add_action( 'edit_user_profile', array( $this, 'render_user_2fa_profile_section' ) );
        add_action( 'personal_options_update', array( $this, 'save_user_2fa_profile_section' ) );
        add_action( 'edit_user_profile_update', array( $this, 'save_user_2fa_profile_section' ) );
    }

    /**
     * Shared logging wrapper so enforcement never fails silently.
     *
     * @param string $event
     * @param string $message
     * @param array  $context
     * @return void
     */
    protected function log_violation( $event, $message, array $context = array() ) {
        // Always attach IP if we can.
        if ( empty( $context['ip'] ) ) {
            $context['ip'] = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        }

        // Make sure logging itself never breaks auth.
        try {
            $logger = $this->plugin ? $this->plugin->get_logger() : null;
            if ( $logger && method_exists( $logger, 'log' ) ) {
                $logger->log( (string) $event, (string) $message, $context );
            }
        } catch ( \Throwable $e ) {
            // Intentionally swallow: enforcement must continue even if logging fails.
        }
    }

    /**
     * Convenience: split and normalize allowed countries list.
     *
     * @param string $raw
     * @return array
     */
    protected function normalize_allowed_countries( $raw ) {
        $raw = strtoupper( trim( (string) $raw ) );
        if ( '' === $raw ) {
            return array();
        }
        $parts = preg_split( '/[\s,;]+/', $raw );
        $out   = array();
        foreach ( (array) $parts as $p ) {
            $p = strtoupper( trim( (string) $p ) );
            if ( preg_match( '/^[A-Z]{2}$/', $p ) ) {
                $out[] = $p;
            }
        }
        return array_values( array_unique( $out ) );
    }

    /**
     * Country lookup hook point (so Geo enforcement works even if provider lives elsewhere).
     * Implementations can hook: apply_filters('aegisshield_geo_country', '', $ip)
     *
     * @param string $ip
     * @return string Two-letter country code or empty string if unavailable.
     */
    protected function get_country_for_ip( $ip ) {
        $cc = apply_filters( 'aegisshield_geo_country', '', $ip );
        $cc = strtoupper( trim( (string) $cc ) );
        return preg_match( '/^[A-Z]{2}$/', $cc ) ? $cc : '';
    }

	/**
     * Honeypot addon
     */
    public function add_honeypot_query_vars( $vars ) {
        $vars[] = 'aegisshield_hp_login';
        return $vars;
    }

    public function register_honeypot_login_rewrite() {
        $settings = $this->plugin->get_settings();
        $enabled  = (string) $settings->get( 'login_guard', 'pro_honeypot_enabled', 'off' );
        $slug     = (string) $settings->get( 'login_guard', 'pro_honeypot_slug', '' );
        $slug     = sanitize_title( $slug );

        if ( 'on' !== $enabled || '' === $slug ) {
            return;
        }

        // /{slug}/ → index.php?aegisshield_hp_login=1
        add_rewrite_rule( '^' . preg_quote( $slug, '/' ) . '/?$', 'index.php?aegisshield_hp_login=1', 'top' );
    }

    public function handle_honeypot_login_hit() {
        $hit = get_query_var( 'aegisshield_hp_login', '' );
        if ( '' === $hit ) {
            return;
        }

        $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        if ( ! empty( $ip ) ) {
            $settings  = $this->plugin->get_settings();
            $lock_mins = (int) $settings->get( 'login_guard', 'lockout_minutes', 15 );
            $this->lock_ip( $ip, $lock_mins );
        }

        $this->log_violation(
            'login_honeypot_url_hit',
            __( 'Honeypot login URL was accessed (likely automated scanning).', 'aegisshield-security' ),
            array( 'ip' => $ip )
        );

        // Respond like a real endpoint but do not reveal it is a honeypot.
        status_header( 404 );
        nocache_headers();
        exit;
    }

    /**
     * For convenience: get all login_guard settings as an array.
     *
     * @return array
     */
    protected function get_login_guard_settings() {
        $settings = $this->plugin->get_settings()->get( 'login_guard', array() );
        return is_array( $settings ) ? $settings : array();
    }

    /**
     * Check if global MFA is switched on in Login Guard settings.
     *
     * @return bool
     */
    protected function is_global_2fa_enabled() {
        $settings = $this->plugin->get_settings();
        $flag     = (string) $settings->get( 'login_guard', 'pro_2fa_enabled', 'off' );
        return ( 'on' === $flag );
    }

    /**
     * Check whether MFA (TOTP) is required for this user (enforced role OR user opted in).
     *
     * @param \WP_User $user
     * @return bool
     */
    protected function is_2fa_required_for_user( $user ) {
        if ( ! ( $user instanceof \WP_User ) ) {
            return false;
        }

        if ( ! $this->is_global_2fa_enabled() ) {
            return false;
        }

        $settings = $this->plugin->get_settings();
        $section  = 'login_guard';

        $enforced_roles = $settings->get( $section, 'pro_2fa_enforced_roles', array( 'administrator', 'editor' ) );
        if ( ! is_array( $enforced_roles ) ) {
            $enforced_roles = array_filter( array_map( 'trim', explode( ',', (string) $enforced_roles ) ) );
        }

        // If any enforced role matches user's roles → required.
        if ( ! empty( $enforced_roles ) ) {
            foreach ( (array) $user->roles as $role ) {
                if ( in_array( $role, $enforced_roles, true ) ) {
                    return true;
                }
            }
        }

        // Otherwise, user can opt-in individually.
        $enabled_meta = get_user_meta( $user->ID, 'aegisshield_2fa_enabled', true );
        return ( '1' === (string) $enabled_meta );
    }

    /**
     * Is this device trusted for this user?
     *
     * @param \WP_User $user
     * @return bool
     */
    protected function is_trusted_device( $user ) {
        if ( ! ( $user instanceof \WP_User ) ) {
            return false;
        }

        $settings = $this->plugin->get_settings();
        $enabled  = (string) $settings->get( 'login_guard', 'pro_trusted_devices_enabled', 'off' );
        if ( 'on' !== $enabled ) {
            return false;
        }

        $cookie_name = 'aegisshield_trusted_device';
        if ( empty( $_COOKIE[ $cookie_name ] ) ) {
            return false;
        }

        $cookie_val = trim( (string) wp_unslash( $_COOKIE[ $cookie_name ] ) );
        $expected   = hash( 'sha256', $user->ID . wp_salt( 'auth' ) );

        return hash_equals( $expected, $cookie_val );
    }

    /**
     * Set trusted device cookie for this user.
     *
     * @param \WP_User $user
     * @return void
     */
    protected function set_trusted_device_cookie( $user ) {
        if ( ! ( $user instanceof \WP_User ) ) {
            return;
        }

        $settings = $this->plugin->get_settings();
        $days     = (int) $settings->get( 'login_guard', 'pro_trusted_device_lifetime_days', 30 );
        if ( $days <= 0 ) {
            $days = 30;
        }

        $cookie_name  = 'aegisshield_trusted_device';
        $cookie_value = hash( 'sha256', $user->ID . wp_salt( 'auth' ) );
        $expire       = time() + ( $days * DAY_IN_SECONDS );

        setcookie(
            $cookie_name,
            $cookie_value,
            array(
                'expires'  => $expire,
                'path'     => COOKIEPATH,
                'domain'   => COOKIE_DOMAIN,
                'secure'   => is_ssl(),
                'httponly' => true,
                'samesite' => 'Lax',
            )
        );
    }

    /**
     * Login form extras: FREE honeypot + PRO MFA (TOTP) panel with logo.
     */
    public function output_login_form_extras() {
        $settings   = $this->plugin->get_settings();
        $honeypot   = (int) $settings->get( 'login_guard', 'honeypot_enabled', 1 );
        $field_name = 'aegisshield_hp_email';

        if ( $honeypot ) {
            echo '<p style="display:none !important;"><label>Anti-spam: <input type="text" name="' . esc_attr( $field_name ) . '" value="" autocomplete="off" /></label></p>';
        }
		
		// Nonce to protect Login Guard actions (MFA fields, email OTP request, etc).
		wp_nonce_field( 'aegisshield_login_guard', 'aegisshield_login_guard_nonce' );

        if ( ! $this->is_global_2fa_enabled() ) {
            return;
        }
		
		// MFA methods – determine if Email OTP is globally allowed.
		$methods = $settings->get( 'login_guard', 'pro_2fa_methods', array( 'totp' ) );
		if ( ! is_array( $methods ) ) {
			$methods = array( 'totp' );
		}
		$email_allowed = in_array( 'email', $methods, true );

        // ------------------------------------------------------------
        // AEGISSHIELD LOGO — CHANGE LOGO HERE
        // To replace the logo, change the file path below.
        // Example: /assets/images/my_new_logo.png
        // ------------------------------------------------------------
        $logo_url = plugins_url(
            'assets/images/aegis_shield.png',
            dirname( __FILE__, 3 ) . '/aegisshield-security.php'
        );

        ?>
        <style>
            .aegisshield-2fa-panel {
                margin-top: 16px;
                padding: 16px 18px;
                background: #111;
                border-radius: 8px;
                text-align: center;
                box-shadow: 0 2px 8px rgba(0,0,0,0.35);
                color: #f1f1f1;
            }
            .aegisshield-2fa-panel img {
                max-width: 80px;
                height: auto;
                display: block;
                margin: 0 auto 8px auto;
            }
            .aegisshield-2fa-panel h2 {
                font-size: 18px;
                margin: 4px 0 8px 0;
                color: #f5f5f5;
            }
            .aegisshield-2fa-panel p {
                font-size: 13px;
                margin: 0 0 8px 0;
                color: #d0d0d0;
            }
            .aegisshield-2fa-panel input[type="text"],
            .aegisshield-2fa-panel input[type="number"] {
                width: 100%;
                max-width: 260px;
                text-align: center;
            }
            .aegisshield-2fa-panel .aegis-2fa-row {
                margin-bottom: 8px;
            }
            .aegisshield-2fa-panel label {
                font-size: 12px;
                color: #ccc;
            }
        </style>
        <div class="aegisshield-2fa-panel">
            <img src="<?php echo esc_url( $logo_url ); ?>" alt="<?php esc_attr_e( 'AegisShield Security', 'aegisshield-security' ); ?>" />
            <h2><?php esc_html_e( 'AegisShield MFA Protection', 'aegisshield-security' ); ?></h2>
            <p><?php esc_html_e( 'If your account requires multi-factor authentication, enter your 6-digit code or a backup code below.', 'aegisshield-security' ); ?></p>

            <div class="aegis-2fa-row">
                <input type="text"
                       name="aegisshield_2fa_code"
                       maxlength="6"
                       pattern="[0-9]{6}"
                       inputmode="numeric"
                       placeholder="<?php esc_attr_e( '6-digit MFA code', 'aegisshield-security' ); ?>" />
            </div>

            <div class="aegis-2fa-row">
                <input type="text"
                       name="aegisshield_2fa_backup"
                       maxlength="16"
                       placeholder="<?php esc_attr_e( 'Backup code (if needed)', 'aegisshield-security' ); ?>" />
            </div>
			
			<?php if ( $email_allowed ) : ?>
				<div class="aegis-2fa-row">
					<input type="text"
						   name="aegisshield_mfa_email_code"
						   maxlength="6"
						   pattern="[0-9]{6}"
						   inputmode="numeric"
						   placeholder="<?php esc_attr_e( 'Email code (optional)', 'aegisshield-security' ); ?>" />
				</div>

				<div class="aegis-2fa-row">
					<button type="submit"
							name="aegisshield_send_email_otp"
							class="button button-secondary">
						<?php esc_html_e( 'Send code to my email', 'aegisshield-security' ); ?>
					</button>
				</div>
			<?php endif; ?>
	
            <div class="aegis-2fa-row">
                <label>
                    <input type="checkbox" name="aegisshield_trust_device" value="1" />
                    <?php esc_html_e( 'Trust this device for future logins', 'aegisshield-security' ); ?>
                </label>
            </div>
        </div>
        <?php
    }

    /**
     * Combined login checks: FREE checks + PRO MFA enforcement.
     *
     * @param \WP_User|\WP_Error|null $user
     * @param string                  $username
     * @param string                  $password
     *
     * @return \WP_User|\WP_Error|null
     */
    public function check_login_attempt_and_2fa( $user, $username, $password ) {
		$username = (string) $username;
        $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';

        // If a previous auth filter already produced an error, respect it.
        if ( is_wp_error( $user ) ) {
            return $user;
        }

        $settings   = $this->plugin->get_settings();
        $max        = (int) $settings->get( 'login_guard', 'max_attempts', 5 );
        $lock_mins  = (int) $settings->get( 'login_guard', 'lockout_minutes', 15 );
        $block_unk  = (int) $settings->get( 'login_guard', 'block_unknown_user', 1 );
        $field_name = 'aegisshield_hp_email';

		// Only enforce nonce when our Login Guard fields/actions are being used,
		// so we don't break XML-RPC / REST / other auth paths that won't include our nonce.
		$login_guard_posted = (
			isset( $_POST[ $field_name ] ) ||
			isset( $_POST['aegisshield_2fa_code'] ) ||
			isset( $_POST['aegisshield_2fa_backup'] ) ||
			isset( $_POST['aegisshield_mfa_email_code'] ) ||
			isset( $_POST['aegisshield_send_email_otp'] ) ||
			isset( $_POST['aegisshield_trust_device'] )
		);

		if ( $login_guard_posted ) {
			$nonce = isset( $_POST['aegisshield_login_guard_nonce'] )
				? sanitize_text_field( wp_unslash( $_POST['aegisshield_login_guard_nonce'] ) )
				: '';

            if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'aegisshield_login_guard' ) ) {
                $this->log_violation(
                    'login_blocked_invalid_nonce',
                    __( 'Login blocked due to invalid or missing nonce.', 'aegisshield-security' ),
                    array(
                        'ip'       => $ip,
                        'username' => (string) $username,
                    )
                );

                return new \WP_Error(
                    'aegisshield_invalid_nonce',
                    __( '<strong>Error</strong>: Security check failed. Please refresh the page and try again.', 'aegisshield-security' )
                );
            }

		}

        if ( empty( $ip ) ) {
            $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        }

        // Honeypot check (FREE).
        if ( isset( $_POST[ $field_name ] ) && '' !== trim( (string) $_POST[ $field_name ] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
            $this->lock_ip( $ip, $lock_mins );
            $this->log_violation(
                'login_blocked_honeypot',
                __( 'Login attempt blocked due to honeypot trigger.', 'aegisshield-security' ),
                array( 'ip' => $ip )
            );
            return new \WP_Error(
                'aegisshield_honeypot',
                __( '<strong>Error</strong>: Your login request was blocked.', 'aegisshield-security' )
            );
        }

        // IP lockout check (FREE / PRO).
        if ( $this->is_ip_locked( $ip ) ) {
            $this->plugin->get_logger()->log(
                'login_blocked_lockout',
                __( 'Login attempt blocked because the IP is locked out.', 'aegisshield-security' ),
                array( 'ip' => $ip )
            );
            return new \WP_Error(
                'aegisshield_locked',
                __( '<strong>Error</strong>: Too many failed login attempts. Please try again later.', 'aegisshield-security' )
            );
        }

        // ---------- PRO GEO ENFORCEMENT ----------
        $geo_mode = (string) $settings->get( 'login_guard', 'pro_geo_mode', 'off' ); // off|alert|block|both
        $allowed_raw = (string) $settings->get( 'login_guard', 'pro_allowed_countries', '' );
        $allowed = $this->normalize_allowed_countries( $allowed_raw );

        // Only enforce if enabled and allowlist is configured.
        if ( 'off' !== $geo_mode && ! empty( $allowed ) && ! empty( $ip ) ) {
            $cc = $this->get_country_for_ip( $ip );

            if ( '' === $cc ) {
                // Provider missing/unavailable: log that enforcement couldn't run.
                $this->log_violation(
                    'login_geo_lookup_unavailable',
                    __( 'Geo rule enabled but no country provider returned a result for this IP.', 'aegisshield-security' ),
                    array( 'ip' => $ip )
                );
            } elseif ( ! in_array( $cc, $allowed, true ) ) {
                // Outside allowed countries.
                $this->log_violation(
                    'login_geo_violation',
                    __( 'Login attempt originated from a disallowed country.', 'aegisshield-security' ),
                    array(
                        'ip'       => $ip,
                        'country'  => $cc,
                        'allowed'  => implode( ',', $allowed ),
                        'username' => (string) $username,
                    )
                );

                if ( 'block' === $geo_mode || 'both' === $geo_mode ) {
                    $this->lock_ip( $ip, $lock_mins );

                    return new \WP_Error(
                        'aegisshield_geo_blocked',
                        __( '<strong>Error</strong>: Your login request was blocked.', 'aegisshield-security' )
                    );
                }
                // alert-only continues.
            }
        }

        // Block unknown usernames if enabled (FREE).
        if ( $block_unk && ! empty( $username ) ) {
            $user_obj = get_user_by( 'login', $username );
            if ( ! $user_obj ) {
                $this->record_failed_login( $username );
                $this->log_violation(
                    'login_blocked_unknown_user',
                    __( 'Login attempt with unknown username was blocked.', 'aegisshield-security' ),
                    array( 'ip' => $ip, 'username' => $username )
                );
                return new \WP_Error(
                    'aegisshield_unknown_user',
                    __( '<strong>Error</strong>: Invalid username or password.', 'aegisshield-security' )
                );
            }
        }

        // At this point, core auth may still not have authenticated the user.
        if ( ! ( $user instanceof \WP_User ) ) {
            return $user;
        }

		/**
         * If user clicked “Send Email OTP”, generate & email the code.
         */
        if ( isset( $_POST['aegisshield_send_email_otp'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
            AS_MFA_Email::send_email_otp( $user );
        }

        // ---------- PRO MFA (TOTP) ENFORCEMENT STARTS HERE ----------

        // If MFA is not globally enabled, or not required for this user → allow.
        if ( ! $this->is_2fa_required_for_user( $user ) ) {
            return $user;
        }

        // Trusted device bypass (if enabled).
        if ( $this->is_trusted_device( $user ) ) {
            return $user;
        }

        // User must have a secret configured; otherwise they are required to enroll.
        $secret = (string) get_user_meta( $user->ID, 'aegisshield_2fa_secret', true );
        if ( '' === $secret ) {
            $this->log_violation(
                'login_2fa_required_not_enrolled',
                __( 'MFA required for this user, but no secret is enrolled.', 'aegisshield-security' ),
                array(
                    'ip'      => $ip,
                    'user_id' => $user->ID,
                )
            );

            return new \WP_Error(
                'aegisshield_2fa_required',
                __( '<strong>Error</strong>: Multi-factor authentication is required for your account. Please configure it from your profile page.', 'aegisshield-security' )
            );
        }


		/**
		 * Unified MFA validation (TOTP, Email OTP, WebAuthn, Backup Codes)
		 */
		$mfa_valid = AS_MFA_Engine::validate( $user, $_POST );

        if ( ! $mfa_valid ) {
            $this->log_violation(
                'login_mfa_invalid',
                __( 'Invalid or missing MFA factor supplied during login.', 'aegisshield-security' ),
                array(
                    'ip'      => $ip,
                    'user_id' => $user->ID,
                )
            );

            return new \WP_Error(
                'aegisshield_mfa_invalid',
                __( '<strong>Error</strong>: Invalid or missing authentication factor.', 'aegisshield-security' )
            );
        }


        // On successful MFA, optionally trust this device.
        if ( isset( $_POST['aegisshield_trust_device'] ) ) {
            $this->set_trusted_device_cookie( $user );
        }

        $this->plugin->get_logger()->log(
            'login_2fa_success',
            __( 'Multi-factor authentication passed for this login.', 'aegisshield-security' ),
            array(
                'ip'      => $ip,
                'user_id' => $user->ID,
            )
        );

        return $user;
    }

    /**
     * Record failed login attempt; retains your existing logic.
     *
     * @param string $username
     */
    public function record_failed_login( $username ) {
        global $wpdb;

        $settings  = $this->plugin->get_settings();
        $max       = (int) $settings->get( 'login_guard', 'max_attempts', 5 );
        $lock_mins = (int) $settings->get( 'login_guard', 'lockout_minutes', 15 );
		
		// PRO thresholds (if set) — these are saved by the admin UI and must be enforced here.
        $pro_known   = (int) $settings->get( 'login_guard', 'pro_lockout_known_threshold', 0 );
        $pro_unknown = (int) $settings->get( 'login_guard', 'pro_lockout_unknown_threshold', 0 );
        $pro_admin   = (int) $settings->get( 'login_guard', 'pro_lockout_admin_threshold', 0 );

        $progressive = (string) $settings->get( 'login_guard', 'pro_lockout_progressive_enabled', 'off' );

        // Decide which threshold applies for THIS username.
        $threshold = $max;

        $u = null;
        if ( ! empty( $username ) ) {
            $u = get_user_by( 'login', $username );
        }

        if ( $u instanceof \WP_User ) {
            // If the target is an admin, prefer the stricter admin threshold.
            $is_admin = in_array( 'administrator', (array) $u->roles, true );
            if ( $is_admin && $pro_admin > 0 ) {
                $threshold = $pro_admin;
            } elseif ( $pro_known > 0 ) {
                $threshold = $pro_known;
            }
        } else {
            if ( $pro_unknown > 0 ) {
                $threshold = $pro_unknown;
            }
        }

        if ( $threshold < 1 ) {
            $threshold = $max;
        }

        $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        if ( empty( $ip ) ) {
            return;
        }

        $table = $wpdb->prefix . 'aegisshield_login_attempts';

        $row = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM {$table} WHERE ip_address = %s",
                $ip
            )
        );

        $now = current_time( 'mysql' );

        if ( $row ) {
            $attempts = (int) $row->attempts + 1;
            $data     = array(
                'attempts'     => $attempts,
                'last_attempt' => $now,
            );
            $wpdb->update( $table, $data, array( 'ip_address' => $ip ) );
        } else {
            $attempts = 1;
            $data     = array(
                'ip_address'   => $ip,
                'attempts'     => $attempts,
                'last_attempt' => $now,
            );
            $wpdb->insert( $table, $data );
        }

        if ( $attempts >= $threshold ) {
            // Progressive lockouts: increase minutes as attempts exceed threshold.
            $effective_lock_mins = $lock_mins;
            if ( 'on' === $progressive ) {
                $mult = (int) floor( max( 0, $attempts - $threshold ) / max( 1, $threshold ) ) + 1;
                $effective_lock_mins = max( 1, (int) ( $lock_mins * $mult ) );
            }

            $this->lock_ip( $ip, $effective_lock_mins );

            $this->log_violation(
                'login_ip_locked',
                __( 'IP address has been temporarily locked due to too many failed login attempts.', 'aegisshield-security' ),
                array(
                    'ip'        => $ip,
                    'attempts'  => $attempts,
                    'threshold' => $threshold,
                    'minutes'   => $effective_lock_mins,
                    'username'  => (string) $username,
                )
            );
        }

    }

    /**
     * Clear attempts on successful login, and log it.
     */
    public function clear_login_attempts( $user_login, $user ) {
        global $wpdb;

        $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        if ( empty( $ip ) ) {
            return;
        }

        $table = $wpdb->prefix . 'aegisshield_login_attempts';
        $wpdb->delete( $table, array( 'ip_address' => $ip ) );

        $this->plugin->get_logger()->log(
            'login_success_clear_attempts',
            __( 'Login attempts for IP were cleared after successful login.', 'aegisshield-security' ),
            array( 'ip' => $ip, 'user_id' => $user->ID )
        );
    }

    /**
     * Is IP locked out?
     */
    protected function is_ip_locked( $ip ) {
        global $wpdb;

        if ( empty( $ip ) ) {
            return false;
        }

        $table = $wpdb->prefix . 'aegisshield_login_attempts';

        $row = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT locked_until FROM {$table} WHERE ip_address = %s",
                $ip
            )
        );

        if ( ! $row || ! $row->locked_until ) {
            return false;
        }

        $locked_until_ts = strtotime( $row->locked_until );
        $now_ts          = current_time( 'timestamp' );

        if ( $locked_until_ts && $locked_until_ts > $now_ts ) {
            return true;
        }

        // Lock expired, clear it.
        $wpdb->update(
            $table,
            array(
                'locked_until' => null,
                'attempts'     => 0,
            ),
            array( 'ip_address' => $ip )
        );

        return false;
    }

    /**
     * Lock this IP for N minutes.
     */
    protected function lock_ip( $ip, $minutes ) {
        global $wpdb;

        if ( empty( $ip ) ) {
            return;
        }

        $table       = $wpdb->prefix . 'aegisshield_login_attempts';
        $locked_time = gmdate( 'Y-m-d H:i:s', time() + ( max( 1, (int) $minutes ) * MINUTE_IN_SECONDS ) );

        $row = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM {$table} WHERE ip_address = %s",
                $ip
            )
        );

        if ( $row ) {
            $wpdb->update(
                $table,
                array(
                    'locked_until' => $locked_time,
                ),
                array( 'ip_address' => $ip )
            );
        } else {
            $wpdb->insert(
                $table,
                array(
                    'ip_address'   => $ip,
                    'attempts'     => 0,
                    'last_attempt' => current_time( 'mysql' ),
                    'locked_until' => $locked_time,
                )
            );
        }
    }

    /* ==========================================================
     *  USER PROFILE – MFA ENROLLMENT UI (TOTP + BACKUP CODES)
     * ========================================================== */

    /**
     * Render MFA section on user profile page.
     *
     * @param \WP_User $user
     */
    public function render_user_2fa_profile_section( $user ) {
        if ( ! $this->is_global_2fa_enabled() ) {
            return;
        }

        // Only show if current user can edit this profile (or it's their own).
        if ( ! current_user_can( 'edit_user', $user->ID ) ) {
            return;
        }

		$secret  = (string) get_user_meta( $user->ID, 'aegisshield_2fa_secret', true );
		$enabled = get_user_meta( $user->ID, 'aegisshield_2fa_enabled', true );

		// Normalise enabled meta to a simple "0" or "1" string to avoid array-to-string notices.
		if ( is_array( $enabled ) ) {
			$enabled = reset( $enabled );
		}
		$enabled      = ( '' === $enabled ) ? '0' : (string) $enabled;

		// Per-user Email OTP toggle (default ON if not explicitly set).
		$email_enabled = (string) get_user_meta( $user->ID, 'aegisshield_mfa_email_enabled', '' );
		if ( '' === $email_enabled ) {
			$email_enabled = '1';
		}

		$status_label = ( '1' === $enabled ) ? __( 'Enabled', 'aegisshield-security' ) : __( 'Disabled', 'aegisshield-security' );
		$new_codes    = isset( $this->last_generated_backup_codes[ $user->ID ] ) ? $this->last_generated_backup_codes[ $user->ID ] : array();

        $site_name = wp_specialchars_decode( get_bloginfo( 'name' ), ENT_QUOTES );
        $label     = rawurlencode( $site_name . ':' . $user->user_login );
        $issuer    = rawurlencode( $site_name );

        $otpauth = '';
        $qr_url  = '';

		if ( '' !== $secret ) {
			$otpauth = sprintf(
				'otpauth://totp/%s?secret=%s&issuer=%s&digits=6&period=30',
				$label,
				rawurlencode( $secret ),
				$issuer
			);

			// QR Engine Selection (Local SVG or Google Charts).
			$engine = $this->plugin->get_settings()->get( 'login_guard', 'pro_mfa_qr_engine', 'google' );

			// Sanitize: enforce string.
			if ( is_array( $engine ) ) {
			$engine = $engine[0] ?? 'local';
			}

			if ( 'google' === $engine ) {
				// Google Charts API QR.
				$qr_url = 'https://chart.googleapis.com/chart?chs=220x220&chld=M|0&cht=qr&chl=' . rawurlencode( $otpauth );
			} else {
				// Local SVG QR (recommended).
				// If AS_QR isn't available or returns empty, fall back to Google so QR always renders.
				$svg = '';

				if ( class_exists( '\AegisShield\Utils\AS_QR' ) && method_exists( '\AegisShield\Utils\AS_QR', 'svg_base64' ) ) {
					$svg = (string) AS_QR::svg_base64( $otpauth );
				}

				if ( '' !== trim( $svg ) ) {
					$qr_url = 'data:image/svg+xml;base64,' . $svg;
				} else {
					// Fallback (prevents "no QR code" scenarios).
					$qr_url = 'https://chart.googleapis.com/chart?chs=220x220&chld=M|0&cht=qr&chl=' . rawurlencode( $otpauth );
				}
			}
		}

        // Prepare masked secret for toggle view.
        $masked_secret = '';
        if ( '' !== $secret ) {
            $len           = strlen( $secret );
            $masked_secret = str_repeat( '•', max( 8, min( $len, 32 ) ) );
        }

        // ------------------------------------------------------------
        // AEGISSHIELD LOGO — CHANGE LOGO HERE IF YOU SWAP THE IMAGE
        // Path: /assets/images/aegis_shield.png
        // ------------------------------------------------------------
        $logo_url = plugins_url(
            'assets/images/aegis_shield.png',
            dirname( __FILE__, 3 ) . '/aegisshield-security.php'
        );
        ?>
        <h2><?php esc_html_e( 'AegisShield Multi-Factor Authentication (MFA)', 'aegisshield-security' ); ?></h2>
        <table class="form-table" role="presentation">
            <tr>
                <th scope="row"><?php esc_html_e( 'MFA Status', 'aegisshield-security' ); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="aegisshield_2fa_enabled" value="1" <?php checked( $enabled, '1' ); ?> />
                        <?php esc_html_e( 'Enable MFA for this account', 'aegisshield-security' ); ?>
                    </label>
					<p class="description">
						<?php esc_html_e( 'If enabled, this user can request a 6-digit login code via email on the login screen.', 'aegisshield-security' ); ?>
					</p>
                    <p class="description">
                        <?php
                        printf(
                            /* translators: %s: current MFA status. */
                            esc_html__( 'Current status: %s', 'aegisshield-security' ),
                            esc_html( $status_label )
                        );
                        ?>
                    </p>

                    <div class="aegisshield-2fa-setup-panel" style="margin-top:15px; max-width:640px; border-radius:8px; background:#111; padding:18px 20px; color:#f1f1f1; box-shadow:0 2px 8px rgba(0,0,0,0.35);">
                        <div style="display:flex; gap:18px; align-items:center; flex-wrap:wrap;">
                            <div style="flex:0 0 auto; text-align:center;">
                                <img src="<?php echo esc_url( $logo_url ); ?>" alt="<?php esc_attr_e( 'AegisShield Security', 'aegisshield-security' ); ?>" style="max-width:80px; height:auto; display:block; margin:0 auto 6px auto;" />
                                <strong style="font-size:13px; letter-spacing:0.03em; text-transform:uppercase; color:#ffb347;">
                                    <?php esc_html_e( 'Premium Login Protection', 'aegisshield-security' ); ?>
                                </strong>
                            </div>

                            <div style="flex:1 1 260px; min-width:260px;">
                                <p style="font-size:13px; margin-top:0; margin-bottom:10px; color:#e0e0e0;">
                                    <?php esc_html_e( 'Scan the QR code with your authenticator app (Google Authenticator, Authy, Microsoft Authenticator, 1Password, etc.).', 'aegisshield-security' ); ?>
                                </p>

                                <?php if ( '' === $secret ) : ?>
                                    <p style="margin:0; color:#ffdddd; font-size:12px;">
                                        <?php esc_html_e( 'To generate a QR code, enable MFA above and click “Update Profile”. A secret will be created automatically.', 'aegisshield-security' ); ?>
                                    </p>
                                <?php else : ?>
                                    <div style="display:flex; gap:16px; align-items:center; flex-wrap:wrap;">
                                        <div style="flex:0 0 auto;">
											<?php if ( '' !== $qr_url ) : ?>
												<?php
												// If we are using local SVG ("data:image/..."), esc_url() would strip it.
												// Use esc_attr() for data: URLs, esc_url() for normal http/https URLs.
												$qr_url = (string) $qr_url;
												$qr_src = ( 0 === strpos( $qr_url, 'data:image' ) )
													? esc_attr( $qr_url )
													: esc_url( $qr_url );
												?>
												<img src="<?php echo $qr_src; ?>"
													 alt="<?php esc_attr_e( 'Scan this QR code with your authenticator app', 'aegisshield-security' ); ?>"
													 style="width:220px; height:220px; background:#fff; padding:6px; border-radius:6px;" />
											<?php else : ?>
												<p style="margin:0; color:#ffdddd; font-size:12px;">
													<?php esc_html_e( 'QR code could not be loaded. You can still add this MFA account using the secret key below.', 'aegisshield-security' ); ?>
												</p>
											<?php endif; ?>
                                        </div>
                                        <div style="flex:1 1 180px; min-width:180px;">
                                            <p style="margin:0 0 6px 0; font-size:12px; color:#ccc;">
                                                <?php esc_html_e( 'Secret key (for manual entry):', 'aegisshield-security' ); ?>
                                            </p>
                                            <div class="aegis-2fa-secret-wrap" style="margin-bottom:6px;">
                                                <?php if ( '' !== $secret ) : ?>
                                                    <code id="aegis_2fa_secret_mask" style="display:inline-block; padding:3px 6px; background:#000; border-radius:4px; font-size:12px;">
                                                        <?php echo esc_html( $masked_secret ); ?>
                                                    </code>
                                                    <code id="aegis_2fa_secret_actual" style="display:none; padding:3px 6px; background:#000; border-radius:4px; font-size:12px;">
                                                        <?php echo esc_html( $secret ); ?>
                                                    </code>
                                                    <a href="#" id="aegis_2fa_secret_toggle" style="margin-left:8px; font-size:11px; text-decoration:none;">
                                                        <?php esc_html_e( 'Show', 'aegisshield-security' ); ?>
                                                    </a>
                                                <?php else : ?>
                                                    <code style="display:inline-block; padding:3px 6px; background:#000; border-radius:4px; font-size:12px;">
                                                        <?php esc_html_e( 'Secret will be generated after you enable MFA and save.', 'aegisshield-security' ); ?>
                                                    </code>
                                                <?php endif; ?>
                                            </div>

                                            <p style="margin:0 0 4px 0; font-size:12px; color:#ccc;">
                                                <?php esc_html_e( 'You can also use backup codes if you lose access to your authenticator app.', 'aegisshield-security' ); ?>
                                            </p>
                                        </div>
                                    </div>
                                <?php endif; ?>

                                <div style="margin-top:12px;">
                                    <label style="display:block; font-size:12px; color:#ccc;">
                                        <input type="checkbox" name="aegisshield_2fa_generate_backup" value="1" />
                                        <?php esc_html_e( 'Generate new backup codes on save (this will invalidate any existing backup codes).', 'aegisshield-security' ); ?>
                                    </label>

                                    <?php if ( ! empty( $new_codes ) ) : ?>
                                        <div style="margin-top:10px; padding:10px; border:1px solid #333; background:#181818; border-radius:4px;">
                                            <strong style="font-size:12px; color:#ffe082;">
                                                <?php esc_html_e( 'New backup codes (store these somewhere safe – they will not be shown again):', 'aegisshield-security' ); ?>
                                            </strong>
                                            <ul style="margin-top:8px; column-count:2; list-style:disc; padding-left:18px;">
                                                <?php foreach ( $new_codes as $code ) : ?>
                                                    <li><code><?php echo esc_html( $code ); ?></code></li>
                                                <?php endforeach; ?>
                                            </ul>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>

                    <script type="text/javascript">
                        (function() {
                            var toggle   = document.getElementById( 'aegis_2fa_secret_toggle' );
                            var maskEl   = document.getElementById( 'aegis_2fa_secret_mask' );
                            var actualEl = document.getElementById( 'aegis_2fa_secret_actual' );

                            if ( ! toggle || ! maskEl || ! actualEl ) {
                                return;
                            }

                            toggle.addEventListener( 'click', function( e ) {
                                e.preventDefault();
                                if ( actualEl.style.display === 'none' || actualEl.style.display === '' ) {
                                    actualEl.style.display = 'inline-block';
                                    maskEl.style.display   = 'none';
                                    toggle.textContent     = '<?php echo esc_js( __( 'Hide', 'aegisshield-security' ) ); ?>';
                                } else {
                                    actualEl.style.display = 'none';
                                    maskEl.style.display   = 'inline-block';
                                    toggle.textContent     = '<?php echo esc_js( __( 'Show', 'aegisshield-security' ) ); ?>';
                                }
                            });
                        })();
                    </script>
                </td>
            </tr>
        </table>
        <?php
    }

    /**
     * Handle saving of MFA (2FA meta) settings on user profile.
     *
     * @param int $user_id
     */
    public function save_user_2fa_profile_section( $user_id ) {
        if ( ! $this->is_global_2fa_enabled() ) {
            return;
        }

        if ( ! current_user_can( 'edit_user', $user_id ) ) {
            return;
        }

        // We rely on WordPress core profile nonce; do not add extra nonce here.
        $enable_2fa = isset( $_POST['aegisshield_2fa_enabled'] ) ? '1' : '0'; // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $gen_backup = isset( $_POST['aegisshield_2fa_generate_backup'] );     // phpcs:ignore WordPress.Security.NonceVerification.Missing

        update_user_meta( $user_id, 'aegisshield_2fa_enabled', $enable_2fa );
		
		$email_enabled = isset( $_POST['aegisshield_mfa_email_enabled'] ) ? '1' : '0'; // phpcs:ignore WordPress.Security.NonceVerification.Missing
		update_user_meta( $user_id, 'aegisshield_mfa_email_enabled', $email_enabled );


        $secret = (string) get_user_meta( $user_id, 'aegisshield_2fa_secret', true );
        if ( '1' === $enable_2fa && '' === $secret ) {
            $secret = $this->generate_2fa_secret();
            update_user_meta( $user_id, 'aegisshield_2fa_secret', $secret );
        }

        if ( '0' === $enable_2fa ) {
            // Leave any existing secret / backup hashes so user can re-enable quickly if desired.
            return;
        }

        if ( $gen_backup ) {
            $plain_codes = $this->generate_backup_codes( 10 );
            $hashes      = array();

            foreach ( $plain_codes as $code ) {
                $hashes[] = wp_hash_password( $code );
            }

            update_user_meta( $user_id, 'aegisshield_2fa_backup_hashes', $hashes );

            // Store in memory for this request so we can display them once after save.
            $this->last_generated_backup_codes[ $user_id ] = $plain_codes;
        }
    }

    /* ==========================================================
     *  TOTP + BACKUP CODE HELPERS
     * ========================================================== */

    /**
     * Generate base32 secret for TOTP.
     *
     * @return string
     */
    protected function generate_2fa_secret() {
        $bytes = random_bytes( 16 );
        return $this->base32_encode( $bytes );
    }

    /**
     * Verify TOTP code (6 digits) with small time window.
     *
     * @param string $secret Base32 secret.
     * @param string $code   6-digit code.
     * @return bool
     */
    protected function verify_totp_code( $secret, $code ) {
        $code = preg_replace( '/[^0-9]/', '', $code );
        if ( strlen( $code ) !== 6 ) {
            return false;
        }

        $secret_bin = $this->base32_decode( $secret );
        if ( '' === $secret_bin ) {
            return false;
        }

        $time_step    = 30;
        $time_counter = floor( time() / $time_step );

        // Allow small clock drift: previous, current, next.
        for ( $i = -1; $i <= 1; $i++ ) {
            $counter = $time_counter + $i;
            $calc    = $this->totp_from_counter( $secret_bin, $counter );

            if ( hash_equals( $calc, $code ) ) {
                return true;
            }
        }

        return false;
    }

    /**
     * Generate 6-digit TOTP from secret binary + counter.
     *
     * @param string $secret_bin
     * @param int    $counter
     * @return string
     */
    protected function totp_from_counter( $secret_bin, $counter ) {
        $bin_counter = pack( 'N*', 0 ) . pack( 'N*', $counter );
        $hash        = hash_hmac( 'sha1', $bin_counter, $secret_bin, true );
        $offset      = ord( substr( $hash, -1 ) ) & 0x0F;
        $truncated   = substr( $hash, $offset, 4 );

        $value = unpack( 'N', $truncated );
        $value = $value[1] & 0x7FFFFFFF;
        $mod   = $value % 1000000;

        return str_pad( (string) $mod, 6, '0', STR_PAD_LEFT );
    }

    /**
     * Base32 encode.
     *
     * @param string $data
     * @return string
     */
    protected function base32_encode( $data ) {
        $alphabet = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567';
        $bits     = '';
        $result   = '';

        foreach ( str_split( $data ) as $char ) {
            $bits .= str_pad( decbin( ord( $char ) ), 8, '0', STR_PAD_LEFT );
        }

        $chunks = str_split( $bits, 5 );
        foreach ( $chunks as $chunk ) {
            if ( strlen( $chunk ) < 5 ) {
                $chunk = str_pad( $chunk, 5, '0', STR_PAD_RIGHT );
            }
            $index   = bindec( $chunk );
            $result .= $alphabet[ $index ];
        }

        return $result;
    }

    /**
     * Base32 decode.
     *
     * @param string $data
     * @return string
     */
    protected function base32_decode( $data ) {
        $alphabet = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567';
        $data     = strtoupper( preg_replace( '/[^A-Z2-7]/', '', $data ) );
        if ( '' === $data ) {
            return '';
        }

        $bits = '';
        foreach ( str_split( $data ) as $char ) {
            $pos = strpos( $alphabet, $char );
            if ( false === $pos ) {
                return '';
            }
            $bits .= str_pad( decbin( $pos ), 5, '0', STR_PAD_LEFT );
        }

        $bytes  = '';
        $chunks = str_split( $bits, 8 );
        foreach ( $chunks as $chunk ) {
            if ( strlen( $chunk ) < 8 ) {
                continue;
            }
            $bytes .= chr( bindec( $chunk ) );
        }

        return $bytes;
    }

    /**
     * Generate an array of plain backup codes.
     *
     * @param int $count
     * @return string[]
     */
    protected function generate_backup_codes( $count = 10 ) {
        $codes = array();
        $count = max( 1, (int) $count );

        for ( $i = 0; $i < $count; $i++ ) {
            // 10-character code, numeric and uppercase letters.
            $bytes   = bin2hex( random_bytes( 5 ) );
            $codes[] = strtoupper( $bytes );
        }

        return $codes;
    }

    /**
     * Use a backup code (one-time); returns true if valid and removes it.
     *
     * @param int    $user_id
     * @param string $code
     * @return bool
     */
    protected function use_backup_code( $user_id, $code ) {
        $code = trim( (string) $code );
        if ( '' === $code ) {
            return false;
        }

        $hashes = get_user_meta( $user_id, 'aegisshield_2fa_backup_hashes', true );
        if ( ! is_array( $hashes ) || empty( $hashes ) ) {
            return false;
        }

        $used_index = null;

        foreach ( $hashes as $index => $hash ) {
            if ( wp_check_password( $code, $hash ) ) {
                $used_index = $index;
                break;
            }
        }

        if ( null === $used_index ) {
            return false;
        }

        // Remove used code and save.
        unset( $hashes[ $used_index ] );
        update_user_meta( $user_id, 'aegisshield_2fa_backup_hashes', array_values( $hashes ) );

        return true;
    }
}
