<?php
namespace AegisShield\Modules;

use AegisShield\AS_Plugin;
use AegisShield\Modules\Malware\AS_Malware_Profiles;
use AegisShield\Modules\Malware\AS_Malware_Incremental;

defined( 'ABSPATH' ) || exit;

/**
 * Malware Heuristics module.
 *
 * Lightweight heuristic scanner for common malware patterns.
 */
class AS_Module_Malware implements AS_Module_Interface {

    /**
     * Plugin instance.
     *
     * @var AS_Plugin
     */
    protected $plugin;

    /**
     * Constructor.
     *
     * @param AS_Plugin $plugin Plugin instance.
     */
    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Module slug.
     *
     * @return string
     */
    public function get_slug() {
        return 'malware';
    }

	public function register_settings() {
		$settings = $this->plugin->get_settings();
		$section  = 'malware';

		// Enable module by default so manual scans + UI work on fresh installs.
		if ( null === $settings->get( $section, 'enabled', null ) ) {
			$settings->set( $section, 'enabled', 'yes' );
		}

		// Provide sane default scan locations so full scans return results.
		if ( null === $settings->get( $section, 'scan_dirs', null ) ) {
			$settings->set( $section, 'scan_dirs', array( 'plugins', 'themes' ) );
		}

		// Optional: enable auto-scan for Free legacy behavior (your code supports it).
		if ( null === $settings->get( $section, 'auto_scan', null ) ) {
			$settings->set( $section, 'auto_scan', 'yes' );
		}

		// Default handling mode to read-only (safe on shared hosting).
		if ( null === $settings->get( $section, 'handling_mode', null ) ) {
			$settings->set( $section, 'handling_mode', 'a' );
		}

		$settings->save();
	}

    /**
     * Initialize hooks.
     *
     * @return void
     */
    public function init() {
        $settings = $this->plugin->get_settings();
        $section  = 'malware';

        $enabled = (string) $settings->get( $section, 'enabled', '' );
        if ( ! in_array( $enabled, array( 'on', '1', 'yes' ), true ) ) {
            return;
        }

        // Hook into hourly maintenance to optionally run scheduled scans.
        add_action( 'aegisshield_hourly_maintenance', array( $this, 'maybe_run_auto_scan' ) );
    }

    /**
     * Possibly run an automatic malware scan (according to schedule).
     *
     * Pro:
     *  - Incremental quick scans: enabled flag + configurable frequency (daily/weekly/monthly/quarterly).
     *  - Weekly full scans (if enabled).
     *
     * Free:
     *  - Daily full scan (legacy behaviour).
     *
     * @return void
     */
    public function maybe_run_auto_scan() {
        $settings = $this->plugin->get_settings();
        $section  = 'malware';

        $enabled = (string) $settings->get( $section, 'enabled', '' );
        if ( ! in_array( $enabled, array( 'on', '1', 'yes' ), true ) ) {
            return;
        }

        $auto_scan = (string) $settings->get( $section, 'auto_scan', '' );
        if ( ! in_array( $auto_scan, array( 'on', '1', 'yes' ), true ) ) {
            return;
        }

        $now     = time();
        $is_pro  = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();
        $did_run = false;

        if ( $is_pro ) {
            // Pro: honour lightweight scheduled quick scans and weekly full scans.
            $quick_enabled  = (string) $settings->get( $section, 'schedule_quick_enabled', '' );
            $weekly_enabled = (string) $settings->get( $section, 'schedule_weekly_enabled', '' );

            $quick_enabled_bool  = in_array( $quick_enabled, array( 'on', '1', 'yes' ), true );
            $weekly_enabled_bool = in_array( $weekly_enabled, array( 'on', '1', 'yes' ), true );

            // Frequency for incremental quick scans (admin-controlled).
            $quick_frequency = (string) $settings->get( $section, 'quick_schedule_frequency', 'daily' );
            if ( ! in_array( $quick_frequency, array( 'daily', 'weekly', 'monthly', 'quarterly' ), true ) ) {
                $quick_frequency = 'daily';
            }

            switch ( $quick_frequency ) {
                case 'weekly':
                    $quick_interval = WEEK_IN_SECONDS;
                    break;
                case 'monthly':
                    // Approximate month as 30 days for cron safety.
                    $quick_interval = 30 * DAY_IN_SECONDS;
                    break;
                case 'quarterly':
                    // Approximate quarter as 90 days.
                    $quick_interval = 90 * DAY_IN_SECONDS;
                    break;
                case 'daily':
                default:
                    $quick_interval = DAY_IN_SECONDS;
                    break;
            }

            $last_quick = (int) $settings->get( $section, 'last_quick_scan', 0 );
            $last_full  = (int) $settings->get( $section, 'last_full_scan', 0 );

            // Incremental quick scan according to chosen frequency.
            if ( $quick_enabled_bool && ( ! $last_quick || ( $now - $last_quick ) >= $quick_interval ) ) {
                $this->run_incremental_scan( 'auto_quick' );
                $settings->set( $section, 'last_quick_scan', $now );
                $did_run = true;
            }

            // Weekly full scan (if enabled).
            if ( $weekly_enabled_bool && ( ! $last_full || ( $now - $last_full ) >= WEEK_IN_SECONDS ) ) {
                $this->run_scan( 'auto' );
                $settings->set( $section, 'last_full_scan', $now );
                $did_run = true;
            }

            // If no specific Pro schedule is enabled, fall back to the legacy daily auto-scan.
            if ( ! $quick_enabled_bool && ! $weekly_enabled_bool ) {
                $last_auto = (int) $settings->get( $section, 'last_auto_scan', 0 );

                if ( ! $last_auto || ( $now - $last_auto ) >= DAY_IN_SECONDS ) {
                    $this->run_scan( 'auto' );
                    $settings->set( $section, 'last_auto_scan', $now );
                    $did_run = true;
                }
            }
        } else {
            // Free: preserve original behaviour – at most one full scan per day.
            $last_auto = (int) $settings->get( $section, 'last_auto_scan', 0 );

            if ( $last_auto && ( $now - $last_auto ) < DAY_IN_SECONDS ) {
                return;
            }

            $this->run_scan( 'auto' );
            $settings->set( $section, 'last_auto_scan', $now );
            $did_run = true;
        }

        if ( $did_run ) {
            $settings->save();
        }
    }

    /**
     * Run an incremental "quick" malware scan limited to recently changed files.
     *
     * This relies on the File Monitor table so that only new or modified
     * files are scanned, keeping resource usage low on shared hosting.
     *
     * @param string $scan_type Scan type label (e.g. 'auto_quick', 'manual_quick').
     * @return array List of results.
     */
    public function run_incremental_scan( $scan_type ) {
        $settings = $this->plugin->get_settings();
        $logger   = $this->plugin->get_logger();
        $section  = 'malware';

        $started_at    = time();
        $scanned_files = 0;
        $is_partial    = false;

        $is_pro = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();
        if ( ! $is_pro ) {
            return array();
        }

        $handling_mode = (string) $settings->get( $section, 'handling_mode', '' );
        if ( ! in_array( $handling_mode, array( 'a', 'b', 'c' ), true ) ) {
            $handling_mode = 'a';
        }

        $ignore_list = $settings->get( $section, 'ignore_list', array() );
        if ( ! is_array( $ignore_list ) ) {
            $ignore_list = array();
        }

        // Determine active profile threshold.
        $profile_key = AS_Malware_Profiles::get_selected_profile( $settings, $section );
        $threshold   = AS_Malware_Profiles::get_threshold( $profile_key );

        // Fetch incremental targets from helper (already capped to 300).
        $limit   = 300;
        $targets = AS_Malware_Incremental::get_incremental_targets();

        if ( empty( $targets ) || ! is_array( $targets ) ) {
            // No changed files: log and exit without touching last_results / meta.
            if ( $logger ) {
                $logger->log(
                    'malware_quick_scan_skipped',
                    __( 'Incremental quick scan skipped: no changed files were found in the File Monitor table.', 'aegisshield-security' ),
                    array(
                        'scan_type' => $scan_type,
                    )
                );
            }
            return array();
        }

        if ( count( $targets ) >= $limit ) {
            $is_partial = true;
        }

        $results = array();

        foreach ( $targets as $rel_path ) {
            $rel_path = ltrim( (string) $rel_path, '/' );

            if ( in_array( $rel_path, $ignore_list, true ) ) {
                continue;
            }

            $abs_path = trailingslashit( ABSPATH ) . $rel_path;

            if ( ! file_exists( $abs_path ) || ! is_file( $abs_path ) ) {
                continue;
            }

            $ext = strtolower( pathinfo( $abs_path, PATHINFO_EXTENSION ) );
            if ( ! in_array( $ext, array( 'php', 'phtml', 'php5', 'php7' ), true ) ) {
                continue;
            }

            // Skip files inside quarantine directory.
            if ( false !== strpos( $abs_path, DIRECTORY_SEPARATOR . 'aegisshield-quarantine' . DIRECTORY_SEPARATOR ) ) {
                continue;
            }

            $scanned_files++;

            $analysis = $this->analyze_file( $abs_path, $rel_path );
            if ( ! $analysis ) {
                continue;
            }

            // Apply profile threshold.
            if ( isset( $analysis['score'] ) && (int) $analysis['score'] < $threshold ) {
                continue;
            }

            $results[] = $analysis;

            // Auto-quarantine for mode C and high-risk files.
            if ( 'c' === $handling_mode && isset( $analysis['score'] ) && $analysis['score'] >= 8 ) {
                $this->quarantine_file( $abs_path, $rel_path );
                $analysis['quarantined'] = true;
            }

            // Log finding.
            if ( $logger ) {
                $logger->log(
                    'malware_detected',
                    sprintf(
                        /* translators: 1: file path, 2: score */
                        __( 'Suspicious file detected during quick scan: %1$s (score %2$d)', 'aegisshield-security' ),
                        $rel_path,
                        (int) $analysis['score']
                    ),
                    array(
                        'file'      => $rel_path,
                        'score'     => (int) $analysis['score'],
                        'level'     => isset( $analysis['level'] ) ? $analysis['level'] : '',
                        'reasons'   => isset( $analysis['reasons'] ) ? $analysis['reasons'] : array(),
                        'scan_type' => $scan_type,
                    )
                );
            }
        }

        // Persist last results for UI display (same schema as full scan).
        $settings->set( $section, 'last_results', $results );
        $settings->set(
            $section,
            'last_results_meta',
            array(
                'scan_type'     => $scan_type,
                'started_at'    => $started_at,
                'completed_at'  => time(),
                'file_count'    => $scanned_files,
                'suspect_count' => count( $results ),
                'partial'       => $is_partial ? 1 : 0,
            )
        );
        $settings->save();

        if ( $is_partial && $logger ) {
            $logger->log(
                'malware_quick_scan_partial',
                __( 'Quick malware scan cap reached; remaining files will be scanned in a later run.', 'aegisshield-security' ),
                array(
                    'limit'         => $limit,
                    'file_count'    => $scanned_files,
                    'suspect_count' => count( $results ),
                    'scan_type'     => $scan_type,
                )
            );
        }

        return $results;
    }

    /**
     * Run a manual malware scan from the admin UI.
     *
     * @return array List of results.
     */
    public function run_manual_scan() {
        return $this->run_scan( 'manual' );
    }

	/**
	 * Public quarantine helper for UI actions.
	 * Ensures we quarantine AND replace the original with a safe placeholder (as advertised).
	 *
	 * @param string $file_rel Relative path from ABSPATH (wp-content/...).
	 * @return bool True on success.
	 */
	public function quarantine_rel( $file_rel ) {
		$file_rel = (string) $file_rel;
		$file_rel = ltrim( $file_rel, "/\\ \t\n\r\0\x0B" );

		if ( '' === $file_rel ) {
			$this->log_violation( 'malware_quarantine_failed_empty', __( 'Quarantine failed: empty file path.', 'aegisshield-security' ) );
			return false;
		}

		$absolute = trailingslashit( ABSPATH ) . $file_rel;

		if ( ! file_exists( $absolute ) ) {
			$this->log_violation(
				'malware_quarantine_failed_missing',
				__( 'Quarantine failed: file does not exist.', 'aegisshield-security' ),
				array( 'file' => $file_rel )
			);
			return false;
		}

		if ( ! is_readable( $absolute ) ) {
			$this->log_violation(
				'malware_quarantine_failed_unreadable',
				__( 'Quarantine failed: file not readable.', 'aegisshield-security' ),
				array( 'file' => $file_rel )
			);
			return false;
		}

		return (bool) $this->quarantine_file( $absolute, $file_rel );
	}

    /**
     * Core scan routine.
     *
     * @param string $scan_type 'manual' or 'auto'.
     * @return array
     */
    protected function run_scan( $scan_type ) {
        $settings = $this->plugin->get_settings();
        $logger   = $this->plugin->get_logger();
        $section  = 'malware';

        $started_at    = time();
        $scanned_files = 0;

        $scan_dirs = $settings->get( $section, 'scan_dirs', array() );
        if ( ! is_array( $scan_dirs ) ) {
            $scan_dirs = array();
        }

        $is_pro = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();

        $custom_dirs = $is_pro ? $settings->get( $section, 'custom_dirs', array() ) : array();
        if ( ! is_array( $custom_dirs ) ) {
            $custom_dirs = array();
        }

		// Back-compat / enforcement: older UI (class-as-page-malware.php) saves to 'directories' and 'custom_directories'.
		// If scan_dirs/custom_dirs are empty, fall back and LOG the mismatch so you can catch it.
		if ( empty( $scan_dirs ) ) {
			$fallback_dirs = $settings->get( $section, 'directories', array() );
			if ( is_array( $fallback_dirs ) && ! empty( $fallback_dirs ) ) {
				$scan_dirs = $fallback_dirs;

				$this->log_violation(
					'malware_scan_dirs_fallback_used',
					__( 'Scanner used fallback settings key "directories" because "scan_dirs" was empty.', 'aegisshield-security' ),
					array( 'scan_type' => $scan_type )
				);
			}
		}

		if ( $is_pro && empty( $custom_dirs ) ) {
			$fallback_custom = $settings->get( $section, 'custom_directories', array() );
			if ( is_array( $fallback_custom ) && ! empty( $fallback_custom ) ) {
				$custom_dirs = $fallback_custom;

				$this->log_violation(
					'malware_custom_dirs_fallback_used',
					__( 'Scanner used fallback settings key "custom_directories" because "custom_dirs" was empty.', 'aegisshield-security' ),
					array( 'scan_type' => $scan_type )
				);
			}
		}

        // If nothing is selected, nothing to do.
		if ( empty( $scan_dirs ) && empty( $custom_dirs ) ) {
			$this->log_violation(
				'malware_scan_skipped_no_dirs',
				__( 'Scan skipped because no directories were selected (scan_dirs/custom_dirs empty).', 'aegisshield-security' ),
				array( 'scan_type' => $scan_type )
			);
			return array();
		}

        $handling_mode = (string) $settings->get( $section, 'handling_mode', '' );
        if ( ! in_array( $handling_mode, array( 'a', 'b', 'c' ), true ) ) {
            $handling_mode = 'a'; // Fallback to read-only if not configured.
        }

        $ignore_list = $settings->get( $section, 'ignore_list', array() );
        if ( ! is_array( $ignore_list ) ) {
            $ignore_list = array();
        }

        // Determine active profile threshold (Pro only; free behaves like balanced).
        $profile_key = null;
        $threshold   = null;

        if ( $is_pro ) {
            $profile_key = AS_Malware_Profiles::get_selected_profile( $settings, $section );
            $threshold   = AS_Malware_Profiles::get_threshold( $profile_key );
        }

        $roots = $this->get_scan_roots( $scan_dirs, $custom_dirs );
        if ( empty( $roots ) ) {
            return array();
        }

        $results = array();
        foreach ( $roots as $root_label => $root_path ) {
            if ( ! $root_path || ! is_dir( $root_path ) ) {
                continue;
            }
            $root_path = rtrim( $root_path, DIRECTORY_SEPARATOR );

            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator(
                    $root_path,
                    \RecursiveDirectoryIterator::SKIP_DOTS
                ),
                \RecursiveIteratorIterator::SELF_FIRST
            );

            foreach ( $iterator as $file ) {
                if ( ! $file->isFile() ) {
                    continue;
                }

                $path = $file->getPathname();

                // Only inspect PHP-like files.
                $ext = strtolower( pathinfo( $path, PATHINFO_EXTENSION ) );
                if ( ! in_array( $ext, array( 'php', 'phtml', 'php5', 'php7' ), true ) ) {
                    continue;
                }

                // Skip files inside quarantine directory.
                if ( false !== strpos( $path, DIRECTORY_SEPARATOR . 'aegisshield-quarantine' . DIRECTORY_SEPARATOR ) ) {
                    continue;
                }

                $rel_path = $this->relative_path( $path );

                if ( in_array( $rel_path, $ignore_list, true ) ) {
                    continue;
                }

                $scanned_files++;

                $analysis = $this->analyze_file( $path, $rel_path );
                if ( ! $analysis ) {
                    continue;
                }

                // Apply profile threshold for Pro sites.
                if ( $is_pro && null !== $threshold && isset( $analysis['score'] ) && (int) $analysis['score'] < $threshold ) {
                    continue;
                }

                $results[] = $analysis;

                // Auto-quarantine for mode C and high-risk files.
                if ( 'c' === $handling_mode && $analysis['score'] >= 8 ) {
                    $this->quarantine_file( $path, $rel_path );
                    $analysis['quarantined'] = true;
                }

                // Log finding.
                $logger->log(
                    'malware_detected',
                    sprintf(
                        /* translators: 1: file path, 2: score */
                        __( 'Suspicious file detected: %1$s (score %2$d)', 'aegisshield-security' ),
                        $rel_path,
                        (int) $analysis['score']
                    ),
                    array(
                        'file'      => $rel_path,
                        'score'     => (int) $analysis['score'],
                        'level'     => $analysis['level'],
                        'reasons'   => $analysis['reasons'],
                        'scan_type' => $scan_type,
                    )
                );
            }
        }

        // Persist last results for UI display.
        $settings->set( $section, 'last_results', $results );
        $settings->set(
            $section,
            'last_results_meta',
            array(
                'scan_type'     => $scan_type,
                'started_at'    => $started_at,
                'completed_at'  => time(),
                'file_count'    => $scanned_files,
                'suspect_count' => count( $results ),
            )
        );
        $settings->save();

        return $results;
    }

    /**
     * Determine roots to scan based on admin settings.
     *
     * @param array $scan_dirs Selected directories.
     * @param array $custom_dirs Custom directories (Pro).
     * @return array
     */
    protected function get_scan_roots( array $scan_dirs, array $custom_dirs = array() ) {
        $roots = array();

        if ( in_array( 'uploads', $scan_dirs, true ) ) {
            $uploads = wp_get_upload_dir();
            if ( ! empty( $uploads['basedir'] ) && is_dir( $uploads['basedir'] ) ) {
                $roots['uploads'] = $uploads['basedir'];
            }
        }

        if ( in_array( 'plugins', $scan_dirs, true ) && defined( 'WP_PLUGIN_DIR' ) && is_dir( WP_PLUGIN_DIR ) ) {
            $roots['plugins'] = WP_PLUGIN_DIR;
        }

        if ( in_array( 'themes', $scan_dirs, true ) && function_exists( 'get_theme_root' ) ) {
            $themes = get_theme_root();
            if ( $themes && is_dir( $themes ) ) {
                $roots['themes'] = $themes;
            }
        }

        // Additional custom directories (Pro only).
        if ( ! empty( $custom_dirs ) ) {
            $abs_root = wp_normalize_path( ABSPATH );

            foreach ( $custom_dirs as $dir ) {
                $dir = trim( (string) $dir );
                if ( '' === $dir ) {
                    continue;
                }

                $relative = ltrim( $dir, '/\\' );
                if ( '' === $relative ) {
                    continue;
                }

                $abs_path = wp_normalize_path( trailingslashit( $abs_root ) . $relative );

                // Ensure the path stays within the WordPress root and exists.
                if ( 0 !== strpos( $abs_path, $abs_root ) || ! is_dir( $abs_path ) ) {
                    continue;
                }

                $key = 'custom_' . md5( $relative );
                $roots[ $key ] = $abs_path;
            }
        }

        return $roots;
    }

    /**
     * Analyze a file for suspicious patterns.
     *
     * @param string $path Absolute path.
     * @param string $rel  Relative path from ABSPATH.
     * @return array|null
     */
    protected function analyze_file( $path, $rel ) {
        $contents = @file_get_contents( $path );
        if ( false === $contents ) {
            return null;
        }

        // Limit analysis to first ~256KB to avoid memory issues.
        if ( strlen( $contents ) > 262144 ) {
            $contents = substr( $contents, 0, 262144 );
        }

        $score   = 0;
        $reasons = array();

        $lower = strtolower( $contents );

        // Suspicious functions - NOTE: These strings are detection signatures only; not executed.  Malware scanners may have false-positive
        $patterns = array(
            'base64_decode' => 3,
            'gzinflate'     => 3,
            'str_rot13'     => 2,
            'eval('         => 5,
            'preg_replace'  => 2,
            'assert('       => 4,
            'shell_exec'    => 4,
            'system('       => 4,
            'passthru'      => 4,
            'exec('         => 4,
        );

        $hit_functions = 0;
        foreach ( $patterns as $needle => $value ) {
            if ( false !== strpos( $lower, $needle ) ) {
                $score    += $value;
                $hit_functions++;
                $reasons[] = sprintf(
                    /* translators: %s is function name. */
                    __( 'Uses suspicious function: %s', 'aegisshield-security' ),
                    $needle
                );
            }
        }

        // preg_replace with /e modifier.
        if ( preg_match( '#preg_replace\s*\([^)]*/e#i', $contents ) ) {
            $score    += 4;
            $reasons[] = __( 'Uses preg_replace with /e modifier', 'aegisshield-security' );
        }

        // Long encoded-looking strings.
        if ( preg_match_all( '#[A-Za-z0-9+/]{100,}={0,2}#', $contents, $matches ) ) {
            if ( ! empty( $matches[0] ) ) {
                $score    += 2;
                $reasons[] = __( 'Contains long encoded-looking strings', 'aegisshield-security' );
            }
        }

        // Suspicious wp-login clones.
        if ( preg_match( '#wp-login\.php$#i', $rel ) && strpos( $rel, 'wp-login.php' ) !== strlen( $rel ) - strlen( 'wp-login.php' ) ) {
            $score    += 5;
            $reasons[] = __( 'wp-login.php clone in non-standard location', 'aegisshield-security' );
        }

        // Randomized-looking filenames.
        $basename = basename( $rel );
        $name     = pathinfo( $basename, PATHINFO_FILENAME );
        if ( strlen( $name ) >= 10 && strlen( $name ) <= 40 && preg_match( '#^[a-z0-9]+$#i', $name ) ) {
            $score    += 3;
            $reasons[] = __( 'Random-looking filename', 'aegisshield-security' );
        }

        // Large file with obfuscated traits.
        if ( $hit_functions >= 3 ) {
            $score    += 5;
            $reasons[] = __( 'Multiple high-risk functions present', 'aegisshield-security' );
        }

        if ( $score <= 0 ) {
            return null;
        }

        if ( $score >= 8 ) {
            $level = 'high';
        } elseif ( $score >= 4 ) {
            $level = 'medium';
        } else {
            $level = 'low';
        }

        return array(
            'file'       => $rel,
            'score'      => (int) $score,
            'level'      => $level,
            'reasons'    => $reasons,
            'modified'   => @filemtime( $path ) ?: 0,
            'scan_time'  => time(),
            'quarantined'=> false,
        );
    }

    /**
     * Quarantine a suspicious file:
     * - Move to wp-content/aegisshield-quarantine/
     * - Replace original with harmless placeholder.
     *
     * @param string $path Absolute path.
     * @param string $rel  Relative path.
     * @return bool
     */
    public function quarantine_file( $path, $rel ) {
        if ( ! file_exists( $path ) ) {
            return false;
        }

        if ( ! defined( 'WP_CONTENT_DIR' ) ) {
            return false;
        }

        $quarantine_root = WP_CONTENT_DIR . '/aegisshield-quarantine';

        if ( ! is_dir( $quarantine_root ) && ! wp_mkdir_p( $quarantine_root ) ) {
            return false;
        }

        // Preserve relative structure inside quarantine.
        $target_path = $quarantine_root . '/' . str_replace( array( '\\', ':' ), array( '/', '' ), $rel );
        $target_dir  = dirname( $target_path );

        if ( ! is_dir( $target_dir ) && ! wp_mkdir_p( $target_dir ) ) {
            return false;
        }

        // Move the original file.
        if ( ! @rename( $path, $target_path ) ) {
            return false;
        }

        // Create harmless placeholder.
        $placeholder  = "<?php\n";
        $placeholder .= "// AegisShield Security placeholder.\n";
        $placeholder .= "// Original file quarantined on " . gmdate( 'c' ) . "\n";
        $placeholder .= "exit;\n";

        @file_put_contents( $path, $placeholder );

        return true;
    }

	/**
	 * Single shared violation logger for Malware module.
	 *
	 * Logs when something is misconfigured, skipped, fails to enforce, or fails to execute.
	 * Use this for both "not working" and "enforcement failed" situations.
	 *
	 * @param string $event   Machine event key (ex: malware_scan_skipped_no_dirs)
	 * @param string $message Human-readable message
	 * @param array  $context Extra metadata
	 */
	public function log_violation( $event, $message, $context = array() ) {
		$logger = $this->plugin ? $this->plugin->get_logger() : null;

		$payload = is_array( $context ) ? $context : array();
		$payload['module'] = 'malware';

		if ( $logger ) {
			$logger->log( (string) $event, (string) $message, $payload );
			return;
		}

		// Fallback for environments where logger isn't available.
		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( '[AegisShield Malware][' . $event . '] ' . $message . ' ' . wp_json_encode( $payload ) );
		}
	}


    /**
     * Get a path relative to ABSPATH.
     *
     * @param string $path Absolute path.
     * @return string
     */
    protected function relative_path( $path ) {
        $path     = str_replace( array( '\\', '/' ), DIRECTORY_SEPARATOR, $path );
        $abs_path = str_replace( array( '\\', '/' ), DIRECTORY_SEPARATOR, ABSPATH );

        if ( 0 === strpos( $path, $abs_path ) ) {
            $rel = substr( $path, strlen( $abs_path ) );
        } else {
            $rel = $path;
        }

        $rel = ltrim( str_replace( array( '\\', DIRECTORY_SEPARATOR ), '/', $rel ), '/' );

        return $rel;
    }
}