<?php
namespace AegisShield\Modules\Malware;

defined( 'ABSPATH' ) || exit;

/**
 * Malware heuristic profiles helper.
 *
 * Provides simple profile selection and score thresholds
 * so that Pro users can choose how sensitive the malware
 * scanner should be.
 */
class AS_Malware_Profiles {

    /**
     * Get the default profile key.
     *
     * @return string
     */
    public static function get_default_profile() {
        return 'balanced';
    }

    /**
     * Retrieve available profile definitions.
     *
     * @return array
     */
    public static function get_profiles() {
        return array(
            'conservative' => array(
                'label'       => __( 'Conservative', 'aegisshield-security' ),
                'description' => __( 'Lower sensitivity and minimal false positives.', 'aegisshield-security' ),
            ),
            'balanced'     => array(
                'label'       => __( 'Balanced', 'aegisshield-security' ),
                'description' => __( 'Recommended default balance of coverage and noise.', 'aegisshield-security' ),
            ),
            'aggressive'   => array(
                'label'       => __( 'Aggressive', 'aegisshield-security' ),
                'description' => __( 'Higher sensitivity; may report more false positives.', 'aegisshield-security' ),
            ),
        );
    }

    /**
     * Get the currently selected profile key from settings.
     *
     * Falls back to the default if the stored value is invalid.
     *
     * @param object|null $settings Settings instance (must support ->get()).
     * @param string      $section  Settings section key.
     * @return string
     */
    public static function get_selected_profile( $settings = null, $section = 'malware' ) {
        $profiles = self::get_profiles();
        $key      = '';

        if ( $settings && is_object( $settings ) && method_exists( $settings, 'get' ) ) {
            $stored = (string) $settings->get( $section, 'scan_profile', '' );
            if ( isset( $profiles[ $stored ] ) ) {
                $key = $stored;
            }
        }

        if ( '' === $key ) {
            $key = self::get_default_profile();
        }

        return $key;
    }

    /**
     * Get the minimum score threshold for a profile.
     *
     * Any finding with a score below this value will be
     * ignored for that profile.
     *
     * @param string $profile_key Profile key.
     * @return int
     */
    public static function get_threshold( $profile_key ) {
        $profile_key = (string) $profile_key;

        $map = array(
            'conservative' => 6,
            'balanced'     => 4,
            'aggressive'   => 2,
        );

        if ( isset( $map[ $profile_key ] ) ) {
            return (int) $map[ $profile_key ];
        }

        return (int) $map['balanced'];
    }
}
