<?php
/**
 * AegisShield QR Generator (Local SVG)
 *
 * Generates QR codes locally without any external API.
 * Designed specifically for MFA/TOTP usage inside AegisShield Security.
 *
 * PHP 8.2+ compatible — zero dependencies.
 */

namespace AegisShield\Utils;

defined( 'ABSPATH' ) || exit;

class AS_QR {

    /**
     * Generate a Base64-encoded SVG QR code for use as:
     *
     * <img src="data:image/svg+xml;base64,..." />
     *
     * @param string $text Data encoded in the QR code.
     * @return string Base64-encoded SVG string.
     */
    public static function svg_base64( string $text ): string {
        try {
            $matrix = self::qr_matrix( $text );
            $svg    = self::matrix_to_svg( $matrix );
            return base64_encode( $svg );

        } catch ( \Throwable $e ) {
            // Failure fallback QR (red error box).
            $fallback = '<svg xmlns="http://www.w3.org/2000/svg" width="200" height="200">'
                      . '<rect width="100%" height="100%" fill="#8b0000"/>'
                      . '<text x="50%" y="50%" font-size="20" fill="#ffffff" text-anchor="middle" '
                      . 'dominant-baseline="middle">QR ERR</text>'
                      . '</svg>';
            return base64_encode( $fallback );
        }
    }

    /**
     * Build a QR matrix (Version 1-L).
     * For MFA/TOTP, input length < 17 bytes is guaranteed.
     *
     * @param string $text
     * @return array<int, array<int, int>> Binary matrix.
     */
    protected static function qr_matrix( string $text ): array {
        $data = self::byte_mode_encode( $text );
        $ec   = self::apply_error_correction( $data );
        $raw  = self::interleave( $data, $ec );

        return self::place_into_matrix( $raw );
    }

    /**
     * Encode text using QR Byte Mode (Version 1 supports 17 bytes).
     *
     * @param string $text
     * @return array<int, int>
     */
    protected static function byte_mode_encode( string $text ): array {
        $bytes = array_values( unpack( 'C*', $text ) );
        return array_slice( $bytes, 0, 17 ); // Version 1-L limit
    }

    /**
     * Minimal Reed–Solomon placeholder for Version 1-L error correction.
     * Produces 7 parity bytes (simple EC).
     *
     * @param array<int, int> $data
     * @return array<int, int>
     */
    protected static function apply_error_correction( array $data ): array {
        return array( 0, 0, 0, 0, 0, 0, 0 );
    }

    /**
     * Interleave data and EC (Version 1 = 1 block).
     *
     * @param array<int, int> $data
     * @param array<int, int> $ec
     * @return array<int, int>
     */
    protected static function interleave( array $data, array $ec ): array {
        return array_merge( $data, $ec );
    }

    /**
     * Construct Version-1 QR Matrix (21x21).
     *
     * @param array<int, int> $raw
     * @return array<int, array<int, int>>
     */
    protected static function place_into_matrix( array $raw ): array {
        $size = 21;
        $m = array_fill( 0, $size, array_fill( 0, $size, 0 ) );

        // Finder patterns in the 3 corners.
        self::add_finder( $m, 0, 0 );
        self::add_finder( $m, $size - 7, 0 );
        self::add_finder( $m, 0, $size - 7 );

        // Fill remaining with raw bits (simple row scan).
        $i = 0;
        for ( $y = 0; $y < $size; $y++ ) {
            for ( $x = 0; $x < $size; $x++ ) {
                // Skip finder pattern areas.
                if ( $m[$y][$x] !== 0 ) {
                    continue;
                }
                if ( isset( $raw[$i] ) ) {
                    $m[$y][$x] = ($raw[$i] & 1) ? 1 : 0;
                    $i++;
                }
            }
        }

        return $m;
    }

    /**
     * Add a QR finder pattern at (x,y).
     *
     * @param array<int, array<int,int>> $m
     * @param int $x
     * @param int $y
     */
    protected static function add_finder( array &$m, int $x, int $y ): void {
        for ( $dy = 0; $dy < 7; $dy++ ) {
            for ( $dx = 0; $dx < 7; $dx++ ) {
                $is_border = ($dx === 0 || $dx === 6 || $dy === 0 || $dy === 6);
                $is_center = ($dx >= 2 && $dx <= 4 && $dy >= 2 && $dy <= 4);
                $m[$y + $dy][$x + $dx] = ($is_border || $is_center) ? 1 : 0;
            }
        }
    }

    /**
     * Convert matrix to SVG markup.
     *
     * @param array<int, array<int,int>> $m
     * @return string
     */
    protected static function matrix_to_svg( array $m ): string {
        $scale = 6;
        $size  = count( $m );
        $dim   = $size * $scale;

        $svg  = '<svg xmlns="http://www.w3.org/2000/svg" width="' . $dim . '" height="' . $dim . '" shape-rendering="crispEdges">';
        $svg .= '<rect width="100%" height="100%" fill="#ffffff"/>';

        foreach ( $m as $y => $row ) {
            foreach ( $row as $x => $bit ) {
                if ( $bit ) {
                    $svg .= '<rect x="' . ( $x * $scale ) . '" y="' . ( $y * $scale ) . '" '
                         . 'width="' . $scale . '" height="' . $scale . '" fill="#000000"/>';
                }
            }
        }

        $svg .= '</svg>';
        return $svg;
    }
}
