<?php
if (!defined('ABSPATH')) { exit; }

use AegisLink\Shortlinks\Shortlinks;
use AegisLink\Shortlinks\Bulk_Terms;

if (!current_user_can('edit_posts')) {
    echo '<p>' . esc_html__('Forbidden', 'aegislink') . '</p>';
    return;
}

$license_obj = (function_exists('aegislink') && isset(aegislink()->license)) ? aegislink()->license : null;
$is_pro = (is_object($license_obj) && method_exists($license_obj, 'is_pro') && $license_obj->is_pro());
$pro_locked = !$is_pro;

$errors = array();
$notices = array();

// PRO lock: allow viewing, but block all mutations (save/delete/import/export).
if ($pro_locked) {
    if (($_SERVER['REQUEST_METHOD'] ?? '') === 'POST') {
        $errors[] = __('PRO feature: ShortURL Links is view-only in the free version. Activate a license to unlock editing and settings.', 'aegislink');
        $_POST = array();
    }
    if (isset($_GET['export'])) {
        $errors[] = __('PRO feature: Export is available in PRO.', 'aegislink');
        unset($_GET['export']);
    }
    if (isset($_GET['delete'])) {
        $errors[] = __('PRO feature: Deleting Short Links is available in PRO.', 'aegislink');
        unset($_GET['delete']);
    }
    if (isset($_GET['import'])) {
        $errors[] = __('PRO feature: Import is available in PRO.', 'aegislink');
        unset($_GET['import']);
    }
}

$short = new Shortlinks();
$base_short_url = $short->get_base_url();

$view = isset($_GET['view']) ? sanitize_key((string) $_GET['view']) : 'manage';
if (!in_array($view, array('manage','bulk','analytics','importexport'), true)) { $view = 'manage'; }


$sanitize_csv_list = function($s) {
    $s = (string) $s;
    $parts = preg_split('/\s*,\s*/', $s, -1, PREG_SPLIT_NO_EMPTY);
    $out = array();
    foreach ($parts as $p) {
        $p = trim($p);
        if ($p === '') { continue; }
        $out[] = sanitize_text_field($p);
    }
    return array_values(array_unique($out));
};

if ($view === 'importexport' && isset($_GET['export']) && (int) $_GET['export'] === 1) {
    check_admin_referer('aegislink_shortlinks_export');

    $q = new WP_Query(array(
        'post_type' => Shortlinks::CPT,
        'post_status' => 'publish',
        'posts_per_page' => -1,
        'orderby' => 'title',
        'order' => 'ASC',
        'no_found_rows' => true,
    ));

    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=aegislink-shortlinks-export-' . gmdate('Ymd-His') . '.csv');
    $out = fopen('php://output', 'w');

    fputcsv($out, array(
        'slug','title','target_url','redirect','track_clicks','rel_nofollow','rel_sponsored','notes',
        'groups','tags',
        'utm_enable','utm_source','utm_medium','utm_campaign','utm_term','utm_content'
    ));

    foreach ($q->posts as $p) {
        $groups = wp_get_post_terms($p->ID, 'aegislink_link_group', array('fields' => 'names'));
        $tags   = wp_get_post_terms($p->ID, 'aegislink_link_tag', array('fields' => 'names'));

        fputcsv($out, array(
            $p->post_name,
            $p->post_title,
            (string) get_post_meta($p->ID, '_aegislink_target', true),
            (string) get_post_meta($p->ID, '_aegislink_redirect', true),
            (string) get_post_meta($p->ID, '_aegislink_track', true),
            (string) get_post_meta($p->ID, '_aegislink_rel_nofollow', true),
            (string) get_post_meta($p->ID, '_aegislink_rel_sponsored', true),
            (string) get_post_meta($p->ID, '_aegislink_notes', true),
            implode(',', (array) $groups),
            implode(',', (array) $tags),
            (string) get_post_meta($p->ID, '_aegislink_utm_enable', true),
            (string) get_post_meta($p->ID, '_aegislink_utm_source', true),
            (string) get_post_meta($p->ID, '_aegislink_utm_medium', true),
            (string) get_post_meta($p->ID, '_aegislink_utm_campaign', true),
            (string) get_post_meta($p->ID, '_aegislink_utm_term', true),
            (string) get_post_meta($p->ID, '_aegislink_utm_content', true),
        ));
    }
    fclose($out);
    exit;
}

if (isset($_POST['aegislink_shortlinks_settings_save'])) {
    check_admin_referer('aegislink_shortlinks_settings_save');

    $prefix = isset($_POST['prefix']) ? sanitize_title((string) wp_unslash($_POST['prefix'])) : 'go';
    if ($prefix === '') { $prefix = 'go'; }

    $default_redirect = isset($_POST['default_redirect']) ? (int) $_POST['default_redirect'] : 302;
    if (!in_array($default_redirect, array(301,302,307), true)) { $default_redirect = 302; }

    $track_clicks = isset($_POST['track_clicks']) ? 1 : 0;
    $rel_nf = isset($_POST['default_rel_nofollow']) ? 1 : 0;
    $rel_sp = isset($_POST['default_rel_sponsored']) ? 1 : 0;

    $qr_provider = isset($_POST['qr_provider']) ? sanitize_text_field((string) wp_unslash($_POST['qr_provider'])) : 'qrserver';
    if (!in_array($qr_provider, array('qrserver','disabled'), true)) { $qr_provider = 'qrserver'; }

    $woo_share_enabled = isset($_POST['woo_share_enabled']) ? 1 : 0;

    $settings = array(
        'prefix' => $prefix,
        'default_redirect' => $default_redirect,
        'track_clicks' => $track_clicks,
        'default_rel_nofollow' => $rel_nf,
        'default_rel_sponsored' => $rel_sp,
        'qr_provider' => $qr_provider,
        'woo_share_enabled' => $woo_share_enabled,
    );

    update_option(Shortlinks::OPT_SETTINGS, $settings, false);
    delete_option('aegislink_shortlinks_rewrite_flushed'); // ensure flush happens once
    $notices[] = __('Settings saved.', 'aegislink');
}

$edit_id = isset($_GET['edit']) ? absint($_GET['edit']) : 0;
$editing = null;
if ($edit_id > 0) {
    $p = get_post($edit_id);
    if ($p && $p->post_type === Shortlinks::CPT) { $editing = $p; }
}

if (isset($_POST['aegislink_shortlink_save'])) {
    check_admin_referer('aegislink_shortlink_save');

    $title  = isset($_POST['title']) ? sanitize_text_field((string) wp_unslash($_POST['title'])) : '';
    $target = isset($_POST['target_url']) ? esc_url_raw((string) wp_unslash($_POST['target_url'])) : '';
    $slug   = isset($_POST['slug']) ? sanitize_title((string) wp_unslash($_POST['slug'])) : '';
    $redir  = isset($_POST['redirect']) ? (int) $_POST['redirect'] : 302;
    $track  = isset($_POST['track_clicks']) ? 1 : 0;
    $rel_nf = isset($_POST['rel_nofollow']) ? 1 : 0;
    $rel_sp = isset($_POST['rel_sponsored']) ? 1 : 0;
    $notes  = isset($_POST['notes']) ? sanitize_textarea_field((string) wp_unslash($_POST['notes'])) : '';

    $groups = isset($_POST['groups']) ? $sanitize_csv_list(wp_unslash($_POST['groups'])) : array();
    $tags   = isset($_POST['tags']) ? $sanitize_csv_list(wp_unslash($_POST['tags'])) : array();

    $utm_enable  = isset($_POST['utm_enable']) ? 1 : 0;
    $utm_source  = isset($_POST['utm_source']) ? sanitize_text_field((string) wp_unslash($_POST['utm_source'])) : '';
    $utm_medium  = isset($_POST['utm_medium']) ? sanitize_text_field((string) wp_unslash($_POST['utm_medium'])) : '';
    $utm_campaign= isset($_POST['utm_campaign']) ? sanitize_text_field((string) wp_unslash($_POST['utm_campaign'])) : '';
    $utm_term    = isset($_POST['utm_term']) ? sanitize_text_field((string) wp_unslash($_POST['utm_term'])) : '';
    $utm_content = isset($_POST['utm_content']) ? sanitize_text_field((string) wp_unslash($_POST['utm_content'])) : '';

    $incoming_edit_id = isset($_POST['edit_id']) ? absint($_POST['edit_id']) : 0;

    if ($title === '') { $errors[] = __('Title is required.', 'aegislink'); }
    if ($target === '' || !preg_match('#^https?://#i', $target)) { $errors[] = __('Target URL must be a valid http(s) URL.', 'aegislink'); }
    if (!in_array($redir, array(301,302,307), true)) { $redir = 302; }

    if ($slug === '') { $slug = $short->generate_slug($title); }
    if ($short->slug_exists($slug, $incoming_edit_id)) {
        $errors[] = __('That Slug is already in use. Please choose another.', 'aegislink');
    }

    if (empty($errors)) {
        $postarr = array(
            'post_type' => Shortlinks::CPT,
            'post_title' => $title,
            'post_status' => 'publish',
            'post_name' => $slug,
        );

        if ($incoming_edit_id > 0) {
            $postarr['ID'] = $incoming_edit_id;
            $saved_id = wp_update_post($postarr, true);
        } else {
            $saved_id = wp_insert_post($postarr, true);
        }

        if (is_wp_error($saved_id)) {
            $errors[] = $saved_id->get_error_message();
        } else {
            update_post_meta($saved_id, '_aegislink_target', $target);
            update_post_meta($saved_id, '_aegislink_redirect', $redir);
            update_post_meta($saved_id, '_aegislink_track', $track);
            update_post_meta($saved_id, '_aegislink_rel_nofollow', $rel_nf);
            update_post_meta($saved_id, '_aegislink_rel_sponsored', $rel_sp);
            update_post_meta($saved_id, '_aegislink_notes', $notes);

            update_post_meta($saved_id, '_aegislink_utm_enable', $utm_enable);
            update_post_meta($saved_id, '_aegislink_utm_source', $utm_source);
            update_post_meta($saved_id, '_aegislink_utm_medium', $utm_medium);
            update_post_meta($saved_id, '_aegislink_utm_campaign', $utm_campaign);
            update_post_meta($saved_id, '_aegislink_utm_term', $utm_term);
            update_post_meta($saved_id, '_aegislink_utm_content', $utm_content);
            wp_set_post_terms($saved_id, $groups, 'aegislink_link_group', false);
            wp_set_post_terms($saved_id, $tags, 'aegislink_link_tag', false);
            update_post_meta($p->ID, '_aegislink_shortlink_id', $saved_id);
            $notices[] = __('Short Link saved.', 'aegislink');

            wp_safe_redirect(admin_url('admin.php?page=aegislink&tab=shorturl&view=manage'));
            exit;
        }
    }
}

if (isset($_GET['delete']) && (int) $_GET['delete'] > 0 && isset($_GET['_wpnonce'])) {
    $del_id = absint($_GET['delete']);
    if (wp_verify_nonce((string) $_GET['_wpnonce'], 'aegislink_shortlink_delete_' . $del_id)) {
        $p = get_post($del_id);
        if ($p && $p->post_type === Shortlinks::CPT) {
            wp_trash_post($del_id);
            $notices[] = __('Short Link deleted.', 'aegislink');
            $view = 'manage';
        }
    }
}

if ($view === 'bulk' && isset($_POST['aegislink_bulk_create'])) {
    check_admin_referer('aegislink_bulk_create');

    $mode = isset($_POST['bulk_mode']) ? sanitize_key((string) wp_unslash($_POST['bulk_mode'])) : 'urls';

    $redir  = isset($_POST['bulk_redirect']) ? (int) $_POST['bulk_redirect'] : 302;
    if (!in_array($redir, array(301,302,307), true)) { $redir = 302; }

    $track  = isset($_POST['bulk_track']) ? 1 : 0;
    $rel_nf = isset($_POST['bulk_rel_nofollow']) ? 1 : 0;
    $rel_sp = isset($_POST['bulk_rel_sponsored']) ? 1 : 0;

    $groups = isset($_POST['bulk_groups']) ? $sanitize_csv_list(wp_unslash($_POST['bulk_groups'])) : array();
    $tags   = isset($_POST['bulk_tags']) ? $sanitize_csv_list(wp_unslash($_POST['bulk_tags'])) : array();

    $created = 0;
    $skipped = 0;

    if ($mode === 'posts') {
        $post_type = isset($_POST['bulk_post_type']) ? sanitize_key((string) wp_unslash($_POST['bulk_post_type'])) : 'post';
        $limit = isset($_POST['bulk_limit']) ? absint($_POST['bulk_limit']) : 100;

        $pq = new WP_Query(array(
            'post_type' => $post_type,
            'post_status' => 'publish',
            'posts_per_page' => $limit,
            'orderby' => 'date',
            'order' => 'DESC',
            'no_found_rows' => true,
        ));

        foreach ($pq->posts as $p) {
            $target = get_permalink($p);
            if (!$target) { $skipped++; continue; }
            $slug = sanitize_title($p->post_name);
            if ($slug === '') { $slug = $short->generate_slug($p->post_title); }
            if ($short->slug_exists($slug, 0)) { $skipped++; continue; }

            $saved_id = wp_insert_post(array(
                'post_type' => Shortlinks::CPT,
                'post_title' => $p->post_title,
                'post_status' => 'publish',
                'post_name' => $slug,
            ), true);

            if (is_wp_error($saved_id)) { $skipped++; continue; }

            update_post_meta($saved_id, '_aegislink_target', esc_url_raw($target));
            update_post_meta($saved_id, '_aegislink_redirect', $redir);
            update_post_meta($saved_id, '_aegislink_track', $track);
            update_post_meta($saved_id, '_aegislink_rel_nofollow', $rel_nf);
            update_post_meta($saved_id, '_aegislink_rel_sponsored', $rel_sp);

            wp_set_post_terms($saved_id, $groups, 'aegislink_link_group', false);
            wp_set_post_terms($saved_id, $tags, 'aegislink_link_tag', false);

            $created++;
        }
    } elseif ($mode === 'terms') {
        $taxonomy = isset($_POST['bulk_taxonomy']) ? sanitize_key((string) wp_unslash($_POST['bulk_taxonomy'])) : '';
        if ($taxonomy === '' || !taxonomy_exists($taxonomy)) {
            $skipped++;
        } else {
            $limit = isset($_POST['bulk_term_limit']) ? absint($_POST['bulk_term_limit']) : 100;
            if ($limit <= 0) { $limit = 100; }

            $strategy = isset($_POST['bulk_term_slug_strategy']) ? sanitize_key((string) wp_unslash($_POST['bulk_term_slug_strategy'])) : 'term';
            if (!in_array($strategy, array('term','cat','tag'), true)) { $strategy = 'term'; }

            $tq = get_terms(array(
                'taxonomy' => $taxonomy,
                'hide_empty' => false,
                'number' => $limit,
                'orderby' => 'count',
                'order' => 'DESC',
            ));

            if (is_wp_error($tq) || !is_array($tq)) {
                $skipped++;
            } else {
                foreach ($tq as $term) {
                    if (!is_object($term) || !isset($term->term_id)) { $skipped++; continue; }

                    $target = get_term_link($term);
                    if (is_wp_error($target) || !$target) { $skipped++; continue; }

                    $slug = Bulk_Terms::build_slug_for_term($term, $strategy);
                    if ($slug === '' || $short->slug_exists($slug, 0)) { $skipped++; continue; }

                    $title = isset($term->name) ? (string) $term->name : $slug;

                    $saved_id = wp_insert_post(array(
                        'post_type' => Shortlinks::CPT,
                        'post_title' => $title,
                        'post_status' => 'publish',
                        'post_name' => $slug,
                    ), true);

                    if (is_wp_error($saved_id)) { $skipped++; continue; }

                    update_post_meta($saved_id, '_aegislink_target', esc_url_raw((string) $target));
                    update_post_meta($saved_id, '_aegislink_redirect', $redir);
                    update_post_meta($saved_id, '_aegislink_track', $track);
                    update_post_meta($saved_id, '_aegislink_rel_nofollow', $rel_nf);
                    update_post_meta($saved_id, '_aegislink_rel_sponsored', $rel_sp);

                    wp_set_post_terms($saved_id, $groups, 'aegislink_link_group', false);
                    wp_set_post_terms($saved_id, $tags, 'aegislink_link_tag', false);

                    update_term_meta((int) $term->term_id, '_aegislink_shortlink_id', $saved_id);

                    $created++;
                }
            }
        }

    } else {
        $raw = isset($_POST['bulk_urls']) ? (string) wp_unslash($_POST['bulk_urls']) : '';
        $lines = preg_split('/\r\n|\r|\n/', $raw);
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '') { continue; }

            $parts = str_getcsv($line);
            $target = isset($parts[0]) ? esc_url_raw(trim((string) $parts[0])) : '';
            $slug = isset($parts[1]) ? sanitize_title(trim((string) $parts[1])) : '';
            $title = isset($parts[2]) ? sanitize_text_field(trim((string) $parts[2])) : '';

            if ($target === '' || !preg_match('#^https?://#i', $target)) { $skipped++; continue; }

            if ($title === '') { $title = $target; }
            if ($slug === '') { $slug = $short->generate_slug($title); }

            if ($short->slug_exists($slug, 0)) { $skipped++; continue; }

            $saved_id = wp_insert_post(array(
                'post_type' => Shortlinks::CPT,
                'post_title' => $title,
                'post_status' => 'publish',
                'post_name' => $slug,
            ), true);

            if (is_wp_error($saved_id)) { $skipped++; continue; }

            update_post_meta($saved_id, '_aegislink_target', $target);
            update_post_meta($saved_id, '_aegislink_redirect', $redir);
            update_post_meta($saved_id, '_aegislink_track', $track);
            update_post_meta($saved_id, '_aegislink_rel_nofollow', $rel_nf);
            update_post_meta($saved_id, '_aegislink_rel_sponsored', $rel_sp);

            wp_set_post_terms($saved_id, $groups, 'aegislink_link_group', false);
            wp_set_post_terms($saved_id, $tags, 'aegislink_link_tag', false);

            $created++;
        }
    }

    $notices[] = sprintf(__('Bulk create finished. Created: %d, Skipped: %d', 'aegislink'), (int) $created, (int) $skipped);
}

if ($view === 'importexport' && isset($_POST['aegislink_import_csv'])) {
    check_admin_referer('aegislink_shortlinks_import');

    if (!isset($_FILES['csv_file']) || empty($_FILES['csv_file']['tmp_name'])) {
        $errors[] = __('Please choose a CSV file.', 'aegislink');
    } else {
        $tmp = (string) $_FILES['csv_file']['tmp_name'];
        $fh = fopen($tmp, 'r');
        if (!$fh) {
            $errors[] = __('Could not read CSV.', 'aegislink');
        } else {
            $header = fgetcsv($fh);
            $map = array();
            if (is_array($header)) {
                foreach ($header as $i => $col) {
                    $map[strtolower(trim((string) $col))] = $i;
                }
            }

            $created = 0;
            $updated = 0;
            $skipped = 0;

            while (($row = fgetcsv($fh)) !== false) {
                $slug  = isset($map['slug']) ? sanitize_title((string) ($row[$map['slug']] ?? '')) : '';
                $title = isset($map['title']) ? sanitize_text_field((string) ($row[$map['title']] ?? '')) : '';
                $target= isset($map['target_url']) ? esc_url_raw((string) ($row[$map['target_url']] ?? '')) : '';

                if ($target === '' || !preg_match('#^https?://#i', $target)) { $skipped++; continue; }
                if ($title === '') { $title = $target; }
                if ($slug === '') { $slug = $short->generate_slug($title); }

                $existing = get_page_by_path($slug, OBJECT, Shortlinks::CPT);

                $postarr = array(
                    'post_type' => Shortlinks::CPT,
                    'post_title' => $title,
                    'post_status' => 'publish',
                    'post_name' => $slug,
                );

                if ($existing && $existing->ID) {
                    $postarr['ID'] = (int) $existing->ID;
                    $saved_id = wp_update_post($postarr, true);
                } else {
                    $saved_id = wp_insert_post($postarr, true);
                }

                if (is_wp_error($saved_id)) { $skipped++; continue; }

                $redir = isset($map['redirect']) ? (int) ($row[$map['redirect']] ?? 302) : 302;
                if (!in_array($redir, array(301,302,307), true)) { $redir = 302; }

                update_post_meta($saved_id, '_aegislink_target', $target);
                update_post_meta($saved_id, '_aegislink_redirect', $redir);
                update_post_meta($saved_id, '_aegislink_track', isset($map['track_clicks']) ? (int) ($row[$map['track_clicks']] ?? 1) : 1);
                update_post_meta($saved_id, '_aegislink_rel_nofollow', isset($map['rel_nofollow']) ? (int) ($row[$map['rel_nofollow']] ?? 1) : 1);
                update_post_meta($saved_id, '_aegislink_rel_sponsored', isset($map['rel_sponsored']) ? (int) ($row[$map['rel_sponsored']] ?? 0) : 0);
                update_post_meta($saved_id, '_aegislink_notes', isset($map['notes']) ? sanitize_textarea_field((string) ($row[$map['notes']] ?? '')) : '');

                update_post_meta($saved_id, '_aegislink_utm_enable', isset($map['utm_enable']) ? (int) ($row[$map['utm_enable']] ?? 0) : 0);
                update_post_meta($saved_id, '_aegislink_utm_source', isset($map['utm_source']) ? sanitize_text_field((string) ($row[$map['utm_source']] ?? '')) : '');
                update_post_meta($saved_id, '_aegislink_utm_medium', isset($map['utm_medium']) ? sanitize_text_field((string) ($row[$map['utm_medium']] ?? '')) : '');
                update_post_meta($saved_id, '_aegislink_utm_campaign', isset($map['utm_campaign']) ? sanitize_text_field((string) ($row[$map['utm_campaign']] ?? '')) : '');
                update_post_meta($saved_id, '_aegislink_utm_term', isset($map['utm_term']) ? sanitize_text_field((string) ($row[$map['utm_term']] ?? '')) : '');
                update_post_meta($saved_id, '_aegislink_utm_content', isset($map['utm_content']) ? sanitize_text_field((string) ($row[$map['utm_content']] ?? '')) : '');

                $groups = isset($map['groups']) ? $sanitize_csv_list((string) ($row[$map['groups']] ?? '')) : array();
                $tags   = isset($map['tags']) ? $sanitize_csv_list((string) ($row[$map['tags']] ?? '')) : array();
                if (!empty($groups)) { wp_set_post_terms($saved_id, $groups, 'aegislink_link_group', false); }
                if (!empty($tags)) { wp_set_post_terms($saved_id, $tags, 'aegislink_link_tag', false); }

                if ($existing && $existing->ID) { $updated++; } else { $created++; }
            }
            fclose($fh);

            $notices[] = sprintf(__('Import finished. Created: %d, Updated: %d, Skipped: %d', 'aegislink'), (int) $created, (int) $updated, (int) $skipped);
        }
    }
}

$settings = get_option(Shortlinks::OPT_SETTINGS, array());
$settings = wp_parse_args($settings, array(
    'prefix' => 'go',
    'default_redirect' => 302,
    'track_clicks' => 1,
    'default_rel_nofollow' => 1,
    'default_rel_sponsored' => 1,
    'qr_provider' => 'qrserver',
));

?>
<div class="wrap aegislink-wrap">
    <h1><?php echo esc_html__('External Linking', 'aegislink'); ?></h1>

    <div class="nav-tab-wrapper">
        <a class="nav-tab <?php echo $view==='manage'?'nav-tab-active':''; ?>" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=manage')); ?>"><?php echo esc_html__('Manage', 'aegislink'); ?></a>
        <a class="nav-tab <?php echo $view==='bulk'?'nav-tab-active':''; ?>" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=bulk')); ?>"><?php echo esc_html__('Bulk', 'aegislink'); ?></a>
        <a class="nav-tab <?php echo $view==='analytics'?'nav-tab-active':''; ?>" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=analytics')); ?>"><?php echo esc_html__('Analytics', 'aegislink'); ?></a>
        <a class="nav-tab <?php echo $view==='importexport'?'nav-tab-active':''; ?>" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=importexport')); ?>"><?php echo esc_html__('Import/Export', 'aegislink'); ?></a>
    </div>

    <?php if (!empty($errors)) : ?>
        <div class="notice notice-error"><p><?php echo esc_html(implode(' ', $errors)); ?></p></div>
    <?php endif; ?>

    <?php if ($pro_locked) : ?>
        <div class="notice notice-warning" style="margin-top:14px;">
            <p><strong><?php echo esc_html__('PRO Feature:', 'aegislink'); ?></strong> <?php echo esc_html__('ShortURL Links is view-only in the free version. Activate a license to unlock editing, import/export, and settings.', 'aegislink'); ?></p>
            <p style="margin:8px 0 0 0;"><a class="button button-primary" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=tools')); ?>"><?php echo esc_html__('Activate License', 'aegislink'); ?></a></p>
        </div>
        <div class="aegislink-pro-locked" aria-disabled="true">
    <?php endif; ?>

    <?php if ($view === 'manage') : ?>

        <h2><?php echo esc_html__('Settings', 'aegislink'); ?></h2>
        <form method="post">
            <?php wp_nonce_field('aegislink_shortlinks_settings_save'); ?>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="prefix"><?php echo esc_html__('Short Link Prefix', 'aegislink'); ?></label></th>
                    <td>
                        <input type="text" class="regular-text" id="prefix" name="prefix" value="<?php echo esc_attr($settings['prefix']); ?>" />
                        <p class="description"><?php echo esc_html__('Example: "go" creates URLs like /go/your-slug', 'aegislink'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Default Redirect', 'aegislink'); ?></th>
                    <td>
                        <select name="default_redirect">
                            <?php foreach (array(301,302,307) as $c) : ?>
                                <option value="<?php echo (int) $c; ?>" <?php selected((int) $settings['default_redirect'], (int) $c); ?>><?php echo (int) $c; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Default Click Tracking', 'aegislink'); ?></th>
                    <td><label><input type="checkbox" name="track_clicks" value="1" <?php checked((int) $settings['track_clicks'], 1); ?> /> <?php echo esc_html__('Enabled', 'aegislink'); ?></label></td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Default rel attributes', 'aegislink'); ?></th>
                    <td>
                        <label style="margin-right:12px;"><input type="checkbox" name="default_rel_nofollow" value="1" <?php checked((int) $settings['default_rel_nofollow'], 1); ?> /> <?php echo esc_html__('nofollow', 'aegislink'); ?></label>
                        <label><input type="checkbox" name="default_rel_sponsored" value="1" <?php checked((int) $settings['default_rel_sponsored'], 1); ?> /> <?php echo esc_html__('sponsored', 'aegislink'); ?></label>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('QR Provider', 'aegislink'); ?></th>
                    <td>
                        <select name="qr_provider">
                            <option value="qrserver" <?php selected($settings['qr_provider'], 'qrserver'); ?>><?php echo esc_html__('QRServer (external image)', 'aegislink'); ?></option>
                            <option value="disabled" <?php selected($settings['qr_provider'], 'disabled'); ?>><?php echo esc_html__('Disabled', 'aegislink'); ?></option>
                        </select>
                        <p class="description"><?php echo esc_html__('QR code images are shown in Share modal. Disable if you prefer not to call external QR image endpoints.', 'aegislink'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('WooCommerce Share Button', 'aegislink'); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="woo_share_enabled" value="1" <?php checked(!empty($settings['woo_share_enabled'])); ?> />
                            <?php echo esc_html__('Enable Share button on WooCommerce single product pages when a product short link exists.', 'aegislink'); ?>
                        </label>
                        <p class="description"><?php echo esc_html__('Bulk-generate product short links to auto-enable sharing. The button will only appear if the product has a mapped short link.', 'aegislink'); ?></p>
                    </td>
                </tr>

            </table>
            <p><button type="submit" class="button button-primary" name="aegislink_shortlinks_settings_save" value="1"><?php echo esc_html__('Save Settings', 'aegislink'); ?></button></p>
        </form>

        <hr />

        <h2><?php echo $editing ? esc_html__('Edit Short Link', 'aegislink') : esc_html__('Add Short Link', 'aegislink'); ?></h2>

        <?php
        $ed_target = $editing ? (string) get_post_meta($editing->ID, '_aegislink_target', true) : '';
        $ed_redir  = $editing ? (int) get_post_meta($editing->ID, '_aegislink_redirect', true) : (int) $settings['default_redirect'];
        $ed_track  = $editing ? (int) get_post_meta($editing->ID, '_aegislink_track', true) : (int) $settings['track_clicks'];
        $ed_nf     = $editing ? (int) get_post_meta($editing->ID, '_aegislink_rel_nofollow', true) : (int) $settings['default_rel_nofollow'];
        $ed_sp     = $editing ? (int) get_post_meta($editing->ID, '_aegislink_rel_sponsored', true) : (int) $settings['default_rel_sponsored'];
        $ed_notes  = $editing ? (string) get_post_meta($editing->ID, '_aegislink_notes', true) : '';

        $ed_groups = $editing ? wp_get_post_terms($editing->ID, 'aegislink_link_group', array('fields' => 'names')) : array();
        $ed_tags   = $editing ? wp_get_post_terms($editing->ID, 'aegislink_link_tag', array('fields' => 'names')) : array();

        $ed_utm_enable = $editing ? (int) get_post_meta($editing->ID, '_aegislink_utm_enable', true) : 0;
        $ed_utm_source = $editing ? (string) get_post_meta($editing->ID, '_aegislink_utm_source', true) : '';
        $ed_utm_medium = $editing ? (string) get_post_meta($editing->ID, '_aegislink_utm_medium', true) : '';
        $ed_utm_campaign = $editing ? (string) get_post_meta($editing->ID, '_aegislink_utm_campaign', true) : '';
        $ed_utm_term = $editing ? (string) get_post_meta($editing->ID, '_aegislink_utm_term', true) : '';
        $ed_utm_content = $editing ? (string) get_post_meta($editing->ID, '_aegislink_utm_content', true) : '';
        ?>

        <form method="post">
            <?php wp_nonce_field('aegislink_shortlink_save'); ?>
            <input type="hidden" name="edit_id" value="<?php echo $editing ? (int) $editing->ID : 0; ?>" />

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="title"><?php echo esc_html__('Title', 'aegislink'); ?></label></th>
                    <td><input type="text" class="regular-text" id="title" name="title" value="<?php echo esc_attr($editing ? $editing->post_title : ''); ?>" required /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="target_url"><?php echo esc_html__('Target URL', 'aegislink'); ?></label></th>
                    <td><input type="url" class="large-text" id="target_url" name="target_url" value="<?php echo esc_attr($ed_target); ?>" placeholder="https://example.com/..." required /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="slug"><?php echo esc_html__('Short Slug', 'aegislink'); ?></label></th>
                    <td>
                        <input type="text" class="regular-text" id="slug" name="slug" value="<?php echo esc_attr($editing ? $editing->post_name : ''); ?>" />
                        <p class="description"><?php echo esc_html__('Leave blank to auto-generate from title.', 'aegislink'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Redirect', 'aegislink'); ?></th>
                    <td>
                        <select name="redirect">
                            <?php foreach (array(301,302,307) as $c) : ?>
                                <option value="<?php echo (int) $c; ?>" <?php selected((int) $ed_redir, (int) $c); ?>><?php echo (int) $c; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php echo esc_html__('Tracking', 'aegislink'); ?></th>
                    <td><label><input type="checkbox" name="track_clicks" value="1" <?php checked((int) $ed_track, 1); ?> /> <?php echo esc_html__('Track clicks for this link', 'aegislink'); ?></label></td>
                </tr>

                <tr>
                    <th scope="row"><?php echo esc_html__('rel attributes', 'aegislink'); ?></th>
                    <td>
                        <label style="margin-right:12px;"><input type="checkbox" name="rel_nofollow" value="1" <?php checked((int) $ed_nf, 1); ?> /> <?php echo esc_html__('nofollow', 'aegislink'); ?></label>
                        <label><input type="checkbox" name="rel_sponsored" value="1" <?php checked((int) $ed_sp, 1); ?> /> <?php echo esc_html__('sponsored', 'aegislink'); ?></label>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="groups"><?php echo esc_html__('Groups (comma-separated)', 'aegislink'); ?></label></th>
                    <td><input type="text" class="regular-text" id="groups" name="groups" value="<?php echo esc_attr(implode(', ', (array) $ed_groups)); ?>" placeholder="Affiliates, Partners, Products" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="tags"><?php echo esc_html__('Tags (comma-separated)', 'aegislink'); ?></label></th>
                    <td><input type="text" class="regular-text" id="tags" name="tags" value="<?php echo esc_attr(implode(', ', (array) $ed_tags)); ?>" placeholder="black-friday, promo, instagram" /></td>
                </tr>

                <tr>
                    <th scope="row"><?php echo esc_html__('UTM Builder', 'aegislink'); ?></th>
                    <td>
                        <label><input type="checkbox" name="utm_enable" value="1" <?php checked((int) $ed_utm_enable, 1); ?> /> <?php echo esc_html__('Append UTM params on redirect', 'aegislink'); ?></label>
                        <div style="margin-top:10px; display:grid; grid-template-columns: 1fr 1fr; gap:10px; max-width: 760px;">
                            <input type="text" name="utm_source" class="regular-text" placeholder="utm_source" value="<?php echo esc_attr($ed_utm_source); ?>" />
                            <input type="text" name="utm_medium" class="regular-text" placeholder="utm_medium" value="<?php echo esc_attr($ed_utm_medium); ?>" />
                            <input type="text" name="utm_campaign" class="regular-text" placeholder="utm_campaign" value="<?php echo esc_attr($ed_utm_campaign); ?>" />
                            <input type="text" name="utm_term" class="regular-text" placeholder="utm_term" value="<?php echo esc_attr($ed_utm_term); ?>" />
                            <input type="text" name="utm_content" class="regular-text" placeholder="utm_content" value="<?php echo esc_attr($ed_utm_content); ?>" />
                        </div>
                        <p class="description"><?php echo esc_html__('These are added at redirect-time so your stored Target URL remains clean.', 'aegislink'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="notes"><?php echo esc_html__('Notes', 'aegislink'); ?></label></th>
                    <td><textarea class="large-text" rows="3" id="notes" name="notes"><?php echo esc_textarea($ed_notes); ?></textarea></td>
                </tr>
            </table>

            <p>
                <button type="submit" class="button button-primary" name="aegislink_shortlink_save" value="1"><?php echo esc_html__('Save Short Link', 'aegislink'); ?></button>
                <?php if ($editing) : ?>
                    <a class="button" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=manage')); ?>"><?php echo esc_html__('Cancel', 'aegislink'); ?></a>
                <?php endif; ?>
            </p>
        </form>

        <hr />

        <h2><?php echo esc_html__('Short Links', 'aegislink'); ?></h2>

        <?php
        $q = new WP_Query(array(
            'post_type' => Shortlinks::CPT,
            'post_status' => 'publish',
            'posts_per_page' => 50,
            'orderby' => 'modified',
            'order' => 'DESC',
            'no_found_rows' => true,
        ));
        ?>

        <table class="widefat striped">
            <thead>
                <tr>
                    <th><?php echo esc_html__('Short URL', 'aegislink'); ?></th>
                    <th><?php echo esc_html__('Target URL', 'aegislink'); ?></th>
                    <th><?php echo esc_html__('Groups / Tags', 'aegislink'); ?></th>
                    <th><?php echo esc_html__('Health', 'aegislink'); ?></th>
                    <th><?php echo esc_html__('Clicks', 'aegislink'); ?></th>
                    <th><?php echo esc_html__('Actions', 'aegislink'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($q->posts)) : ?>
                    <tr><td colspan="6"><?php echo esc_html__('No short links yet.', 'aegislink'); ?></td></tr>
                <?php else : foreach ($q->posts as $p) : 
                    $short_url = $base_short_url . $p->post_name;
                    $target = (string) get_post_meta($p->ID, '_aegislink_target', true);
                    $clicks = $short->get_click_count($p->ID);

                    $groups = wp_get_post_terms($p->ID, 'aegislink_link_group', array('fields' => 'names'));
                    $tags   = wp_get_post_terms($p->ID, 'aegislink_link_tag', array('fields' => 'names'));

                    $health_ok = (int) get_post_meta($p->ID, '_aegislink_health_ok', true);
                    $health_code = (int) get_post_meta($p->ID, '_aegislink_health_code', true);
                    $health_checked = (string) get_post_meta($p->ID, '_aegislink_health_checked', true);
                    $health_label = $health_checked ? ($health_ok ? __('OK', 'aegislink') : __('Broken', 'aegislink')) : __('Pending', 'aegislink');
                    $health_tip = $health_checked ? sprintf(__('HTTP %d last checked %s', 'aegislink'), $health_code, $health_checked) : __('Not checked yet.', 'aegislink');
                ?>
                    <tr>
                        <td><code><?php echo esc_html($short_url); ?></code></td>
                        <td style="max-width:520px; overflow:hidden; text-overflow:ellipsis;">
                            <a href="<?php echo esc_url($target); ?>" target="_blank" rel="noopener"><?php echo esc_html($target); ?></a>
                        </td>
                        <td>
                            <?php if (!empty($groups)) : ?>
                                <div><strong><?php echo esc_html__('Groups:', 'aegislink'); ?></strong> <?php echo esc_html(implode(', ', $groups)); ?></div>
                            <?php endif; ?>
                            <?php if (!empty($tags)) : ?>
                                <div><strong><?php echo esc_html__('Tags:', 'aegislink'); ?></strong> <?php echo esc_html(implode(', ', $tags)); ?></div>
                            <?php endif; ?>
                            <?php if (empty($groups) && empty($tags)) : ?>
                                <span style="opacity:.7;"><?php echo esc_html__('—', 'aegislink'); ?></span>
                            <?php endif; ?>
                        </td>
                        <td title="<?php echo esc_attr($health_tip); ?>">
                            <?php if ($health_label === 'OK') : ?>
                                <span class="aegislink-pill aegislink-pill-ok"><?php echo esc_html($health_label); ?></span>
                            <?php elseif ($health_label === 'Broken') : ?>
                                <span class="aegislink-pill aegislink-pill-bad"><?php echo esc_html($health_label); ?></span>
                            <?php else : ?>
                                <span class="aegislink-pill aegislink-pill-pending"><?php echo esc_html($health_label); ?></span>
                            <?php endif; ?>
                        </td>
                        <td><?php echo (int) $clicks; ?></td>
                        <td>
                            <a class="button-link" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=manage&edit=' . (int) $p->ID)); ?>"><?php echo esc_html__('Edit', 'aegislink'); ?></a>
                            <span class="sep">|</span>
                            <a class="button-link" href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=manage&delete=' . (int) $p->ID), 'aegislink_shortlink_delete_' . (int) $p->ID)); ?>" onclick="return confirm('<?php echo esc_js(__('Delete this short link?', 'aegislink')); ?>');"><?php echo esc_html__('Delete', 'aegislink'); ?></a>
                            <span class="sep">|</span>
                            <a href="#" class="button-link aegislink-share" data-short="<?php echo esc_attr($short_url); ?>" data-title="<?php echo esc_attr($p->post_title); ?>" data-qr="<?php echo esc_attr($settings['qr_provider']); ?>"><?php echo esc_html__('Share', 'aegislink'); ?></a>
                        </td>
                    </tr>
                <?php endforeach; endif; ?>
            </tbody>
        </table>

        <p class="description" style="margin-top:10px;">
            <?php echo esc_html__('Tip: Health checks run daily in small batches. Broken targets show a warning.', 'aegislink'); ?>
        </p>

        <div id="aegislink-share-modal" class="aegislink-modal" style="display:none;">
            <div class="aegislink-modal-inner">
                <button type="button" class="button-link aegislink-modal-close" aria-label="<?php echo esc_attr__('Close', 'aegislink'); ?>">×</button>
                <h2 id="aegislink-share-title"><?php echo esc_html__('Share Short Link', 'aegislink'); ?></h2>
                <p><input type="text" class="large-text" id="aegislink-share-url" readonly /></p>
                <p>
                    <button type="button" class="button button-primary" id="aegislink-copy-btn"><?php echo esc_html__('Copy', 'aegislink'); ?></button>
                </p>
                <div id="aegislink-qr-wrap" style="margin-top:12px;"></div>
                <div style="margin-top:14px;">
                    <a href="#" target="_blank" rel="noopener" id="aegislink-share-twitter">X</a>
                    <span class="sep">|</span>
                    <a href="#" target="_blank" rel="noopener" id="aegislink-share-facebook">Facebook</a>
                    <span class="sep">|</span>
                    <a href="#" target="_blank" rel="noopener" id="aegislink-share-linkedin">LinkedIn</a>
                </div>
            </div>
        </div>

    <?php elseif ($view === 'bulk') : ?>

        <h2><?php echo esc_html__('Bulk Short Links', 'aegislink'); ?></h2>
        <p class="description"><?php echo esc_html__('Create many short links at once (perfect for WooCommerce product URLs and category-heavy sites).', 'aegislink'); ?></p>

        <form method="post">
            <?php wp_nonce_field('aegislink_bulk_create'); ?>

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php echo esc_html__('Source Type', 'aegislink'); ?></th>
                    <td>
                        <label><input type="radio" name="bulk_mode" value="urls" checked /> <?php echo esc_html__('Paste URLs', 'aegislink'); ?></label>
                        <br />
                        <label><input type="radio" name="bulk_mode" value="posts" /> <?php echo esc_html__('Generate from Posts (Products, Pages, etc.)', 'aegislink'); ?></label>
                        <br />
                        <label><input type="radio" name="bulk_mode" value="terms" <?php checked($bulk_mode, 'terms'); ?> /> <?php echo esc_html__('Generate from Terms (Categories/Tags/Taxonomies)', 'aegislink'); ?></label>
                    </td>
                </tr>

                <tr class="aegislink-bulk-urls">
                    <th scope="row"><?php echo esc_html__('URLs List', 'aegislink'); ?></th>
                    <td>
                        <textarea class="large-text" rows="8" name="bulk_urls" placeholder="https://example.com/product/blue-shoes/,blue-shoes,Blue Shoes&#10;https://example.com/category/mens/"> </textarea>
                        <p class="description"><?php echo esc_html__('One per line. Optional CSV columns: URL,slug,title', 'aegislink'); ?></p>
                    </td>
                </tr>

                <tr class="aegislink-bulk-posts" style="display:none;">
                    <th scope="row"><?php echo esc_html__('Post Type', 'aegislink'); ?></th>
                    <td>
                        <?php
                        $pts = get_post_types(array('public' => true), 'objects');
                        ?>
                        <select name="bulk_post_type">
                            <?php foreach ($pts as $pt => $obj) : ?>
                                <option value="<?php echo esc_attr($pt); ?>"><?php echo esc_html($obj->labels->singular_name); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description"><?php echo esc_html__('If WooCommerce is active, choose Products.', 'aegislink'); ?></p>
                    </td>
                </tr>
                <tr class="aegislink-bulk-terms" style="display:none;">
                    <th scope="row"><?php echo esc_html__('Taxonomy', 'aegislink'); ?></th>
                    <td>
                        <?php
                        $taxes = Bulk_Terms::get_taxonomies();
                        $sel_tax = isset($_POST['bulk_taxonomy']) ? sanitize_key((string) wp_unslash($_POST['bulk_taxonomy'])) : '';
                        if ($sel_tax === '') { $sel_tax = taxonomy_exists('product_cat') ? 'product_cat' : 'category'; }
                        $sel_strategy = isset($_POST['bulk_term_slug_strategy']) ? sanitize_key((string) wp_unslash($_POST['bulk_term_slug_strategy'])) : 'term';
                        if (!in_array($sel_strategy, array('term','cat','tag'), true)) { $sel_strategy = 'term'; }
                        $sel_limit = isset($_POST['bulk_term_limit']) ? absint($_POST['bulk_term_limit']) : 100;
                        if ($sel_limit <= 0) { $sel_limit = 100; }
                        ?>
                        <select name="bulk_taxonomy">
                            <?php foreach ($taxes as $tx) : ?>
                                <option value="<?php echo esc_attr($tx['name']); ?>" <?php selected($sel_tax, $tx['name']); ?>>
                                    <?php echo esc_html($tx['label'] . ' (' . $tx['name'] . ')'); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description"><?php echo esc_html__('Select which taxonomy terms to generate short links for (WooCommerce product categories/tags are supported when WooCommerce is active).', 'aegislink'); ?></p>
                    </td>
                </tr>

                <tr class="aegislink-bulk-terms" style="display:none;">
                    <th scope="row"><?php echo esc_html__('Slug Strategy', 'aegislink'); ?></th>
                    <td>
                        <select name="bulk_term_slug_strategy">
                            <option value="term" <?php selected($sel_strategy, 'term'); ?>><?php echo esc_html__('Use term slug', 'aegislink'); ?></option>
                            <option value="cat" <?php selected($sel_strategy, 'cat'); ?>><?php echo esc_html__('Prefix cat-{slug}', 'aegislink'); ?></option>
                            <option value="tag" <?php selected($sel_strategy, 'tag'); ?>><?php echo esc_html__('Prefix tag-{slug}', 'aegislink'); ?></option>
                        </select>
                        <p class="description"><?php echo esc_html__('Default uses the term slug. Optional prefixes help avoid collisions between similar term slugs.', 'aegislink'); ?></p>
                    </td>
                </tr>

                <tr class="aegislink-bulk-terms" style="display:none;">
                    <th scope="row"><?php echo esc_html__('Limit', 'aegislink'); ?></th>
                    <td>
                        <input type="number" min="1" max="2000" name="bulk_term_limit" value="<?php echo esc_attr($sel_limit); ?>" />
                        <p class="description"><?php echo esc_html__('How many terms to process (starts with most-used terms).', 'aegislink'); ?></p>
                    </td>
                </tr>


                <tr class="aegislink-bulk-posts" style="display:none;">
                    <th scope="row"><?php echo esc_html__('Limit', 'aegislink'); ?></th>
                    <td><input type="number" min="1" max="5000" name="bulk_limit" value="100" /></td>
                </tr>

                <tr>
                    <th scope="row"><?php echo esc_html__('Defaults', 'aegislink'); ?></th>
                    <td>
                        <label><?php echo esc_html__('Redirect:', 'aegislink'); ?>
                            <select name="bulk_redirect">
                                <?php foreach (array(301,302,307) as $c) : ?>
                                    <option value="<?php echo (int) $c; ?>" <?php selected((int) $settings['default_redirect'], (int) $c); ?>><?php echo (int) $c; ?></option>
                                <?php endforeach; ?>
                            </select>
                        </label>
                        <div style="margin-top:8px;">
                            <label style="margin-right:12px;"><input type="checkbox" name="bulk_track" value="1" <?php checked((int) $settings['track_clicks'], 1); ?> /> <?php echo esc_html__('Track clicks', 'aegislink'); ?></label>
                            <label style="margin-right:12px;"><input type="checkbox" name="bulk_rel_nofollow" value="1" <?php checked((int) $settings['default_rel_nofollow'], 1); ?> /> <?php echo esc_html__('nofollow', 'aegislink'); ?></label>
                            <label><input type="checkbox" name="bulk_rel_sponsored" value="1" <?php checked((int) $settings['default_rel_sponsored'], 1); ?> /> <?php echo esc_html__('sponsored', 'aegislink'); ?></label>
                        </div>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php echo esc_html__('Assign Groups / Tags', 'aegislink'); ?></th>
                    <td>
                        <input type="text" class="regular-text" name="bulk_groups" placeholder="Groups (comma-separated)" />
                        <input type="text" class="regular-text" name="bulk_tags" placeholder="Tags (comma-separated)" />
                    </td>
                </tr>
            </table>

            <p><button type="submit" class="button button-primary" name="aegislink_bulk_create" value="1"><?php echo esc_html__('Create Short Links', 'aegislink'); ?></button></p>
        </form>

    <?php elseif ($view === 'analytics') : ?>

        <h2><?php echo esc_html__('Advanced Analytics', 'aegislink'); ?></h2>
        <?php
        global $wpdb;
        $table = $wpdb->prefix . 'aegislink_clicks';

        $total_clicks = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table}");
        $since = gmdate('Y-m-d H:i:s', time() - 30*DAY_IN_SECONDS);

        $daily = $wpdb->get_results($wpdb->prepare("
            SELECT DATE(created_at) AS d, COUNT(*) AS c
            FROM {$table}
            WHERE created_at >= %s
            GROUP BY DATE(created_at)
            ORDER BY d DESC
            LIMIT 30
        ", $since));

        $top_ref = $wpdb->get_results($wpdb->prepare("
            SELECT referer, COUNT(*) AS c
            FROM {$table}
            WHERE created_at >= %s AND referer <> '' AND referer IS NOT NULL
            GROUP BY referer
            ORDER BY c DESC
            LIMIT 20
        ", $since));

        $top_ua = $wpdb->get_results($wpdb->prepare("
            SELECT ua, COUNT(*) AS c
            FROM {$table}
            WHERE created_at >= %s AND ua <> '' AND ua IS NOT NULL
            GROUP BY ua
            ORDER BY c DESC
            LIMIT 10
        ", $since));
        ?>

        <p><strong><?php echo esc_html__('Total clicks recorded:', 'aegislink'); ?></strong> <?php echo (int) $total_clicks; ?></p>

        <h3><?php echo esc_html__('Daily Clicks (Last 30 Days)', 'aegislink'); ?></h3>
        <table class="widefat striped" style="max-width:420px;">
            <thead><tr><th><?php echo esc_html__('Date', 'aegislink'); ?></th><th><?php echo esc_html__('Clicks', 'aegislink'); ?></th></tr></thead>
            <tbody>
            <?php if (empty($daily)) : ?>
                <tr><td colspan="2"><?php echo esc_html__('No clicks yet.', 'aegislink'); ?></td></tr>
            <?php else : foreach ($daily as $r) : ?>
                <tr><td><?php echo esc_html($r->d); ?></td><td><?php echo (int) $r->c; ?></td></tr>
            <?php endforeach; endif; ?>
            </tbody>
        </table>

        <h3 style="margin-top:16px;"><?php echo esc_html__('Top Referrers (Last 30 Days)', 'aegislink'); ?></h3>
        <table class="widefat striped">
            <thead><tr><th><?php echo esc_html__('Referrer', 'aegislink'); ?></th><th><?php echo esc_html__('Clicks', 'aegislink'); ?></th></tr></thead>
            <tbody>
            <?php if (empty($top_ref)) : ?>
                <tr><td colspan="2"><?php echo esc_html__('No referrers recorded.', 'aegislink'); ?></td></tr>
            <?php else : foreach ($top_ref as $r) : ?>
                <tr><td style="max-width:720px; overflow:hidden; text-overflow:ellipsis;"><?php echo esc_html((string) $r->referer); ?></td><td><?php echo (int) $r->c; ?></td></tr>
            <?php endforeach; endif; ?>
            </tbody>
        </table>

        <h3 style="margin-top:16px;"><?php echo esc_html__('Top User Agents (Last 30 Days)', 'aegislink'); ?></h3>
        <table class="widefat striped">
            <thead><tr><th><?php echo esc_html__('User Agent', 'aegislink'); ?></th><th><?php echo esc_html__('Clicks', 'aegislink'); ?></th></tr></thead>
            <tbody>
            <?php if (empty($top_ua)) : ?>
                <tr><td colspan="2"><?php echo esc_html__('No user agents recorded.', 'aegislink'); ?></td></tr>
            <?php else : foreach ($top_ua as $r) : ?>
                <tr><td style="max-width:720px; overflow:hidden; text-overflow:ellipsis;"><?php echo esc_html((string) $r->ua); ?></td><td><?php echo (int) $r->c; ?></td></tr>
            <?php endforeach; endif; ?>
            </tbody>
        </table>

        <p class="description" style="margin-top:12px;">
            <?php echo esc_html__('Privacy note: clicks store referrer + user-agent + an IP hash (not the raw IP).', 'aegislink'); ?>
        </p>

    <?php else : // importexport ?>

        <h2><?php echo esc_html__('Import / Export', 'aegislink'); ?></h2>

        <h3><?php echo esc_html__('Export', 'aegislink'); ?></h3>
        <p>
            <a class="button button-secondary" href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=importexport&export=1'), 'aegislink_shortlinks_export')); ?>">
                <?php echo esc_html__('Download CSV Export', 'aegislink'); ?>
            </a>
        </p>

        <hr />

        <h3><?php echo esc_html__('Import', 'aegislink'); ?></h3>
        <form method="post" enctype="multipart/form-data">
            <?php wp_nonce_field('aegislink_shortlinks_import'); ?>
            <input type="file" name="csv_file" accept=".csv,text/csv" required />
            <p class="description"><?php echo esc_html__('Columns supported: slug,title,target_url,redirect,track_clicks,rel_nofollow,rel_sponsored,notes,groups,tags,utm_*', 'aegislink'); ?></p>
            <p><button type="submit" class="button button-primary" name="aegislink_import_csv" value="1"><?php echo esc_html__('Import CSV', 'aegislink'); ?></button></p>
        </form>

    <?php endif; ?>

    <?php if ($pro_locked) : ?>
        </div>
        <script>
        (function(){
            try {
                var wrap = document.querySelector('.aegislink-pro-locked');
                if(!wrap) return;
                // ensure all form controls are disabled (defense in depth)
                wrap.querySelectorAll('input,select,textarea,button').forEach(function(el){ el.setAttribute('disabled','disabled'); });
            } catch(e) {}
        })();
        </script>
    <?php endif; ?>

</div>
