<?php
namespace AegisLink;

use AegisLink\Admin\Admin_Menu;
use AegisLink\Utils\License;
use AegisLink\SEO\Linking;
use AegisLink\Shortcodes\Shortcodes;
use AegisLink\Shortlinks\Shortlinks;

if (!defined('ABSPATH')) { exit; }

class Plugin {

    private static $instance = null;

    /** @var License */
    public $license;

    /** @var Shortlinks */
    public $shortlinks;

    /** @var Linking */
    public $linking;

    /** @var Admin_Menu */
    public $admin_menu;

    /** @var Shortcodes */
    public $shortcodes;

    public static function instance() : self {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function boot() : void {
        $this->includes();

        $this->license = new License();
		add_action('admin_init', function () {
			// Daily CLM check: refresh license status + (optional) free-install registration
			$this->license->maybe_daily_check(AEGISLINK_VERSION);
		});

		$this->linking = new Linking();
        $this->shortcodes = new Shortcodes();
        $this->shortlinks = new Shortlinks();

        // WooCommerce Share integration (optional)
        $opts = (array) get_option(Shortlinks::OPT_SETTINGS, array());
        if (!empty($opts['woo_share_enabled']) && class_exists('WooCommerce')) {
            $woo = new \AegisLink\Integrations\Woo_Share();
            $woo->hooks();
        }

        if (is_admin()) {
            $this->admin_menu = new Admin_Menu($this->license);
        }

        // Update checker (CLM)
        add_filter('pre_set_site_transient_update_plugins', array($this, 'filter_update_check'), 10, 1);
        add_filter('plugins_api', array($this, 'filter_plugins_api'), 10, 3);
    }

    private function includes() : void {
        require_once AEGISLINK_DIR . 'includes/utils/class-aegislink-license.php';
        require_once AEGISLINK_DIR . 'includes/seo/class-aegislink-linking.php';
        require_once AEGISLINK_DIR . 'includes/admin/class-aegislink-admin-menu.php';
        require_once AEGISLINK_DIR . 'includes/class-aegisify-bundled-plugins-page.php';
        require_once AEGISLINK_DIR . 'includes/shortcodes/class-aegislink-shortcodes.php';
        require_once AEGISLINK_DIR . 'includes/shortlinks/class-aegislink-shortlinks.php';
        require_once AEGISLINK_DIR . 'includes/shortlinks/class-aegislink-shortlinks-bulk-terms.php';
        require_once AEGISLINK_DIR . 'includes/integrations/class-aegislink-woo-share.php';
    }

    public function filter_update_check($transient) {
        if (!is_object($transient)) { return $transient; }

        // Avoid hammering endpoint if WP calls frequently
        $cached = get_transient('aegislink_update_check');
        if (is_array($cached)) {
            if (!empty($cached['response'])) {
                $transient->response = array_merge((array) $transient->response, (array) $cached['response']);
            }
            return $transient;
        }

        $status = $this->license->get_status();
        $license_key = $this->license->get_key();

        $payload = array(
            'slug'            => AEGISLINK_SLUG,
            'product'         => AEGISLINK_SLUG,
            'version'         => AEGISLINK_VERSION,
            'domain'          => (string) wp_parse_url(home_url(), PHP_URL_HOST),
            'site_url'        => (string) home_url(),
            'license_key'     => (string) $license_key,
            'email'           => (string) get_option('aegislink_license_email', ''),
            'wp_version'      => (string) get_bloginfo('version'),
            'php_version'     => (string) PHP_VERSION,
            'plugin_basename' => (string) AEGISLINK_BASENAME,
            'is_pro'          => !empty($status['is_pro']) ? 1 : 0,
        );

        $resp = wp_remote_post(AEGISLINK_CLM_UPDATE_ENDPOINT, array(
            'timeout' => 20,
            'headers' => array('Content-Type' => 'application/json'),
            'body'    => wp_json_encode($payload),
        ));

        $response = array();

        if (!is_wp_error($resp)) {
            $code = wp_remote_retrieve_response_code($resp);
            $body = wp_remote_retrieve_body($resp);
            $json = json_decode($body, true);

            if ($code >= 200 && $code < 300 && is_array($json)) {
                // Expect CLM update-check schema: { success, update: { new_version, package, tested, requires, ... } }
                if (!empty($json['update']['new_version']) && !empty($json['update']['package'])) {
                    $response[AEGISLINK_BASENAME] = array(
                        'slug'        => 'aegislink',
                        'plugin'      => AEGISLINK_BASENAME,
                        'new_version' => (string) $json['update']['new_version'],
                        'package'     => (string) $json['update']['package'],
                        'tested'      => !empty($json['update']['tested']) ? (string) $json['update']['tested'] : '',
                        'requires'    => !empty($json['update']['requires']) ? (string) $json['update']['requires'] : '',
                        'url'         => !empty($json['update']['homepage']) ? (string) $json['update']['homepage'] : '',
                    );
                }
            }
        }

        set_transient('aegislink_update_check', array('response' => $response), DAY_IN_SECONDS);

        if (!empty($response)) {
            $transient->response = array_merge((array) $transient->response, $response);
        }
        return $transient;
    }

    public function filter_plugins_api($result, $action, $args) {
        if ($action !== 'plugin_information') { return $result; }
        if (!is_object($args) || empty($args->slug) || $args->slug !== 'aegislink') { return $result; }

        // Minimal plugin info (WordPress expects an object)
        $info = new \stdClass();
        $info->name = 'AegisLink';
        $info->slug = 'aegislink';
        $info->version = AEGISLINK_VERSION;
        $info->author = '<a href="https://aegisify.com/">Aegisify</a>';
        $info->homepage = 'https://aegisify.com/';
        $info->requires = '6.8';
        $info->tested = get_bloginfo('version');
        $info->requires_php = '8.2';
        $info->sections = array(
            'description' => 'Smart internal linking + linking assistant by Aegisify.',
        );

        return $info;
    }
}