<?php
namespace AegisLink\Shortcodes;

if (!defined('ABSPATH')) { exit; }

class Shortcodes {

    public function __construct() {
        add_shortcode('aegislink_keywords_block', array($this, 'render_keywords_block'));
    }
	
    public function render_keywords_block($atts = array(), $content = null) : string {
        $opt = get_option('aegislink_wordcloud_defaults', array());
        if (!is_array($opt)) { $opt = array(); }

        $defaults = array(
            'title'       => isset($opt['title']) ? (string) $opt['title'] : __('Keyword Word Cloud', 'aegislink'),
            'limit'       => isset($opt['limit']) ? (int) $opt['limit'] : 20,
            'min_len'     => isset($opt['min_len']) ? (int) $opt['min_len'] : 3,
            'max_len'     => isset($opt['max_len']) ? (int) $opt['max_len'] : 32,
            'show_footer' => isset($opt['show_footer']) ? (int) $opt['show_footer'] : 1,

            'lg_size'     => isset($opt['lg_size']) ? (int) $opt['lg_size'] : 42,
            'md_size'     => isset($opt['md_size']) ? (int) $opt['md_size'] : 28,
            'sm_size'     => isset($opt['sm_size']) ? (int) $opt['sm_size'] : 18,

            'lg_count'    => isset($opt['lg_count']) ? (int) $opt['lg_count'] : 10,
            'md_count'    => isset($opt['md_count']) ? (int) $opt['md_count'] : 20,

            'post_id'     => 0,
        );

        $atts = shortcode_atts($defaults, (array) $atts, 'aegislink_keywords_block');

        if (!empty($_GET['aegislink_wc']) && (string) $_GET['aegislink_wc'] === '1') {
            return '';
        }

        $limit = max(5, min(80, (int) $atts['limit']));
        $min_len = max(2, min(20, (int) $atts['min_len']));
        $max_len = max($min_len, min(64, (int) $atts['max_len']));
		$post_id = (int) $atts['post_id'];

		if ($post_id <= 0) {
			$post_id = (int) get_queried_object_id();
		}

		if ($post_id <= 0) {
			$maybe = get_the_ID();
			if (!empty($maybe)) {
				$post_id = (int) $maybe;
			}
		}

		if ($post_id <= 0) {
			global $post;
			if (!empty($post) && !empty($post->ID)) {
				$post_id = (int) $post->ID;
			}
		}

        $raw = '';
        if ($post_id > 0) {
            $raw = (string) get_post_field('post_content', $post_id);
        } elseif (is_string($content) && $content !== '') {
            $raw = (string) $content;
        }

        $raw_original = $raw;

        if ($raw === '') {

			return '<div class="aegislink-wordcloud" style="padding:14px;border:1px solid #e5e5e5;border-radius:12px;">'
				. '<div style="text-align:center;font-size:22px;margin-bottom:10px;">' . esc_html($atts['title']) . '</div>'
				. '<div style="text-align:center;font-size:13px;opacity:.75;">' . esc_html__('No keywords found (page content may be generated outside post_content). Try: [aegislink_keywords_block post_id="123"]', 'aegislink') . '</div>'
				. '<div style="text-align:center;margin-top:10px;font-size:11px;color:#b0b0b0;">'
				. '<a href="https://www.aegisify.com/aegisseo/" target="_blank" rel="noopener noreferrer" style="color:#b0b0b0;text-decoration:none;">Powered by AegisSEO</a>'
				. '</div>'
				. '</div>';
		}

		$rendered = $raw_original;

		if ($post_id > 0) {
			$orig = $GLOBALS['shortcode_tags']['aegislink_keywords_block'] ?? null;
			if (isset($GLOBALS['shortcode_tags']['aegislink_keywords_block'])) {
				unset($GLOBALS['shortcode_tags']['aegislink_keywords_block']);
			}

			$rendered = apply_filters('the_content', $raw_original);

			if ($orig) {
				$GLOBALS['shortcode_tags']['aegislink_keywords_block'] = $orig;
			}
		}

		$rendered = html_entity_decode((string) $rendered, ENT_QUOTES | ENT_HTML5, get_bloginfo('charset'));
		$rendered = wp_strip_all_tags((string) $rendered, true);
		$rendered = preg_replace('/\s+/', ' ', (string) $rendered);
		$rendered = trim((string) $rendered);
		
		$text_for_tokens = strtolower($rendered);

		if ($rendered === '') {

            $alt = $this->fetch_visible_text_from_permalink($post_id);

            if ($alt === '') {
                $alt = trim((string) get_the_title($post_id) . ' ' . (string) get_post_field('post_excerpt', $post_id));
            }

			$meta_keys = array(
				'_elementor_data',           
				'_fusion',                   
				'_fusion_builder_status',    
				'_fusion_builder_content',  
			);

			foreach ($meta_keys as $k) {
				$v = get_post_meta($post_id, $k, true);
				if (empty($v)) { continue; }

				if (is_string($v)) {
					$json = json_decode($v, true);
					if (is_array($json)) {
						$alt .= ' ' . $this->flatten_text_from_array($json);
					} else {
						$alt .= ' ' . $v;
					}
				} elseif (is_array($v)) {
					$alt .= ' ' . $this->flatten_text_from_array($v);
				}
			}

			$alt = html_entity_decode((string) $alt, ENT_QUOTES | ENT_HTML5, get_bloginfo('charset'));
			$alt = wp_strip_all_tags((string) $alt, true);
			$alt = preg_replace('/\s+/', ' ', (string) $alt);
			$alt = trim((string) $alt);

			if ($alt === '') {
				return '<div class="aegislink-wordcloud" style="padding:14px;border:1px solid #e5e5e5;border-radius:12px;">'
					. '<div style="text-align:center;font-size:22px;margin-bottom:10px;">' . esc_html($atts['title']) . '</div>'
					. '<div style="text-align:center;font-size:13px;opacity:.75;">'
					. esc_html__('No readable text found. This page likely contains only the shortcode in post_content. Try adding page text in the editor OR use: [aegislink_keywords_block post_id="123"]', 'aegislink')
					. '</div>'
					. '<div style="text-align:center;margin-top:10px;font-size:11px;color:#b0b0b0;">'
					. '<a href="https://www.aegisify.com/aegisseo/" target="_blank" rel="noopener noreferrer" style="color:#b0b0b0;text-decoration:none;">Powered by AegisSEO</a>'
					. '</div>'
					. '</div>';
			}

			$rendered = $alt;
			$text_for_tokens = strtolower($rendered);
		}

		$rank_sources = $this->ranking_keyword_sources((int) $post_id);
		if (!empty($rank_sources)) {
			$text_for_tokens .= ' ' . strtolower(implode(' ', $rank_sources)) . ' ' . strtolower(implode(' ', $rank_sources));
		}

		$tokens = preg_split('/[^a-z0-9]+/i', $text_for_tokens);
		if (!is_array($tokens) || empty($tokens)) {

			return '<div class="aegislink-wordcloud" style="padding:14px;border:1px solid #e5e5e5;border-radius:12px;">'
				. '<div style="text-align:center;font-size:22px;margin-bottom:10px;">' . esc_html($atts['title']) . '</div>'
				. '<div style="text-align:center;font-size:13px;opacity:.75;">' . esc_html__('No readable keywords found for this page.', 'aegislink') . '</div>'
				. '<div style="text-align:center;margin-top:10px;font-size:11px;color:#b0b0b0;">'
				. '<a href="https://www.aegisify.com/aegisseo/" target="_blank" rel="noopener noreferrer" style="color:#b0b0b0;text-decoration:none;">Powered by AegisSEO</a>'
				. '</div>'
				. '</div>';
		}

        $stop = $this->stopwords();
        $freq = array();

        foreach ($tokens as $t) {
            $t = trim($t);
            if ($t === '') { continue; }
            if (strlen($t) < $min_len) { continue; }
            if (strlen($t) > $max_len) { continue; }
            if (isset($stop[$t])) { continue; }
            if (preg_match('/^\d+$/', $t)) { continue; }

            if (!isset($freq[$t])) { $freq[$t] = 0; }
            $freq[$t]++;
        }

        if (empty($freq)) {
            return '';
        }

        arsort($freq);
        $freq = array_slice($freq, 0, $limit, true);

        $counts = array_values($freq);
        $max = max($counts);
        $min = min($counts);

        $sorted_counts = $counts;
        rsort($sorted_counts);
        $n = count($sorted_counts);
        $t_large = $sorted_counts[(int) floor($n * 0.25)] ?? $sorted_counts[0];
        $t_med   = $sorted_counts[(int) floor($n * 0.60)] ?? end($sorted_counts);

        $lg_size = max(16, min(120, (int) $atts['lg_size']));
        $md_size = max(14, min(100, (int) $atts['md_size']));
        $sm_size = max(10, min(80, (int) $atts['sm_size']));

        $lg_count = max(0, (int) $atts['lg_count']);
        $md_count = max(0, (int) $atts['md_count']);
        $use_counts = ($lg_count > 0 || $md_count > 0);

        $class_by_word = array();
        if ($use_counts) {
            $i = 0;
            foreach ($freq as $w => $c) {
                $class = 'aegislink-word-sm';
                if ($i < $lg_count) {
                    $class = 'aegislink-word-lg';
                } elseif ($i < ($lg_count + $md_count)) {
                    $class = 'aegislink-word-md';
                }
                $class_by_word[$w] = $class;
                $i++;
            }
        }

        $id = 'aegislink-kb-' . wp_generate_uuid4();

        ob_start();
        ?>
        <div id="<?php echo esc_attr($id); ?>" class="aegislink-keywords-block">
            <h2 class="aegislink-kb-title"><?php echo esc_html($atts['title']); ?></h2>
            <div class="aegislink-kb-cloud" role="list" aria-label="<?php echo esc_attr($atts['title']); ?>">
                <?php foreach ($freq as $word => $count): 
                    $class = isset($class_by_word[$word]) ? $class_by_word[$word] : 'aegislink-word-sm';
                    if (!$use_counts) {
                        if ($count >= $t_large) {
                            $class = 'aegislink-word-lg';
                        } elseif ($count >= $t_med) {
                            $class = 'aegislink-word-md';
                        }
                    }
                ?>
                    <span class="aegislink-word <?php echo esc_attr($class); ?>" role="listitem" style="color: <?php echo esc_attr($this->color_for_word($word)); ?>;">
                        <?php echo esc_html($word); ?>
                    </span>
                <?php endforeach; ?>
            </div>

            <?php if (!empty($atts['show_footer'])) : ?>
            <div class="aegislink-kb-powered">
                <a href="https://www.aegisify.com/aegisseo/" target="_blank" rel="noopener noreferrer">
                    <?php echo esc_html__('Powered by AegisSEO', 'aegislink'); ?>
                </a>
            </div>
            <?php endif; ?>
        </div>

        <style>
            #<?php echo esc_attr($id); ?>.aegislink-keywords-block{
                padding: 18px 16px;
                border: 1px solid rgba(0,0,0,.08);
                border-radius: 14px;
                background: transparent;
                max-width: 100%;
                margin: 16px auto;
            }
            #<?php echo esc_attr($id); ?> .aegislink-kb-title{
                text-align:center;
                margin: 0 0 14px 0;
                font-size: 32px;
                line-height: 1.1;
                font-weight: 700;
            }
            #<?php echo esc_attr($id); ?> .aegislink-kb-cloud{
                display:flex;
                flex-wrap: wrap;
                gap: 10px 18px;
                justify-content: center;
                align-items: center;
                padding: 8px 6px 2px 6px;
            }
            #<?php echo esc_attr($id); ?> .aegislink-word{
                display:inline-block;
                padding: 4px 8px;
                border-radius: 999px;
                background: rgba(0,0,0,0.035);
                transition: transform .12s ease, background .12s ease;
                text-decoration: none;
                color: inherit;
                font-weight: 700;
                line-height: 1;
                opacity: 0.96;
                letter-spacing: .2px;
                user-select: none;
            }
            #<?php echo esc_attr($id); ?> .aegislink-word-sm{ font-size: <?php echo (int) $sm_size; ?>px; }
            #<?php echo esc_attr($id); ?> .aegislink-word-md{ font-size: <?php echo (int) $md_size; ?>px; }
            #<?php echo esc_attr($id); ?> .aegislink-word-lg{ font-size: <?php echo (int) $lg_size; ?>px; }
            #<?php echo esc_attr($id); ?> .aegislink-word:hover{ transform: translateY(-1px) scale(1.02); background: rgba(0,0,0,0.06); }


            #<?php echo esc_attr($id); ?> .aegislink-kb-powered{
                text-align:center;
                margin-top: 12px;
                font-size: 11px;
                color: #b0b0b0;
                opacity: 1;
            }
            #<?php echo esc_attr($id); ?> .aegislink-kb-powered a{
                color: #b0b0b0;
                text-decoration:none;
            }
            #<?php echo esc_attr($id); ?> .aegislink-kb-powered a:hover{
                color: #9a9a9a;
                text-decoration:underline;
            }
        </style>
        <?php
        return (string) ob_get_clean();
    }

private function flatten_text_from_array($data) : string {
	$out = '';
	if (is_array($data)) {
		foreach ($data as $k => $v) {
			if (is_string($v)) {
				$val = trim($v);
				if ($val !== '' && $this->looks_like_human_text($val)) {
					$out .= ' ' . $val;
				}
			} elseif (is_array($v)) {
				$out .= ' ' . $this->flatten_text_from_array($v);
			}
		}
	}
	return trim($out);
}

    private function fetch_visible_text_from_permalink(int $post_id) : string {
        if ($post_id <= 0) {
            return '';
        }

        $cache_key = 'aegislink_wc_text_' . $post_id;
        $cached = get_transient($cache_key);
        if (is_string($cached) && $cached !== '') {
            return $cached;
        }

        $url = get_permalink($post_id);
        if (!$url) {
            return '';
        }

        $url = add_query_arg('aegislink_wc', '1', $url);

        $resp = wp_remote_get($url, array(
            'timeout' => 8,
            'redirection' => 3,
            'headers' => array(
                'X-AegisLink-Wordcloud' => '1',
            ),
        ));

        if (is_wp_error($resp)) {
            return '';
        }

        $code = (int) wp_remote_retrieve_response_code($resp);
        if ($code < 200 || $code >= 300) {
            return '';
        }

        $html = (string) wp_remote_retrieve_body($resp);
        if ($html === '') {
            return '';
        }

        $text = $this->extract_visible_text_from_html($html);
        if ($text !== '') {
            set_transient($cache_key, $text, 10 * MINUTE_IN_SECONDS);
        }
        return $text;
    }

    private function extract_visible_text_from_html(string $html) : string {
        if ($html === '') { return ''; }

        $prev = libxml_use_internal_errors(true);

        $dom = new \DOMDocument();
        $dom->loadHTML('<?xml encoding="utf-8" ?>' . $html, LIBXML_NOWARNING | LIBXML_NOERROR);

        $xpath = new \DOMXPath($dom);

        foreach ($xpath->query('//script|//style|//noscript|//svg|//meta|//link|//head') as $node) {
            $node->parentNode?->removeChild($node);
        }

        foreach ($xpath->query('//*[contains(concat(" ", normalize-space(@class), " "), " header ") or contains(concat(" ", normalize-space(@class), " "), " footer ") or contains(concat(" ", normalize-space(@class), " "), " nav ") or self::nav]') as $node) {
            $links = $xpath->query('.//a', $node);
            if ($links && $links->length >= 3) {
                $node->parentNode?->removeChild($node);
            }
        }

        $body = $xpath->query('//body')->item(0);
        $text = $body ? $body->textContent : $dom->textContent;

        libxml_clear_errors();
        libxml_use_internal_errors($prev);

        $text = html_entity_decode((string) $text, ENT_QUOTES | ENT_HTML5, get_bloginfo('charset'));
        $text = preg_replace('/\s+/', ' ', (string) $text);
        $text = trim((string) $text);

        $text = preg_replace('/\[[^\]]+\]/', ' ', $text);

        return trim(preg_replace('/\s+/', ' ', (string) $text));
    }

private function looks_like_human_text(string $s) : bool {
	$s = trim($s);
	if ($s === '') { return false; }

	$lower = strtolower($s);

	$noise = array(
		'default','sidebar','small','medium','large','yes','no','true','false',
		'px','em','rem','vh','vw','rgba','rgb','transparent','solid','none',
		'aegislink','aegisseo','wordpress','http','https'
	);
	if (in_array($lower, $noise, true)) { return false; }

	if (preg_match('#^https?://#i', $s)) { return false; }
	if (preg_match('/^#[0-9a-f]{3,8}$/i', $s)) { return false; }
	if (preg_match('/^[0-9a-f]{6,}$/i', $s)) { return false; } 
	if (preg_match('/^\{.*\}$|^\[.*\]$/s', $s)) { return false; }

	$letters = preg_match_all('/[a-z]/i', $s);
	if ($letters < 3) { return false; }

	return true;
}

private function clean_builder_text(string $s) : string {
	$s = html_entity_decode($s, ENT_QUOTES | ENT_HTML5, get_bloginfo('charset'));
	$s = wp_strip_all_tags($s, true);
	$s = preg_replace('/\s+/', ' ', (string) $s);
	return trim((string) $s);
}

    
private function color_palette() : array {
    return array(
        '#6B8DD6', // soft blue
        '#59B3C1', // soft teal
        '#8E7CC3', // soft purple
        '#D88AA5', // soft pink
        '#7CBFA0', // soft green
        '#D2A86E', // soft tan
        '#9AA6B2', // soft slate
    );
}

private function color_for_word(string $word) : string {
    $palette = $this->color_palette();
    $h = md5(strtolower($word));
    $n = hexdec(substr($h, 0, 6));
    $idx = $n % max(1, count($palette));
    return (string) ($palette[$idx] ?? '#6B8DD6');
}

private function ranking_keyword_sources(int $post_id) : array {
    if ($post_id <= 0) { return array(); }

    $sources = array();

    $meta_keys = array(
        'aegisseo_focus_phrase',
        '_aegisseo_focus_phrase',
        'aegisseo_focus_keyword',
        '_aegisseo_focus_keyword',
        'aegisseo_target_keywords',
        '_aegisseo_target_keywords',
    );

    foreach ($meta_keys as $k) {
        $v = get_post_meta($post_id, $k, true);
        if (is_string($v) && trim($v) !== '') {
            $sources[] = $v;
        } elseif (is_array($v) && !empty($v)) {
            $sources[] = implode(' ', array_map('strval', $v));
        }
    }

    $terms = get_the_terms($post_id, 'category');
    if (is_array($terms)) {
        foreach ($terms as $t) {
            if (!empty($t->name)) { $sources[] = (string) $t->name; }
        }
    }
    $terms = get_the_terms($post_id, 'post_tag');
    if (is_array($terms)) {
        foreach ($terms as $t) {
            if (!empty($t->name)) { $sources[] = (string) $t->name; }
        }
    }

    $gsc_keys = array(
        'aegisseo_gsc_queries',
        '_aegisseo_gsc_queries',
        'aegisseo_top_queries',
        '_aegisseo_top_queries',
    );
    foreach ($gsc_keys as $k) {
        $v = get_post_meta($post_id, $k, true);
        if (is_string($v) && $v !== '') {
            $json = json_decode($v, true);
            if (is_array($json)) {
                foreach ($json as $row) {
                    if (is_string($row)) {
                        $sources[] = $row;
                    } elseif (is_array($row) && !empty($row['query'])) {
                        $sources[] = (string) $row['query'];
                    }
                }
            }
        }
    }

    $out = array();
    foreach ($sources as $s) {
        $s = trim(wp_strip_all_tags((string) $s));
        if ($s === '') { continue; }
        $out[] = $s;
    }

    return array_values(array_unique($out));
}

private function stopwords() : array {
        $list = array(
            'the','and','for','with','that','this','from','are','was','were','have','has','had','you','your','our','their',
            'but','not','can','will','just','they','them','then','than','there','here','when','what','where','which','who',
            'how','why','into','onto','over','under','about','after','before','again','also','more','most','some','any',
            'all','each','only','such','these','those','its','it','as','at','by','of','to','in','on','or','is','be','we',
            'i','me','my','us','a','an'
        );
        $stop = array();
        foreach ($list as $w) { $stop[$w] = true; }
        return $stop;
    }
}