<?php
namespace AegisLink\Shortlinks;

if (!defined('ABSPATH')) { exit; }

class Shortlinks {

    const CPT = 'aegislink_link';
    const OPT_SETTINGS = 'aegislink_shortlinks_settings';
    const QV = 'aegislink_go';

    private $settings = array();

    public function __construct() {
        $this->settings = $this->get_settings();

        add_action('init', array($this, 'register_cpt'), 0);
        add_action('init', array($this, 'register_taxonomies'), 0);
        add_action('init', array($this, 'register_rewrite'), 1);
        add_filter('query_vars', array($this, 'register_query_var'));
        add_action('template_redirect', array($this, 'maybe_redirect'), 0);
        add_action('aegislink_shortlinks_health_check', array($this, 'run_health_check'));

        if (is_admin()) {
            add_action('admin_init', array($this, 'maybe_flush_rewrite_once'));
        }
    }

    public static function activate() : void {
        self::maybe_create_clicks_table();
        self::ensure_default_settings();
        self::unschedule_health_check();
        self::maybe_schedule_health_check();
        flush_rewrite_rules(false);
    }

    public static function deactivate() : void {
                self::unschedule_health_check();
flush_rewrite_rules(false);
    }

    public function register_cpt() : void {
        $labels = array(
            'name' => __('Short Links', 'aegislink'),
            'singular_name' => __('Short Link', 'aegislink'),
        );

        register_post_type(self::CPT, array(
            'labels' => $labels,
            'public' => false,
            'show_ui' => false, 
            'show_in_menu' => false,
            'supports' => array('title'),
            'capability_type' => 'post',
            'map_meta_cap' => true,
        ));
    }


    public function register_taxonomies() : void {

        register_taxonomy('aegislink_link_group', self::CPT, array(
            'label' => __('Groups', 'aegislink'),
            'public' => false,
            'show_ui' => false, 
            'show_in_nav_menus' => false,
            'show_admin_column' => false,
            'hierarchical' => true,
            'rewrite' => false,
        ));

        register_taxonomy('aegislink_link_tag', self::CPT, array(
            'label' => __('Tags', 'aegislink'),
            'public' => false,
            'show_ui' => false, 
            'show_in_nav_menus' => false,
            'show_admin_column' => false,
            'hierarchical' => false,
            'rewrite' => false,
        ));
    }

    public function register_rewrite() : void {
        $prefix = $this->get_prefix();
        add_rewrite_rule('^' . preg_quote($prefix, '/') . '/([^/]+)/?$', 'index.php?' . self::QV . '=$matches[1]', 'top');
    }

    public function register_query_var($vars) {
        $vars[] = self::QV;
        return $vars;
    }

    private function get_prefix() : string {
        $s = $this->settings;
        $prefix = isset($s['prefix']) ? (string) $s['prefix'] : 'go';
        $prefix = trim($prefix);
        $prefix = preg_replace('/[^a-zA-Z0-9\-]/', '', $prefix);
        if ($prefix === '') { $prefix = 'go'; }
        return strtolower($prefix);
    }

    public function get_base_url() : string {
        return trailingslashit(home_url('/' . $this->get_prefix()));
    }

    public function get_settings() : array {
        $defaults = array(
            'prefix' => 'go',
            'default_redirect' => 302,
            'track_clicks' => 1,
            'default_rel_nofollow' => 1,
            'default_rel_sponsored' => 0,
            'qr_provider' => 'qrserver',
        );
        $opt = get_option(self::OPT_SETTINGS, array());
        if (!is_array($opt)) { $opt = array(); }
        return array_merge($defaults, $opt);
    }

    public static function ensure_default_settings() : void {
        $defaults = array(
            'prefix' => 'go',
            'default_redirect' => 302,
            'track_clicks' => 1,
            'default_rel_nofollow' => 1,
            'default_rel_sponsored' => 0,
            'qr_provider' => 'qrserver',
        );
        $opt = get_option(self::OPT_SETTINGS, null);
        if (!is_array($opt)) {
            update_option(self::OPT_SETTINGS, $defaults, false);
            return;
        }
        $merged = array_merge($defaults, $opt);
        if ($merged !== $opt) {
            update_option(self::OPT_SETTINGS, $merged, false);
        }
    }

    public function maybe_flush_rewrite_once() : void {
        $flag = 'aegislink_shortlinks_rewrite_flushed';
        if (get_option($flag, 0)) { return; }
        update_option($flag, 1, false);
        flush_rewrite_rules(false);
    }

    public static function maybe_create_clicks_table() : void {
        global $wpdb;
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $table = $wpdb->prefix . 'aegislink_clicks';
        $charset = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            link_id BIGINT UNSIGNED NOT NULL,
            created_at DATETIME NOT NULL,
            referer TEXT NULL,
            ua TEXT NULL,
            ip_hash VARCHAR(64) NULL,
            PRIMARY KEY  (id),
            KEY link_id (link_id),
            KEY created_at (created_at)
        ) {$charset};";

        dbDelta($sql);
    }

    public function maybe_redirect() : void {
        $slug = get_query_var(self::QV);
        if (!$slug) { return; }

        $slug = sanitize_title((string) $slug);
        if ($slug === '') {
            status_header(404);
            nocache_headers();
            exit;
        }

        $post = get_page_by_path($slug, OBJECT, self::CPT);
        if (!$post || $post->post_status !== 'publish') {
            status_header(404);
            nocache_headers();
            exit;
        }

        $target = (string) get_post_meta($post->ID, '_aegislink_target', true);
        $target = trim($target);

        if ($target === '' || !preg_match('#^https?://#i', $target)) {
            status_header(404);
            nocache_headers();
            exit;
        }

        $code = (int) get_post_meta($post->ID, '_aegislink_redirect', true);
        if (!in_array($code, array(301,302,307), true)) {
            $code = (int) $this->settings['default_redirect'];
            if (!in_array($code, array(301,302,307), true)) { $code = 302; }
        }

        $track = (int) get_post_meta($post->ID, '_aegislink_track', true);
        if ($track === 0 && !empty($this->settings['track_clicks'])) {
            $track = (int) $this->settings['track_clicks'];
        }

        if ($track) {
            $this->record_click($post->ID);
        }

        $target = $this->apply_utm($post->ID, $target);

        nocache_headers();
        header('X-Robots-Tag: noindex, nofollow', true);
        wp_redirect($target, $code);
        exit;
    }


    private function apply_utm(int $link_id, string $target) : string {
        $enable = (int) get_post_meta($link_id, '_aegislink_utm_enable', true);
        if (!$enable) { return $target; }

        $params = array();
        $fields = array(
            'utm_source'   => '_aegislink_utm_source',
            'utm_medium'   => '_aegislink_utm_medium',
            'utm_campaign' => '_aegislink_utm_campaign',
            'utm_term'     => '_aegislink_utm_term',
            'utm_content'  => '_aegislink_utm_content',
        );
        foreach ($fields as $k => $meta_key) {
            $v = (string) get_post_meta($link_id, $meta_key, true);
            $v = trim($v);
            if ($v !== '') { $params[$k] = $v; }
        }
        if (empty($params)) { return $target; }

        return add_query_arg($params, $target);
    }

    public function run_health_check() : void {
        if (!function_exists('wp_remote_head')) { return; }

        $batch = (int) apply_filters('aegislink_shortlinks_health_batch', 25);
        if ($batch < 1) { $batch = 25; }
        $q = new \WP_Query(array(
            'post_type' => self::CPT,
            'post_status' => 'publish',
            'posts_per_page' => $batch,
            'orderby' => 'modified',
            'order' => 'ASC',
            'meta_query' => array(
                'relation' => 'OR',
                array(
                    'key' => '_aegislink_health_checked',
                    'compare' => 'NOT EXISTS',
                ),
                array(
                    'key' => '_aegislink_health_checked',
                    'value' => gmdate('Y-m-d H:i:s', time() - 7*DAY_IN_SECONDS),
                    'compare' => '<',
                    'type' => 'DATETIME',
                ),
            ),
            'no_found_rows' => true,
        ));

        if (empty($q->posts)) { return; }

        foreach ($q->posts as $p) {
            $target = (string) get_post_meta($p->ID, '_aegislink_target', true);
            $target = trim($target);
            if ($target === '' || !preg_match('#^https?://#i', $target)) {
                update_post_meta($p->ID, '_aegislink_health_ok', 0);
                update_post_meta($p->ID, '_aegislink_health_code', 0);
                update_post_meta($p->ID, '_aegislink_health_checked', gmdate('Y-m-d H:i:s'));

            $checked++;
            if ($ok) { $ok_count++; } else { $broken_count++; }
                continue;
            }

            $final = $this->apply_utm($p->ID, $target);

            $args = array(
                'timeout' => 8,
                'redirection' => 3,
                'user-agent' => 'AegisLink/' . (defined('AEGISLINK_VERSION') ? AEGISLINK_VERSION : 'unknown') . ' (+WordPress)',
            );
            $res = wp_remote_head($final, $args);
            if (is_wp_error($res)) {
                $res = wp_remote_get($final, $args);
            }

            $code = 0;
            $ok = 0;
            if (!is_wp_error($res)) {
                $code = (int) wp_remote_retrieve_response_code($res);
                if ($code >= 200 && $code < 400) { $ok = 1; }
            }

            update_post_meta($p->ID, '_aegislink_health_ok', $ok);
            update_post_meta($p->ID, '_aegislink_health_code', $code);
            update_post_meta($p->ID, '_aegislink_health_checked', gmdate('Y-m-d H:i:s'));

            $checked++;
            if ($ok) { $ok_count++; } else { $broken_count++; }
        }
    

        // Record last run summary for admin UI
        update_option('aegislink_health_last_run_gmt', $last_run_gmt, false);
        update_option('aegislink_health_last_run_checked', (int) $checked, false);
        update_option('aegislink_health_last_run_ok', (int) $ok_count, false);
        update_option('aegislink_health_last_run_broken', (int) $broken_count, false);

    }

    private function record_click(int $link_id) : void {
        global $wpdb;
        $table = $wpdb->prefix . 'aegislink_clicks';

        $referer = isset($_SERVER['HTTP_REFERER']) ? (string) wp_unslash($_SERVER['HTTP_REFERER']) : '';
        $ua      = isset($_SERVER['HTTP_USER_AGENT']) ? (string) wp_unslash($_SERVER['HTTP_USER_AGENT']) : '';
        $ip      = isset($_SERVER['REMOTE_ADDR']) ? (string) wp_unslash($_SERVER['REMOTE_ADDR']) : '';

        $salt = wp_salt('aegislink');
        $ip_hash = $ip !== '' ? hash('sha256', $salt . '|' . $ip) : null;

        if (strlen($referer) > 2000) { $referer = substr($referer, 0, 2000); }
        if (strlen($ua) > 255) { $ua = substr($ua, 0, 255); }

        $wpdb->insert(
            $table,
            array(
                'link_id' => (int) $link_id,
                'created_at' => current_time('mysql'),
                'referer' => $referer,
                'ua' => $ua,
                'ip_hash' => $ip_hash,
            ),
            array('%d','%s','%s','%s','%s')
        );
    }

    public function generate_slug(string $title = '') : string {
        $title = trim($title);
        $base = $title !== '' ? sanitize_title($title) : '';
        if ($base === '') {
            $base = strtolower(wp_generate_password(6, false, false));
        }

        $slug = $base;
        $i = 1;
        while ($this->slug_exists($slug)) {
            $i++;
            $slug = $base . '-' . $i;
            if ($i > 50) {
                $slug = $base . '-' . strtolower(wp_generate_password(4, false, false));
                break;
            }
        }
        return $slug;
    }

	public static function maybe_schedule_health_check() {
		$hook = 'aegislink_shortlinks_health_check';

		if (!wp_next_scheduled($hook)) {
			wp_schedule_event(time() + 300, 'daily', $hook);
		}
	}

	public static function unschedule_health_check() {
		$hook = 'aegislink_shortlinks_health_check';

		$ts = wp_next_scheduled($hook);
		while ($ts) {
			wp_unschedule_event($ts, $hook);
			$ts = wp_next_scheduled($hook);
		}
	}

    private function slug_exists(string $slug) : bool {
        $slug = sanitize_title($slug);
        if ($slug === '') { return true; }
        $existing = get_page_by_path($slug, OBJECT, self::CPT);
        return !empty($existing);
    }

    public function get_click_count(int $link_id) : int {
        global $wpdb;
        $table = $wpdb->prefix . 'aegislink_clicks';
        $n = (int) $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$table} WHERE link_id=%d", $link_id));
        return max(0, $n);
    }
}
