<?php
namespace AegisLink\Utils;

if (!defined('ABSPATH')) { exit; }

class License {

    const OPTION_KEY = 'aegislink_license_key';
    const TRANSIENT_STATUS = 'aegislink_license_status';
	const OPTION_FREE_OPTIN  = 'aegislink_free_reg_optin';
    const OPTION_FREE_STATUS = 'aegislink_free_reg_status'; // unregistered|registered|error
    const OPTION_LAST_DAILY_CHECK = 'aegislink_clm_last_daily_check';

    public function get_key() : string {
        return (string) get_option(self::OPTION_KEY, '');
    }

    public function set_key(string $key) : void {
        update_option(self::OPTION_KEY, trim($key), false);
        delete_transient(self::TRANSIENT_STATUS);
    }

    public function get_cached_status() : array {
        $s = get_transient(self::TRANSIENT_STATUS);
        return is_array($s) ? $s : array();
    }

    public function get_status() : array {
        return $this->get_cached_status();
    }

    public function is_pro() : bool {
        $s = $this->get_cached_status();
        if (isset($s['is_pro'])) {
            return (bool) $s['is_pro'];
        }
        return false;
    }

    public function check_now(string $current_version) : array {
        $site_url = home_url();
        $domain   = wp_parse_url($site_url, PHP_URL_HOST);

        $license_key = $this->get_key();

        $payload = array(
            'slug'            => defined('AEGISLINK_SLUG') ? AEGISLINK_SLUG : 'aegislink',
            'product'         => defined('AEGISLINK_SLUG') ? AEGISLINK_SLUG : 'aegislink',
            'version'         => (string) $current_version,
            'domain'          => (string) $domain,
            'site_url'        => (string) $site_url,
            'license_key'     => (string) $license_key,
			'email' 		  => (string) get_option('aegislink_license_email', ''),
            'wp_version'      => (string) get_bloginfo('version'),
            'php_version'     => (string) PHP_VERSION,
            'plugin_basename' => defined('AEGISLINK_BASENAME') ? (string) AEGISLINK_BASENAME : '',
        );

        $args = array(
            'timeout' => 20,
            'headers' => array( 'Content-Type' => 'application/json' ),
            'body'    => wp_json_encode($payload),
        );

		$endpoint = defined('AEGISLINK_CLM_STATUS_ENDPOINT') ? AEGISLINK_CLM_STATUS_ENDPOINT : '';
		if (!$endpoint && defined('AEGISLINK_CLM_UPDATE_ENDPOINT')) {
			$endpoint = AEGISLINK_CLM_UPDATE_ENDPOINT;
		}

        $resp = wp_remote_post($endpoint, $args);
        if (is_wp_error($resp)) {
            return array('success'=>false,'status'=>'wp_error','message'=>$resp->get_error_message());
        }

        $code = (int) wp_remote_retrieve_response_code($resp);
        $body = (string) wp_remote_retrieve_body($resp);

        if ($code < 200 || $code >= 300) {
            return array('success'=>false,'status'=>'http','message'=>'CLM HTTP '.$code,'raw'=>$body);
        }

        $data = json_decode($body, true);
        if (!is_array($data)) {
            return array('success'=>false,'status'=>'bad_json','message'=>'CLM returned invalid JSON','raw'=>$body);
        }

        $is_pro = false;
        $candidates_true = array(
            $data['is_pro'] ?? null,
            $data['pro'] ?? null,
            $data['license_active'] ?? null,
            $data['active'] ?? null,
        );
        foreach ($candidates_true as $v) {
            if ($v === true || $v === 1 || $v === '1' || $v === 'true') { $is_pro = true; break; }
        }
        if (!$is_pro) {
            $status = strtolower((string)($data['license_status'] ?? $data['status'] ?? ''));
            $plan   = strtolower((string)($data['plan'] ?? $data['tier'] ?? $data['license_type'] ?? ''));
            if (in_array($status, array('active','valid','licensed'), true) && in_array($plan, array('pro','premium','paid'), true)) {
                $is_pro = true;
            }
            if (in_array($plan, array('pro','premium','paid'), true) && $status === '') {
                $is_pro = true;
            }
        }

        $status_row = array(
            'success'     => !empty($data['success']) ? true : true, // some CLM builds omit success for license checks
            'checked_at'  => time(),
            'is_pro'      => (bool) $is_pro,
            'license_key' => (string) $license_key,
            'status'      => (string)($data['license_status'] ?? $data['status'] ?? ''),
            'plan'        => (string)($data['plan'] ?? $data['tier'] ?? $data['license_type'] ?? ''),
            'expires'     => (string)($data['expires'] ?? $data['expires_at'] ?? ''),
            'raw'         => $data,
        );

        set_transient(self::TRANSIENT_STATUS, $status_row, DAY_IN_SECONDS);

        return $status_row;
    }
		private function clm_post(string $endpoint, array $payload, int $timeout = 20) : array {
			if ($endpoint === '') {
				return array('success' => false, 'status' => 'no_endpoint', 'message' => 'Missing CLM endpoint.');
			}

			$resp = wp_remote_post($endpoint, array(
				'timeout' => $timeout,
				'headers' => array('Content-Type' => 'application/json'),
				'body'    => wp_json_encode($payload),
			));

			if (is_wp_error($resp)) {
				aegislink_update_log('CLM request WP_Error.', array(
					'endpoint' => $endpoint,
					'error' => $resp->get_error_message(),
				), 'ERROR');

				return array('success'=>false,'status'=>'wp_error','message'=>$resp->get_error_message());
			}

			$code = (int) wp_remote_retrieve_response_code($resp);
			$raw  = (string) wp_remote_retrieve_body($resp);
			$data = json_decode($raw, true);

			if (!is_array($data)) {
				aegislink_update_log('CLM invalid JSON.', array(
					'endpoint' => $endpoint,
					'http_code' => $code,
					'raw' => $raw,
				), 'ERROR');

				return array('success'=>false,'status'=>'bad_json','message'=>'Invalid JSON from CLM','http_code'=>$code,'raw'=>$raw);
			}

			if (!isset($data['success'])) {
				$data['success'] = ($code >= 200 && $code < 300);
			}

			$data['http_code'] = $code;
			return $data;
		}
		
		public function activate_now(string $current_version) : array {
			$site_url = home_url();
			$domain   = wp_parse_url($site_url, PHP_URL_HOST);

			$payload = array(
				'slug'        => defined('AEGISLINK_SLUG') ? AEGISLINK_SLUG : 'aegislink',
				'product'     => defined('AEGISLINK_SLUG') ? AEGISLINK_SLUG : 'aegislink',
				'version'     => (string) $current_version,
				'domain'      => (string) $domain,
				'site_url'    => (string) $site_url,
				'license_key' => (string) $this->get_key(),
				'email'       => (string) get_option('aegislink_license_email', get_option('admin_email', '')),
			);

            $email = (string) ($payload['email'] ?? '');
            $key   = (string) ($payload['license_key'] ?? '');
            if ($key === '' || $email === '' || !is_email($email)) {
                return array('success'=>false,'status'=>'invalid_input','message'=>'License key and a valid email are required.');
            }

			$data = $this->clm_post(defined('AEGISLINK_CLM_ACTIVATE_ENDPOINT') ? AEGISLINK_CLM_ACTIVATE_ENDPOINT : '', $payload, 20);

			aegislink_update_log('CLM activate response.', array(
				'http_code' => $data['http_code'] ?? null,
				'success' => !empty($data['success']),
				'data' => $data,
			), !empty($data['success']) ? 'INFO' : 'ERROR');

			// Refresh cached status after activation attempt
			$this->check_now($current_version);

            // IMPORTANT: Email must match the license on CLM. If CLM reports not active, do NOT treat as activated.
            $st = $this->get_cached_status();
            $st_state = strtolower((string)($st['status'] ?? ''));
            $st_plan  = strtolower((string)($st['plan'] ?? ''));
            $active_after = (!empty($st['is_pro']) || in_array($st_state, array('active','valid','licensed'), true));
            if (!$active_after) {
                delete_transient(self::TRANSIENT_STATUS);
                $data['success'] = false;
                $data['status']  = 'email_or_license_mismatch';
                $data['message'] = $data['message'] ?? 'License activation failed: email and license key must match.';
            }

			return $data;
		}

		public function deactivate_now(string $current_version) : array {
			$site_url = home_url();
			$domain   = wp_parse_url($site_url, PHP_URL_HOST);

			$payload = array(
				'slug'        => defined('AEGISLINK_SLUG') ? AEGISLINK_SLUG : 'aegislink',
				'product'     => defined('AEGISLINK_SLUG') ? AEGISLINK_SLUG : 'aegislink',
				'version'     => (string) $current_version,
				'domain'      => (string) $domain,
				'site_url'    => (string) $site_url,
				'license_key' => (string) $this->get_key(),
				'email'       => (string) get_option('aegislink_license_email', get_option('admin_email', '')),
			);

			$data = $this->clm_post(defined('AEGISLINK_CLM_DEACTIVATE_ENDPOINT') ? AEGISLINK_CLM_DEACTIVATE_ENDPOINT : '', $payload, 20);

			aegislink_update_log('CLM deactivate response.', array(
				'http_code' => $data['http_code'] ?? null,
				'success' => !empty($data['success']),
				'data' => $data,
			), !empty($data['success']) ? 'INFO' : 'ERROR');

			delete_transient(self::TRANSIENT_STATUS);
			$this->check_now($current_version);

			return $data;
		}
	    public function maybe_send_free_register(string $current_version) : void {

        // Only send if user explicitly consented on License page
        $optin = (int) get_option(self::OPTION_FREE_OPTIN, 0);
        if ($optin !== 1) { return; }

		if (get_transient('aegislink_clm_free_reg_sent')) { return; }

		$site_url = home_url();
		$domain   = wp_parse_url($site_url, PHP_URL_HOST);

		$email = (string) get_option('aegislink_license_email', '');
		if (!$email) {
			$email = (string) get_option('admin_email', '');
		}

			$payload = array(
				'product'     => defined('AEGISLINK_SLUG') ? AEGISLINK_SLUG : 'aegislink',
				'email'       => (string) $email,
				'site_url'    => (string) $site_url,
				'domain'      => (string) ($domain ? $domain : ''),
				'version'     => (string) $current_version,
				'wp_version'  => (string) get_bloginfo('version'),
				'php_version' => (string) PHP_VERSION,
			);

		$endpoint = defined('AEGISLINK_CLM_FREE_REG_ENDPOINT') ? AEGISLINK_CLM_FREE_REG_ENDPOINT : '';
		if (!$endpoint) { return; }

		$args = array(
			'timeout' => 15,
			'headers' => array('Content-Type' => 'application/json'),
			'body'    => wp_json_encode($payload),
		);

		$resp = wp_remote_post($endpoint, $args);

        if (is_wp_error($resp)) {
            update_option(self::OPTION_FREE_STATUS, 'error', false);

            aegislink_update_log('CLM free-register failed (WP_Error).', array(
                'endpoint' => $endpoint,
                'error'    => $resp->get_error_message(),
            ), 'ERROR');

            return;
        }

		$code = (int) wp_remote_retrieve_response_code($resp);
		$body = (string) wp_remote_retrieve_body($resp);

		// Backoff (avoid spamming)
		$backoff = ($code >= 500) ? HOUR_IN_SECONDS : DAY_IN_SECONDS;
		set_transient('aegislink_clm_free_reg_sent', 1, $backoff);

		// Try to read JSON response (CLM normally returns JSON)
		$data = json_decode($body, true);
		$success = null;

		if (is_array($data) && array_key_exists('success', $data)) {
			$success = (bool) $data['success'];
		}

		// Prefer explicit JSON success when available; otherwise fall back to HTTP 2xx
		if ($success === true || ($success === null && $code >= 200 && $code < 300)) {
			update_option(self::OPTION_FREE_STATUS, 'registered', false);

			aegislink_update_log('CLM free-register success.', array(
				'endpoint'  => $endpoint,
				'http_code' => $code,
				'data'      => is_array($data) ? $data : null,
				'body'      => $body,
			), 'INFO');

		} else {
			update_option(self::OPTION_FREE_STATUS, 'error', false);

			aegislink_update_log('CLM free-register failed.', array(
				'endpoint'  => $endpoint,
				'http_code' => $code,
				'data'      => is_array($data) ? $data : null,
				'body'      => $body,
			), 'ERROR');
		}

	}





    /**
     * Run a CLM status check at most once per day (for PRO gating + update eligibility).
     * Safe to call on every request.
     */
    public function maybe_daily_check(string $current_version) : void {
        $last = (int) get_option(self::OPTION_LAST_DAILY_CHECK, 0);
        // Also respect the status transient if it is still fresh.
        if ($last > 0 && (time() - $last) < DAY_IN_SECONDS) {
            return;
        }
        update_option(self::OPTION_LAST_DAILY_CHECK, time(), false);

        // Refresh license status.
        $this->check_now($current_version);

        // Free-install registration is OPTIONAL and separate from licensing.
        $this->maybe_send_free_register($current_version);
    }

	public function hook_update_check() : void {
		add_filter('pre_set_site_transient_update_plugins', function($transient) {
			if (!is_object($transient) || empty($transient->checked) || !is_array($transient->checked)) {
				return $transient;
			}
			if (empty($transient->checked[defined('AEGISLINK_BASENAME') ? AEGISLINK_BASENAME : ''])) {
				return $transient;
			}
			$this->check_now(defined('AEGISLINK_VERSION') ? AEGISLINK_VERSION : '');
			return $transient;
		});
	}

    public static function render_pro_banner(string $title = 'Upgrade to PRO', string $message = '') : void {
        if ($message === '') {
            $message = __('PRO feature: This section is available in AegisSEO PRO. Unlock automation, evidence reporting, and advanced guardrails.', 'aegislink');
        }
        ?>
        <div class="aegisseo-pro-banner">
            <div class="aegisseo-pro-banner__left">
                <div class="aegisseo-pro-badge"><?php echo esc_html($title); ?></div>
                <div class="aegisseo-pro-text"><?php echo esc_html($message); ?></div>
            </div>
            <div class="aegisseo-pro-banner__right">
                <a class="button button-primary" href="<?php echo esc_url(admin_url('admin.php?page=aegisseo&tab=license')); ?>">
                    <?php echo esc_html__('Activate License', 'aegislink'); ?>
                </a>
            </div>
        </div>
        <?php
    }
}
