<?php
/**
 * Plugin Name: AegisLink
 * Description: Smart internal linking + linking assistant by Aegisify. Create keyword→URL smart links and generate context-aware internal link suggestions.
 * Version: 1.1.5
 * Requires at least: 6.8
 * Requires PHP: 8.2
 * Author:      Aegisify | AegisWAF | AegisShield | AegisSEO
 * Author URI:  https://aegisify.com/
 * Text Domain: aegislink
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) { exit; }

define('AEGISLINK_VERSION', '1.1.5');
define('AEGISLINK_FILE', __FILE__);
define('AEGISLINK_DIR', plugin_dir_path(__FILE__));
define('AEGISLINK_URL', plugin_dir_url(__FILE__));
define('AEGISLINK_BASENAME', plugin_basename(__FILE__));
define('AEGISLINK_SLUG', 'aegislink');

// CLM (Chue License Manager) endpoints
define('AEGISLINK_CLM_BASE', 'https://aegisify.com');
define('AEGISLINK_CLM_FREE_REG_ENDPOINT',  AEGISLINK_CLM_BASE . '/wp-json/chue-license/v1/aegislink/free-register');
define('AEGISLINK_CLM_UPDATE_ENDPOINT',    AEGISLINK_CLM_BASE . '/wp-json/chue-license/v1/aegislink/update-check');
define('AEGISLINK_CLM_ACTIVATE_ENDPOINT',  AEGISLINK_CLM_BASE . '/wp-json/chue-license/v1/aegislink/activate');
define('AEGISLINK_CLM_DEACTIVATE_ENDPOINT',AEGISLINK_CLM_BASE . '/wp-json/chue-license/v1/aegislink/deactivate');
define('AEGISLINK_CLM_STATUS_ENDPOINT',    AEGISLINK_CLM_BASE . '/wp-json/chue-license/v1/aegislink/status');

require_once AEGISLINK_DIR . 'includes/class-aegislink-plugin.php';

function aegislink() : \AegisLink\Plugin {
    return \AegisLink\Plugin::instance();
}

register_activation_hook(__FILE__, function () {
    require_once AEGISLINK_DIR . 'includes/utils/class-aegislink-license.php';

    $lic = new \AegisLink\Utils\License();
    $lic->maybe_send_free_register(AEGISLINK_VERSION);
});

function aegislink_clm_check_for_updates($transient) {
    if (!is_object($transient) || empty($transient->checked) || !is_array($transient->checked)) {
        return $transient;
    }

    if (!defined('AEGISLINK_BASENAME') || empty($transient->checked[AEGISLINK_BASENAME])) {
        return $transient;
    }

    $cache_key = 'aegislink_clm_update_cache';
    $cached = get_site_transient($cache_key);

    if (is_object($cached) && !empty($cached->new_version)) {
        $installed = (string) $transient->checked[AEGISLINK_BASENAME];
        if ($installed !== '' && version_compare($installed, (string) $cached->new_version, '>=')) {
            delete_site_transient($cache_key);
        } else {
            $transient->response[AEGISLINK_BASENAME] = $cached;
            return $transient;
        }
    }

    $current_version = (string) $transient->checked[AEGISLINK_BASENAME];

    $payload = array(
        'product'     => defined('AEGISLINK_SLUG') ? AEGISLINK_SLUG : 'aegislink',
        'slug'        => defined('AEGISLINK_SLUG') ? AEGISLINK_SLUG : 'aegislink',
        'version'     => $current_version,
        'site_url'    => home_url(),
        'license_key' => (string) get_option('aegislink_license_key', ''),
        'email'       => (string) get_option('aegislink_license_email', get_option('admin_email', '')),
    );

    $resp = wp_remote_post(AEGISLINK_CLM_UPDATE_ENDPOINT, array(
        'timeout' => 20,
        'headers' => array('Content-Type' => 'application/json'),
        'body'    => wp_json_encode($payload),
    ));

    if (is_wp_error($resp)) {
        return $transient;
    }

    $code = (int) wp_remote_retrieve_response_code($resp);
    if ($code < 200 || $code >= 300) {
        return $transient;
    }

	$data = json_decode((string) wp_remote_retrieve_body($resp), true);
	if (!is_array($data)) {
		return $transient;
	}

	// CLM returns the fields at the top level (not nested under "data")
	$d = $data;

	// (Optional compatibility: if any older wrapper returns {"data": {...}}, accept it too)
	if (!empty($data['data']) && is_array($data['data'])) {
		$d = $data['data'];
	}

	if (empty($d['success']) || empty($d['update_available']) || empty($d['version']) || empty($d['download_url'])) {
		return $transient;
	}

    $update              = new stdClass();
    $update->slug        = AEGISLINK_SLUG;
    $update->plugin      = AEGISLINK_BASENAME;
    $update->new_version = (string) $d['version'];
    $update->package     = (string) $d['download_url'];
    $update->url         = !empty($d['homepage']) ? (string) $d['homepage'] : 'https://aegisify.com';

    $transient->response[AEGISLINK_BASENAME] = $update;

    // cache for stability (short TTL to avoid sticky notices)
    set_site_transient($cache_key, $update, 30 * MINUTE_IN_SECONDS);

    return $transient;
}
add_filter('pre_set_site_transient_update_plugins', 'aegislink_clm_check_for_updates', 25, 1);

// Clear cached update after plugin upgrade
add_action('upgrader_process_complete', function($upgrader, $hook_extra) {
    if (!empty($hook_extra['action']) && $hook_extra['action'] === 'update'
        && !empty($hook_extra['type']) && $hook_extra['type'] === 'plugin'
        && !empty($hook_extra['plugins']) && is_array($hook_extra['plugins'])
        && in_array(AEGISLINK_BASENAME, $hook_extra['plugins'], true)
    ) {
        delete_site_transient('aegislink_clm_update_cache');
    }
}, 10, 2);

add_action('plugins_loaded', function() {
    add_action('init', function() {
        load_plugin_textdomain('aegislink', false, dirname(AEGISLINK_BASENAME) . '/languages');
    });

    aegislink()->boot();
});
