<?php

if (!defined('ABSPATH')) { exit; }

if (!class_exists('Aegisify_Bundled_Plugins_Page')) {

class Aegisify_Bundled_Plugins_Page {

	private static $config = array(
		'parent_slug' => 'aegisseo',

		'menu_slug'   => 'aegisify-bundled-plugins',

		'page_title'  => 'Aegisify Plugins',
		'menu_title'  => 'Aegisify Plugins',

		'view_cap'    => 'read',

		'products' => array(
			'aegisshield' => array(
				'plugin_file' => 'aegisshield-security/aegisshield-security.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2026/01/aegisshield-security-7.1.7.zip',
				'title'       => 'AegisShield Free & PRO',
				'desc'        => 'Full WordPress security suite with Pro Upgrade: hardening, monitoring, malware checks, integrity tools, and protection layers built for real admins.',
				'image'       => '', // set your image URL if you want
			),
			'aegiswaf' => array(
				'plugin_file' => 'aegiswaf/aegiswaf.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2026/01/aegiswaf-1.8.10.zip',
				'title'       => 'AegisWAF Free & PRO',
				'desc'        => 'Web Application Firewall for WordPress, DDOS Layer7, Bot Control, API Shield and Logs Attack Story. Advanced features upgrades',
				'image'       => '',
			),
			'aegisseo' => array(
				'plugin_file' => 'aegisseo/aegisseo.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2026/01/AegisSEO-1.6.1.zip',
				'title'       => 'AegisSEO Free & PRO',
				'desc'        => 'SEO OPs Center, Global SEO Automation, Social Meta, Schema, Advanced SiteMAPS, Issues & Fixes, Evidence, Linking Assistance and FREE advanced features.',
				'image'       => '',
			),
			'aegisspam' => array(
				'plugin_file' => 'aegisspam/aegisspam.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2026/01/aegisspam-1.1.5.zip',
				'title'       => 'AegisSpam Free & PRO',
				'desc'        => 'Local-first anti-spam protection for WordPress comments, registrations, and Contact Forms with transparent scoring, logs, allow/deny lists, firewall, and local Machine learning.',
				'image'       => '', // set your image URL if you want
			),
			'aegisticket' => array(
				'plugin_file' => 'aegisticket/support-ticket-system.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2026/01/aegisticket-1.8.1.zip',
				'title'       => 'AegisTicket Free',
				'desc'        => 'Free Support ticketing plugin built for ops: CPT-based tickets, admin workflows, and customer-friendly submission + tracking. NOTE: Use at your own risks.',
				'image'       => '', // set your image URL if you want
			),
			'aegislink' => array(
				'plugin_file' => 'AegisLink/aegislink.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2026/01/AegisLink-1.1.5.zip',
				'title'       => 'AegisLink Free',
				'desc'        => 'Free Smart internal linking + linking assistant by Aegisify. Create keyword URL smart links and generate context-aware internal link suggestions.',
				'image'       => '', // set your image URL if you want
			),
			'aegissitemap' => array(
				'plugin_file' => 'aegissitemap/aegissitemap.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2026/01/aegissitemap-1.2.8.zip',
				'title'       => 'AegisSiteMap Free',
				'desc'        => 'Free XML sitemap engine by Aegisify: fast, cache-friendly sitemap index, HTML sitemap, images, and optional video/news sitemaps.',
				'image'       => '', // set your image URL if you want
			),
		),
	);
	
	public static function init(array $override = array()) : void {
		if (!is_admin()) { return; }

		$override_products = array();
		if (isset($override['products']) && is_array($override['products'])) {
			$override_products = $override['products'];
			unset($override['products']);
		}

		self::$config = array_merge(self::$config, $override);

		if (!empty($override_products)) {
			self::$config['products'] = array_merge((array) self::$config['products'], $override_products);
		}

		$products = (array) self::$config['products'];

		if (isset($products['aegisseo']) && is_array($products['aegisseo'])) {
			if (defined('AEGISSEO_BASENAME') && AEGISSEO_BASENAME) {
				$products['aegisseo']['plugin_file'] = AEGISSEO_BASENAME;
			}
		}

		self::$config['products'] = $products;

		add_action('admin_post_aegisify_products_install', array(__CLASS__, 'handle_install'));
	}

	private static function maybe_handle_inline_install() : void {
		if (empty($_GET['aegisify_install'])) {
			return;
		}

		if (!current_user_can('install_plugins')) {
			wp_die('You do not have permission to install plugins.');
		}

		$key = sanitize_key((string) $_GET['aegisify_install']);
		check_admin_referer('aegisify_inline_install_' . $key);

		$products = self::get_products();
		if (empty($products[$key])) {
			wp_die('Invalid product.');
		}

		$p = $products[$key];
		$plugin_file = !empty($p['plugin_file']) ? (string) $p['plugin_file'] : '';
		if ($plugin_file && self::is_installed($plugin_file)) {
			self::redirect_back('Plugin is already installed.', 'updated');
		}

		$download = !empty($p['zip_url']) ? esc_url_raw((string) $p['zip_url']) : '';
		if (!$download || !preg_match('#^https?://#i', $download)) {
			self::redirect_back('Install failed: Missing or invalid zip_url for this product.', 'error');
		}

		$check = self::validate_zip_url($download);
		if (empty($check['ok'])) {
			self::redirect_back('Install failed: ' . $check['error'], 'error');
		}

		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		$url = remove_query_arg(array('aegisify_notice','aegisify_type'), wp_unslash($_SERVER['REQUEST_URI']));
		$creds = request_filesystem_credentials($url);
		if ($creds === false) {
			exit;
		}
		if (!WP_Filesystem($creds)) {
			request_filesystem_credentials($url, '', true);
			exit;
		}

		$skin = new Automatic_Upgrader_Skin();
		$upgrader = new Plugin_Upgrader($skin);

		$result = $upgrader->install($download);

		if (is_wp_error($result)) {
			self::redirect_back('Install failed: ' . $result->get_error_message(), 'error');
		}

		if (!$result) {
			self::redirect_back('Install failed: WordPress could not install the package (permissions / invalid zip).', 'error');
		}

		self::redirect_back('Installed successfully.', 'updated');
	}

	public static function add_submenu() : void {
		$parent = (string) self::$config['parent_slug'];
		$slug   = (string) self::$config['menu_slug'];

		add_submenu_page(
			$parent,
			(string) self::$config['page_title'],
			(string) self::$config['menu_title'],
			(string) self::$config['view_cap'],
			$slug,
			array(__CLASS__, 'render_page')
		);
	}

	private static function get_products() : array {
		$products = (array) self::$config['products'];

		if (has_filter('aegisify_bundled_plugins_products')) {
			$products = (array) apply_filters('aegisify_bundled_plugins_products', $products, self::$config);
		}

		return $products;
	}
	
	private static function resolve_plugin_file(string $plugin_file): string {
		$plugin_file = ltrim($plugin_file, '/');
		if (!$plugin_file) { return ''; }

		// If exact path exists, use it.
		if (file_exists(WP_PLUGIN_DIR . '/' . $plugin_file)) {
			return $plugin_file;
		}

		// Try case-insensitive match against installed plugins list.
		if (!function_exists('get_plugins')) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$target = strtolower($plugin_file);
		foreach (array_keys(get_plugins()) as $pf) {
			if (strtolower($pf) === $target) {
				return $pf; // return the real-cased path, e.g. "AegisLink/aegislink.php"
			}
		}

		return $plugin_file;
	}

	private static function is_installed(string $plugin_file) : bool {
		$plugin_file = self::resolve_plugin_file($plugin_file);
		if (!$plugin_file) { return false; }
		return file_exists(WP_PLUGIN_DIR . '/' . $plugin_file);
	}
	
	private static function is_active_plugin(string $plugin_file) : bool {
		$plugin_file = self::resolve_plugin_file($plugin_file);
		if (!$plugin_file) { return false; }

		if (!function_exists('is_plugin_active')) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		return is_plugin_active($plugin_file);
	}

	private static function validate_zip_url(string $url): array {

		if (!preg_match('#^https?://#i', $url)) {
			return array(
				'ok'    => false,
				'error' => 'Zip URL is not a valid http(s) URL.',
			);
		}

		$head = wp_remote_head($url, array(
			'timeout'     => 15,
			'redirection' => 5,
		));

		if (is_wp_error($head)) {
			return array(
				'ok'    => false,
				'error' => 'Zip URL request failed: ' . $head->get_error_message(),
			);
		}

		$code = wp_remote_retrieve_response_code($head);
		if ($code !== 200) {
			return array(
				'ok'    => false,
				'error' => 'Zip URL returned HTTP ' . $code . ' (blocked or redirected).',
			);
		}

		$content_type = wp_remote_retrieve_header($head, 'content-type');
		if ($content_type && stripos($content_type, 'text/html') !== false) {
			return array(
				'ok'    => false,
				'error' => 'Zip URL returned HTML instead of a ZIP file.',
			);
		}

		$get = wp_remote_get($url, array(
			'timeout'     => 15,
			'redirection' => 5,
			'stream'      => true,
			'limit_response_size' => 2,
		));

		if (is_wp_error($get)) {
			return array(
				'ok'    => false,
				'error' => 'Zip URL download failed: ' . $get->get_error_message(),
			);
		}

		$body = wp_remote_retrieve_body($get);
		if (substr($body, 0, 2) !== 'PK') {
			return array(
				'ok'    => false,
				'error' => 'Downloaded file is not a valid ZIP archive.',
			);
		}

		return array(
			'ok' => true,
		);
	}

	private static function redirect_back(string $notice = '', string $type = 'updated') : void {
		$parent = (string) self::$config['parent_slug'];
		$slug   = (string) self::$config['menu_slug'];

		$url = admin_url('admin.php?page=' . rawurlencode($slug));

		if ($notice) {
			$url = add_query_arg(array(
				'aegisify_notice' => rawurlencode($notice),
				'aegisify_type'   => rawurlencode($type),
			), $url);
		}

		wp_safe_redirect($url);
		exit;
	}
	
	private static function get_wp_activate_url(string $plugin_file) : string {
		$plugin_file = ltrim($plugin_file, '/');

		$url = admin_url('plugins.php?action=activate&plugin=' . rawurlencode($plugin_file));

		return wp_nonce_url($url, 'activate-plugin_' . $plugin_file);
	}

	public static function handle_install() : void {
		if (!current_user_can('install_plugins')) {
			self::redirect_back('You do not have permission to install plugins.', 'error');
		}
		check_admin_referer('aegisify_products_install');

		$key = isset($_POST['product_key']) ? sanitize_key((string) $_POST['product_key']) : '';
		$products = self::get_products();

		if (!$key || !isset($products[$key])) {
			self::redirect_back('Invalid product.', 'error');
		}
		
		$p = $products[$key];

		$download = isset($p['zip_url']) ? esc_url_raw((string) $p['zip_url']) : '';
		if (!$download || !preg_match('#^https?://#i', $download)) {
			self::redirect_back('Install failed: Missing or invalid zip_url for this product.', 'error');
		}

		$check = self::validate_zip_url($download);
		if (empty($check['ok'])) {
			self::add_notice('Install failed: ' . $check['error'], 'error');
			return;
		}

		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/misc.php';
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		$skin = new WP_Ajax_Upgrader_Skin();

		$upgrader = new Plugin_Upgrader($skin);
		$result = $upgrader->install($download);

		if (is_wp_error($result)) {
			self::redirect_back('Install failed: ' . $result->get_error_message(), 'error');
		}

		if (!$result) {
			$err = (isset($skin->result) && is_wp_error($skin->result))
				? $skin->result->get_error_message()
				: 'Unknown installer failure.';
			self::redirect_back('Install failed: ' . $err, 'error');
		}

		self::redirect_back('Installed successfully.', 'updated');

		$p = $products[$key];
		$plugin_file = isset($p['plugin_file']) ? (string) $p['plugin_file'] : '';

		if ($plugin_file && self::is_installed($plugin_file)) {
			self::redirect_back('Plugin is already installed.', 'updated');
		}
		
	}
	
	public static function render_page() : void {
		self::maybe_handle_inline_install();
		if (!current_user_can((string) self::$config['view_cap'])) {
			wp_die(esc_html__('Sorry, you are not allowed to access this page.', 'aegisseo'));
		}

		$notice = isset($_GET['aegisify_notice']) ? (string) wp_unslash($_GET['aegisify_notice']) : '';
		$type   = isset($_GET['aegisify_type']) ? sanitize_key((string) $_GET['aegisify_type']) : '';

		$products = self::get_products();
		?>
		<div class="wrap">
			<h1 style="margin-bottom:8px;"><?php echo esc_html(self::$config['page_title']); ?></h1>
			<div class="aegisify-banner" style="margin: 12px 0 20px 0;">
				<img
					src="<?php echo esc_url( plugin_dir_url( __FILE__ ) . '/banner.png' ); ?>"
					alt="<?php echo esc_attr__( 'Aegisify Plugin Suite', 'aegisseo' ); ?>"
					style="
						width:70%;
						max-width:650px;
						align: center;
						height:auto;
						display:block;
						border-radius:8px;
						box-shadow:0 2px 6px rgba(0,0,0,0.08);
					"
				/>
			</div>			
			<p style="max-width: 980px;">
			    <h1 style="margin-bottom:8px;"><?php echo esc_html('Aegisify Products: Built for the Community. Free by Design. Powerful by Nature.'); ?></h1>
				<?php echo esc_html__('Welcome to Aegisify, and thank you for choosing our WordPress products. Every Aegisify plugin is carefully engineered, thoroughly tested, and built with a deep commitment to delivering real value to the WordPress community. We believe powerful tools should be accessible to everyone, which is why each product includes meaningful, high-impact features that are free forever!  No forced upgrades, no hidden restrictions, and no pressure to go Pro. You’ll continue to receive updates, improvements, and critical bug fixes whether you upgrade or not. Pro features are available for those who want advanced capabilities, but the choice is always yours. We’re grateful for your trust and proud to support your WordPress site with tools designed to protect, optimize, and grow your business today and into the future.', 'aegisseo'); ?>
				<h3 style="margin-bottom:8px;"><?php echo esc_html('Thank you for being part of the Aegisify community.'); ?></h3></p>
				<div style="margin-bottom: 8px; font-size: 14px;">
					<strong><?php echo esc_html__('Aegisify:', 'aegisseo'); ?></strong>
					<a href="https://aegisify.com" target="_blank" rel="noopener noreferrer">https://aegisify.com</a>
				</div>

				<?php
					$quicklinks = array(
						array(
							'label' => __('Register Account', 'aegisseo'),
							'url'   => 'https://aegisify.com/register/',
							'icon'  => 'dashicons-id',
						),
						array(
							'label' => __('Buy Now', 'aegisseo'),
							'url'   => 'https://aegisify.com/profile/',
							'icon'  => 'dashicons-cart',
						),
						array(
							'label' => __('Open a Ticket', 'aegisseo'),
							'url'   => 'https://aegisify.com/profile/',
							'icon'  => 'dashicons-sos',
						),
						array(
							'label' => __('Report a Bug', 'aegisseo'),
							'url'   => 'https://aegisify.com/report-a-bug/',
							'icon'  => 'dashicons-dismiss',
						),
						array(
							'label' => __('Donate', 'aegisseo'),
							'url'   => 'https://www.paypal.com/ncp/payment/8ZU8J9GTCALCU',
							'icon'  => 'dashicons-heart',
						),
					);
				?>

				<div style="display:flex; flex-wrap:wrap; gap:10px;">
					<?php foreach ($quicklinks as $q): ?>
						<a
							href="<?php echo esc_url($q['url']); ?>"
							target="_blank"
							rel="noopener noreferrer"
							style="
								display:inline-flex;
								align-items:center;
								gap:6px;
								padding:8px 10px;
								border:1px solid #dcdcde;
								border-radius:8px;
								background:#fff;
								text-decoration:none;
								font-size:13px;
							"
						>
							<span class="dashicons <?php echo esc_attr($q['icon']); ?>" style="font-size:18px; width:18px; height:18px;"></span>
							<span><?php echo esc_html($q['label']); ?></span>
						</a>
					<?php endforeach; ?>
				</div>

			</p>

			<?php if ($notice): ?>
				<div class="<?php echo esc_attr(($type === 'error') ? 'notice notice-error' : 'notice notice-success'); ?> is-dismissible">
					<p><?php echo esc_html($notice); ?></p>
				</div>
			<?php endif; ?>

			<div class="notice notice-info" style="max-width:980px;">
				<p>
					<?php echo esc_html__('Tip: WordPress may request filesystem credentials (FTP) depending on server permissions. That is normal for plugin installs on some hosts.', 'aegisseo'); ?>
				</p>
			</div>

			<style>
				.aegisify-grid{
					display:grid;
					grid-template-columns: repeat(4, minmax(0, 1fr));
					gap: 16px;
					max-width: 1200px;
				}
				@media (max-width: 1200px){
					.aegisify-grid{ grid-template-columns: repeat(3, minmax(0,1fr)); }
				}
				@media (max-width: 900px){
					.aegisify-grid{ grid-template-columns: repeat(2, minmax(0,1fr)); }
				}
				@media (max-width: 520px){
					.aegisify-grid{ grid-template-columns: 1fr; }
				}
				.aegisify-card{
					background:#fff;
					border:1px solid #e5e5e5;
					border-radius:8px;
					overflow:hidden;
					box-shadow: 0 1px 2px rgba(0,0,0,0.04);
					display:flex;
					flex-direction:column;
					min-height: 280px;
				}
				.aegisify-card__top{
					position:relative;
					padding: 18px 18px 10px;
					border-bottom: 1px solid #f0f0f0;
					display:flex;
					align-items:center;
					justify-content:center;
					min-height: 120px;
				}
				.aegisify-card__badge{
					position:absolute;
					top:10px;
					left:10px;
					background:#2271b1;
					color:#fff;
					font-size: 11px;
					padding: 4px 8px;
					border-radius: 4px;
					letter-spacing: .3px;
					text-transform: uppercase;
				}
				.aegisify-card__img{
					max-width: 78%;
					max-height: 78px;
					width:auto;
					height:auto;
					object-fit:contain;
				}
				.aegisify-card__body{
					padding: 14px 16px 0;
					flex:1;
				}
				.aegisify-card__title{
					margin: 0 0 8px;
					font-size: 14px;
					font-weight: 700;
				}
				.aegisify-card__desc{
					margin: 0 0 14px;
					color:#4b5563;
					font-size: 12.5px;
					line-height: 1.45;
					min-height: 54px;
				}
				.aegisify-card__footer{
					padding: 12px 16px 16px;
					display:flex;
					align-items:center;
					justify-content:space-between;
					gap: 10px;
				}
				.aegisify-status{
					font-size: 12px;
					color:#111827;
				}
				.aegisify-status strong{ font-weight:700; }
				.aegisify-actions form{ display:inline; }
			</style>

			<div class="aegisify-grid">
				<?php foreach ($products as $k => $p): ?>
					<?php
						$k = (string) $k;
						$title = isset($p['title']) ? (string) $p['title'] : $k;
						$desc  = isset($p['desc']) ? (string) $p['desc'] : '';
						$img   = isset($p['image']) ? (string) $p['image'] : '';
						$plugin_file = isset($p['plugin_file']) ? (string) $p['plugin_file'] : '';

						$installed = ($plugin_file ? self::is_installed($plugin_file) : false);
						$active = ($installed && $plugin_file) ? self::is_active_plugin($plugin_file) : false;
					?>
					<div class="aegisify-card">
						<div class="aegisify-card__top">
							<div class="aegisify-card__badge"><?php echo esc_html($installed ? 'Installed' : 'Available'); ?></div>
							<?php if ($img): ?>
								<img class="aegisify-card__img" src="<?php echo esc_url($img); ?>" alt="<?php echo esc_attr($title); ?>">
							<?php else: ?>
								<div style="font-weight:800;font-size:18px;opacity:.25;"><?php echo esc_html($title); ?></div>
							<?php endif; ?>
						</div>

						<div class="aegisify-card__body">
							<div class="aegisify-card__title"><?php echo esc_html($title); ?></div>
							<div class="aegisify-card__desc"><?php echo esc_html($desc); ?></div>
						</div>

						<div class="aegisify-card__footer">
							<div class="aegisify-status">
								<strong><?php echo esc_html($installed ? 'Status:' : 'Status:'); ?></strong>
								<?php echo esc_html(!$installed ? ' Not Installed' : ($active ? ' Active' : ' Installed')); ?>
							</div>

							<div class="aegisify-actions">
								<?php if (!$installed): ?>
									<?php
										$install_url = add_query_arg(
											array('aegisify_install' => $k),
											menu_page_url((string) self::$config['menu_slug'], false)
										);
										$install_url = wp_nonce_url($install_url, 'aegisify_inline_install_' . $k);
									?>
									<a href="<?php echo esc_url($install_url); ?>" class="button button-primary">
										<?php echo esc_html__('Install Plugin', 'aegisseo'); ?>
									</a>

								<?php elseif (!$active): ?>
									<?php $activate_url = self::get_wp_activate_url($plugin_file); ?>
									<a href="<?php echo esc_url($activate_url); ?>" class="button button-secondary">
										<?php echo esc_html__('Activate Plugin', 'aegisseo'); ?>
									</a>

								<?php else: ?>
									<a href="<?php echo esc_url(admin_url('plugins.php')); ?>" class="button">
										<?php echo esc_html__('Go To Plugins', 'aegisseo'); ?>
									</a>
								<?php endif; ?>
							</div>
						</div>
					</div>
				<?php endforeach; ?>
			</div>

		</div>
		<?php
	}
}

} 
