<?php
namespace AegisLink\SEO;

if (!defined('ABSPATH')) { exit; }

class Linking {
	
    public function __construct() {
        if (is_admin()) { return; }

        add_filter('the_content', array($this, 'apply_smart_links'), 99);
        add_filter('the_excerpt', array($this, 'apply_smart_links'), 99);
        add_filter('widget_text_content', array($this, 'apply_smart_links'), 99);

        // Elementor: apply on rendered widget content
        add_filter('elementor/widget/render_content', array($this, 'apply_smart_links'), 99, 1);
        // Some Elementor themes run this filter
        add_filter('elementor/frontend/the_content', array($this, 'apply_smart_links'), 99, 1);
    }
	
    public function suggest_links($post_id, $limit = 8) {
        $post = get_post($post_id);
        if (!$post) return array();
        $keywords = $this->extract_keywords($post_id);
        if (empty($keywords)) return array();

        $q = new \WP_Query(array(
            'post_type' => 'any',
            'post_status' => 'publish',
            'posts_per_page' => max(1,(int)$limit),
            'post__not_in' => array((int)$post_id),
            's' => implode(' ', array_slice($keywords,0,6)),
            'no_found_rows' => true,
        ));

        $out = array();
        foreach ((array)$q->posts as $p) {
            $out[] = array('ID'=>$p->ID,'title'=>get_the_title($p->ID),'url'=>get_permalink($p->ID));
        }
        wp_reset_postdata();
        return $out;
    }

    private function extract_keywords($post_id) {
        $title = strtolower((string)get_the_title($post_id));
        $terms = wp_get_post_terms($post_id, array('category','post_tag'), array('fields'=>'names'));
        if (!is_array($terms)) $terms = array();
        $text = $title . ' ' . strtolower(implode(' ', $terms));
        $text = preg_replace('/[^a-z0-9\s]/', ' ', $text);
        $words = preg_split('/\s+/', $text, -1, PREG_SPLIT_NO_EMPTY);

        $stop = array('the','and','for','with','that','this','from','your','you','are','was','were','but','not','have','has','had','into','onto','over','under','about','how','why','what','when','where','which','who','a','an','to','in','on','of','is','it');
        $k = array();
        foreach ($words as $w) {
            if (strlen($w) < 3) continue;
            if (in_array($w,$stop,true)) continue;
            $k[] = $w;
        }
        return array_values(array_unique($k));
    }
	
    private function get_smart_links_options() : array {
        $opts = get_option('aegislink_smart_links', array('enabled' => 0, 'map' => array()));
        if (!is_array($opts)) { $opts = array('enabled' => 0, 'map' => array()); }
        if (empty($opts['enabled'])) { return array('enabled' => 0, 'map' => array()); }
        if (!isset($opts['map']) || !is_array($opts['map'])) { $opts['map'] = array(); }

        // Normalize map to: lower_keyword => url
        $map = array();
        foreach ($opts['map'] as $row) {
            if (!is_array($row)) { continue; }
            $kw = isset($row['keyword']) ? trim((string) $row['keyword']) : '';
            $url = isset($row['url']) ? trim((string) $row['url']) : '';
            if ($kw === '' || $url === '') { continue; }
            if (preg_match('/\s/', $kw)) { continue; } // one word only
            $map[strtolower($kw)] = esc_url_raw($url);
        }

        return array('enabled' => 1, 'map' => $map);
    }

    public function apply_smart_links($content) {
        if (!is_string($content) || $content === '') { return $content; }
        if (is_admin() || wp_doing_ajax() || is_feed()) { return $content; }

        $opts = $this->get_smart_links_options();
        if (empty($opts['enabled']) || empty($opts['map'])) { return $content; }

        $map = $opts['map'];

        // Build one regex for all keywords (longest first prevents partial matches)
        $keywords = array_keys($map);
        usort($keywords, function($a, $b) { return strlen($b) <=> strlen($a); });

        $escaped = array_map(function($k) { return preg_quote($k, '/'); }, $keywords);
        $pattern = '/\b(' . implode('|', $escaped) . ')\b/ui';

        // Split HTML into tags/text while tracking "inside" states (a, code, pre, script, style)
        $parts = preg_split('/(<[^>]+>)/', $content, -1, PREG_SPLIT_DELIM_CAPTURE);
        if (!is_array($parts) || count($parts) < 2) {
            // No tags; safe to process directly
            return preg_replace_callback($pattern, function($m) use ($map) {
                $key = strtolower($m[1]);
                if (empty($map[$key])) { return $m[0]; }
                $url = $map[$key];
                return '<a href="' . esc_url($url) . '" class="aegislink-smartlink" rel="noopener">' . esc_html($m[0]) . '</a>';
            }, $content);
        }

        $inside_a = false;
        $inside_code = false;
        $inside_pre = false;
        $inside_script = false;
        $inside_style = false;

        for ($i = 0; $i < count($parts); $i++) {
            $p = $parts[$i];
            if ($p === '') { continue; }

            if ($p[0] === '<') {
                $tag = strtolower($p);

                // Track open/close states
                if (preg_match('/^<a\b/i', $p)) { $inside_a = true; }
                if (preg_match('/^<\/a\b/i', $p)) { $inside_a = false; }

                if (preg_match('/^<code\b/i', $p)) { $inside_code = true; }
                if (preg_match('/^<\/code\b/i', $p)) { $inside_code = false; }

                if (preg_match('/^<pre\b/i', $p)) { $inside_pre = true; }
                if (preg_match('/^<\/pre\b/i', $p)) { $inside_pre = false; }

                if (preg_match('/^<script\b/i', $p)) { $inside_script = true; }
                if (preg_match('/^<\/script\b/i', $p)) { $inside_script = false; }

                if (preg_match('/^<style\b/i', $p)) { $inside_style = true; }
                if (preg_match('/^<\/style\b/i', $p)) { $inside_style = false; }

                continue; // tags unchanged
            }

            // Only modify plain text that is NOT inside link/code/pre/script/style
            if ($inside_a || $inside_code || $inside_pre || $inside_script || $inside_style) {
                continue;
            }

            $parts[$i] = preg_replace_callback($pattern, function($m) use ($map) {
                $key = strtolower($m[1]);
                if (empty($map[$key])) { return $m[0]; }
                $url = $map[$key];

                return '<a href="' . esc_url($url) . '" class="aegislink-smartlink" rel="noopener">' . esc_html($m[0]) . '</a>';
            }, $p);
        }


        // Optional debug: log replacements count when WP_DEBUG_LOG is enabled
        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            $new_html = implode('', $parts);
            if ($new_html !== $content) {
                // quick count of our marker class
                $cnt = substr_count($new_html, 'aegislink-smartlink');
                error_log('[AegisLink] Smart Links applied. links_added=' . $cnt);
            }
            return $new_html;
        }

        return implode('', $parts);
    }

}
