<?php
/**
 * Independent submenu page: Aegisify Bundled Plugins
 * - Single file, portable across Aegisify products.
 * - Renders plugin cards and supports Install/Remove via CLM download.
 */

if (!defined('ABSPATH')) { exit; }

if (!class_exists('Aegisify_Bundled_Plugins_Page')) {

class Aegisify_Bundled_Plugins_Page {

	private static $config = array(
		'parent_slug' => 'aegisseo',

		'menu_slug'   => 'aegisify-bundled-plugins',

		'page_title'  => 'Aegisify Plugins',
		'menu_title'  => 'Aegisify Plugins',

		'view_cap'    => 'read',

		'products' => array(
			'aegisshield' => array(
				'plugin_file' => 'aegisshield-security/aegisshield-security.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2025/12/aegisshield-security-7.1.2.zip',
				'title'       => 'AegisShield',
				'desc'        => 'Full WordPress security suite: hardening, monitoring, malware checks, integrity tools, and protection layers built for real admins.',
				'image'       => '', // set your image URL if you want
			),
			'aegiswaf' => array(
				'plugin_file' => 'aegiswaf/aegiswaf.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2025/12/aegiswaf-1.8.3.zip',
				'title'       => 'AegisWAF',
				'desc'        => 'Web Application Firewall for WordPress, DDOS Layer7, Bot Control, API Shield and Logs Attack Story.',
				'image'       => '',
			),
			'aegisseo' => array(
				'plugin_file' => 'aegisseo/aegisseo.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2025/12/AegisSEO-1.4.8.zip',
				'title'       => 'AegisSEO',
				'desc'        => 'SEO OPs Center, Global SEO Automation, Social Meta, Schema, Advanced SiteMAPS, Issues & Fixes, Evidence, Linking Assistance and more FREE advanced features...',
				'image'       => '',
			),
			'aegisticket' => array(
				'plugin_file' => 'aegisticket/aegisticket.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2025/12/aegiswaf-1.8.2.zip',
				'title'       => 'AegisTicket',
				'desc'        => 'Support ticketing plugin built for ops: CPT-based tickets, admin workflows, and customer-friendly submission + tracking.',
				'image'       => '', // set your image URL if you want
			),
		),
	);
	
	public static function init(array $override = array()) : void {
		if (!is_admin()) { return; }

		self::$config = array_merge(self::$config, $override);

		$products = (array) self::$config['products'];

		if (isset($products['aegisseo']) && is_array($products['aegisseo'])) {
			if (defined('AEGISSEO_BASENAME') && AEGISSEO_BASENAME) {
				$products['aegisseo']['plugin_file'] = AEGISSEO_BASENAME;
			}
			if (empty($products['aegisseo']['image']) && defined('AEGISSEO_URL') && AEGISSEO_URL) {
				$products['aegisseo']['image'] = rtrim(AEGISSEO_URL, '/') . '/assets/images/logo_shield.png';
			}
		}

		self::$config['products'] = $products;

		add_action('admin_post_aegisify_products_install', array(__CLASS__, 'handle_install'));
	}

	private static function maybe_handle_inline_install() : void {
		if (empty($_GET['aegisify_install'])) {
			return;
		}

		if (!current_user_can('install_plugins')) {
			wp_die('You do not have permission to install plugins.');
		}

		$key = sanitize_key((string) $_GET['aegisify_install']);
		check_admin_referer('aegisify_inline_install_' . $key);

		$products = self::get_products();
		if (empty($products[$key])) {
			wp_die('Invalid product.');
		}

		$p = $products[$key];
		$plugin_file = !empty($p['plugin_file']) ? (string) $p['plugin_file'] : '';
		if ($plugin_file && self::is_installed($plugin_file)) {
			self::redirect_back('Plugin is already installed.', 'updated');
		}

		$download = !empty($p['zip_url']) ? esc_url_raw((string) $p['zip_url']) : '';
		if (!$download || !preg_match('#^https?://#i', $download)) {
			self::redirect_back('Install failed: Missing or invalid zip_url for this product.', 'error');
		}

		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		$url = remove_query_arg(array('aegisify_notice','aegisify_type'), wp_unslash($_SERVER['REQUEST_URI']));
		$creds = request_filesystem_credentials($url);
		if ($creds === false) {
			exit;
		}
		if (!WP_Filesystem($creds)) {
			request_filesystem_credentials($url, '', true);
			exit;
		}

		$skin = new Automatic_Upgrader_Skin();
		$upgrader = new Plugin_Upgrader($skin);

		$result = $upgrader->install($download);

		if (is_wp_error($result)) {
			self::redirect_back('Install failed: ' . $result->get_error_message(), 'error');
		}

		if (!$result) {
			self::redirect_back('Install failed: WordPress could not install the package (permissions / invalid zip).', 'error');
		}

		self::redirect_back('Installed successfully.', 'updated');
	}

	public static function add_submenu() : void {
		$parent = (string) self::$config['parent_slug'];
		$slug   = (string) self::$config['menu_slug'];

		add_submenu_page(
			$parent,
			(string) self::$config['page_title'],
			(string) self::$config['menu_title'],
			(string) self::$config['view_cap'],
			$slug,
			array(__CLASS__, 'render_page')
		);
	}

	private static function get_products() : array {
		$products = (array) self::$config['products'];

		if (has_filter('aegisify_bundled_plugins_products')) {
			$products = (array) apply_filters('aegisify_bundled_plugins_products', $products, self::$config);
		}

		return $products;
	}

	private static function is_installed(string $plugin_file) : bool {
		$plugin_file = ltrim($plugin_file, '/');
		if (!$plugin_file) { return false; }
		return file_exists(WP_PLUGIN_DIR . '/' . $plugin_file);
	}
	
	private static function is_active_plugin(string $plugin_file) : bool {
		$plugin_file = ltrim($plugin_file, '/');
		if (!$plugin_file) { return false; }

		if (!function_exists('is_plugin_active')) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		return is_plugin_active($plugin_file);
	}

	private static function redirect_back(string $notice = '', string $type = 'updated') : void {
		$parent = (string) self::$config['parent_slug'];
		$slug   = (string) self::$config['menu_slug'];

		$url = admin_url('admin.php?page=' . rawurlencode($slug));

		if ($notice) {
			$url = add_query_arg(array(
				'aegisify_notice' => rawurlencode($notice),
				'aegisify_type'   => rawurlencode($type),
			), $url);
		}

		wp_safe_redirect($url);
		exit;
	}
	
	private static function get_wp_activate_url(string $plugin_file) : string {
		$plugin_file = ltrim($plugin_file, '/');

		$url = admin_url('plugins.php?action=activate&plugin=' . rawurlencode($plugin_file));

		return wp_nonce_url($url, 'activate-plugin_' . $plugin_file);
	}

	public static function handle_install() : void {
		if (!current_user_can('install_plugins')) {
			self::redirect_back('You do not have permission to install plugins.', 'error');
		}
		check_admin_referer('aegisify_products_install');

		$key = isset($_POST['product_key']) ? sanitize_key((string) $_POST['product_key']) : '';
		$products = self::get_products();

		if (!$key || !isset($products[$key])) {
			self::redirect_back('Invalid product.', 'error');
		}

		$download = isset($p['zip_url']) ? esc_url_raw((string) $p['zip_url']) : '';
		if (!$download || !preg_match('#^https?://#i', $download)) {
			self::redirect_back('Install failed: Missing or invalid zip_url for this product.', 'error');
		}

		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/misc.php';
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		$skin = new WP_Ajax_Upgrader_Skin();

		$upgrader = new Plugin_Upgrader($skin);
		$result = $upgrader->install($download);

		if (is_wp_error($result)) {
			self::redirect_back('Install failed: ' . $result->get_error_message(), 'error');
		}

		if (!$result) {
			$err = (isset($skin->result) && is_wp_error($skin->result))
				? $skin->result->get_error_message()
				: 'Unknown installer failure.';
			self::redirect_back('Install failed: ' . $err, 'error');
		}

		self::redirect_back('Installed successfully.', 'updated');

		$p = $products[$key];
		$plugin_file = isset($p['plugin_file']) ? (string) $p['plugin_file'] : '';

		if ($plugin_file && self::is_installed($plugin_file)) {
			self::redirect_back('Plugin is already installed.', 'updated');
		}
		
	}
	
	public static function render_page() : void {
		self::maybe_handle_inline_install();
		if (!current_user_can((string) self::$config['view_cap'])) {
			wp_die(esc_html__('Sorry, you are not allowed to access this page.', 'aegisseo'));
		}

		$notice = isset($_GET['aegisify_notice']) ? (string) wp_unslash($_GET['aegisify_notice']) : '';
		$type   = isset($_GET['aegisify_type']) ? sanitize_key((string) $_GET['aegisify_type']) : '';

		$products = self::get_products();
		?>
		<div class="wrap">
			<h1 style="margin-bottom:8px;"><?php echo esc_html(self::$config['page_title']); ?></h1>
			<div class="aegisify-banner" style="margin: 12px 0 20px 0;">
				<img
					src="<?php echo esc_url( plugin_dir_url( __FILE__ ) . '../../../assets/banner.png' ); ?>"
					alt="<?php echo esc_attr__( 'Aegisify Plugin Suite', 'aegisseo' ); ?>"
					style="
						width:100%;
						max-width:980px;
						height:auto;
						display:block;
						border-radius:8px;
						box-shadow:0 2px 6px rgba(0,0,0,0.08);
					"
				/>
			</div>			
			<p style="max-width: 980px;">
				<?php echo esc_html__('Install and manage your Aegisify product suite from one place. Each card pulls the latest package from CLM and installs it into WordPress like a standard plugin.', 'aegisseo'); ?>
			</p>

			<?php if ($notice): ?>
				<div class="<?php echo esc_attr(($type === 'error') ? 'notice notice-error' : 'notice notice-success'); ?> is-dismissible">
					<p><?php echo esc_html($notice); ?></p>
				</div>
			<?php endif; ?>

			<div class="notice notice-info" style="max-width:980px;">
				<p>
					<?php echo esc_html__('Tip: WordPress may request filesystem credentials (FTP) depending on server permissions. That is normal for plugin installs on some hosts.', 'aegisseo'); ?>
				</p>
			</div>

			<style>
				.aegisify-grid{
					display:grid;
					grid-template-columns: repeat(4, minmax(0, 1fr));
					gap: 16px;
					max-width: 1200px;
				}
				@media (max-width: 1200px){
					.aegisify-grid{ grid-template-columns: repeat(3, minmax(0,1fr)); }
				}
				@media (max-width: 900px){
					.aegisify-grid{ grid-template-columns: repeat(2, minmax(0,1fr)); }
				}
				@media (max-width: 520px){
					.aegisify-grid{ grid-template-columns: 1fr; }
				}
				.aegisify-card{
					background:#fff;
					border:1px solid #e5e5e5;
					border-radius:8px;
					overflow:hidden;
					box-shadow: 0 1px 2px rgba(0,0,0,0.04);
					display:flex;
					flex-direction:column;
					min-height: 280px;
				}
				.aegisify-card__top{
					position:relative;
					padding: 18px 18px 10px;
					border-bottom: 1px solid #f0f0f0;
					display:flex;
					align-items:center;
					justify-content:center;
					min-height: 120px;
				}
				.aegisify-card__badge{
					position:absolute;
					top:10px;
					left:10px;
					background:#2271b1;
					color:#fff;
					font-size: 11px;
					padding: 4px 8px;
					border-radius: 4px;
					letter-spacing: .3px;
					text-transform: uppercase;
				}
				.aegisify-card__img{
					max-width: 78%;
					max-height: 78px;
					width:auto;
					height:auto;
					object-fit:contain;
				}
				.aegisify-card__body{
					padding: 14px 16px 0;
					flex:1;
				}
				.aegisify-card__title{
					margin: 0 0 8px;
					font-size: 14px;
					font-weight: 700;
				}
				.aegisify-card__desc{
					margin: 0 0 14px;
					color:#4b5563;
					font-size: 12.5px;
					line-height: 1.45;
					min-height: 54px;
				}
				.aegisify-card__footer{
					padding: 12px 16px 16px;
					display:flex;
					align-items:center;
					justify-content:space-between;
					gap: 10px;
				}
				.aegisify-status{
					font-size: 12px;
					color:#111827;
				}
				.aegisify-status strong{ font-weight:700; }
				.aegisify-actions form{ display:inline; }
			</style>

			<div class="aegisify-grid">
				<?php foreach ($products as $k => $p): ?>
					<?php
						$k = (string) $k;
						$title = isset($p['title']) ? (string) $p['title'] : $k;
						$desc  = isset($p['desc']) ? (string) $p['desc'] : '';
						$img   = isset($p['image']) ? (string) $p['image'] : '';
						$plugin_file = isset($p['plugin_file']) ? (string) $p['plugin_file'] : '';

						$installed = ($plugin_file ? self::is_installed($plugin_file) : false);
						$active = ($installed && $plugin_file) ? self::is_active_plugin($plugin_file) : false;
					?>
					<div class="aegisify-card">
						<div class="aegisify-card__top">
							<div class="aegisify-card__badge"><?php echo esc_html($installed ? 'Installed' : 'Available'); ?></div>
							<?php if ($img): ?>
								<img class="aegisify-card__img" src="<?php echo esc_url($img); ?>" alt="<?php echo esc_attr($title); ?>">
							<?php else: ?>
								<div style="font-weight:800;font-size:18px;opacity:.25;"><?php echo esc_html($title); ?></div>
							<?php endif; ?>
						</div>

						<div class="aegisify-card__body">
							<div class="aegisify-card__title"><?php echo esc_html($title); ?></div>
							<div class="aegisify-card__desc"><?php echo esc_html($desc); ?></div>
						</div>

						<div class="aegisify-card__footer">
							<div class="aegisify-status">
								<strong><?php echo esc_html($installed ? 'Status:' : 'Status:'); ?></strong>
								<?php echo esc_html(!$installed ? ' Not Installed' : ($active ? ' Active' : ' Installed')); ?>
							</div>

							<div class="aegisify-actions">
								<?php if (!$installed): ?>
									<form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
										<input type="hidden" name="action" value="aegisify_products_install" />
										<input type="hidden" name="product_key" value="<?php echo esc_attr($k); ?>" />
										<?php wp_nonce_field('aegisify_products_install'); ?>
										<button type="submit" class="button button-primary">
											<?php echo esc_html__('Install Plugin', 'aegisseo'); ?>
										</button>
									</form>

								<?php elseif (!$active): ?>
									<?php $activate_url = self::get_wp_activate_url($plugin_file); ?>
									<a href="<?php echo esc_url($activate_url); ?>" class="button button-secondary">
										<?php echo esc_html__('Activate Plugin', 'aegisseo'); ?>
									</a>

								<?php else: ?>
									<a href="<?php echo esc_url(admin_url('plugins.php')); ?>" class="button">
										<?php echo esc_html__('Go To Plugins', 'aegisseo'); ?>
									</a>
								<?php endif; ?>
							</div>
						</div>
					</div>
				<?php endforeach; ?>
			</div>

		</div>
		<?php
	}
}

} 
