<?php
namespace AegisSEO\Admin;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Admin_Bulk {

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;

        add_action('admin_post_aegisseo_bulk_save', array($this, 'save'));
        add_action('admin_post_aegisseo_bulk_export', array($this, 'export_csv'));
        add_action('admin_post_aegisseo_bulk_import', array($this, 'import_csv'));
    }

    private function cap() {
        return current_user_can('manage_options');
    }

    public function save() {
        if (!$this->cap()) { wp_die('Forbidden'); }
        check_admin_referer('aegisseo_bulk_save');

        $rows = isset($_POST['rows']) && is_array($_POST['rows']) ? (array) $_POST['rows'] : array();

        foreach ($rows as $post_id => $data) {
            $post_id = (int) $post_id;
            if ($post_id <= 0) continue;
            if (!current_user_can('edit_post', $post_id)) continue;

            $title = isset($data['title']) ? sanitize_text_field(wp_unslash($data['title'])) : '';
            $desc  = isset($data['description']) ? sanitize_textarea_field(wp_unslash($data['description'])) : '';
            $canon = isset($data['canonical']) ? esc_url_raw(wp_unslash($data['canonical'])) : '';
            $noindex = !empty($data['noindex']) ? 1 : 0;
            $nofollow= !empty($data['nofollow']) ? 1 : 0;

            update_post_meta($post_id, '_aegisseo_title', $title);
            update_post_meta($post_id, '_aegisseo_description', $desc);
            update_post_meta($post_id, '_aegisseo_canonical', $canon);
            update_post_meta($post_id, '_aegisseo_noindex', $noindex);
            update_post_meta($post_id, '_aegisseo_nofollow', $nofollow);
        }

        wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=bulk&saved=1'));
        exit;
    }

    public function export_csv() {
        if (!$this->cap()) { wp_die('Forbidden'); }
        check_admin_referer('aegisseo_bulk_export');

        $post_type = isset($_GET['post_type']) ? sanitize_key($_GET['post_type']) : 'any';

        $args = array(
            'post_type'      => ($post_type === 'any' ? get_post_types(array('public'=>true),'names') : $post_type),
            'post_status'    => array('publish','draft','pending','future','private'),
            'posts_per_page' => 2000,
            'fields'         => 'ids',
        );
        $ids = get_posts($args);

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=aegisseo-bulk-export.csv');

        $out = fopen('php://output', 'w');
        fputcsv($out, array('post_id','post_type','post_title','seo_title','meta_description','canonical','noindex','nofollow'));

        foreach ($ids as $id) {
            $row = array(
                $id,
                get_post_type($id),
                get_the_title($id),
                (string) get_post_meta($id, '_aegisseo_title', true),
                (string) get_post_meta($id, '_aegisseo_description', true),
                (string) get_post_meta($id, '_aegisseo_canonical', true),
                (int) get_post_meta($id, '_aegisseo_noindex', true),
                (int) get_post_meta($id, '_aegisseo_nofollow', true),
            );
            fputcsv($out, $row);
        }

        fclose($out);
        exit;
    }

    public function import_csv() {
        if (!$this->cap()) { wp_die('Forbidden'); }
        check_admin_referer('aegisseo_bulk_import');

        if (empty($_FILES['csv_file']['tmp_name'])) {
            wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=bulk&import=0'));
            exit;
        }

        $fh = fopen($_FILES['csv_file']['tmp_name'], 'r');
        if (!$fh) {
            wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=bulk&import=0'));
            exit;
        }

        $header = fgetcsv($fh);
        $count = 0;

        while (($row = fgetcsv($fh)) !== false) {
            $map = array();
            foreach ($header as $i => $k) {
                $map[$k] = $row[$i] ?? '';
            }
            $post_id = isset($map['post_id']) ? (int) $map['post_id'] : 0;
            if ($post_id <= 0) continue;
            if (!current_user_can('edit_post', $post_id)) continue;

            update_post_meta($post_id, '_aegisseo_title', sanitize_text_field($map['seo_title'] ?? ''));
            update_post_meta($post_id, '_aegisseo_description', sanitize_textarea_field($map['meta_description'] ?? ''));
            update_post_meta($post_id, '_aegisseo_canonical', esc_url_raw($map['canonical'] ?? ''));
            update_post_meta($post_id, '_aegisseo_noindex', !empty($map['noindex']) ? 1 : 0);
            update_post_meta($post_id, '_aegisseo_nofollow', !empty($map['nofollow']) ? 1 : 0);
            $count++;
        }
        fclose($fh);

        wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=bulk&import=' . (int)$count));
        exit;
    }
}
