<?php
namespace AegisSEO\Admin;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Admin_Migration {

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;

        add_action('admin_post_aegisseo_migrate_meta', array($this, 'migrate_meta'));
        add_action('admin_post_aegisseo_migrate_redirection', array($this, 'migrate_redirection'));
    }

    private function cap() {
        return current_user_can('manage_options');
    }

    public function migrate_meta() {
        if (!$this->cap()) { wp_die('Forbidden'); }
        check_admin_referer('aegisseo_migrate_meta');

        $source = isset($_POST['source']) ? sanitize_key($_POST['source']) : '';
        $limit  = isset($_POST['limit']) ? max(1, (int) $_POST['limit']) : 500;
        $offset = isset($_POST['offset']) ? max(0, (int) $_POST['offset']) : 0;

        $keys = $this->get_meta_map($source);
        if (!$keys) {
            wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=migration&meta=0'));
            exit;
        }

        $ids = get_posts(array(
            'post_type'      => get_post_types(array('public'=>true),'names'),
            'post_status'    => array('publish','draft','pending','future','private'),
            'posts_per_page' => $limit,
            'offset'         => $offset,
            'fields'         => 'ids',
        ));

        $migrated = 0;

        foreach ($ids as $id) {
            if (!empty($keys['title'])) {
                $v = $this->first_meta($id, $keys['title']);
                if ($v !== '' && get_post_meta($id, '_aegisseo_title', true) === '') {
                    update_post_meta($id, '_aegisseo_title', (string) $v);
                }
            }
            if (!empty($keys['desc'])) {
                $v = $this->first_meta($id, $keys['desc']);
                if ($v !== '' && get_post_meta($id, '_aegisseo_description', true) === '') {
                    update_post_meta($id, '_aegisseo_description', (string) $v);
                }
            }
            if (!empty($keys['canon'])) {
                $v = $this->first_meta($id, $keys['canon']);
                if ($v !== '' && get_post_meta($id, '_aegisseo_canonical', true) === '') {
                    update_post_meta($id, '_aegisseo_canonical', esc_url_raw((string) $v));
                }
            }
            $noindex = null;
            $nofollow= null;
            if (!empty($keys['noindex'])) {
                $v = $this->first_meta($id, $keys['noindex']);
                $noindex = $this->truthy($v);
            }
            if (!empty($keys['nofollow'])) {
                $v = $this->first_meta($id, $keys['nofollow']);
                $nofollow = $this->truthy($v);
            }
            if ($noindex !== null && get_post_meta($id, '_aegisseo_noindex', true) === '') {
                update_post_meta($id, '_aegisseo_noindex', $noindex ? 1 : 0);
            }
            if ($nofollow !== null && get_post_meta($id, '_aegisseo_nofollow', true) === '') {
                update_post_meta($id, '_aegisseo_nofollow', $nofollow ? 1 : 0);
            }

            if (!empty($keys['og_title'])) {
                $v = $this->first_meta($id, $keys['og_title']);
                if ($v !== '' && get_post_meta($id, '_aegisseo_og_title', true) === '') {
                    update_post_meta($id, '_aegisseo_og_title', (string) $v);
                }
            }
            if (!empty($keys['og_desc'])) {
                $v = $this->first_meta($id, $keys['og_desc']);
                if ($v !== '' && get_post_meta($id, '_aegisseo_og_description', true) === '') {
                    update_post_meta($id, '_aegisseo_og_description', (string) $v);
                }
            }
            if (!empty($keys['og_img'])) {
                $v = $this->first_meta($id, $keys['og_img']);
                $url = $this->maybe_image_url($v);
                if ($url !== '' && get_post_meta($id, '_aegisseo_og_image', true) === '') {
                    update_post_meta($id, '_aegisseo_og_image', esc_url_raw($url));
                }
            }
            if (!empty($keys['tw_title'])) {
                $v = $this->first_meta($id, $keys['tw_title']);
                if ($v !== '' && get_post_meta($id, '_aegisseo_twitter_title', true) === '') {
                    update_post_meta($id, '_aegisseo_twitter_title', (string) $v);
                }
            }
            if (!empty($keys['tw_desc'])) {
                $v = $this->first_meta($id, $keys['tw_desc']);
                if ($v !== '' && get_post_meta($id, '_aegisseo_twitter_description', true) === '') {
                    update_post_meta($id, '_aegisseo_twitter_description', (string) $v);
                }
            }
            if (!empty($keys['tw_img'])) {
                $v = $this->first_meta($id, $keys['tw_img']);
                $url = $this->maybe_image_url($v);
                if ($url !== '' && get_post_meta($id, '_aegisseo_twitter_image', true) === '') {
                    update_post_meta($id, '_aegisseo_twitter_image', esc_url_raw($url));
                }
            }

            $migrated++;
        }

        wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=migration&meta=' . (int)$migrated . '&next=' . (int)($offset + $limit)));
        exit;
    }

    public function migrate_redirection() {
        if (!$this->cap()) { wp_die('Forbidden'); }
        check_admin_referer('aegisseo_migrate_redirection');

        global $wpdb;
        $items = $wpdb->get_results("SELECT url, action_data, action_type, match_url FROM {$wpdb->prefix}redirection_items WHERE status='enabled' LIMIT 5000", ARRAY_A);
        if (!is_array($items)) { $items = array(); }

        $imported = 0;
        foreach ($items as $it) {
            $from = $it['url'] ?? '';
            $to   = $it['action_data'] ?? '';
            if ($from === '' || $to === '') continue;

            if (function_exists('aegisseo') && isset(aegisseo()->redirects) && aegisseo()->redirects) {
                aegisseo()->redirects->upsert(array(
                    'source' => $from,
                    'target' => $to,
                    'type'   => 301,
                    'hits'   => 0,
                    'status' => 1,
                    'note'   => 'Imported from Redirection',
                    'created_at' => current_time('mysql'),
                ));
                $imported++;
            }
        }

        wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=migration&redir=' . (int)$imported));
        exit;
    }

		private function truthy($v) {
			if ($v === '' || $v === null) return null;

			if (is_bool($v)) return $v;

			if (is_array($v)) {
				$vv = array_map('strtolower', array_map('trim', array_map('strval', $v)));
				if (in_array('noindex', $vv, true) || in_array('nofollow', $vv, true)) return true;
				if (in_array('0', $vv, true) || in_array('no', $vv, true) || in_array('false', $vv, true)) return false;
				return null;
			}

			if (is_numeric($v)) return ((int) $v) === 1;

			$s = strtolower(trim((string) $v));
			if ($s === 'yes' || $s === '1' || $s === 'true' || $s === 'noindex' || $s === 'nofollow') return true;
			if ($s === 'no' || $s === '0' || $s === 'false') return false;

			return null;
		}

		private function first_meta($post_id, $key_or_keys) {
        if (empty($key_or_keys)) return '';
        $keys = is_array($key_or_keys) ? $key_or_keys : array($key_or_keys);
        foreach ($keys as $k) {
            $k = (string) $k;
            if ($k === '') continue;
            $v = get_post_meta($post_id, $k, true);
            if (is_string($v)) {
                $v = trim($v);
                if ($v !== '') return $v;
            } elseif (!empty($v)) {
                return $v;
            }
        }
        return '';
    }

    private function maybe_image_url($v) {
        if ($v === '' || $v === null) return '';

        if (is_numeric($v)) {
            $url = wp_get_attachment_url((int) $v);
            return $url ? (string) $url : '';
        }
        if (is_array($v)) {
            if (!empty($v['url'])) return (string) $v['url'];
            if (!empty($v[0])) return (string) $v[0];
            return '';
        }
        $v = trim((string) $v);
        return $v;
    }

    private function get_meta_map($source) {
        switch ($source) {
            case 'yoast':
                return array(
                    'title'  => '_yoast_wpseo_title',
                    'desc'   => '_yoast_wpseo_metadesc',
                    'canon'  => '_yoast_wpseo_canonical',
                    'noindex'=> '_yoast_wpseo_meta-robots-noindex',
                    'nofollow'=> '_yoast_wpseo_meta-robots-nofollow',
                
                    'og_title' => '_yoast_wpseo_opengraph-title',
                    'og_desc'  => '_yoast_wpseo_opengraph-description',
                    'og_img'   => '_yoast_wpseo_opengraph-image',
                    'tw_title' => '_yoast_wpseo_twitter-title',
                    'tw_desc'  => '_yoast_wpseo_twitter-description',
                    'tw_img'   => '_yoast_wpseo_twitter-image',
);
            case 'rankmath':
                return array(
                    'title' => 'rank_math_title',
                    'desc'  => 'rank_math_description',
                    'canon' => 'rank_math_canonical_url',
                    'noindex' => 'rank_math_robots',
                    'nofollow'=> 'rank_math_robots',
                
                    'og_title' => 'rank_math_facebook_title',
                    'og_desc'  => 'rank_math_facebook_description',
                    'og_img'   => 'rank_math_facebook_image',
                    'tw_title' => 'rank_math_twitter_title',
                    'tw_desc'  => 'rank_math_twitter_description',
                    'tw_img'   => 'rank_math_twitter_image',
);
            case 'aioseo':
                return array(
                    'title' => array('_aioseo_title', 'aioseo_title'),
                    'desc'  => array('_aioseo_description', 'aioseo_description'),
                    'canon' => array('_aioseo_canonical_url', '_aioseo_canonical'),
                    'noindex' => array('_aioseo_robots_noindex', '_aioseo_noindex'),
                    'nofollow'=> array('_aioseo_robots_nofollow', '_aioseo_nofollow'),

                    'og_title' => '_aioseo_og_title',
                    'og_desc'  => '_aioseo_og_description',
                    'og_img'   => array('_aioseo_og_image', '_aioseo_og_image_url'),
                    'tw_title' => '_aioseo_twitter_title',
                    'tw_desc'  => '_aioseo_twitter_description',
                    'tw_img'   => array('_aioseo_twitter_image', '_aioseo_twitter_image_url'),
                );
            case 'seopress':
                return array(
                    'title' => '_seopress_titles_title',
                    'desc'  => '_seopress_titles_desc',
                    'canon' => '_seopress_robots_canonical',
                    'noindex' => '_seopress_robots_index',
                    'nofollow'=> '_seopress_robots_follow',
                
                    'og_title' => '_seopress_social_fb_title',
                    'og_desc'  => '_seopress_social_fb_desc',
                    'og_img'   => '_seopress_social_fb_img',
                    'tw_title' => '_seopress_social_twitter_title',
                    'tw_desc'  => '_seopress_social_twitter_desc',
                    'tw_img'   => '_seopress_social_twitter_img',
);
            default:
                return null;
        }
    }
}
