<?php
namespace AegisSEO\SEO;

if (!defined('ABSPATH')) { exit; }

class Events {

    private $tracked_meta_keys = array(
        '_aegisseo_title',
        '_aegisseo_description',
        '_aegisseo_canonical',
        '_aegisseo_robots',
        '_aegisseo_focus_phrase',
        '_aegisseo_schema_mode',
        '_aegisseo_schema_type',
        '_aegisseo_schema_data',
        '_aegisseo_noindex',
        '_aegisseo_nofollow',
        '_aegisseo_og_title',
        '_aegisseo_og_description',
        '_aegisseo_og_image',
        '_aegisseo_twitter_title',
        '_aegisseo_twitter_description',
        '_aegisseo_twitter_image',
    );

    public function __construct() {
        add_filter('add_post_metadata', array($this, 'capture_add_post_meta'), 10, 5);
        add_filter('update_post_metadata', array($this, 'capture_update_post_meta'), 10, 5);
        add_filter('delete_post_metadata', array($this, 'capture_delete_post_meta'), 10, 5);
        add_action('post_updated', array($this, 'capture_post_updated'), 10, 3);
    }

    private function is_tracked_meta($meta_key) {
        return in_array($meta_key, $this->tracked_meta_keys, true);
    }

    public function capture_add_post_meta($check, $object_id, $meta_key, $meta_value, $unique) {
        if (!$this->is_tracked_meta($meta_key)) { return $check; }
        $this->log_event('meta_add', 'post', (int) $object_id, $meta_key, null, $meta_value);
        return $check;
    }

    public function capture_update_post_meta($check, $object_id, $meta_key, $meta_value, $prev_value) {
        if (!$this->is_tracked_meta($meta_key)) { return $check; }
        $old = get_post_meta((int) $object_id, $meta_key, true);
        $this->log_event('meta_update', 'post', (int) $object_id, $meta_key, $old, $meta_value);
        return $check;
    }

    public function capture_delete_post_meta($check, $object_id, $meta_key, $meta_value, $delete_all) {
        if (!$this->is_tracked_meta($meta_key)) { return $check; }
        $old = get_post_meta((int) $object_id, $meta_key, true);
        $this->log_event('meta_delete', 'post', (int) $object_id, $meta_key, $old, null);
        return $check;
    }

    public function log_event($event_type, $object_type, $object_id = 0, $meta_key = null, $old_value = null, $new_value = null, $user_id = null) {
        global $wpdb;

        $table = $wpdb->prefix . 'aegisseo_events';

        if (null === $user_id) {
            $user_id = get_current_user_id();
        }

        $data = array(
            'event_type'   => (string) $event_type,
            'object_type'  => (string) $object_type,
            'object_id'    => (int) $object_id,
            'meta_key'     => $meta_key ? (string) $meta_key : null,
            'old_value'    => is_scalar($old_value) || null === $old_value ? $old_value : wp_json_encode($old_value),
            'new_value'    => is_scalar($new_value) || null === $new_value ? $new_value : wp_json_encode($new_value),
            'user_id'      => (int) $user_id,
            'created_at'   => current_time('mysql'),
        );

        $formats = array('%s','%s','%d','%s','%s','%s','%d','%s');

        if (null === $data['meta_key']) {
            $formats[3] = '%s';
            $data['meta_key'] = null;
        }

        $wpdb->insert($table, $data, $formats);
    }

public function capture_post_updated($post_id, $post_after, $post_before) {
    if (!is_object($post_after) || !is_object($post_before)) { return; }
    $post_id = (int) $post_id;
    if ($post_id <= 0) { return; }

    $pt = get_post_type($post_id);
    if (!$pt) { return; }
    $pto = get_post_type_object($pt);
    if (!$pto || empty($pto->public)) { return; }

    $old_title = (string) $post_before->post_title;
    $new_title = (string) $post_after->post_title;
    if ($old_title !== $new_title) {
        $this->log_event('title_changed', 'post', $post_id, 'post_title', $old_title, $new_title);
    }

    $old_slug = (string) $post_before->post_name;
    $new_slug = (string) $post_after->post_name;
    if ($old_slug !== '' && $new_slug !== '' && $old_slug !== $new_slug) {
        $this->log_event('slug_changed', 'post', $post_id, 'post_slug', $old_slug, $new_slug);
    }

    $old_content = (string) $post_before->post_content;
    $new_content = (string) $post_after->post_content;

    $old_text = trim(wp_strip_all_tags($old_content));
    $new_text = trim(wp_strip_all_tags($new_content));

    $old_words = $old_text !== '' ? str_word_count($old_text) : 0;
    $new_words = $new_text !== '' ? str_word_count($new_text) : 0;

    $old_h = 0; $new_h = 0;
    if ($old_content !== '') { $old_h = preg_match_all('/<h[1-6][^>]*>/i', $old_content); }
    if ($new_content !== '') { $new_h = preg_match_all('/<h[1-6][^>]*>/i', $new_content); }

    if ($old_words !== $new_words || $old_h !== $new_h) {
        $summary = sprintf('Words: %d→%d; Headings: %d→%d', (int)$old_words, (int)$new_words, (int)$old_h, (int)$new_h);
        $this->log_event('onpage_changed', 'post', $post_id, 'onpage_diff', null, $summary);
    }
}

    public function get_recent($limit = 50, $object_type = null, $object_id = null) {
        global $wpdb;

        $limit = max(1, (int) $limit);
        $table = $wpdb->prefix . 'aegisseo_events';

        $where = '1=1';
        $params = array();

        if ($object_type) {
            $where .= ' AND object_type = %s';
            $params[] = (string) $object_type;
        }
        if (null !== $object_id) {
            $where .= ' AND object_id = %d';
            $params[] = (int) $object_id;
        }

        $sql = "SELECT * FROM {$table} WHERE {$where} ORDER BY created_at DESC, id DESC LIMIT %d";
        $params[] = $limit;

        return $params ? (array) $wpdb->get_results($wpdb->prepare($sql, $params), ARRAY_A) : array();
    }

    public function get_event($id) {
        global $wpdb;

        $id = (int) $id;
        if ($id <= 0) { return null; }

        $table = $wpdb->prefix . 'aegisseo_events';
        $sql = "SELECT * FROM {$table} WHERE id = %d";
        $row = $wpdb->get_row($wpdb->prepare($sql, $id), ARRAY_A);

        return $row ? (array) $row : null;
    }


}