<?php
namespace AegisSEO\SEO;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Frontend {

    private $options;
    private $schema;

    public function __construct(Options $options, Schema $schema) {
        $this->options = $options;
        $this->schema  = $schema;

        add_action('wp_head', array($this, 'output_head'), 1);
        add_filter('document_title_parts', array($this, 'filter_document_title_parts'), 99);

        add_action('admin_bar_menu', array($this, 'admin_bar_indicator'), 100);
    }

    public function admin_bar_indicator($wp_admin_bar) {
        if (!is_admin_bar_showing()) { return; }
        if (!current_user_can('manage_options')) { return; }
        if (!(int) $this->options->get('show_admin_bar_indicator', 1)) { return; }

        $wp_admin_bar->add_node(array(
            'id'    => 'aegisseo-indicator',
            'title' => 'AegisSEO',
            'href'  => admin_url('admin.php?page=aegisseo'),
            'meta'  => array('title' => 'Open AegisSEO settings'),
        ));
    }

    public function filter_document_title_parts($parts) {
        $computed = $this->compute_title();
        if (!empty($computed)) {
            $parts['title'] = $computed;
        }
        return $parts;
    }

    public function output_head() {
        if (is_admin()) { return; }
        if (is_feed())  { return; }

        $meta = $this->compute_meta();

        if (!empty($meta['title'])) {
            echo "\n<!-- AegisSEO -->\n";
            echo '<meta name="aegisseo:title" content="' . esc_attr($meta['title']) . '" />' . "\n";
        }

        if (!empty($meta['description'])) {
            echo '<meta name="description" content="' . esc_attr($meta['description']) . '" />' . "\n";
        }

        if (!empty($meta['canonical'])) {
            echo '<link rel="canonical" href="' . esc_url($meta['canonical']) . '" />' . "\n";
        }

        if (!empty($meta['robots'])) {
            echo '<meta name="robots" content="' . esc_attr($meta['robots']) . '" />' . "\n";
        }

        if (!empty($meta['og']) && is_array($meta['og'])) {
            foreach ($meta['og'] as $property => $content) {
                if (is_array($content)) {
                    foreach ($content as $c) {
                        if ($c === '') { continue; }
                        echo '<meta property="' . esc_attr($property) . '" content="' . esc_attr($c) . '" />' . "\n";
                    }
                    continue;
                }
                if ($content === '') continue;
                echo '<meta property="' . esc_attr($property) . '" content="' . esc_attr($content) . '" />' . "\n";
            }
        }

        if (!empty($meta['twitter']) && is_array($meta['twitter'])) {
            foreach ($meta['twitter'] as $name => $content) {
                if ($content === '') continue;
                echo '<meta name="' . esc_attr($name) . '" content="' . esc_attr($content) . '" />' . "\n";
            }
        }

        if (!empty($meta['schema'])) {
            echo '<script type="application/ld+json">' . wp_json_encode($meta['schema']) . '</script>' . "\n";
        }

        echo "<!-- /AegisSEO -->\n";
    }

    private function compute_title() {
        $opts = $this->options->get_all();
        $sep  = $opts['separator'];

        if (is_singular()) {
            $post_id = get_queried_object_id();
            if ($post_id) {
                $custom = get_post_meta($post_id, '_aegisseo_title', true);
                if (is_string($custom) && trim($custom) !== '') {
                    return trim(wp_strip_all_tags($custom));
                }
            }
        }

        if (is_front_page() || is_home()) {
            return Variables::replace($opts['title_home'], array('sep' => $sep));
        }

        if (is_singular('post')) {
            $post_id = get_queried_object_id();
            $title = get_the_title($post_id);
            $category = '';
            $cats = get_the_category($post_id);
            if (!empty($cats) && !is_wp_error($cats)) { $category = $cats[0]->name; }
            return Variables::replace($opts['title_single'], array('title' => $title, 'category' => $category, 'sep' => $sep));
        }

        if (is_singular()) {
            $post_id = get_queried_object_id();
            $title = get_the_title($post_id);
            return Variables::replace($opts['title_page'], array('title' => $title, 'sep' => $sep));
        }

        if (is_category() || is_tag() || is_tax()) {
            $term = single_term_title('', false);
            return Variables::replace($opts['title_archive'], array('term' => $term, 'sep' => $sep));
        }

        return '';
    }

    private function compute_meta() {
        $opts = $this->options->get_all();
        $sep  = $opts['separator'];

        $context = array(
            'is_singular' => is_singular(),
            'post_id'     => is_singular() ? (int) get_queried_object_id() : 0,
        );

        $title = $this->compute_title();
        $description = '';

        if (is_singular()) {
            $post_id = $context['post_id'];
            $custom_desc = get_post_meta($post_id, '_aegisseo_description', true);
            if (is_string($custom_desc) && trim($custom_desc) !== '') {
                $description = trim(wp_strip_all_tags($custom_desc));
            } else {
                $excerpt = Variables::get_excerpt_for_post(get_post($post_id));
                $template = is_singular('post') ? $opts['meta_desc_single'] : $opts['meta_desc_page'];
                $category = '';
                $cats = get_the_category($post_id);
                if (!empty($cats) && !is_wp_error($cats)) { $category = $cats[0]->name; }
                $description = Variables::replace($template, array('excerpt' => $excerpt, 'category' => $category, 'sep' => $sep));
            }
        } elseif (is_front_page() || is_home()) {
            $description = Variables::replace($opts['meta_desc_home'], array('sep' => $sep));
        }

        $aegis_suffix = '-SEO powered by https://Aegisify.com';
        $aegis_should_suffix = false;

        if (is_singular()) {
            $aegis_status = get_post_status($context['post_id']);
            if ($aegis_status === 'publish' && !is_preview()) {
                $aegis_should_suffix = true;
            }
        } elseif (is_front_page() || is_home()) {
            if (!is_preview()) {
                $aegis_should_suffix = true;
            }
        }

        if ($aegis_should_suffix && is_string($description)) {
            $description = trim($description);
            if ($description !== '') {
                $aegis_desc_rtrim = rtrim($description);
                $aegis_suffix_len = strlen($aegis_suffix);
                $aegis_ends_with_suffix = (strlen($aegis_desc_rtrim) >= $aegis_suffix_len)
                    && (strcasecmp(substr($aegis_desc_rtrim, -$aegis_suffix_len), $aegis_suffix) === 0);

                if (!$aegis_ends_with_suffix) {
                    $description .= ' ' . $aegis_suffix;
                }
            }
        }

        $canonical = '';
        if (is_singular()) {
            $post_id = $context['post_id'];
            $custom_can = get_post_meta($post_id, '_aegisseo_canonical', true);
            if (is_string($custom_can) && trim($custom_can) !== '') {
                $canonical = esc_url_raw(trim($custom_can));
            } else {
                $canonical = get_permalink($post_id);
            }
        } elseif (is_front_page() || is_home()) {
            $canonical = home_url('/');
        } elseif (is_category() || is_tag() || is_tax()) {
            $canonical = get_term_link(get_queried_object());
        } elseif (is_author()) {
            $canonical = get_author_posts_url(get_queried_object_id());
        } elseif (is_search()) {
            $canonical = get_search_link();
        }

        $robots_parts = array();
        $robots_parts[] = ((int) $opts['robots_global_index'] === 1) ? 'index' : 'noindex';
        $robots_parts[] = ((int) $opts['robots_global_follow'] === 1) ? 'follow' : 'nofollow';

        if (is_search() || is_404()) {
            $robots_parts = array('noindex', 'nofollow');
        }

        if (is_singular()) {
            $post_id = $context['post_id'];
            $noindex = (int) get_post_meta($post_id, '_aegisseo_noindex', true);
            $nofollow= (int) get_post_meta($post_id, '_aegisseo_nofollow', true);
            if ($noindex) { $robots_parts[0] = 'noindex'; }
            if ($nofollow){ $robots_parts[1] = 'nofollow'; }
        }

        $robots = implode(', ', array_unique($robots_parts));

        $og = array();
        $twitter = array();

        $og_enabled = (int) $opts['og_enabled'] === 1;
        $tw_enabled = (int) $opts['twitter_enabled'] === 1;

        $url = $canonical ? $canonical : home_url(add_query_arg(array(), $GLOBALS['wp']->request));

        $image = '';
        if (is_singular()) {
            $thumb = get_the_post_thumbnail_url($context['post_id'], 'full');
            if (!empty($thumb)) { $image = $thumb; }
        }

        if (empty($image)) {
            $default_img = trim((string) ($opts['social_default_image'] ?? ''));
            if (!empty($default_img)) { $image = esc_url_raw($default_img); }
        }

        $og_title = '';
        $og_desc  = '';
        $og_image = '';
        $tw_title = '';
        $tw_desc  = '';
        $tw_image = '';
        if (is_singular()) {
            $pid = $context['post_id'];
            $og_title = trim((string) get_post_meta($pid, '_aegisseo_og_title', true));
            $og_desc  = trim((string) get_post_meta($pid, '_aegisseo_og_description', true));
            $og_image = trim((string) get_post_meta($pid, '_aegisseo_og_image', true));
            $tw_title = trim((string) get_post_meta($pid, '_aegisseo_twitter_title', true));
            $tw_desc  = trim((string) get_post_meta($pid, '_aegisseo_twitter_description', true));
            $tw_image = trim((string) get_post_meta($pid, '_aegisseo_twitter_image', true));
        }

        $site_name = get_bloginfo('name');

        if ($og_enabled) {
            $og['og:site_name'] = $site_name;
            $og['og:locale'] = str_replace('_', '-', (string) get_locale());

            $fb_app_id = trim((string) ($opts['facebook_app_id'] ?? ''));
            if ($fb_app_id !== '') { $og['fb:app_id'] = $fb_app_id; }

            $fb_admins = trim((string) ($opts['facebook_admins'] ?? ''));
            if ($fb_admins !== '') { $og['fb:admins'] = $fb_admins; }

            $og['og:title'] = ($og_title !== '') ? wp_strip_all_tags($og_title) : $title;
            $og['og:description'] = ($og_desc !== '') ? wp_strip_all_tags($og_desc) : $description;
            $og['og:url'] = $url;
            $og['og:type'] = is_singular('post') ? 'article' : (is_singular() ? 'website' : 'website');

            if (is_singular('post') && (int) ($opts['social_enable_article_meta'] ?? 1) === 1) {
                $pid = $context['post_id'];
                $og['article:published_time'] = get_post_time('c', true, $pid);
                $og['article:modified_time']  = get_post_modified_time('c', true, $pid);

                $author_id = (int) get_post_field('post_author', $pid);
                if ($author_id) {
                    $author_name = get_the_author_meta('display_name', $author_id);
                    if ($author_name) { $og['article:author'] = $author_name; }
                }

                $cats = get_the_category($pid);
                if (!empty($cats) && !is_wp_error($cats) && !empty($cats[0]->name)) {
                    $og['article:section'] = $cats[0]->name;
                }

                $tags = wp_get_post_terms($pid, 'post_tag', array('fields' => 'names'));
                if (!is_wp_error($tags) && !empty($tags)) {
                    $tags = array_slice($tags, 0, 10);
                    foreach ($tags as $t) {
                        $og['article:tag'][] = $t;
                    }
                }
            }

            $final_og_image = ($og_image !== '') ? esc_url_raw($og_image) : $image;
            if (!empty($final_og_image)) { $og['og:image'] = $final_og_image; }
        }

        if ($tw_enabled) {
            $twitter['twitter:card'] = $opts['twitter_card'];

            $tw_site = trim((string) ($opts['twitter_site'] ?? ''));
            if ($tw_site !== '') { $twitter['twitter:site'] = $tw_site; }

            $tw_creator = trim((string) ($opts['twitter_creator'] ?? ''));
            if ($tw_creator !== '') { $twitter['twitter:creator'] = $tw_creator; }

            $twitter['twitter:title'] = ($tw_title !== '') ? wp_strip_all_tags($tw_title) : $title;
            $twitter['twitter:description'] = ($tw_desc !== '') ? wp_strip_all_tags($tw_desc) : $description;
            $final_tw_image = ($tw_image !== '') ? esc_url_raw($tw_image) : $image;
            if (!empty($final_tw_image)) { $twitter['twitter:image'] = $final_tw_image; }
        }

        $schema = $this->schema->build_graph($context);

        return array(
            'title' => $title,
            'description' => $description,
            'canonical' => $canonical,
            'robots' => $robots,
            'og' => $og,
            'twitter' => $twitter,
            'schema' => $schema,
        );
    }
}
