<?php
namespace AegisSEO\SEO;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class GSC {
    private $options;

    public function __construct(Options $options) {
        $this->options = $options;
        add_action('admin_init', array($this, 'maybe_handle_oauth_callback'));
    }

    public function maybe_handle_oauth_callback() {
        if (!is_admin() || !current_user_can('manage_options')) return;
        if (!isset($_GET['page'], $_GET['tab']) || $_GET['page'] !== 'aegisseo' || $_GET['tab'] !== 'search') return;
        if (empty($_GET['code'])) return;

        $code = sanitize_text_field((string)$_GET['code']);
        $opts = $this->options->get_all();
        if (empty($opts['gsc_client_id']) || empty($opts['gsc_client_secret'])) return;

        $token = $this->exchange_code_for_tokens($code, $opts['gsc_client_id'], $opts['gsc_client_secret'], admin_url('admin.php?page=aegisseo&tab=search'));
        if (is_wp_error($token)) return;

        if (!empty($token['refresh_token'])) {
            $this->options->update(array('gsc_refresh_token' => $token['refresh_token']));
        }
    }

    private function exchange_code_for_tokens($code, $client_id, $client_secret, $redirect_uri) {
        $resp = wp_remote_post('https://oauth2.googleapis.com/token', array(
            'timeout' => 20,
            'headers' => array('Content-Type'=>'application/x-www-form-urlencoded'),
            'body' => array(
                'code' => $code,
                'client_id' => $client_id,
                'client_secret' => $client_secret,
                'redirect_uri' => $redirect_uri,
                'grant_type' => 'authorization_code',
            ),
        ));
        if (is_wp_error($resp)) return $resp;
        $data = json_decode(wp_remote_retrieve_body($resp), true);
        if (!is_array($data)) return new \WP_Error('aegisseo_gsc', 'Invalid token response');
        if (!empty($data['error'])) return new \WP_Error('aegisseo_gsc', 'Google token error: ' . $data['error']);
        return $data;
    }

    private function refresh_access_token() {
        $opts = $this->options->get_all();
        if (empty($opts['gsc_refresh_token']) || empty($opts['gsc_client_id']) || empty($opts['gsc_client_secret'])) {
            return new \WP_Error('aegisseo_gsc', 'Missing GSC credentials/refresh token.');
        }

        $resp = wp_remote_post('https://oauth2.googleapis.com/token', array(
            'timeout' => 20,
            'headers' => array('Content-Type'=>'application/x-www-form-urlencoded'),
            'body' => array(
                'client_id' => $opts['gsc_client_id'],
                'client_secret' => $opts['gsc_client_secret'],
                'refresh_token' => $opts['gsc_refresh_token'],
                'grant_type' => 'refresh_token',
            ),
        ));
        if (is_wp_error($resp)) return $resp;
        $data = json_decode(wp_remote_retrieve_body($resp), true);
        if (!is_array($data) || empty($data['access_token'])) return new \WP_Error('aegisseo_gsc', 'Failed to refresh access token.');
        return $data['access_token'];
    }

    public function fetch_summary($property) {
        $access = $this->refresh_access_token();
        if (is_wp_error($access)) return $access;

        $property = rtrim((string)$property,'/') . '/';
        $url = 'https://www.googleapis.com/webmasters/v3/sites/' . rawurlencode($property) . 'searchAnalytics/query';

        $end = gmdate('Y-m-d');
        $start = gmdate('Y-m-d', strtotime('-28 days'));
        $body = array('startDate'=>$start,'endDate'=>$end,'dimensions'=>array(),'rowLimit'=>1);

        $resp = wp_remote_post($url, array(
            'timeout' => 20,
            'headers' => array('Authorization'=>'Bearer ' . $access,'Content-Type'=>'application/json'),
            'body' => wp_json_encode($body),
        ));
        if (is_wp_error($resp)) return $resp;
        $data = json_decode(wp_remote_retrieve_body($resp), true);
        if (!is_array($data)) return new \WP_Error('aegisseo_gsc', 'Invalid GSC response.');

        $clicks=0;$impr=0;$ctr=0;$pos=0;
        if (!empty($data['rows'][0])) {
            $row=$data['rows'][0];
            $clicks=isset($row['clicks'])?round($row['clicks'],2):0;
            $impr=isset($row['impressions'])?round($row['impressions'],2):0;
            $ctr=isset($row['ctr'])?round($row['ctr']*100,2):0;
            $pos=isset($row['position'])?round($row['position'],2):0;
        }
        return array('clicks'=>$clicks,'impressions'=>$impr,'ctr'=>$ctr,'position'=>$pos);
    }

public function get_access_token() {
    $opts = $this->options->get_all();
    if (empty($opts['gsc_client_id']) || empty($opts['gsc_client_secret']) || empty($opts['gsc_refresh_token'])) {
        return new \WP_Error('aegisseo_gsc', 'Missing GSC credentials.');
    }
    $token = $this->refresh_access_token($opts['gsc_refresh_token'], $opts['gsc_client_id'], $opts['gsc_client_secret']);
    if (is_wp_error($token)) return $token;
    if (empty($token['access_token'])) return new \WP_Error('aegisseo_gsc', 'Could not obtain access token.');
    return (string)$token['access_token'];
}

public function fetch_page_summary($property, $page_url, $start, $end) {
    $access = $this->get_access_token();
    if (is_wp_error($access)) return $access;

    $property = rtrim((string)$property,'/') . '/';
    $url = 'https://www.googleapis.com/webmasters/v3/sites/' . rawurlencode($property) . 'searchAnalytics/query';

    $page_url = (string) $page_url;
    $start = preg_replace('/[^0-9\-]/','',(string)$start);
    $end   = preg_replace('/[^0-9\-]/','',(string)$end);

    $body = array(
        'startDate'=>$start,
        'endDate'=>$end,
        'dimensions'=>array('page'),
        'dimensionFilterGroups'=>array(
            array(
                'filters'=>array(
                    array(
                        'dimension'=>'page',
                        'operator'=>'equals',
                        'expression'=>$page_url,
                    )
                )
            )
        ),
        'rowLimit'=>1
    );

    $resp = wp_remote_post($url, array(
        'timeout' => 20,
        'headers' => array('Authorization'=>'Bearer ' . $access,'Content-Type'=>'application/json'),
        'body' => wp_json_encode($body),
    ));
    if (is_wp_error($resp)) return $resp;

    $code = (int) wp_remote_retrieve_response_code($resp);
    $raw = (string) wp_remote_retrieve_body($resp);

    if ($code === 429) {
        return new \WP_Error('aegisseo_gsc_rate_limited', 'Rate limited by Google Search Console.');
    }
    if ($code < 200 || $code >= 300) {
        return new \WP_Error('aegisseo_gsc', 'GSC error: HTTP ' . $code);
    }

    $data = json_decode($raw, true);
    if (!is_array($data)) return new \WP_Error('aegisseo_gsc', 'Invalid GSC response.');

    $clicks=0;$impr=0;$ctr=0;$pos=0;
    if (!empty($data['rows'][0])) {
        $row=$data['rows'][0];
        $clicks=isset($row['clicks'])?round((float)$row['clicks'],2):0;
        $impr=isset($row['impressions'])?round((float)$row['impressions'],2):0;
        $ctr=isset($row['ctr'])?round((float)$row['ctr']*100,2):0;
        $pos=isset($row['position'])?round((float)$row['position'],2):0;
    }
    return array('clicks'=>$clicks,'impressions'=>$impr,'ctr'=>$ctr,'position'=>$pos);
}

    public function fetch_page_queries($property, $page_url, $start_date, $end_date, $limit = 10) {
        $token = $this->get_access_token();
        if (!$token) { return new \WP_Error('aegisseo_gsc_no_token', 'No GSC access token.'); }

        $property = (string)$property;
        $page_url = (string)$page_url;
        if ($property === '' || $page_url === '') {
            return new \WP_Error('aegisseo_gsc_bad_args', 'Missing property or page URL.');
        }

        $body = array(
            'startDate' => (string)$start_date,
            'endDate'   => (string)$end_date,
            'dimensions' => array('query'),
            'rowLimit' => (int)$limit,
            'dimensionFilterGroups' => array(
                array(
                    'filters' => array(
                        array(
                            'dimension' => 'page',
                            'operator' => 'equals',
                            'expression' => $page_url,
                        ),
                    ),
                ),
            ),
        );

        $endpoint = 'https://searchconsole.googleapis.com/webmasters/v3/sites/' . rawurlencode($property) . '/searchAnalytics/query';
        $res = wp_remote_post($endpoint, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $token,
                'Content-Type'  => 'application/json',
            ),
            'timeout' => 20,
            'body' => wp_json_encode($body),
        ));

        if (is_wp_error($res)) return $res;
        $code = wp_remote_retrieve_response_code($res);
        $raw  = wp_remote_retrieve_body($res);
        $data = json_decode($raw, true);
        if ($code < 200 || $code >= 300) {
            return new \WP_Error('aegisseo_gsc_http', 'GSC query failed: ' . $code);
        }

        $rows = array();
        foreach ((array)($data['rows'] ?? array()) as $r) {
            $q = isset($r['keys'][0]) ? (string)$r['keys'][0] : '';
            $rows[] = array(
                'query' => $q,
                'clicks' => (float)($r['clicks'] ?? 0),
                'impressions' => (float)($r['impressions'] ?? 0),
                'ctr' => (float)($r['ctr'] ?? 0),
                'position' => (float)($r['position'] ?? 0),
            );
        }
        return $rows;
    }

}
