<?php
namespace AegisSEO\SEO;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Redirects {

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;

        add_action('template_redirect', array($this, 'maybe_redirect'), 0);
    }

    private function is_enabled() {
        return ((int)$this->options->get('redirects_enabled', 1) === 1);
    }

    private function request_path_query() {
        $uri = isset($_SERVER['REQUEST_URI']) ? wp_unslash($_SERVER['REQUEST_URI']) : '';
        $uri = is_string($uri) ? trim($uri) : '';
        if ($uri === '') { return '/'; }

        if ($uri[0] !== '/') { $uri = '/' . $uri; }
        return $uri;
    }

    public function maybe_redirect() {
        if (!$this->is_enabled()) { return; }
        if (is_admin() || is_feed() || (defined('DOING_AJAX') && DOING_AJAX)) { return; }

        global $wpdb;
        $tred = $wpdb->prefix . 'aegisseo_redirects';

        $req = $this->request_path_query();

        $hash = md5('exact:' . $req);
        $row = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$tred} WHERE enabled=1 AND match_type='exact' AND source_hash=%s LIMIT 1",
            $hash
        ), ARRAY_A);

        if ($row) {
            $this->do_redirect($row, $req);
            return;
        }

        $rules = $wpdb->get_results("SELECT * FROM {$tred} WHERE enabled=1 AND match_type='regex' ORDER BY id DESC LIMIT 200", ARRAY_A);
        if (!$rules) { return; }

        foreach ($rules as $r) {
            $pattern = $r['source'];
            if (!$this->is_valid_regex($pattern)) { continue; }
            if (@preg_match($pattern, $req)) {
                $this->do_redirect($r, $req);
                return;
            }
        }
    }

    private function is_valid_regex($pattern) {
        if (!is_string($pattern) || trim($pattern) === '') { return false; }
        $delim = substr($pattern, 0, 1);
        if (!in_array($delim, array('#','~','/'), true)) { return false; }
        $ok = @preg_match($pattern, '/test') !== false;
        return $ok;
    }

    private function do_redirect($row, $req) {
        $status = (int) $row['status_code'];
        if (!in_array($status, array(301,302,307,410,451), true)) { $status = 301; }

        $target = (string) $row['target'];

        if ($row['match_type'] === 'regex' && $this->is_valid_regex($row['source'])) {
            $replaced = @preg_replace($row['source'], $target, $req);
            if (is_string($replaced) && $replaced !== '') {
                $target = $replaced;
            }
        }

        if ($status === 410 || $status === 451) {
            status_header($status);
            if (!empty($target)) {
                header('Location: ' . esc_url_raw($this->normalize_target($target)), true, $status);
            }
            exit;
        }

        if (empty($target)) { return; }
        wp_safe_redirect($this->normalize_target($target), $status);
        exit;
    }

    private function normalize_target($target) {
        $target = trim((string)$target);
        if ($target === '') { return home_url('/'); }

        if (preg_match('#^https?://#i', $target)) {
            return esc_url_raw($target);
        }

        if ($target[0] !== '/') { $target = '/' . $target; }
        return home_url($target);
    }

    public function upsert($data) {
        global $wpdb;
        $tred = $wpdb->prefix . 'aegisseo_redirects';

        $id = isset($data['id']) ? (int)$data['id'] : 0;
        $source = isset($data['source']) ? (string)$data['source'] : '';
        $target = isset($data['target']) ? (string)$data['target'] : '';
        $match_type = isset($data['match_type']) ? (string)$data['match_type'] : 'exact';
        $status = isset($data['status_code']) ? (int)$data['status_code'] : 301;
        $enabled = isset($data['enabled']) ? (int)$data['enabled'] : 1;
        $is_suggestion = isset($data['is_suggestion']) ? (int)$data['is_suggestion'] : 0;
        $suggestion_status = isset($data['suggestion_status']) ? (string)$data['suggestion_status'] : 'approved';
        $source_post_id = isset($data['source_post_id']) ? (int)$data['source_post_id'] : 0;
        $reason = isset($data['reason']) ? (string)$data['reason'] : '';

        $source = trim($source);
        $target = trim($target);

        if ($match_type !== 'regex') { $match_type = 'exact'; }

        if ($match_type === 'exact') {
            if ($source === '' || $source[0] !== '/') { $source = '/' . ltrim($source, '/'); }
        } else {
            if (!$this->is_valid_regex($source)) { return new \WP_Error('invalid_regex', 'Invalid regex pattern.'); }
        }

        if (!in_array($status, array(301,302,307,410,451), true)) { $status = 301; }

        $now = current_time('mysql');
        $hash = md5($match_type . ':' . $source);

        if ($id > 0) {
            $wpdb->update(
                $tred,
                array(
                    'source' => $source,
                    'source_hash' => $hash,
                    'target' => $target,
                    'match_type' => $match_type,
                    'status_code' => $status,
                    'enabled' => $enabled ? 1 : 0,
                    'updated_at' => $now,
                'is_suggestion' => $is_suggestion,
                'suggestion_status' => $suggestion_status,
                'source_post_id' => $source_post_id,
                'reason' => $reason,

                    'is_suggestion' => $is_suggestion,
                    'suggestion_status' => $suggestion_status,
                    'source_post_id' => $source_post_id,
                    'reason' => $reason,

                ),
                array('id' => $id),
                array('%s','%s','%s','%s','%d','%d','%s','%d','%s','%d','%s'),
                array('%d')
            );
            return $id;
        }

        $wpdb->insert(
            $tred,
            array(
                'source' => $source,
                'source_hash' => $hash,
                'target' => $target,
                'match_type' => $match_type,
                'status_code' => $status,
                'enabled' => $enabled ? 1 : 0,
                'created_at' => $now,
                'updated_at' => $now,
                'is_suggestion' => $is_suggestion,
                'suggestion_status' => $suggestion_status,
                'source_post_id' => $source_post_id,
                'reason' => $reason,

                    'is_suggestion' => $is_suggestion,
                    'suggestion_status' => $suggestion_status,
                    'source_post_id' => $source_post_id,
                    'reason' => $reason,

            ),
            array('%s','%s','%s','%s','%d','%d','%s','%s','%d','%s','%d','%s')
        );

        return (int) $wpdb->insert_id;
    }

    public function delete($id) {
        global $wpdb;
        $tred = $wpdb->prefix . 'aegisseo_redirects';
        return $wpdb->delete($tred, array('id' => (int)$id), array('%d'));
    }

    public function get($id) {
        global $wpdb;
        $tred = $wpdb->prefix . 'aegisseo_redirects';
        return $wpdb->get_row($wpdb->prepare("SELECT * FROM {$tred} WHERE id=%d", (int)$id), ARRAY_A);
    }

    public function list($args = array()) {
        global $wpdb;
        $tred = $wpdb->prefix . 'aegisseo_redirects';

        $where = "1=1";
        if (isset($args['enabled']) && $args['enabled'] !== '') {
            $where .= $wpdb->prepare(" AND enabled=%d", (int)$args['enabled']);
        }
        if (!empty($args['match_type'])) {
            $mt = ($args['match_type'] === 'regex') ? 'regex' : 'exact';
            $where .= $wpdb->prepare(" AND match_type=%s", $mt);
        }
        $limit = isset($args['limit']) ? max(1, min(500, (int)$args['limit'])) : 50;
        $offset = isset($args['offset']) ? max(0, (int)$args['offset']) : 0;

        return $wpdb->get_results($wpdb->prepare("SELECT * FROM {$tred} WHERE {$where} ORDER BY id DESC LIMIT %d OFFSET %d", $limit, $offset), ARRAY_A);
    }
}
