<?php
namespace AegisSEO\SEO;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Schema {

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;
        add_filter('aegisseo_schema_graph', array($this, 'filter_graph'), 10, 2);
    }

    public function resolve_schema_type($post_id) {
    $post_id = (int) $post_id;
    $opts = $this->options->get_all();

    $schema_type = (string) get_post_meta($post_id, '_aegisseo_schema_type', true);
    if ($schema_type !== '') { return $schema_type; }

    $pt = get_post_type($post_id);
    $map = $opts['schema_post_type_map'] ?? array();
    if ($pt && is_array($map) && !empty($map[$pt])) {
        return (string) $map[$pt];
    }

    if ($pt === 'post') { return 'Article'; }
    return 'WebPage';
}

public function get_schema_templates() {
    return array(
        'WebPage' => __('WebPage', 'aegisseo'),
        'Article' => __('Article', 'aegisseo'),
        'BlogPosting' => __('BlogPosting', 'aegisseo'),
        'FAQPage' => __('FAQPage', 'aegisseo'),
        'HowTo' => __('HowTo', 'aegisseo'),
        'Product' => __('Product', 'aegisseo'),
        'LocalBusiness' => __('LocalBusiness', 'aegisseo'),
        'Organization' => __('Organization', 'aegisseo'),
        'Person' => __('Person', 'aegisseo'),
        'Event' => __('Event', 'aegisseo'),
    );
}

public function validate_schema($post_id, $schema_type = null) {
    $post_id = (int) $post_id;
    $type = $schema_type ? (string)$schema_type : $this->resolve_schema_type($post_id);
    $data = get_post_meta($post_id, '_aegisseo_schema_data', true);
    $data = is_string($data) ? json_decode($data, true) : (is_array($data) ? $data : array());
    if (!is_array($data)) { $data = array(); }

    $issues = array();

    if ($type === 'Product') {
        if (empty($data['price'])) $issues[] = __('Product: missing price', 'aegisseo');
        if (empty($data['currency'])) $issues[] = __('Product: missing currency', 'aegisseo');
    } elseif ($type === 'LocalBusiness') {
        if (empty($data['telephone']) && empty($data['phone'])) $issues[] = __('LocalBusiness: missing phone', 'aegisseo');
        if (empty($data['address'])) $issues[] = __('LocalBusiness: missing address', 'aegisseo');
    } elseif ($type === 'Event') {
        if (empty($data['startDate'])) $issues[] = __('Event: missing start date', 'aegisseo');
        if (empty($data['location_name']) && empty($data['location'])) $issues[] = __('Event: missing location', 'aegisseo');
    } elseif ($type === 'FAQPage') {
        $faqs = $this->extract_faq_pairs($post_id);
        if (empty($faqs)) $issues[] = __('FAQPage: no Q/A pairs detected (use "Q: ... A: ...")', 'aegisseo');
    } elseif ($type === 'HowTo') {
        $steps = $this->extract_howto_steps($post_id);
        if (count($steps) < 2) $issues[] = __('HowTo: not enough steps detected (use numbered steps)', 'aegisseo');
    }

    return $issues;
}

public function filter_graph($graph, $context) {
        return $graph;
    }

    public function build_graph($context = array()) {
        $opts = $this->options->get_all();

        $home = home_url('/');
        $site_name = get_bloginfo('name');
        $logo_id = (int) $opts['logo_id'];
        $logo_url = $logo_id ? wp_get_attachment_image_url($logo_id, 'full') : '';

        $publisher = array();
        if ($opts['site_type'] === 'person') {
            $publisher = array(
                '@type' => 'Person',
                'name'  => !empty($opts['person_name']) ? $opts['person_name'] : $site_name,
                'url'   => $home,
            );
        } else {
            $publisher = array(
                '@type' => 'Organization',
                'name'  => !empty($opts['org_name']) ? $opts['org_name'] : $site_name,
                'url'   => $home,
            );
            if (!empty($logo_url)) {
                $publisher['logo'] = array(
                    '@type' => 'ImageObject',
                    'url'   => esc_url($logo_url),
                );
            }
        }

        $same_as = array();
        $profiles = $opts['social_profiles'] ?? array();
        if (is_array($profiles)) {
            foreach ($profiles as $u) {
                $u = trim((string) $u);
                if ($u !== '' && filter_var($u, FILTER_VALIDATE_URL)) {
                    $same_as[] = esc_url_raw($u);
                }
            }
        }
        if (!empty($same_as)) {
            $publisher['sameAs'] = array_values(array_unique($same_as));
        }

$graph = array();

        $graph[] = array(
            '@type' => 'WebSite',
            '@id'   => trailingslashit($home) . '#website',
            'url'   => $home,
            'name'  => $site_name,
            'publisher' => $publisher,
            'potentialAction' => array(
                '@type' => 'SearchAction',
                'target' => array(
                    '@type' => 'EntryPoint',
                    'urlTemplate' => $home . '?s={search_term_string}',
                ),
                'query-input' => 'required name=search_term_string',
            ),
        );

        if (!empty($context['is_singular']) && !empty($context['post_id'])) {
            $post_id = (int) $context['post_id'];
            $url = get_permalink($post_id);

                        $type = $this->resolve_schema_type($post_id);

$node = array(
                '@type' => $type,
                '@id'   => trailingslashit($url) . '#primary',
                'url'   => $url,
                'name'  => get_the_title($post_id),
                'isPartOf' => array('@id' => trailingslashit($home) . '#website'),
                'publisher' => $publisher,
            );

            $date_published = get_the_date('c', $post_id);
            $date_modified  = get_the_modified_date('c', $post_id);

            if ($type === 'Article') {
                $node['datePublished'] = $date_published;
                $node['dateModified']  = $date_modified;
                $thumb = get_the_post_thumbnail_url($post_id, 'full');
                if (!empty($thumb)) {
                    $node['image'] = array(
                        '@type' => 'ImageObject',
                        'url'   => esc_url($thumb),
                    );
                }
            }

if ($type === 'FAQPage') {
    $faqs = $this->extract_faq_pairs($post_id);
    if (!empty($faqs)) {
        $node['mainEntity'] = array();
        foreach ($faqs as $qa) {
            $node['mainEntity'][] = array(
                '@type' => 'Question',
                'name' => $qa['q'],
                'acceptedAnswer' => array(
                    '@type' => 'Answer',
                    'text' => $qa['a'],
                ),
            );
        }
    }
} elseif ($type === 'HowTo') {
    $steps = $this->extract_howto_steps($post_id);
    if (!empty($steps)) {
        $node['step'] = array();
        foreach ($steps as $st) {
            $node['step'][] = array(
                '@type' => 'HowToStep',
                'name' => $st,
                'text' => $st,
            );
        }
    }
}

            if ($type === 'BlogPosting') {
                $node['datePublished'] = $date_published;
                $node['dateModified']  = $date_modified;
                $node['headline'] = get_the_title($post_id);
            } elseif ($type === 'Product') {
                $data = get_post_meta($post_id, '_aegisseo_schema_data', true);
                $data = is_string($data) ? json_decode($data, true) : (is_array($data) ? $data : array());
                if (!is_array($data)) { $data = array(); }

                $node['name'] = !empty($data['name']) ? (string)$data['name'] : get_the_title($post_id);
                if (!empty($data['brand'])) { $node['brand'] = array('@type'=>'Brand','name'=>(string)$data['brand']); }
                if (!empty($data['sku'])) { $node['sku'] = (string)$data['sku']; }

                $imgs = array();
                $thumb = get_the_post_thumbnail_url($post_id, 'full');
                if (!empty($thumb)) { $imgs[] = esc_url($thumb); }
                if (!empty($data['image'])) { $imgs[] = esc_url((string)$data['image']); }
                if (!empty($imgs)) { $node['image'] = array_values(array_unique($imgs)); }

                $price = isset($data['price']) ? (string)$data['price'] : '';
                $currency = isset($data['currency']) ? (string)$data['currency'] : '';
                if ($price !== '' && $currency !== '') {
                    $offer = array(
                        '@type' => 'Offer',
                        'price' => $price,
                        'priceCurrency' => $currency,
                        'url' => $url,
                    );
                    if (!empty($data['availability'])) {
                        $offer['availability'] = 'https://schema.org/' . preg_replace('/[^A-Za-z]/', '', (string)$data['availability']);
                    }
                    $node['offers'] = $offer;
                }
            } elseif ($type === 'LocalBusiness') {
                $data = get_post_meta($post_id, '_aegisseo_schema_data', true);
                $data = is_string($data) ? json_decode($data, true) : (is_array($data) ? $data : array());
                if (!is_array($data)) { $data = array(); }

                $node['name'] = !empty($data['name']) ? (string)$data['name'] : $site_name;
                if (!empty($data['telephone'])) { $node['telephone'] = (string)$data['telephone']; }
                if (!empty($data['address'])) {
                    $node['address'] = array(
                        '@type' => 'PostalAddress',
                        'streetAddress' => (string)$data['address'],
                    );
                }
                if (!empty($data['geo_lat']) && !empty($data['geo_lng'])) {
                    $node['geo'] = array('@type'=>'GeoCoordinates','latitude'=>(string)$data['geo_lat'],'longitude'=>(string)$data['geo_lng']);
                }
            } elseif ($type === 'Event') {
                $data = get_post_meta($post_id, '_aegisseo_schema_data', true);
                $data = is_string($data) ? json_decode($data, true) : (is_array($data) ? $data : array());
                if (!is_array($data)) { $data = array(); }

                if (!empty($data['startDate'])) { $node['startDate'] = (string)$data['startDate']; }
                if (!empty($data['endDate'])) { $node['endDate'] = (string)$data['endDate']; }

                $loc_name = !empty($data['location_name']) ? (string)$data['location_name'] : '';
                $loc_addr = !empty($data['location_address']) ? (string)$data['location_address'] : '';
                if ($loc_name || $loc_addr) {
                    $node['location'] = array('@type'=>'Place','name'=>$loc_name ? $loc_name : $loc_addr);
                    if ($loc_addr) {
                        $node['location']['address'] = array('@type'=>'PostalAddress','streetAddress'=>$loc_addr);
                    }
                }
            }

$graph[] = $node;
        }

        $graph = apply_filters('aegisseo_schema_graph', $graph, $context);

        return array(
            '@context' => 'https://schema.org',
            '@graph'   => $graph,
        );
    }

    private function extract_faq_pairs($post_id) {
        $post = get_post((int) $post_id);
        if (!$post) { return array(); }
        $content = (string) $post->post_content;

        $pairs = array();
        if (preg_match_all('/\bQ\s*[:\-]\s*(.+?)\s*\bA\s*[:\-]\s*(.+?)(?=(\n\s*Q\s*[:\-])|$)/is', $content, $m, PREG_SET_ORDER)) {
            foreach ($m as $row) {
                $q = trim(wp_strip_all_tags($row[1]));
                $a = trim(wp_strip_all_tags($row[2]));
                if ($q !== '' && $a !== '') {
                    $pairs[] = array('q' => $q, 'a' => $a);
                }
            }
        }

        if (count($pairs) > 10) {
            $pairs = array_slice($pairs, 0, 10);
        }

        return $pairs;
    }

    private function extract_howto_steps($post_id) {
        $post = get_post((int) $post_id);
        if (!$post) { return array(); }
        $content = (string) $post->post_content;

        $steps = array();

        if (preg_match_all('/\bStep\s+\d+\b\s*[:\-]?\s*(.+?)(?=(\n\s*Step\s+\d+\b)|$)/is', $content, $m, PREG_SET_ORDER)) {
            foreach ($m as $row) {
                $s = trim(wp_strip_all_tags($row[1]));
                if ($s !== '') { $steps[] = $s; }
            }
        }

        if (empty($steps) && preg_match_all('/<ol[^>]*>(.*?)<\/ol>/is', $content, $ols)) {
            foreach ($ols[1] as $ol_html) {
                if (preg_match_all('/<li[^>]*>(.*?)<\/li>/is', $ol_html, $lis)) {
                    foreach ($lis[1] as $li_html) {
                        $s = trim(wp_strip_all_tags($li_html));
                        if ($s !== '') { $steps[] = $s; }
                    }
                }
                if (!empty($steps)) { break; }
            }
        }

        if (count($steps) > 15) {
            $steps = array_slice($steps, 0, 15);
        }

        return $steps;
    }


}
