<?php
namespace AegisSEO\SEO;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Sitemap {

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;

        add_action('init', array($this, 'register_rewrite'));
        add_action('template_redirect', array($this, 'maybe_output_sitemap'));
        add_filter('query_vars', array($this, 'add_query_vars'));
        add_action('transition_post_status', array($this, 'maybe_ping_search_engines'), 10, 3);

        // Sitemap cache invalidation (strictly sitemap-related)
        add_action('save_post', array($this, 'bump_sitemap_cache_version_on_save'), 10, 3);
        add_action('deleted_post', array($this, 'bump_sitemap_cache_version_simple'), 10, 1);
        add_action('created_term', array($this, 'bump_sitemap_cache_version_simple'), 10, 3);
        add_action('edited_term', array($this, 'bump_sitemap_cache_version_simple'), 10, 3);
        add_action('delete_term', array($this, 'bump_sitemap_cache_version_simple'), 10, 4);
        add_action('updated_option', array($this, 'maybe_bump_sitemap_cache_on_settings_change'), 10, 3);
    }

    public function add_query_vars($vars) {
        $vars[] = 'aegisseo_sitemap';       // index|legacy|child
        $vars[] = 'aegisseo_sitemap_kind';  // pt|tax
        $vars[] = 'aegisseo_sitemap_name';  // post_type|taxonomy
        $vars[] = 'aegisseo_sitemap_page';  // int
        $vars[] = 'aegisseo_html_sitemap';   // 1
        return $vars;
    }

    public function register_rewrite() {
        if (!(int) $this->options->get('sitemap_enabled', 1)) { return; }

        add_rewrite_rule('^sitemap_index\.xml$', 'index.php?aegisseo_sitemap=index', 'top');

        add_rewrite_rule('^sitemap-pt-([^/]+)\.xml$', 'index.php?aegisseo_sitemap=child&aegisseo_sitemap_kind=pt&aegisseo_sitemap_name=$matches[1]', 'top');
        add_rewrite_rule('^sitemap-pt-([^/]+)-([0-9]+)\.xml$', 'index.php?aegisseo_sitemap=child&aegisseo_sitemap_kind=pt&aegisseo_sitemap_name=$matches[1]&aegisseo_sitemap_page=$matches[2]', 'top');

        add_rewrite_rule('^sitemap-tax-([^/]+)\.xml$', 'index.php?aegisseo_sitemap=child&aegisseo_sitemap_kind=tax&aegisseo_sitemap_name=$matches[1]', 'top');
        add_rewrite_rule('^sitemap-tax-([^/]+)-([0-9]+)\.xml$', 'index.php?aegisseo_sitemap=child&aegisseo_sitemap_kind=tax&aegisseo_sitemap_name=$matches[1]&aegisseo_sitemap_page=$matches[2]', 'top');

        if ((int) $this->options->get('sitemap_video_enabled', 1) === 1) {
            add_rewrite_rule('^video-sitemap\.xml$', 'index.php?aegisseo_sitemap=video', 'top');
        }
        if ((int) $this->options->get('sitemap_news_enabled', 0) === 1) {
            add_rewrite_rule('^news-sitemap\.xml$', 'index.php?aegisseo_sitemap=news', 'top');
        }

        if ((int) $this->options->get('sitemap_html_enabled', 1) === 1) {
            add_rewrite_rule('^sitemap/?$', 'index.php?aegisseo_html_sitemap=1', 'top');
        }

        add_rewrite_rule('^sitemap\.xml$', 'index.php?aegisseo_sitemap=legacy', 'top');
    }

    public function maybe_output_sitemap() {
        if (!(int) $this->options->get('sitemap_enabled', 1)) { return; }

        if ((int) $this->options->get('sitemap_html_enabled', 1) === 1 && (string) get_query_var('aegisseo_html_sitemap') === '1') {
            $this->output_html_sitemap();
            exit;
        }

        $mode = (string) get_query_var('aegisseo_sitemap');
        if ($mode === '') { return; }

        $cache_v = (int) $this->options->get('sitemap_cache_v', 1);
        if ($cache_v <= 0) { $cache_v = 1; }

        $cache_key = 'aegisseo_sm_' . $cache_v . '_' . md5(
            $mode . '|' .
            (string) get_query_var('aegisseo_sitemap_kind') . '|' .
            (string) get_query_var('aegisseo_sitemap_name') . '|' .
            (string) get_query_var('aegisseo_sitemap_page') . '|' .
            (int) $this->options->get('sitemap_page_size', 2000)
        );

        $cached = get_transient($cache_key);
        if (is_string($cached) && $cached !== '') {
            nocache_headers();
            header('Content-Type: application/xml; charset=UTF-8');
            echo $cached;
            exit;
        }

        $xml = '';
        if ($mode === 'index') {
            $xml = $this->build_index();
        } elseif ($mode === 'video') {
            $xml = $this->build_video_sitemap();
        } elseif ($mode === 'news') {
            $xml = $this->build_news_sitemap();
        } elseif ($mode === 'child') {
            $kind = (string) get_query_var('aegisseo_sitemap_kind');
            $name = (string) get_query_var('aegisseo_sitemap_name');
            $page = (int) get_query_var('aegisseo_sitemap_page');
            if ($page <= 0) {
                $page = isset($_GET['paged']) ? max(1, (int) $_GET['paged']) : 1;
            }
            $xml = $this->build_child($kind, $name, $page);
        } else {

            if ((int) $this->options->get('sitemap_index_enabled', 1) === 1) {
                $xml = $this->build_index();
            } else {
                $xml = $this->build_sitemap();
            }
        }

        if ($xml === '') { return; }

        // Cache write: 6 hours (invalidated by cache version bump hooks)
        set_transient($cache_key, $xml, 6 * HOUR_IN_SECONDS);

        nocache_headers();
        header('Content-Type: application/xml; charset=UTF-8');
        echo $xml;
        exit;
    }

    private function xml_escape($s) {
        return esc_html((string) $s);
    }
	
	private function powered_by_xml_comment() {
		return "\n  SEO SiteMap powered by Aegisify.com\n";
	}

	/**
	 * Clean up third-party branding in UI labels (do not change slugs).
	 */
	private static function clean_ui_label($label, $slug, $kind = '') {
		$label = (string) $label;

		// Hard overrides for the Avada/slider slugs shown in your screenshot.
		if ($kind === 'post_type' && $slug === 'slide') {
			return __('Slides', 'aegisseo');
		}
		if ($kind === 'taxonomy' && $slug === 'slide-page') {
			return __('Sliders', 'aegisseo');
		}

		// Strip a leading "Avada " if present.
		$label = preg_replace('/^\s*Avada\s+/i', '', $label);

		return $label;
	}

	/**
	 * Return plugin mappings (post types / taxonomies) based on saved sitemap_plugins.
	 * WooCommerce is supported by default. Others can be added via filter.
	 */
	private function get_plugin_mappings($opts) {
		$plugins = array();

		if (!empty($opts['sitemap_plugins']) && is_array($opts['sitemap_plugins'])) {
			foreach ($opts['sitemap_plugins'] as $p) {
				$p = is_string($p) ? trim($p) : '';
				if ($p !== '') $plugins[] = $p;
			}
		}

		$plugins = array_values(array_unique($plugins));

		// Default mappings (truthful + stable): WooCommerce
		$mappings = array(
			// plugin file => mapping
			'woocommerce/woocommerce.php' => array(
				'post_types' => array('product'),
				'taxonomies' => array('product_cat', 'product_tag'),
			),
		);

		/**
		 * Allow other plugins to register sitemap mappings.
		 * Return format:
		 * [
		 *   'plugin-folder/plugin-file.php' => ['post_types'=>[], 'taxonomies'=>[]],
		 * ]
		 */
		$mappings = apply_filters('aegisseo_sitemap_plugin_mappings', $mappings);

		$out = array('post_types' => array(), 'taxonomies' => array());
		foreach ($plugins as $plugin_file) {
			if (isset($mappings[$plugin_file])) {
				$map = $mappings[$plugin_file];
				if (!empty($map['post_types']) && is_array($map['post_types'])) {
					$out['post_types'] = array_merge($out['post_types'], $map['post_types']);
				}
				if (!empty($map['taxonomies']) && is_array($map['taxonomies'])) {
					$out['taxonomies'] = array_merge($out['taxonomies'], $map['taxonomies']);
				}
			}
		}

		$out['post_types'] = array_values(array_unique(array_filter($out['post_types'])));
		$out['taxonomies'] = array_values(array_unique(array_filter($out['taxonomies'])));

		return $out;
	}

	private function is_post_type_enabled($pt, $opts) {
		if (!empty($opts['sitemap_post_types'][$pt]) && (int) $opts['sitemap_post_types'][$pt] === 1) {
			return true;
		}
		$maps = $this->get_plugin_mappings($opts);
		return in_array($pt, $maps['post_types'], true);
	}

	private function is_taxonomy_enabled($tax, $opts) {
		if (!empty($opts['sitemap_taxonomies'][$tax]) && (int) $opts['sitemap_taxonomies'][$tax] === 1) {
			return true;
		}
		$maps = $this->get_plugin_mappings($opts);
		return in_array($tax, $maps['taxonomies'], true);
	}

	private function get_folder_exclusions($opts) {
		$raw = $opts['sitemap_folder_exclusions'] ?? '';
		if (!is_string($raw)) return array();

		$lines = preg_split('/\R+/', $raw);
		$out = array();

		foreach ($lines as $l) {
			$l = trim($l);
			if ($l === '') continue;

			// Normalize: ensure it starts with a slash and has no domain.
			// Examples allowed: /wp-content/uploads/private, uploads/private, /private/
			$l = preg_replace('#^https?://[^/]+#i', '', $l);
			if ($l !== '' && $l[0] !== '/') $l = '/' . $l;

			$out[] = rtrim($l, '/');
		}

		return array_values(array_unique($out));
	}

	private function is_loc_excluded($loc, $opts) {
		$ex = $this->get_folder_exclusions($opts);
		if (empty($ex)) return false;

		$parts = wp_parse_url($loc);
		$path = isset($parts['path']) ? (string) $parts['path'] : '';
		if ($path === '') return false;

		foreach ($ex as $frag) {
			if ($frag === '') continue;
			if (strpos($path, $frag) !== false) {
				return true;
			}
		}
		return false;
	}

    private function build_index() {
        $opts = $this->options->get_all();
        $page_size = (int) ($opts['sitemap_page_size'] ?? 2000);
        if ($page_size < 500) $page_size = 500;
        if ($page_size > 5000) $page_size = 5000;

	$post_types = array();
	if (!empty($opts['sitemap_post_types']) && is_array($opts['sitemap_post_types'])) {
		foreach ($opts['sitemap_post_types'] as $pt => $enabled) {
			if ((int) $enabled === 1) $post_types[] = $pt;
		}
	}

	$taxes = array();
	if (!empty($opts['sitemap_taxonomies']) && is_array($opts['sitemap_taxonomies'])) {
		foreach ($opts['sitemap_taxonomies'] as $tax => $enabled) {
			if ((int) $enabled === 1) $taxes[] = $tax;
		}
	}

	// Merge plugin-mapped post types/taxonomies (e.g. WooCommerce)
	$maps = $this->get_plugin_mappings($opts);
	if (!empty($maps['post_types'])) $post_types = array_merge($post_types, $maps['post_types']);
	if (!empty($maps['taxonomies'])) $taxes = array_merge($taxes, $maps['taxonomies']);

	$post_types = array_values(array_unique($post_types));
	$taxes      = array_values(array_unique($taxes));

        $items = array();

        foreach ($post_types as $pt) {
            $count_obj = wp_count_posts($pt);
            $count = (int) ($count_obj && isset($count_obj->publish) ? $count_obj->publish : 0);
            $pages = max(1, (int) ceil($count / $page_size));
            for ($p = 1; $p <= $pages; $p++) {
                $loc = home_url('/sitemap-pt-' . $pt . ($p > 1 ? '-' . $p : '') . '.xml');
                $items[] = array('loc' => $loc, 'lastmod' => gmdate('c'));
            }
        }

        foreach ($taxes as $tax) {
            $term_count = (int) wp_count_terms($tax, array('hide_empty' => false));
            $pages = max(1, (int) ceil($term_count / $page_size));
            for ($p = 1; $p <= $pages; $p++) {
                $loc = home_url('/sitemap-tax-' . $tax . ($p > 1 ? '-' . $p : '') . '.xml');
                $items[] = array('loc' => $loc, 'lastmod' => gmdate('c'));
            }
        }

        if ((int) ($opts['sitemap_video_enabled'] ?? 1) === 1) {
            $items[] = array('loc' => home_url('/video-sitemap.xml'), 'lastmod' => gmdate('c'));
        }
        if ((int) ($opts['sitemap_news_enabled'] ?? 0) === 1) {
            $items[] = array('loc' => home_url('/news-sitemap.xml'), 'lastmod' => gmdate('c'));
        }

        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
        foreach ($items as $it) {
            $xml .= "  <sitemap>\n";
            $xml .= '    <loc>' . $this->xml_escape($it['loc']) . "</loc>\n";
            if (!empty($it['lastmod'])) {
                $xml .= '    <lastmod>' . $this->xml_escape($it['lastmod']) . "</lastmod>\n";
            }
            $xml .= "  </sitemap>\n";
        }
		$xml .= $this->powered_by_xml_comment();
        $xml .= "</sitemapindex>\n";

        return $xml;
    }

    private function build_child($kind, $name, $page) {
        $opts = $this->options->get_all();
        $page_size = (int) ($opts['sitemap_page_size'] ?? 2000);
        if ($page_size < 500) $page_size = 500;
        if ($page_size > 5000) $page_size = 5000;

        $include_images = (int) ($opts['sitemap_include_images'] ?? 1) === 1;
        $include_attached = (int) ($opts['sitemap_include_attached_images'] ?? 0) === 1;

        $urls = array();

		if ($kind === 'pt') {
			if (!$this->is_post_type_enabled($name, $opts)) {
				return '';
			}

            $q = new \WP_Query(array(
                'post_type'      => $name,
                'post_status'    => 'publish',
                'posts_per_page' => $page_size,
                'paged'          => max(1, (int) $page),
                'orderby'        => 'modified',
                'order'          => 'DESC',
                'no_found_rows'  => false,
            ));

            if ($q->have_posts()) {
                while ($q->have_posts()) {
                    $q->the_post();
                    $id = get_the_ID();

                    $row = array(
                        'loc' => get_permalink($id),
                        'lastmod' => get_post_modified_time('c', true, $id),
                        'images' => array(),
                    );

                    if ($include_images) {
                        $thumb_id = get_post_thumbnail_id($id);
                        if ($thumb_id) {
                            $u = wp_get_attachment_url($thumb_id);
                            if ($u) $row['images'][] = (string) $u;
                        }
                        if ($include_attached) {
                            $atts = get_attached_media('image', $id);
                            if (!empty($atts) && is_array($atts)) {
                                foreach ($atts as $att) {
                                    if (!($att instanceof \WP_Post)) continue;
                                    $u = wp_get_attachment_url($att->ID);
                                    if ($u) $row['images'][] = (string) $u;
                                    if (count($row['images']) >= 10) break;
                                }
                            }
                        }
                        if (!empty($row['images'])) {
                            $row['images'] = array_values(array_unique($row['images']));
                        }
                    }

                    if (!$this->is_loc_excluded($row['loc'], $opts)) {
						$urls[] = $row;
					}
                }
                wp_reset_postdata();
            }
        } elseif ($kind === 'tax') {
			if (!$this->is_taxonomy_enabled($name, $opts)) {
				return '';
			}

            $offset = (max(1, (int) $page) - 1) * $page_size;
            $terms = get_terms(array(
                'taxonomy'   => $name,
                'hide_empty' => false,
                'number'     => $page_size,
                'offset'     => $offset,
            ));
            if (!is_wp_error($terms) && !empty($terms)) {
                foreach ($terms as $t) {
                    $link = get_term_link($t);
                    if (is_wp_error($link)) continue;
                    $urls[] = array(
                        'loc' => $link,
                        'lastmod' => gmdate('c'),
                        'images' => array(),
                    );
                }
            }
        } else {
            return '';
        }

        if ((int) ($opts['sitemap_video_enabled'] ?? 1) === 1) {
            $items[] = array('loc' => home_url('/video-sitemap.xml'), 'lastmod' => gmdate('c'));
        }
        if ((int) ($opts['sitemap_news_enabled'] ?? 0) === 1) {
            $items[] = array('loc' => home_url('/news-sitemap.xml'), 'lastmod' => gmdate('c'));
        }

        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        if ($include_images) {
            $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">' . "\n";
        } else {
            $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
        }

        foreach ($urls as $u) {
            $xml .= "  <url>\n";
            $xml .= '    <loc>' . $this->xml_escape($u['loc']) . "</loc>\n";
            if (!empty($u['lastmod'])) {
                $xml .= '    <lastmod>' . $this->xml_escape($u['lastmod']) . "</lastmod>\n";
            }
            if ($include_images && !empty($u['images'])) {
                foreach ($u['images'] as $img) {
                    $xml .= "    <image:image>\n";
                    $xml .= '      <image:loc>' . $this->xml_escape($img) . "</image:loc>\n";
                    $xml .= "    </image:image>\n";
                }
            }
            $xml .= "  </url>\n";
        }
		$xml .= $this->powered_by_xml_comment();
        $xml .= "</urlset>\n";
        return $xml;
    }

    private function build_sitemap() {
        $opts = $this->options->get_all();

        $urls = array();

        $urls[] = array(
            'loc' => home_url('/'),
            'lastmod' => gmdate('c'),
        );

        $post_types = array();
        if (is_array($opts['sitemap_post_types'])) {
            foreach ($opts['sitemap_post_types'] as $pt => $enabled) {
                if ((int) $enabled === 1) { $post_types[] = $pt; }
            }
        }
        if (!empty($post_types)) {
            $q = new \WP_Query(array(
                'post_type'      => $post_types,
                'post_status'    => 'publish',
                'posts_per_page' => 5000,
                'no_found_rows'  => true,
                'fields'         => 'ids',
                'orderby'        => 'modified',
                'order'          => 'DESC',
            ));
            if (!empty($q->posts)) {
                foreach ($q->posts as $post_id) {
                    $noindex = (int) get_post_meta($post_id, '_aegisseo_noindex', true);
                    if ($noindex) { continue; }

                    $urls[] = array(
                        'loc' => get_permalink($post_id),
                        'lastmod' => get_post_modified_time('c', true, $post_id),
                    );
                }
            }
            wp_reset_postdata();
        }

        $taxes = array();
        if (is_array($opts['sitemap_taxonomies'])) {
            foreach ($opts['sitemap_taxonomies'] as $tax => $enabled) {
                if ((int) $enabled === 1) { $taxes[] = $tax; }
            }
        }
        if (!empty($taxes)) {
            foreach ($taxes as $tax) {
                $terms = get_terms(array(
                    'taxonomy' => $tax,
                    'hide_empty' => true,
                    'number' => 5000,
                ));
                if (is_wp_error($terms)) { continue; }
                foreach ($terms as $term) {
                    $link = get_term_link($term);
                    if (is_wp_error($link)) { continue; }
                    if (!$this->is_loc_excluded($link, $opts)) {
						$urls[] = array(
							'loc' => $link,
							'lastmod' => gmdate('c')
						);
					}
                }
            }
        }

        $urls = apply_filters('aegisseo_sitemap_urls', $urls);

        $xml  = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
        $xml .= "<urlset xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\">\n";
        foreach ($urls as $u) {
            if (empty($u['loc'])) { continue; }
            $xml .= "  <url>\n";
            $xml .= "    <loc>" . esc_url($u['loc']) . "</loc>\n";
            if (!empty($u['lastmod'])) {
                $xml .= "    <lastmod>" . esc_html($u['lastmod']) . "</lastmod>\n";
            }
            $xml .= "  </url>\n";
        }
		$xml .= $this->powered_by_xml_comment();
        $xml .= "</urlset>\n";
        return $xml;
    }
	
	private function output_html_sitemap() {
		if ((int) $this->options->get('sitemap_html_enabled', 1) !== 1) { return; }

		nocache_headers();
		header('Content-Type: text/html; charset=UTF-8');

		$pt_enabled  = (array) $this->options->get('sitemap_post_types', array());
		$tax_enabled = (array) $this->options->get('sitemap_taxonomies', array());

		echo '<!doctype html><html><head><meta charset="utf-8">';
		echo '<meta name="robots" content="index,follow">';
		echo '<title>' . esc_html(get_bloginfo('name') . ' - Sitemap') . '</title>';
		echo '<style>
			body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;margin:24px;line-height:1.5}
			h1{margin:0 0 12px} h2{margin:18px 0 8px}
			ul{margin:0 0 16px 18px}
			.note{color:#646970;font-size:13px;margin:0 0 16px}
			.powered{margin-top:22px;color:#646970;font-size:12px}
		</style>';
		echo '</head><body>';

		echo '<h1>' . esc_html__('HTML Sitemap', 'aegisseo') . '</h1>';
		echo '<p class="note">' . esc_html__('Generated by AegisSEO for humans and crawlers.', 'aegisseo') . '</p>';

		// Posts by post type
		foreach ($pt_enabled as $pt => $on) {
			if ((int) $on !== 1) continue;
			$obj = get_post_type_object($pt);
			if (!$obj || empty($obj->public)) continue;

			echo '<h2>' . esc_html($obj->labels->name) . '</h2>';

			$q = new \WP_Query(array(
				'post_type'      => $pt,
				'post_status'    => 'publish',
				'posts_per_page' => 200,
				'orderby'        => 'modified',
				'order'          => 'DESC',
				'no_found_rows'  => true,
			));

			if ($q->have_posts()) {
				echo '<ul>';
				while ($q->have_posts()) {
					$q->the_post();
					echo '<li><a href="' . esc_url(get_permalink()) . '">' . esc_html(get_the_title()) . '</a></li>';
				}
				echo '</ul>';
				wp_reset_postdata();
			} else {
				echo '<p class="note">' . esc_html__('No published items found.', 'aegisseo') . '</p>';
			}
		}

		// Terms by taxonomy
		foreach ($tax_enabled as $tax => $on) {
			if ((int) $on !== 1) continue;
			$tx = get_taxonomy($tax);
			if (!$tx || empty($tx->public)) continue;

			echo '<h2>' . esc_html($tx->labels->name) . '</h2>';

			$terms = get_terms(array(
				'taxonomy'   => $tax,
				'hide_empty' => true,
				'number'     => 200,
			));

			if (!is_wp_error($terms) && !empty($terms)) {
				echo '<ul>';
				foreach ($terms as $t) {
					$link = get_term_link($t);
					if (is_wp_error($link)) continue;
					echo '<li><a href="' . esc_url($link) . '">' . esc_html($t->name) . '</a></li>';
				}
				echo '</ul>';
			} else {
				echo '<p class="note">' . esc_html__('No terms found.', 'aegisseo') . '</p>';
			}
		}

		echo '<p class="powered">FREE SEO SiteMap powered by Aegisify.com</p>';
		echo '</body></html>';
	}

	private function build_video_sitemap() {
		if ((int) $this->options->get('sitemap_video_enabled', 1) !== 1) { return ''; }

		$page_size = (int) $this->options->get('sitemap_page_size', 2000);
		if ($page_size < 500) $page_size = 500;
		if ($page_size > 5000) $page_size = 5000;

		$pt_enabled = (array) $this->options->get('sitemap_post_types', array('post' => 1));
		$post_types = array();
		foreach ($pt_enabled as $pt => $on) {
			if ((int) $on !== 1) continue;
			$obj = get_post_type_object($pt);
			if ($obj && !empty($obj->public)) $post_types[] = $pt;
		}
		if (empty($post_types)) $post_types = array('post');

		$q = new \WP_Query(array(
			'post_type'      => $post_types,
			'post_status'    => 'publish',
			'posts_per_page' => $page_size,
			'orderby'        => 'modified',
			'order'          => 'DESC',
			'no_found_rows'  => true,
		));

		$items = array();
		if ($q->have_posts()) {
			while ($q->have_posts()) {
				$q->the_post();
				$id = get_the_ID();
				$content = (string) get_post_field('post_content', $id);

				// Basic detection: YouTube/Vimeo URL or <video src="">
				$video_url = '';
				if (preg_match('~https?://(www\.)?(youtube\.com/watch\?v=|youtu\.be/)[^\s"<]+~i', $content, $m)) {
					$video_url = $m[0];
				} elseif (preg_match('~https?://(www\.)?vimeo\.com/[0-9]+~i', $content, $m)) {
					$video_url = $m[0];
				} elseif (preg_match('~<video[^>]+src=["\']([^"\']+)["\']~i', $content, $m)) {
					$video_url = $m[1];
				}

				if ($video_url === '') continue;

				$lastmod_gmt = get_post_field('post_modified_gmt', $id);
				if (empty($lastmod_gmt) || $lastmod_gmt === '0000-00-00 00:00:00') {
					$lastmod_gmt = get_post_field('post_date_gmt', $id);
				}

				$items[] = array(
					'loc'     => get_permalink($id),
					'lastmod' => gmdate('c', strtotime($lastmod_gmt)),
					'title'   => get_the_title($id),
					'desc'    => wp_strip_all_tags(get_the_excerpt($id)),
					'video'   => $video_url,
				);
			}
			wp_reset_postdata();
		}

		$xml  = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
		$xml .= "<urlset xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\" xmlns:video=\"http://www.google.com/schemas/sitemap-video/1.1\">\n";

		foreach ($items as $it) {
			$xml .= "  <url>\n";
			$xml .= '    <loc>' . $this->xml_escape($it['loc']) . "</loc>\n";
			if (!empty($it['lastmod'])) {
				$xml .= '    <lastmod>' . $this->xml_escape($it['lastmod']) . "</lastmod>\n";
			}
			$xml .= "    <video:video>\n";
			$xml .= '      <video:title>' . $this->xml_escape($it['title']) . "</video:title>\n";
			$xml .= '      <video:description>' . $this->xml_escape($it['desc']) . "</video:description>\n";
			$xml .= '      <video:content_loc>' . $this->xml_escape($it['video']) . "</video:content_loc>\n";
			$xml .= "    </video:video>\n";
			$xml .= "  </url>\n";
		}

		$xml .= $this->powered_by_xml_comment();
		$xml .= "</urlset>\n";

		return $xml;
	}

	private function build_news_sitemap() {
		if ((int) $this->options->get('sitemap_news_enabled', 0) !== 1) { return ''; }

		$publication_name = get_bloginfo('name');
		$locale = get_locale();
		$lang = 'en';
		if (is_string($locale) && strpos($locale, '_') !== false) {
			$lang = strtolower(explode('_', $locale)[0]);
		}

		// Keep it “news-appropriate”: last 48 hours
		$since = gmdate('Y-m-d H:i:s', time() - (48 * HOUR_IN_SECONDS));

		$q = new \WP_Query(array(
			'post_type'      => array('post'),
			'post_status'    => 'publish',
			'posts_per_page' => 1000,
			'date_query'     => array(
				array(
					'column'    => 'post_date_gmt',
					'after'     => $since,
					'inclusive' => true,
				),
			),
			'orderby'        => 'date',
			'order'          => 'DESC',
			'no_found_rows'  => true,
		));

		$xml  = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
		$xml .= "<urlset xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\" xmlns:news=\"http://www.google.com/schemas/sitemap-news/0.9\">\n";

		if ($q->have_posts()) {
			while ($q->have_posts()) {
				$q->the_post();
				$id = get_the_ID();
				$date_gmt = get_post_field('post_date_gmt', $id);

				$xml .= "  <url>\n";
				$xml .= '    <loc>' . $this->xml_escape(get_permalink($id)) . "</loc>\n";
				$xml .= "    <news:news>\n";
				$xml .= "      <news:publication>\n";
				$xml .= '        <news:name>' . $this->xml_escape($publication_name) . "</news:name>\n";
				$xml .= '        <news:language>' . $this->xml_escape($lang) . "</news:language>\n";
				$xml .= "      </news:publication>\n";
				$xml .= '      <news:publication_date>' . $this->xml_escape(gmdate('c', strtotime($date_gmt))) . "</news:publication_date>\n";
				$xml .= '      <news:title>' . $this->xml_escape(get_the_title($id)) . "</news:title>\n";
				$xml .= "    </news:news>\n";
				$xml .= "  </url>\n";
			}
			wp_reset_postdata();
		}

		$xml .= $this->powered_by_xml_comment();
		$xml .= "</urlset>\n";

		return $xml;
	}

    public function maybe_ping_search_engines($new_status, $old_status, $post) {
        if (!($post instanceof \WP_Post)) { return; }
        if ($new_status !== 'publish' || $old_status === 'publish') { return; }
        if ((int) $this->options->get('ping_search_engines', 0) !== 1) { return; }

        if (get_transient('aegisseo_last_ping') === '1') { return; }
        set_transient('aegisseo_last_ping', '1', 6 * HOUR_IN_SECONDS);

        if (wp_is_post_revision($post->ID) || wp_is_post_autosave($post->ID)) { return; }

        $use_index = ((int) $this->options->get('sitemap_index_enabled', 1) === 1);
        $sitemap_url = $use_index ? home_url('/sitemap_index.xml') : home_url('/sitemap.xml');

        $targets = apply_filters('aegisseo_ping_targets', array(
            'bing'   => 'https://www.bing.com/ping?sitemap=' . rawurlencode($sitemap_url),
            'google' => 'https://www.google.com/ping?sitemap=' . rawurlencode($sitemap_url),
        ));

        $ok = true;
        $results = array();

        foreach ($targets as $key => $url) {
            $r = wp_remote_get($url, array(
                'timeout' => 5,
                'redirection' => 2,
                'user-agent' => 'AegisSEO/' . AEGISSEO_VERSION . '; ' . home_url('/'),
            ));

            $code = is_wp_error($r) ? 0 : (int) wp_remote_retrieve_response_code($r);
            $results[$key] = $code;
            if ($code < 200 || $code >= 400) $ok = false;
        }

        $this->options->update('sitemap_last_ping', array(
            'time' => time(),
            'ok'   => $ok ? 1 : 0,
            'results' => $results,
            'sitemap' => $sitemap_url,
        ));
    }
    private function get_sitemap_cache_version() {
        $v = (int) $this->options->get('sitemap_cache_v', 1);
        if ($v <= 0) { $v = 1; }
        return $v;
    }

    private function bump_sitemap_cache_version() {
        $v = $this->get_sitemap_cache_version();
        $this->options->update('sitemap_cache_v', $v + 1);
    }

    public function bump_sitemap_cache_version_on_save($post_id, $post, $update) {
        if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) { return; }
        $this->bump_sitemap_cache_version();
    }

    public function bump_sitemap_cache_version_simple() {
        $this->bump_sitemap_cache_version();
    }

    public function maybe_bump_sitemap_cache_on_settings_change($option_name, $old_value, $new_value) {
        // Only react to AegisSEO settings changes (strictly sitemap-related cache behavior)
        if ($option_name !== 'aegisseo_settings') { return; }
        $this->bump_sitemap_cache_version();
    }
// LAST INSERT
	public static function render_sitemap_inventory_panel() {
		$opts = get_option('aegisseo_settings', array());
		$enabled = isset($opts['sitemap_enabled']) ? (int) $opts['sitemap_enabled'] : 1;

		echo '<div class="aegisseo-sitemap-panel">';
		echo '<h2>' . esc_html__('Sitemap Inventory Map', 'aegisseo') . '</h2>';

		if ($enabled !== 1) {
			echo '<p class="aegisseo-sitemap-mini">' . esc_html__('Sitemaps are currently disabled.', 'aegisseo') . '</p>';
			echo '</div>';
			return;
		}

		$page_size = (int) ($opts['sitemap_page_size'] ?? 2000);
		if ($page_size < 500) $page_size = 500;
		if ($page_size > 5000) $page_size = 5000;

		$urls = array();

		// Post type sitemaps
		$pts = array();
		if (!empty($opts['sitemap_post_types']) && is_array($opts['sitemap_post_types'])) {
			foreach ($opts['sitemap_post_types'] as $pt => $en) {
				if ((int) $en === 1) $pts[] = $pt;
			}
		}

		// Taxonomy sitemaps
		$taxes = array();
		if (!empty($opts['sitemap_taxonomies']) && is_array($opts['sitemap_taxonomies'])) {
			foreach ($opts['sitemap_taxonomies'] as $tax => $en) {
				if ((int) $en === 1) $taxes[] = $tax;
			}
		}

		// Plugin mappings (WooCommerce etc.) – display only (build_index/build_child handles generation)
		$plugin_pts = array('product'); // shown as enabled when plugin adds them; actual generation handled elsewhere
		$plugin_tax = array('product_cat', 'product_tag');

		// Build rows for post types
		foreach (array_values(array_unique(array_merge($pts, $plugin_pts))) as $pt) {
			if (!post_type_exists($pt)) continue;

			$obj = get_post_type_object($pt);
			$label = $obj && !empty($obj->labels->singular_name) ? $obj->labels->singular_name : $pt;
			$label = self::clean_ui_label($label, $pt, 'post_type');

			$count_obj = wp_count_posts($pt);
			$cnt = (int) ($count_obj && isset($count_obj->publish) ? $count_obj->publish : 0);
			$pages = max(1, (int) ceil($cnt / $page_size));

			for ($p=1; $p <= $pages; $p++) {
				$u = ($p === 1)
					? home_url('/sitemap-pt-' . rawurlencode($pt) . '.xml')
					: home_url('/sitemap-pt-' . rawurlencode($pt) . '-' . $p . '.xml');
				$urls[] = array('type' => $label, 'from' => 'Index', 'url' => $u, 'count' => $cnt, 'pages' => $pages);
			}
		}

		// Build rows for taxonomies
		foreach (array_values(array_unique(array_merge($taxes, $plugin_tax))) as $tax) {
			if (!taxonomy_exists($tax)) continue;

			$obj = get_taxonomy($tax);
			$label = $obj && !empty($obj->labels->singular_name) ? $obj->labels->singular_name : $tax;
			$label = self::clean_ui_label($label, $tax, 'taxonomy');

			$cnt = (int) wp_count_terms($tax, array('hide_empty' => false));
			$pages = max(1, (int) ceil($cnt / $page_size));

			for ($p=1; $p <= $pages; $p++) {
				$u = ($p === 1)
					? home_url('/sitemap-tax-' . rawurlencode($tax) . '.xml')
					: home_url('/sitemap-tax-' . rawurlencode($tax) . '-' . $p . '.xml');
				$urls[] = array('type' => $label, 'from' => 'Index', 'url' => $u, 'count' => $cnt, 'pages' => $pages);
			}
		}

		echo '<h3 style="margin:14px 0 8px;">' . esc_html__('Last Ping Status', 'aegisseo') . '</h3>';
		$ping = $opts['sitemap_last_ping'] ?? null;
		if (is_array($ping) && !empty($ping['time'])) {
			$ok = !empty($ping['ok']) ? 1 : 0;
			echo '<div style="display:flex;gap:10px;align-items:center;flex-wrap:wrap">';
			echo '<span class="aegisseo-pill ' . ($ok ? 'aegisseo-ok' : 'aegisseo-bad') . '">' . ($ok ? esc_html__('OK', 'aegisseo') : esc_html__('Failed', 'aegisseo')) . '</span>';
			echo '<span class="aegisseo-mono">' . esc_html(gmdate('Y-m-d H:i:s', (int) $ping['time'])) . ' UTC</span>';
			if (!empty($ping['message'])) {
				echo '<span class="aegisseo-sitemap-mini">' . esc_html((string) $ping['message']) . '</span>';
			}
			echo '</div>';
		} else {
			echo '<p class="aegisseo-sitemap-mini">' . esc_html__('No ping has been recorded yet.', 'aegisseo') . '</p>';
		}

		echo '<h3 style="margin:14px 0 8px;">' . esc_html__('Generated Sitemap Endpoints', 'aegisseo') . '</h3>';

		echo '<table class="aegisseo-sitemap-table">';
		echo '<thead><tr>';
		echo '<th>' . esc_html__('Type', 'aegisseo') . '</th>';
		echo '<th>' . esc_html__('URL', 'aegisseo') . '</th>';
		echo '<th>' . esc_html__('Source', 'aegisseo') . '</th>';
		echo '<th>' . esc_html__('Count', 'aegisseo') . '</th>';
		echo '</tr></thead><tbody>';

		foreach ($urls as $r) {
			echo '<tr>';
			echo '<td><span class="aegisseo-pill">' . esc_html($r['type']) . '</span></td>';
			echo '<td><a class="aegisseo-mono" href="' . esc_url($r['url']) . '" target="_blank" rel="noopener noreferrer">' . esc_html($r['url']) . '</a></td>';
			echo '<td>' . esc_html($r['from']) . '</td>';
			echo '<td class="aegisseo-mono">' . esc_html((string) $r['count']) . '</td>';
			echo '</tr>';
		}

		echo '</tbody></table>';
		echo '</div>';
	}

}
