<?php
namespace AegisSEO\Utils;

if (!defined('ABSPATH')) { exit; }

class License {

    const OPTION_KEY = 'aegisseo_license_key';
    const TRANSIENT_STATUS = 'aegisseo_license_status';

    public function get_key() : string {
        return (string) get_option(self::OPTION_KEY, '');
    }

    public function set_key(string $key) : void {
        update_option(self::OPTION_KEY, trim($key), false);
        delete_transient(self::TRANSIENT_STATUS);
    }

    public function get_cached_status() : array {
        $s = get_transient(self::TRANSIENT_STATUS);
        return is_array($s) ? $s : array();
    }

    public function is_pro() : bool {
        $s = $this->get_cached_status();
        if (isset($s['is_pro'])) {
            return (bool) $s['is_pro'];
        }
        return false;
    }

    public function check_now(string $current_version) : array {
        $site_url = home_url();
        $domain   = wp_parse_url($site_url, PHP_URL_HOST);

        $license_key = $this->get_key();

        $payload = array(
            'slug'            => defined('AEGISSEO_SLUG') ? AEGISSEO_SLUG : 'aegisseo',
            'product'         => defined('AEGISSEO_SLUG') ? AEGISSEO_SLUG : 'aegisseo',
            'version'         => (string) $current_version,
            'domain'          => (string) $domain,
            'site_url'        => (string) $site_url,
            'license_key'     => (string) $license_key,
			'email' 		  => (string) get_option('aegisseo_license_email', ''),
            'wp_version'      => (string) get_bloginfo('version'),
            'php_version'     => (string) PHP_VERSION,
            'plugin_basename' => defined('AEGISSEO_BASENAME') ? (string) AEGISSEO_BASENAME : '',
        );

        $args = array(
            'timeout' => 20,
            'headers' => array( 'Content-Type' => 'application/json' ),
            'body'    => wp_json_encode($payload),
        );

		$endpoint = defined('AEGISSEO_CLM_STATUS_ENDPOINT') ? AEGISSEO_CLM_STATUS_ENDPOINT : '';
		if (!$endpoint && defined('AEGISSEO_CLM_UPDATE_ENDPOINT')) {
			$endpoint = AEGISSEO_CLM_UPDATE_ENDPOINT;
		}

        $resp = wp_remote_post($endpoint, $args);
        if (is_wp_error($resp)) {
            return array('success'=>false,'status'=>'wp_error','message'=>$resp->get_error_message());
        }

        $code = (int) wp_remote_retrieve_response_code($resp);
        $body = (string) wp_remote_retrieve_body($resp);

        if ($code < 200 || $code >= 300) {
            return array('success'=>false,'status'=>'http','message'=>'CLM HTTP '.$code,'raw'=>$body);
        }

        $data = json_decode($body, true);
        if (!is_array($data)) {
            return array('success'=>false,'status'=>'bad_json','message'=>'CLM returned invalid JSON','raw'=>$body);
        }

        $is_pro = false;
        $candidates_true = array(
            $data['is_pro'] ?? null,
            $data['pro'] ?? null,
            $data['license_active'] ?? null,
            $data['active'] ?? null,
        );
        foreach ($candidates_true as $v) {
            if ($v === true || $v === 1 || $v === '1' || $v === 'true') { $is_pro = true; break; }
        }
        if (!$is_pro) {
            $status = strtolower((string)($data['license_status'] ?? $data['status'] ?? ''));
            $plan   = strtolower((string)($data['plan'] ?? $data['tier'] ?? $data['license_type'] ?? ''));
            if (in_array($status, array('active','valid','licensed'), true) && in_array($plan, array('pro','premium','paid'), true)) {
                $is_pro = true;
            }
            if (in_array($plan, array('pro','premium','paid'), true) && $status === '') {
                $is_pro = true;
            }
        }

        $status_row = array(
            'success'     => !empty($data['success']) ? true : true, // some CLM builds omit success for license checks
            'checked_at'  => time(),
            'is_pro'      => (bool) $is_pro,
            'license_key' => (string) $license_key,
            'status'      => (string)($data['license_status'] ?? $data['status'] ?? ''),
            'plan'        => (string)($data['plan'] ?? $data['tier'] ?? $data['license_type'] ?? ''),
            'expires'     => (string)($data['expires'] ?? $data['expires_at'] ?? ''),
            'raw'         => $data,
        );

        set_transient(self::TRANSIENT_STATUS, $status_row, HOUR_IN_SECONDS * 6);

        return $status_row;
    }

    public function maybe_send_free_register(string $current_version) : void {
        if (get_transient('aegisseo_clm_free_reg_sent')) { return; }

        $site_url = home_url();
        $domain   = wp_parse_url($site_url, PHP_URL_HOST);

        $payload = array(
            'slug'        => defined('AEGISSEO_SLUG') ? AEGISSEO_SLUG : 'aegisseo',
            'product'     => defined('AEGISSEO_SLUG') ? AEGISSEO_SLUG : 'aegisseo',
            'version'     => (string)$current_version,
            'domain'      => (string)$domain,
            'site_url'    => (string)$site_url,
            'wp_version'  => (string)get_bloginfo('version'),
            'php_version' => (string)PHP_VERSION,
        );

        $endpoint = defined('AEGISSEO_CLM_FREE_REG_ENDPOINT') ? AEGISSEO_CLM_FREE_REG_ENDPOINT : '';
        if (!$endpoint) { return; }

        $args = array(
            'timeout' => 15,
            'headers' => array('Content-Type' => 'application/json'),
            'body'    => wp_json_encode($payload),
        );

        $resp = wp_remote_post($endpoint, $args);
        set_transient('aegisseo_clm_free_reg_sent', 1, DAY_IN_SECONDS);

        if (function_exists('error_log')) {
            $code = is_wp_error($resp) ? 'wp_error' : (int) wp_remote_retrieve_response_code($resp);
            error_log('[AegisSEO] CLM free-register attempted. code=' . $code);
        }
    }



public function hook_update_check() : void {
    add_filter('pre_set_site_transient_update_plugins', function($transient) {
        if (!is_object($transient) || empty($transient->checked) || !is_array($transient->checked)) {
            return $transient;
        }
        if (empty($transient->checked[defined('AEGISSEO_BASENAME') ? AEGISSEO_BASENAME : ''])) {
            return $transient;
        }
        $this->check_now(defined('AEGISSEO_VERSION') ? AEGISSEO_VERSION : '');
        return $transient;
    });
}

    public static function render_pro_banner(string $title = 'Upgrade to PRO', string $message = '') : void {
        if ($message === '') {
            $message = __('PRO feature: This section is available in AegisSEO PRO. Unlock automation, evidence reporting, and advanced guardrails.', 'aegisseo');
        }
        ?>
        <div class="aegisseo-pro-banner">
            <div class="aegisseo-pro-banner__left">
                <div class="aegisseo-pro-badge"><?php echo esc_html($title); ?></div>
                <div class="aegisseo-pro-text"><?php echo esc_html($message); ?></div>
            </div>
            <div class="aegisseo-pro-banner__right">
                <a class="button button-primary" href="<?php echo esc_url(admin_url('admin.php?page=aegisseo&tab=license')); ?>">
                    <?php echo esc_html__('Activate License', 'aegisseo'); ?>
                </a>
            </div>
        </div>
        <?php
    }
}
