<?php

/**
 * Plugin Name: AegisSEO - Search Engine Optimization
 * Plugin URI: https://aegisify.com/aegisseo/
 * Description: AegisSEO is an advanced WordPress SEO optimization plugin that helps improve search visibility through structured data, Search Console integration, automated analysis, and performance-focused SEO tools.
 * Version: 1.6.9
 * Requires at least: 6.8+
 * Requires PHP: 8.2+
 * Author:      Aegisify | AegisWAF | AegisShield | AegisSEO
 * Author URI:  https://aegisify.com/
 * Text Domain: aegisseo
 * License: GPLv2 or later
 */

if (!defined('ABSPATH')) {
    exit;
}

define('AEGISSEO_VERSION', '1.6.9');
define('AEGISSEO_FILE', __FILE__);
define('AEGISSEO_DIR', plugin_dir_path(__FILE__));

require_once AEGISSEO_DIR . 'includes/class-aegisseo-plugin.php';
require_once AEGISSEO_DIR . 'includes/log/class-aegisseo-logger.php';

define('AEGISSEO_URL', plugin_dir_url(__FILE__));
define('AEGISSEO_BASENAME', plugin_basename(__FILE__));

if (!defined('AEGISSEO_WPORG_BUILD')) {
    define('AEGISSEO_WPORG_BUILD', true);
}

define('AEGISSEO_SLUG', 'aegisseo');
define('AEGISSEO_CLM_BASE', 'https://aegisify.com');
define('AEGISSEO_CLM_FREE_REG_ENDPOINT', AEGISSEO_CLM_BASE . '/wp-json/chue-license/v1/aegisseo/free-register');
define('AEGISSEO_CLM_UPDATE_ENDPOINT',    AEGISSEO_CLM_BASE . '/wp-json/chue-license/v1/aegisseo/update-check');

define('AEGISSEO_CLM_ACTIVATE_ENDPOINT',   AEGISSEO_CLM_BASE . '/wp-json/chue-license/v1/aegisseo/activate');
define('AEGISSEO_CLM_DEACTIVATE_ENDPOINT', AEGISSEO_CLM_BASE . '/wp-json/chue-license/v1/aegisseo/deactivate');
define('AEGISSEO_CLM_STATUS_ENDPOINT',     AEGISSEO_CLM_BASE . '/wp-json/chue-license/v1/aegisseo/status');		
																			 
function aegisseo_update_log( string $msg, array $ctx = array(), string $level = 'INFO' ) : void {
    if (class_exists('\AegisSEO\Log\Logger')) {
        switch (strtoupper($level)) {
            case 'ERROR':
                \AegisSEO\Log\Logger::error($msg, $ctx);
                break;
            case 'WARN':
            case 'WARNING':
                \AegisSEO\Log\Logger::warn($msg, $ctx);
                break;
            case 'DEBUG':
                \AegisSEO\Log\Logger::debug($msg, $ctx);
                break;
            default:
                \AegisSEO\Log\Logger::info($msg, $ctx);
        }
        return;
    }

    // Fallback (should be rare)
    error_log('[AegisSEO] ' . $msg . ($ctx ? ' | ' . wp_json_encode($ctx) : ''));
}

function aegisseo_clm_register_free_install(bool $force = false) : array {

    // Prevent spamming endpoint (manual click can override with $force=true)
    if (!$force && get_transient('aegisseo_clm_free_reg_sent')) {
        return array(
            'success' => false,
            'status'  => 'backoff',
            'message' => 'Backoff active (already attempted recently).',
        );
    }

    $site_url = home_url();
    $domain   = wp_parse_url($site_url, PHP_URL_HOST);

    $payload = array(
        'product'        => AEGISSEO_SLUG,
        'email'          => (string) get_option('admin_email'),
        'site_url'       => (string) $site_url,
        'domain'         => (string) $domain,
        'plugin_version' => defined('AEGISSEO_VERSION') ? (string) AEGISSEO_VERSION : 'unknown',
        'wp_version'     => (string) get_bloginfo('version'),
        'php_version'    => (string) PHP_VERSION,
    );

    $args = array(
        'timeout' => 15,
        'headers' => array('Content-Type' => 'application/json'),
        'body'    => wp_json_encode($payload),
    );

    $resp = wp_remote_post(AEGISSEO_CLM_FREE_REG_ENDPOINT, $args);

    if (is_wp_error($resp)) {
        aegisseo_update_log('CLM free-register failed (WP_Error).', array(
            'endpoint' => AEGISSEO_CLM_FREE_REG_ENDPOINT,
            'error'    => $resp->get_error_message(),
        ));

        return array(
            'success' => false,
            'status'  => 'wp_error',
            'message' => $resp->get_error_message(),
        );
    }

    $code = (int) wp_remote_retrieve_response_code($resp);
    $body = (string) wp_remote_retrieve_body($resp);

    $backoff = ($code >= 500) ? HOUR_IN_SECONDS : DAY_IN_SECONDS;
    set_transient('aegisseo_clm_free_reg_sent', 1, $backoff);

    aegisseo_update_log('CLM free-register response.', array(
        'endpoint'  => AEGISSEO_CLM_FREE_REG_ENDPOINT,
        'http_code' => $code,
        'body'      => $body,
        'backoff_s' => $backoff,
    ));

    if ($code >= 200 && $code < 300) {
        return array(
            'success' => true,
            'status'  => 'ok',
            'message' => 'Registered',
        );
    }

    return array(
        'success' => false,
        'status'  => 'http_' . $code,
        'message' => 'Non-2xx response from CLM',
        'body'    => $body,
    );
}

function aegisseo_clm_direct_update_check( string $current_version ) : array {

    $site_url = home_url();
    $domain   = wp_parse_url( $site_url, PHP_URL_HOST );

    $license_key = (string) get_option( 'aegisseo_license_key', '' );

    $payload = array(
        'slug'        => AEGISSEO_SLUG,
        'product'     => AEGISSEO_SLUG,
        'version'     => $current_version,
        'domain'      => $domain ? (string) $domain : '',
        'site_url'    => (string) $site_url,
        'license_key' => $license_key,
    );

    $resp = wp_remote_post( AEGISSEO_CLM_UPDATE_ENDPOINT, array(
        'timeout' => 15,
        'headers' => array( 'Content-Type' => 'application/json' ),
        'body'    => wp_json_encode( $payload ),
    ) );

    if ( is_wp_error( $resp ) ) {
        return array(
            'success' => false,
            'status'  => 'wp_error',
            'message' => $resp->get_error_message(),
        );
    }

    $code = (int) wp_remote_retrieve_response_code( $resp );
    $body = (string) wp_remote_retrieve_body( $resp );

    if ( $code < 200 || $code >= 300 ) {
        return array(
            'success' => false,
            'status'  => 'http_' . $code,
            'message' => 'Non-2xx from CLM',
            'raw'     => $body,
        );
    }

    $data = json_decode( $body, true );
    if ( ! is_array( $data ) ) {
        return array(
            'success' => false,
            'status'  => 'bad_json',
            'message' => 'CLM returned invalid JSON',
            'raw'     => $body,
        );
    }

    if ( empty( $data['version'] ) && ! empty( $data['latest_version'] ) ) {
        $data['version'] = (string) $data['latest_version'];
    }

    return $data;
}

function aegisseo_check_for_updates_once_daily( $transient ) {

    if ( ! is_object( $transient ) || empty( $transient->checked ) || ! is_array( $transient->checked ) ) {
        return $transient;
    }

    if ( empty( $transient->checked[ AEGISSEO_BASENAME ] ) ) {
        return $transient;
    }

    $current_version = (string) $transient->checked[ AEGISSEO_BASENAME ];
    $data            = aegisseo_clm_direct_update_check( $current_version );

    if ( empty( $data['success'] ) || empty( $data['update_available'] ) ) {
        return $transient;
    }

    $remote_version = ! empty( $data['version'] ) ? (string) $data['version'] : '';
    $package        = ! empty( $data['download_url'] ) ? (string) $data['download_url'] : '';

    if ( empty( $remote_version ) || empty( $package ) ) {
        return $transient;
    }

    if ( $current_version && version_compare( $remote_version, $current_version, '<=' ) ) {
        return $transient;
    }

    $update              = new stdClass();
    $update->slug        = AEGISSEO_SLUG;
    $update->plugin      = AEGISSEO_BASENAME;
    $update->new_version = $remote_version;
    $update->package     = $package;
    $update->url         = ! empty( $data['homepage'] ) ? (string) $data['homepage'] : 'https://aegisify.com';

	$transient->response[ AEGISSEO_BASENAME ] = $update;

	$cache_key = 'aegisseo_update_' . md5( AEGISSEO_BASENAME . '|' . $current_version );
	set_site_transient( $cache_key, $update, DAY_IN_SECONDS );

	return $transient;
}

if (!AEGISSEO_WPORG_BUILD) {
    add_filter( 'pre_set_site_transient_update_plugins', 'aegisseo_check_for_updates_once_daily', 25, 1 );
}

function aegisseo() {
    return \AegisSEO\Plugin::instance();
}

	add_action('plugins_loaded', function() {
		\AegisSEO\Log\Logger::init(12);
		aegisseo()->boot();

	}, 5);