<?php
namespace AegisSEO\Log;

if (!defined('ABSPATH')) { exit; }

final class Logger {

	const LEVEL_DEBUG = 'DEBUG';
	const LEVEL_INFO  = 'INFO';
	const LEVEL_WARN  = 'WARN';
	const LEVEL_ERROR = 'ERROR';

	private static $min_level = self::LEVEL_WARN;

	private static function level_rank(string $level) : int {
		switch (strtoupper($level)) {
			case self::LEVEL_ERROR: return 4;
			case self::LEVEL_WARN:  return 3;
			case self::LEVEL_INFO:  return 2;
			case self::LEVEL_DEBUG: return 1;
			default: return 2;
		}
	}

    private static $initialized = false;

    private static $log_dir;

    private static $log_file;

    private static $keep_weeks = 12; // purge older rotated logs (optional)

    public static function init(int $keep_weeks = 12) : void {
        if (self::$initialized) { return; }
        self::$initialized = true;

        self::$keep_weeks = max(1, (int) $keep_weeks);
		if (defined('AEGISSEO_LOG_LEVEL')) {
			$lvl = strtoupper((string) AEGISSEO_LOG_LEVEL);
			if (in_array($lvl, array(self::LEVEL_ERROR, self::LEVEL_WARN, self::LEVEL_INFO, self::LEVEL_DEBUG), true)) {
				self::$min_level = $lvl;
			}
		}

        self::$log_dir  = rtrim(AEGISSEO_DIR, '/\\') . '/includes/log';
        self::$log_file = self::$log_dir . '/aegisseo.log';

        self::ensure_dir();
        self::rotate_if_needed();
        self::purge_old_rotations();

        set_error_handler(array(__CLASS__, 'handle_php_error'));

        set_exception_handler(array(__CLASS__, 'handle_exception'));

        register_shutdown_function(array(__CLASS__, 'handle_shutdown'));

		self::debug('Logger initialized', array(
			'log_file' => self::$log_file,
			'version'  => defined('AEGISSEO_VERSION') ? AEGISSEO_VERSION : 'unknown',
			'min_level'=> self::$min_level,
		));
    }

    public static function debug(string $msg, array $ctx = array()) : void { self::write(self::LEVEL_DEBUG, $msg, $ctx); }
    public static function info(string $msg, array $ctx = array()) : void  { self::write(self::LEVEL_INFO,  $msg, $ctx); }
    public static function warn(string $msg, array $ctx = array()) : void  { self::write(self::LEVEL_WARN,  $msg, $ctx); }
    public static function error(string $msg, array $ctx = array()) : void { self::write(self::LEVEL_ERROR, $msg, $ctx); }

    private static function ensure_dir() : void {
        if (!is_dir(self::$log_dir)) {
            @mkdir(self::$log_dir, 0755, true);
        }
    }

    /**
     * Weekly rotation:
     * - If current aegisseo.log exists and its ISO week differs from now, rename it to aegisseo-YYYY-WW.log
     * - Start a fresh aegisseo.log
     */
    private static function rotate_if_needed() : void {
        if (!file_exists(self::$log_file)) { return; }

        $mtime = @filemtime(self::$log_file);
        if (!$mtime) { return; }

        $then = new \DateTime('@' . $mtime);
        $then->setTimezone(new \DateTimeZone('UTC'));
        $now  = new \DateTime('now', new \DateTimeZone('UTC'));

        $then_key = $then->format('o-W'); // ISO year-week
        $now_key  = $now->format('o-W');

        if ($then_key === $now_key) { return; }

        $rotated = self::$log_dir . '/aegisseo-' . $then->format('o-W') . '.log';
        @rename(self::$log_file, $rotated);
    }

    private static function purge_old_rotations() : void {
        $pattern = self::$log_dir . '/aegisseo-*.log';
        $files = glob($pattern);
        if (!$files) { return; }

        // Keep only newest N weekly logs + current aegisseo.log
        usort($files, function($a, $b) {
            return filemtime($b) <=> filemtime($a);
        });

        $to_delete = array_slice($files, self::$keep_weeks);
        foreach ($to_delete as $f) {
            @unlink($f);
        }
    }

    private static function write(string $level, string $message, array $context = array()) : void {
		if (self::level_rank($level) < self::level_rank(self::$min_level)) {
			return;
		}
        if (!is_dir(self::$log_dir) || (file_exists(self::$log_dir) && !is_writable(self::$log_dir))) {

            error_log('[AegisSEO][LOGGER] Log directory not writable: ' . self::$log_dir);
            return;
        }

        self::rotate_if_needed();

        $ts = gmdate('Y-m-d H:i:s') . ' UTC';
        $ctx = $context ? json_encode($context, JSON_UNESCAPED_SLASHES) : '';
        $line = '[' . $ts . '][' . $level . '] ' . $message . ($ctx ? ' | ' . $ctx : '') . "\n";

        @file_put_contents(self::$log_file, $line, FILE_APPEND);
    }

    /**
     * Only log errors that actually originate from the plugin (AegisSEO directory).
     */
    public static function handle_php_error($errno, $errstr, $errfile, $errline) : bool {
        if (!self::is_plugin_file($errfile)) {
            return false; // let WP / other handlers process
        }

        self::warn('PHP error captured', array(
            'errno' => $errno,
            'error' => $errstr,
            'file'  => $errfile,
            'line'  => $errline,
        ));

        return false; // don’t block normal handling
    }

    public static function handle_exception($ex) : void {
        $file = method_exists($ex, 'getFile') ? $ex->getFile() : '';
        if ($file && !self::is_plugin_file($file)) { return; }

        self::error('Uncaught exception', array(
            'message' => $ex->getMessage(),
            'file'    => $ex->getFile(),
            'line'    => $ex->getLine(),
        ));
    }

    public static function handle_shutdown() : void {
        $err = error_get_last();
        if (!$err) { return; }

        // fatal types
        $fatal_types = array(E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR);
        if (!in_array((int)$err['type'], $fatal_types, true)) { return; }

        if (!self::is_plugin_file($err['file'] ?? '')) { return; }

        self::error('Fatal error at shutdown', array(
            'type' => $err['type'] ?? null,
            'msg'  => $err['message'] ?? '',
            'file' => $err['file'] ?? '',
            'line' => $err['line'] ?? 0,
        ));
    }

    private static function is_plugin_file(string $file) : bool {
        if (!$file) { return false; }
        $base = rtrim(AEGISSEO_DIR, '/\\');
        $file_norm = str_replace('\\', '/', $file);
        $base_norm = str_replace('\\', '/', $base);
        return (strpos($file_norm, $base_norm . '/') !== false);
    }
}
