<?php
namespace AegisSEO\SEO;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Monitor_404 {

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;

        add_action('template_redirect', array($this, 'maybe_log_404'), 1);
        add_action('aegisseo_daily_maintenance', array($this, 'daily_prune'));
    }

    private function is_enabled() {
        return ((int)$this->options->get('monitor_404_enabled', 1) === 1);
    }

    public function maybe_log_404() {
        if (!$this->is_enabled()) { return; }
        if (!is_404()) { return; }
        if (is_admin()) { return; }
        if (defined('DOING_AJAX') && DOING_AJAX) { return; }

        $request_uri = isset($_SERVER['REQUEST_URI']) ? wp_unslash($_SERVER['REQUEST_URI']) : '';
        $request_uri = is_string($request_uri) ? trim($request_uri) : '';
        if ($request_uri === '') { return; }

        $lower = strtolower($request_uri);
        $skip = array('/favicon.ico', '/robots.txt');
        foreach ($skip as $s) {
            if ($lower === $s) { return; }
        }
        if (preg_match('#\.(css|js|map|jpg|jpeg|png|gif|webp|svg|ico|woff|woff2|ttf|eot)$#i', $lower)) {
            return;
        }

        $ref = isset($_SERVER['HTTP_REFERER']) ? wp_unslash($_SERVER['HTTP_REFERER']) : '';
        $ua  = isset($_SERVER['HTTP_USER_AGENT']) ? wp_unslash($_SERVER['HTTP_USER_AGENT']) : '';
        $ip  = isset($_SERVER['REMOTE_ADDR']) ? wp_unslash($_SERVER['REMOTE_ADDR']) : '';

        $this->log($request_uri, $ref, $ua, $ip);
    }

    public function log($url, $referrer = '', $user_agent = '', $ip = '') {
        global $wpdb;

        $t404 = $wpdb->prefix . 'aegisseo_404';

        $url = (string) $url;
        $url = mb_substr($url, 0, 2000);

        $now = current_time('mysql');
        $hash = md5($url);

        $row = $wpdb->get_row($wpdb->prepare("SELECT id, hits FROM {$t404} WHERE url_hash = %s", $hash), ARRAY_A);

        if ($row) {
            $wpdb->update(
                $t404,
                array(
                    'hits' => (int)$row['hits'] + 1,
                    'last_seen' => $now,
                    'referrer' => $referrer ? mb_substr((string)$referrer, 0, 2000) : null,
                    'user_agent' => $user_agent ? mb_substr((string)$user_agent, 0, 2000) : null,
                    'ip' => $ip ? mb_substr((string)$ip, 0, 45) : null,
                ),
                array('id' => (int)$row['id']),
                array('%d','%s','%s','%s','%s'),
                array('%d')
            );
        } else {
            $wpdb->insert(
                $t404,
                array(
                    'url' => $url,
                    'url_hash' => $hash,
                    'referrer' => $referrer ? mb_substr((string)$referrer, 0, 2000) : null,
                    'user_agent' => $user_agent ? mb_substr((string)$user_agent, 0, 2000) : null,
                    'ip' => $ip ? mb_substr((string)$ip, 0, 45) : null,
                    'hits' => 1,
                    'first_seen' => $now,
                    'last_seen' => $now,
                ),
                array('%s','%s','%s','%s','%s','%d','%s','%s')
            );

            $max = (int) $this->options->get('monitor_404_max_rows', 5000);
            if ($max > 0) {
                $count = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$t404}");
                if ($count > $max) {
                    $excess = $count - $max;
                    $wpdb->query($wpdb->prepare("DELETE FROM {$t404} ORDER BY last_seen ASC LIMIT %d", $excess));
                }
            }
        }
    }

    public function daily_prune() {
        if (!$this->is_enabled()) { return; }
        $this->prune();
    }

    public function prune() {
        global $wpdb;
        $t404 = $wpdb->prefix . 'aegisseo_404';

        $days = (int) $this->options->get('monitor_404_retention_days', 30);
        if ($days <= 0) { return 0; }

        $cutoff = gmdate('Y-m-d H:i:s', time() - ($days * DAY_IN_SECONDS));
        return $wpdb->query($wpdb->prepare("DELETE FROM {$t404} WHERE last_seen < %s", $cutoff));
    }

    public function clear_all() {
        global $wpdb;
        $t404 = $wpdb->prefix . 'aegisseo_404';
        return $wpdb->query("TRUNCATE TABLE {$t404}");
    }

    public function export_csv() {
        if (!current_user_can('manage_options')) { wp_die('Forbidden'); }

        global $wpdb;
        $t404 = $wpdb->prefix . 'aegisseo_404';

        $rows = $wpdb->get_results("SELECT url, hits, first_seen, last_seen, referrer, user_agent, ip FROM {$t404} ORDER BY last_seen DESC LIMIT 5000", ARRAY_A);

        nocache_headers();
        header('Content-Type: text/csv; charset=UTF-8');
        header('Content-Disposition: attachment; filename="aegisseo-404-monitor.csv"');

        $out = fopen('php://output', 'w');
        fputcsv($out, array('url','hits','first_seen','last_seen','referrer','user_agent','ip'));
        foreach ($rows as $r) {
            fputcsv($out, $r);
        }
        fclose($out);
        exit;
    }
}
