<?php
namespace AegisSEO\SEO;

if (!defined('ABSPATH')) { exit; }

class Events {

    private $tracked_meta_keys = array(
        '_aegisseo_title',
        '_aegisseo_description',
        '_aegisseo_canonical',
        '_aegisseo_robots',
        '_aegisseo_focus_phrase',
        '_aegisseo_schema_mode',
        '_aegisseo_schema_type',
        '_aegisseo_schema_data',
        '_aegisseo_noindex',
        '_aegisseo_nofollow',
        '_aegisseo_og_title',
        '_aegisseo_og_description',
        '_aegisseo_og_image',
        '_aegisseo_twitter_title',
        '_aegisseo_twitter_description',
        '_aegisseo_twitter_image',
    );

    public function __construct() {
        add_filter('add_post_metadata', array($this, 'capture_add_post_meta'), 10, 5);
        add_filter('update_post_metadata', array($this, 'capture_update_post_meta'), 10, 5);
        add_filter('delete_post_metadata', array($this, 'capture_delete_post_meta'), 10, 5);
        add_action('post_updated', array($this, 'capture_post_updated'), 10, 3);
    }

    private function is_tracked_meta($meta_key) {
        return in_array($meta_key, $this->tracked_meta_keys, true);
    }

	private function is_allowed_post_object($post_id) {
		$post_id = (int) $post_id;
		if ($post_id <= 0) { return false; }

		$pt = get_post_type($post_id);
		if (!$pt) { return false; }

		// Ignore internal/system types + any AegisSEO internal post types.
		$deny = array(
			'revision',
			'nav_menu_item',
			'custom_css',
			'customize_changeset',
			'oembed_cache',
			'user_request',
			'wp_block',
			'wp_template',
			'wp_template_part',
			'wp_navigation',
			'wp_global_styles',
			'attachment',
		);
		if (in_array($pt, $deny, true)) { return false; }
		if (stripos($pt, 'aegisseo') === 0) { return false; }

		// Prefer AegisSEO scope post types (your intended content types).
		$scope = array();
		if (function_exists('aegisseo')) {
			$plugin = aegisseo();
			if ($plugin && isset($plugin->options) && method_exists($plugin->options, 'get_scope_post_types')) {
				$scope = (array) $plugin->options->get_scope_post_types();
			}
		}

		if (!empty($scope)) {
			return in_array($pt, $scope, true);
		}

		// Fallback: allow public post types only if scope not available.
		$public = get_post_types(array('public' => true), 'names');
		if (!is_array($public) || empty($public)) { return false; }

		return in_array($pt, $public, true);
	}

	private function get_allowed_event_post_types_sql_in() {
		static $cached = null;
		if (null !== $cached) { return $cached; }
		// Prefer AegisSEO scope post types so event list only shows real content.
		$scope = array();
		if (function_exists('aegisseo')) {
			$plugin = aegisseo();
			if ($plugin && isset($plugin->options) && method_exists($plugin->options, 'get_scope_post_types')) {
				$scope = (array) $plugin->options->get_scope_post_types();
			}
		}
		if (!empty($scope)) {
			$cached = array_values(array_filter(array_map('sanitize_key', $scope)));
			if (empty($cached)) { $cached = array('post', 'page'); }
			return $cached;
		}

		$public = get_post_types(array('public' => true), 'names');
		if (!is_array($public)) { $public = array(); }

		$deny = array(
			'revision',
			'nav_menu_item',
			'custom_css',
			'customize_changeset',
			'oembed_cache',
			'user_request',
			'wp_block',
			'wp_template',
			'wp_template_part',
			'wp_navigation',
			'wp_global_styles',
			'attachment',
		);

		$allowed = array();
		foreach ($public as $pt) {
			if (!is_string($pt) || $pt === '') { continue; }
			if (in_array($pt, $deny, true)) { continue; }
			if (stripos($pt, 'aegisseo') === 0) { continue; }
			$allowed[] = $pt;
		}

		if (empty($allowed)) { $allowed = array('post','page'); }

		$cached = $allowed;
		return $cached;
	}

	public function capture_add_post_meta($check, $object_id, $meta_key, $meta_value, $unique) {
		if (!$this->is_allowed_post_object((int) $object_id)) { return $check; }
		if (!$this->is_tracked_meta($meta_key)) { return $check; }
		$this->log_event('meta_add', 'post', (int) $object_id, $meta_key, null, $meta_value);
		return $check;
	}

	public function capture_update_post_meta($check, $object_id, $meta_key, $meta_value, $prev_value) {
		if (!$this->is_allowed_post_object((int) $object_id)) { return $check; }
		if (!$this->is_tracked_meta($meta_key)) { return $check; }
        $old = get_post_meta((int) $object_id, $meta_key, true);
        $this->log_event('meta_update', 'post', (int) $object_id, $meta_key, $old, $meta_value);
        return $check;
    }

    public function capture_delete_post_meta($check, $object_id, $meta_key, $meta_value, $delete_all) {
        if (!$this->is_tracked_meta($meta_key)) { return $check; }
        $old = get_post_meta((int) $object_id, $meta_key, true);
        $this->log_event('meta_delete', 'post', (int) $object_id, $meta_key, $old, null);
        return $check;
    }

    public function log_event($event_type, $object_type, $object_id = 0, $meta_key = null, $old_value = null, $new_value = null, $user_id = null) {
        global $wpdb;

        $table = $wpdb->prefix . 'aegisseo_events';

        if (null === $user_id) {
            $user_id = get_current_user_id();
        }

        $data = array(
            'event_type'   => (string) $event_type,
            'object_type'  => (string) $object_type,
            'object_id'    => (int) $object_id,
            'meta_key'     => $meta_key ? (string) $meta_key : null,
            'old_value'    => is_scalar($old_value) || null === $old_value ? $old_value : wp_json_encode($old_value),
            'new_value'    => is_scalar($new_value) || null === $new_value ? $new_value : wp_json_encode($new_value),
            'user_id'      => (int) $user_id,
            'created_at'   => current_time('mysql'),
        );

        $formats = array('%s','%s','%d','%s','%s','%s','%d','%s');

        if (null === $data['meta_key']) {
            $formats[3] = '%s';
            $data['meta_key'] = null;
        }

        $wpdb->insert($table, $data, $formats);
    }

public function capture_post_updated($post_id, $post_after, $post_before) {
	if (!$this->is_allowed_post_object((int) $post_id)) { return; }
    if (!is_object($post_after) || !is_object($post_before)) { return; }
    $post_id = (int) $post_id;
    if ($post_id <= 0) { return; }

    $pt = get_post_type($post_id);
    if (!$pt) { return; }
    $pto = get_post_type_object($pt);
    if (!$pto || empty($pto->public)) { return; }

    $old_title = (string) $post_before->post_title;
    $new_title = (string) $post_after->post_title;
    if ($old_title !== $new_title) {
        $this->log_event('title_changed', 'post', $post_id, 'post_title', $old_title, $new_title);
    }

    $old_slug = (string) $post_before->post_name;
    $new_slug = (string) $post_after->post_name;
    if ($old_slug !== '' && $new_slug !== '' && $old_slug !== $new_slug) {
        $this->log_event('slug_changed', 'post', $post_id, 'post_slug', $old_slug, $new_slug);
    }

    $old_content = (string) $post_before->post_content;
    $new_content = (string) $post_after->post_content;

    $old_text = trim(wp_strip_all_tags($old_content));
    $new_text = trim(wp_strip_all_tags($new_content));

    $old_words = $old_text !== '' ? str_word_count($old_text) : 0;
    $new_words = $new_text !== '' ? str_word_count($new_text) : 0;

    $old_h = 0; $new_h = 0;
    if ($old_content !== '') { $old_h = preg_match_all('/<h[1-6][^>]*>/i', $old_content); }
    if ($new_content !== '') { $new_h = preg_match_all('/<h[1-6][^>]*>/i', $new_content); }

    if ($old_words !== $new_words || $old_h !== $new_h) {
        $summary = sprintf('Words: %d→%d; Headings: %d→%d', (int)$old_words, (int)$new_words, (int)$old_h, (int)$new_h);
        $this->log_event('onpage_changed', 'post', $post_id, 'onpage_diff', null, $summary);
    }
}

    public function get_recent($limit = 50, $object_type = null, $object_id = null, $allowed_post_types = null) {
        global $wpdb;

        $limit = max(1, (int) $limit);
        $table = $wpdb->prefix . 'aegisseo_events';

        $where = '1=1';
        $params = array();

        if ($object_type) {
            $where .= ' AND object_type = %s';
            $params[] = (string) $object_type;
        }
        if (null !== $object_id) {
            $where .= ' AND object_id = %d';
            $params[] = (int) $object_id;
        }

        $allowed = $allowed_post_types;

        if (is_string($allowed) && $allowed !== '') {

            $allowed = array_map('trim', explode(',', $allowed));
        }

        if (is_array($allowed)) {
            $allowed = array_values(array_filter(array_map('sanitize_key', $allowed)));
        } else {
            $allowed = array();
        }

        if (!$object_type && null === $object_id) {
            if (!empty($allowed)) {
                $placeholders = implode(',', array_fill(0, count($allowed), '%s'));

                $where .= " AND (object_type <> 'post' OR object_id IN (SELECT ID FROM {$wpdb->posts} WHERE post_type IN ({$placeholders})))";
                $params = array_merge($params, $allowed);
            } else {

                $content_only = (bool) apply_filters('aegisseo_events_content_only', true);
                if ($content_only) {
                    $allowed_pts = $this->get_allowed_event_post_types_sql_in();
                    $placeholders = implode(',', array_fill(0, count($allowed_pts), '%s'));

                    $where .= " AND (object_type <> 'post' OR object_id IN (SELECT ID FROM {$wpdb->posts} WHERE post_type IN ({$placeholders})))";
                    $params = array_merge($params, $allowed_pts);
                }
            }
        }

        $sql = "SELECT * FROM {$table} WHERE {$where} ORDER BY created_at DESC, id DESC LIMIT %d";
        $params[] = $limit;

        return $params ? (array) $wpdb->get_results($wpdb->prepare($sql, $params), ARRAY_A) : array();
    }

    public function count_all($object_type = null, $object_id = null, $allowed_post_types = null) {
        global $wpdb;

        $table = $wpdb->prefix . 'aegisseo_events';

        $where = '1=1';
        $params = array();

        if ($object_type) {
            $where .= ' AND object_type = %s';
            $params[] = (string) $object_type;
        }
        if (null !== $object_id) {
            $where .= ' AND object_id = %d';
            $params[] = (int) $object_id;
        }

		$allowed = $allowed_post_types;

		if (is_string($allowed) && $allowed !== '') {
			$allowed = array_map('trim', explode(',', $allowed));
		}
		if (is_array($allowed)) {
			$allowed = array_values(array_filter(array_map('sanitize_key', $allowed)));
		} else {
			$allowed = array();
		}

		if (!$object_type && null === $object_id) {
			if (!empty($allowed)) {
				$placeholders = implode(',', array_fill(0, count($allowed), '%s'));
				$where .= " AND (object_type <> 'post' OR object_id IN (SELECT ID FROM {$wpdb->posts} WHERE post_type IN ({$placeholders})))";
				$params = array_merge($params, $allowed);
			} else {
				$content_only = (bool) apply_filters('aegisseo_events_content_only', true);
				if ($content_only) {
					$allowed_pts = $this->get_allowed_event_post_types_sql_in();
					$placeholders = implode(',', array_fill(0, count($allowed_pts), '%s'));
					$where .= " AND (object_type <> 'post' OR object_id IN (SELECT ID FROM {$wpdb->posts} WHERE post_type IN ({$placeholders})))";
					$params = array_merge($params, $allowed_pts);
				}
			}
		}
        $sql = "SELECT COUNT(*) FROM {$table} WHERE {$where}";

        if (!empty($params)) {
            return (int) $wpdb->get_var($wpdb->prepare($sql, $params));
        }

        return (int) $wpdb->get_var($sql);
    }

    public function get_recent_paged($limit = 50, $offset = 0, $object_type = null, $object_id = null, $allowed_post_types = null) {
        global $wpdb;

        $limit  = max(1, (int) $limit);
        $offset = max(0, (int) $offset);

        $table = $wpdb->prefix . 'aegisseo_events';

        $where = '1=1';
        $params = array();

        if ($object_type) {
            $where .= ' AND object_type = %s';
            $params[] = (string) $object_type;
        }
        if (null !== $object_id) {
            $where .= ' AND object_id = %d';
            $params[] = (int) $object_id;
        }
		
		$allowed = $allowed_post_types;

		if (is_string($allowed) && $allowed !== '') {
			$allowed = array_map('trim', explode(',', $allowed));
		}
		if (is_array($allowed)) {
			$allowed = array_values(array_filter(array_map('sanitize_key', $allowed)));
		} else {
			$allowed = array();
		}

		if (!$object_type && null === $object_id) {
			if (!empty($allowed)) {
				$placeholders = implode(',', array_fill(0, count($allowed), '%s'));
				$where .= " AND (object_type <> 'post' OR object_id IN (SELECT ID FROM {$wpdb->posts} WHERE post_type IN ({$placeholders})))";
				$params = array_merge($params, $allowed);
			} else {
				$content_only = (bool) apply_filters('aegisseo_events_content_only', true);
				if ($content_only) {
					$allowed_pts = $this->get_allowed_event_post_types_sql_in();
					$placeholders = implode(',', array_fill(0, count($allowed_pts), '%s'));
					$where .= " AND (object_type <> 'post' OR object_id IN (SELECT ID FROM {$wpdb->posts} WHERE post_type IN ({$placeholders})))";
					$params = array_merge($params, $allowed_pts);
				}
			}
		}
		
        $sql = "SELECT * FROM {$table} WHERE {$where} ORDER BY created_at DESC, id DESC LIMIT %d OFFSET %d";
        $params[] = $limit;
        $params[] = $offset;

        return (array) $wpdb->get_results($wpdb->prepare($sql, $params), ARRAY_A);
    }

	public function get_event($id) {
		global $wpdb;

		$id = (int) $id;
		if ($id <= 0) { return null; }

		$table = $wpdb->prefix . 'aegisseo_events';

		$sql = "SELECT * FROM {$table} WHERE id = %d";
		$row = $wpdb->get_row($wpdb->prepare($sql, $id), ARRAY_A);

		return $row ? (array) $row : null;
	}


}