<?php
namespace AegisSEO\SEO;

if (!defined('ABSPATH')) { exit; }

class Score {

    public function analyze_post($post_id, $focus_phrase = '') {
        $post = get_post($post_id);
        if (!$post) return array('score'=>0,'checks'=>array());

        $title = (string) get_the_title($post_id);
        $content_html = (string) $post->post_content;
        $content_plain = wp_strip_all_tags($content_html);
        $content_plain = preg_replace('/\s{2,}/', ' ', trim($content_plain));

        $desc = get_post_meta($post_id, '_aegisseo_description', true);
        if (!is_string($desc) || trim($desc) === '') {
            $desc = mb_substr($content_plain, 0, 160);
        }

        $checks = array();

        $focus = trim((string) $focus_phrase);
        if ($focus !== '') {
            $in_title = (stripos($title, $focus) !== false);
            $in_content = (stripos($content_plain, $focus) !== false);
            $checks['focus_phrase'] = array(
                'label' => __('Focus phrase used in title/content', 'aegisseo'),
                'ok'    => ($in_title && $in_content),
                'detail'=> ($in_title ? __('In title','aegisseo') : __('Not in title','aegisseo')) . ' / ' . ($in_content ? __('In content','aegisseo') : __('Not in content','aegisseo')),
            );
        } else {
            $checks['focus_phrase'] = array(
                'label' => __('Focus phrase set', 'aegisseo'),
                'ok'    => false,
                'detail'=> __('Set a focus phrase to enable this check.', 'aegisseo'),
            );
        }

        $len_title = mb_strlen($title);
        $checks['title_length'] = array(
            'label' => __('Title length (30–60 chars)', 'aegisseo'),
            'ok'    => ($len_title >= 30 && $len_title <= 60),
            'detail'=> sprintf(__('%d chars', 'aegisseo'), $len_title),
        );

        $len_desc = mb_strlen(wp_strip_all_tags((string)$desc));
        $checks['description_length'] = array(
            'label' => __('Meta description length (120–160 chars)', 'aegisseo'),
            'ok'    => ($len_desc >= 120 && $len_desc <= 160),
            'detail'=> sprintf(__('%d chars', 'aegisseo'), $len_desc),
        );

        preg_match_all('/<h[1-6][^>]*>.*?<\/h[1-6]>/is', $content_html, $m);
        $hcount = isset($m[0]) ? count($m[0]) : 0;
        $checks['headings'] = array(
            'label' => __('Headings used (H2/H3 etc.)', 'aegisseo'),
            'ok'    => ($hcount >= 2),
            'detail'=> sprintf(__('%d headings found', 'aegisseo'), $hcount),
        );

        preg_match_all('/<img[^>]*>/i', $content_html, $imgs);
        $img_tags = isset($imgs[0]) ? $imgs[0] : array();
        $img_total = count($img_tags);
        $img_missing_alt = 0;
        foreach ($img_tags as $tag) {
            if (!preg_match('/\salt\s*=\s*(["\']).*?\1/i', $tag) || preg_match('/\salt\s*=\s*(["\'])\s*\1/i', $tag)) {
                $img_missing_alt++;
            }
        }
        $checks['image_alt'] = array(
            'label' => __('Images have alt text', 'aegisseo'),
            'ok'    => ($img_total === 0) ? true : ($img_missing_alt === 0),
            'detail'=> ($img_total === 0) ? __('No images found','aegisseo') : sprintf(__('%d images, %d missing alt', 'aegisseo'), $img_total, $img_missing_alt),
        );

        $internal = 0; $external = 0;
        preg_match_all('/<a\s[^>]*href=(["\'])(.*?)\1[^>]*>/i', $content_html, $links);
        $hrefs = isset($links[2]) ? $links[2] : array();
        $host = wp_parse_url(home_url('/'), PHP_URL_HOST);
        foreach ($hrefs as $href) {
            $href = trim($href);
            if ($href === '' || strpos($href, '#') === 0) continue;
            $hhost = wp_parse_url($href, PHP_URL_HOST);
            if (empty($hhost) || $hhost === $host) $internal++; else $external++;
        }
        $checks['links'] = array(
            'label' => __('Internal & outbound links', 'aegisseo'),
            'ok'    => ($internal >= 1),
            'detail'=> sprintf(__('%d internal / %d outbound', 'aegisseo'), $internal, $external),
        );

        $checks['readability'] = $this->readability($content_plain);

        $points = 0; $total = 0;
        foreach ($checks as $c) { $total++; if (!empty($c['ok'])) $points++; }
        $score = ($total > 0) ? (int) round(($points / $total) * 100) : 0;

        return array('score'=>$score,'checks'=>$checks);
    }

    public function readability($text) {
        $text = trim((string)$text);
        if ($text === '') return array('label'=>__('Readability (simple)','aegisseo'),'ok'=>false,'detail'=>__('No content to analyze.','aegisseo'));

        $sentences = preg_split('/[.!?]+/u', $text, -1, PREG_SPLIT_NO_EMPTY);
        $words = preg_split('/\s+/u', $text, -1, PREG_SPLIT_NO_EMPTY);
        $sentence_count = max(1, is_array($sentences) ? count($sentences) : 1);
        $word_count = max(1, is_array($words) ? count($words) : 1);
        $avg_sentence = $word_count / $sentence_count;

        $syllables = 0;
        foreach ($words as $w) {
            $w = strtolower(preg_replace('/[^a-z]/', '', $w));
            if ($w === '') continue;
            preg_match_all('/[aeiouy]+/', $w, $mm);
            $syllables += max(1, count($mm[0]));
        }

        $flesch = 206.835 - (1.015 * ($word_count / $sentence_count)) - (84.6 * ($syllables / $word_count));
        $ok = ($flesch >= 60 && $avg_sentence <= 20);

        return array('label'=>__('Readability (simple)','aegisseo'),'ok'=>$ok,'detail'=>sprintf(__('Flesch: %d, Avg sentence: %.1f words','aegisseo'), (int)round($flesch), $avg_sentence));
    }

    public function analyze_raw($post_id, $content_html, $override_title = '', $override_desc = '', $focus_phrase = '') {
        $post = $post_id ? get_post($post_id) : null;

        $title = $override_title !== '' ? (string) $override_title : (string) ($post ? get_the_title($post_id) : '');
        $content_html = (string) $content_html;
        $content_plain = wp_strip_all_tags($content_html);
        $content_plain = preg_replace('/\s{2,}/', ' ', trim($content_plain));

        $meta_desc = $override_desc !== '' ? (string) $override_desc : '';

        $focus = (string) $focus_phrase;
        $checks = array();

        $len_title = strlen($title);
        $checks['title_length'] = array(
            'label' => __('SEO title length (50–60 chars)', 'aegisseo'),
            'ok'    => ($len_title >= 50 && $len_title <= 60),
            'detail'=> sprintf(__('%d chars', 'aegisseo'), $len_title),
            'priority' => 1,
        );

        $len_desc = strlen($meta_desc);
        $checks['description_length'] = array(
            'label' => __('Meta description length (120–160 chars)', 'aegisseo'),
            'ok'    => ($len_desc >= 120 && $len_desc <= 160),
            'detail'=> sprintf(__('%d chars', 'aegisseo'), $len_desc),
            'priority' => 1,
        );

        preg_match_all('/<h[1-6][^>]*>.*?<\/h[1-6]>/is', $content_html, $m);
        $hcount = isset($m[0]) ? count($m[0]) : 0;
        $checks['headings'] = array(
            'label' => __('Headings used (H2/H3 etc.)', 'aegisseo'),
            'ok'    => ($hcount >= 2),
            'detail'=> sprintf(__('%d headings found', 'aegisseo'), $hcount),
            'priority' => 2,
        );

        preg_match_all('/<img[^>]+>/i', $content_html, $imgs);
        $img_count = isset($imgs[0]) ? count($imgs[0]) : 0;
        $checks['images'] = array(
            'label' => __('Images used', 'aegisseo'),
            'ok'    => ($img_count >= 1),
            'detail'=> sprintf(__('%d images found', 'aegisseo'), $img_count),
            'priority' => 4,
        );

        $missing_alt = 0;
        if (!empty($imgs[0])) {
            foreach ($imgs[0] as $tag) {
                if (stripos($tag, ' alt=') === false) { $missing_alt++; continue; }
                if (preg_match('/\salt=(["\'])\s*(.*?)\s*\1/i', $tag, $mm)) {
                    $alt = trim((string) $mm[2]);
                    if ($alt === '') $missing_alt++;
                } else {
                    $missing_alt++;
                }
            }
        }
        $checks['image_alt'] = array(
            'label' => __('Images have alt text', 'aegisseo'),
            'ok'    => ($img_count === 0 ? true : $missing_alt === 0),
            'detail'=> ($img_count === 0 ? __('No images', 'aegisseo') : sprintf(__('%d missing alt', 'aegisseo'), $missing_alt)),
            'priority' => 3,
        );

        $word_count = str_word_count($content_plain);
        $checks['thin_content'] = array(
            'label' => __('Content depth (300+ words)', 'aegisseo'),
            'ok'    => ($word_count >= 300),
            'detail'=> sprintf(__('%d words', 'aegisseo'), $word_count),
            'priority' => 2,
        );

        if ($focus !== '') {
            $focus_l = mb_strtolower($focus);
            $title_l = mb_strtolower($title);
            $content_l = mb_strtolower($content_plain);
            $checks['focus_in_title'] = array(
                'label' => __('Focus phrase in title', 'aegisseo'),
                'ok'    => (strpos($title_l, $focus_l) !== false),
                'detail'=> (strpos($title_l, $focus_l) !== false ? __('Yes', 'aegisseo') : __('No', 'aegisseo')),
                'priority' => 1,
            );
            $checks['focus_in_content'] = array(
                'label' => __('Focus phrase in content', 'aegisseo'),
                'ok'    => (strpos($content_l, $focus_l) !== false),
                'detail'=> (strpos($content_l, $focus_l) !== false ? __('Yes', 'aegisseo') : __('No', 'aegisseo')),
                'priority' => 3,
            );
        }

        preg_match_all('/<a\s[^>]*href=("|\')(.*?)\1[^>]*>/i', $content_html, $links);
        $internal = 0;
        $external = 0;
        if (!empty($links[2])) {
            $home = home_url();
            foreach ($links[2] as $href) {
                if ($href === '' || strpos($href, '#') === 0) continue;
                if (strpos($href, $home) === 0 || strpos($href, '/') === 0) $internal++; else $external++;
            }
        }
        $checks['internal_links'] = array(
            'label' => __('Internal links (2+ recommended)', 'aegisseo'),
            'ok'    => ($internal >= 2),
            'detail'=> sprintf(__('%d internal', 'aegisseo'), $internal),
            'priority' => 4,
        );

        $max = 0;
        $got = 0;
        foreach ($checks as $ck) {
            $w = isset($ck['priority']) ? (int) $ck['priority'] : 3;
            $weight = max(1, 6 - $w); // priority 1 -> 5 weight, 5 -> 1 weight
            $max += $weight;
            if (!empty($ck['ok'])) $got += $weight;
        }
        $score = $max > 0 ? (int) round(($got / $max) * 100) : 0;

        $actions = array();
        foreach ($checks as $key => $ck) {
            if (!empty($ck['ok'])) continue;
            $actions[] = array(
                'key' => $key,
                'label' => $ck['label'],
                'detail' => $ck['detail'] ?? '',
                'priority' => $ck['priority'] ?? 3,
                'hint' => $this->hint_for_check($key),
            );
        }
        usort($actions, function($a, $b) {
            return (int)$a['priority'] <=> (int)$b['priority'];
        });
        $actions = array_slice($actions, 0, 10);

        return array('score' => $score, 'checks' => $checks, 'actions' => $actions);
    }

    private function hint_for_check($key) {
        $hints = array(
            'title_length' => __('Aim for ~55 characters. Use a clear benefit + keyword.', 'aegisseo'),
            'description_length' => __('Write 1–2 sentences describing the page and include the keyword naturally.', 'aegisseo'),
            'headings' => __('Add at least two subheadings (H2/H3) to structure the content.', 'aegisseo'),
            'thin_content' => __('Expand the page to answer the search intent thoroughly (300+ words).', 'aegisseo'),
            'image_alt' => __('Add descriptive alt text to images (helps accessibility and image search).', 'aegisseo'),
            'focus_in_title' => __('Include the focus phrase near the beginning of the title.', 'aegisseo'),
            'focus_in_content' => __('Use the focus phrase naturally in the first 100 words and a subheading.', 'aegisseo'),
            'internal_links' => __('Add 2+ internal links to relevant pages using descriptive anchors.', 'aegisseo'),
        );
        return $hints[$key] ?? '';
    }

}
