<?php
if (!defined('ABSPATH')) { exit; }
if (!current_user_can('manage_options')) { echo '<div class="notice notice-error"><p>Forbidden.</p></div>'; return; }

global $wpdb;

$post_type = isset($_GET['post_type']) ? sanitize_key($_GET['post_type']) : 'any';
$filter = isset($_GET['filter']) ? sanitize_key($_GET['filter']) : '';
$search = isset($_GET['s']) ? sanitize_text_field(wp_unslash($_GET['s'])) : '';

$paged = isset($_GET['paged']) ? max(1, (int) $_GET['paged']) : 1;
$per_page = isset($_GET['per_page']) ? (int) $_GET['per_page'] : 25;
if (!in_array($per_page, array(25,50,100), true)) { $per_page = 25; }
$offset = ($paged - 1) * $per_page;

$post_types = get_post_types(array('public'=>true),'objects');
$all_types = array_keys($post_types);
$chosen_types = ($post_type === 'any' ? $all_types : array($post_type));

function aegisseo_bulk_cache_key($filter, $types, $search) {
    return 'aegisseo_bulk_' . md5(implode('|', array((string)$filter, implode(',', (array)$types), (string)$search, (string)get_current_blog_id())));
}

function aegisseo_bulk_normalize_title($t) {
    $t = wp_strip_all_tags((string) $t);
    $t = strtolower($t);
    $t = preg_replace('/\s+/', ' ', $t);
    return trim($t);
}

function aegisseo_bulk_compute_ids($filter, $types, $search) {
    $cache_key = aegisseo_bulk_cache_key($filter, $types, $search);
    $cached = get_transient($cache_key);
    if (is_array($cached)) return $cached;

    $ids = array();

    $chunk = 500;
    $page = 0;

    do {
        $args = array(
            'post_type'      => (array) $types,
            'post_status'    => array('publish','draft','pending','future','private'),
            'posts_per_page' => $chunk,
            'offset'         => $page * $chunk,
            'orderby'        => 'ID',
            'order'          => 'ASC',
            'fields'         => 'ids',
            's'              => ($search !== '' ? $search : ''),
        );
        $batch_ids = get_posts($args);
        if (empty($batch_ids)) break;

        foreach ($batch_ids as $id) {
            $p = get_post($id);
            if (!$p) continue;

            if ($filter === 'thin') {
                $content_plain = wp_strip_all_tags((string) $p->post_content);
                $wc = str_word_count($content_plain);
                if ($wc < 300) $ids[] = (int) $id;
                continue;
            }

            if ($filter === 'dup_title') {
                $ids[] = (int) $id;
                continue;
            }
        }

        $page++;
    } while (count($batch_ids) === $chunk);

    if ($filter === 'dup_title') {
        $counts = array();
        $norm_by_id = array();

        foreach ($ids as $id) {
            $seo_title = (string) get_post_meta($id, '_aegisseo_title', true);
            $title = ($seo_title !== '' ? $seo_title : (string) get_the_title($id));
            $norm = aegisseo_bulk_normalize_title($title);
            if ($norm === '') continue;
            $norm_by_id[$id] = $norm;
            if (!isset($counts[$norm])) $counts[$norm] = 0;
            $counts[$norm]++;
        }

        $dup_ids = array();
        foreach ($norm_by_id as $id => $norm) {
            if (!empty($counts[$norm]) && $counts[$norm] > 1) $dup_ids[] = (int) $id;
        }
        $ids = $dup_ids;
    }

    set_transient($cache_key, $ids, 10 * MINUTE_IN_SECONDS);
    return $ids;
}

$rows = array();
$total = 0;

if ($filter === '' || $filter === 'no_meta' || $filter === 'noindex') {
    $posts_table = $wpdb->posts;
    $pm_table = $wpdb->postmeta;

    $types_in = implode(',', array_fill(0, count($chosen_types), '%s'));
    $statuses_in = implode(',', array_fill(0, 5, '%s'));

    $params = array_merge($chosen_types, array('publish','draft','pending','future','private'));

    $where = "p.post_type IN ($types_in) AND p.post_status IN ($statuses_in)";
    if ($search !== '') {
        $where .= " AND (p.post_title LIKE %s OR p.post_content LIKE %s)";
        $like = '%' . $wpdb->esc_like($search) . '%';
        $params[] = $like;
        $params[] = $like;
    }

    $join = "";
    $having = "";
    if ($filter === 'noindex') {
        $join .= " INNER JOIN $pm_table pm_noindex ON (pm_noindex.post_id = p.ID AND pm_noindex.meta_key = '_aegisseo_noindex')";
        $where .= " AND pm_noindex.meta_value = '1'";
    } elseif ($filter === 'no_meta') {
        $join .= " LEFT JOIN $pm_table pm_title ON (pm_title.post_id = p.ID AND pm_title.meta_key = '_aegisseo_title')";
        $join .= " LEFT JOIN $pm_table pm_desc  ON (pm_desc.post_id = p.ID AND pm_desc.meta_key = '_aegisseo_description')";
        $where .= " AND (pm_title.meta_value IS NULL OR pm_title.meta_value = '') AND (pm_desc.meta_value IS NULL OR pm_desc.meta_value = '')";
    }

    $sql_total = "SELECT COUNT(DISTINCT p.ID) FROM $posts_table p $join WHERE $where";
    $total = (int) $wpdb->get_var($wpdb->prepare($sql_total, $params));

    $sql_ids = "SELECT DISTINCT p.ID FROM $posts_table p $join WHERE $where ORDER BY p.ID DESC LIMIT %d OFFSET %d";
    $params_ids = array_merge($params, array($per_page, $offset));
    $ids = $wpdb->get_col($wpdb->prepare($sql_ids, $params_ids));

    if (!empty($ids)) {
        $posts = get_posts(array(
            'post__in'       => array_map('intval', $ids),
            'orderby'        => 'post__in',
            'posts_per_page' => count($ids),
            'post_type'      => $chosen_types,
            'post_status'    => array('publish','draft','pending','future','private'),
        ));
    } else {
        $posts = array();
    }

    foreach ($posts as $p) {
        $id = $p->ID;
        $seo_title = (string) get_post_meta($id, '_aegisseo_title', true);
        $seo_desc  = (string) get_post_meta($id, '_aegisseo_description', true);
        $canon     = (string) get_post_meta($id, '_aegisseo_canonical', true);
        $noindex   = (int) get_post_meta($id, '_aegisseo_noindex', true);
        $nofollow  = (int) get_post_meta($id, '_aegisseo_nofollow', true);

        $content_plain = wp_strip_all_tags((string) $p->post_content);
        $word_count = str_word_count($content_plain);

        $rows[] = array($p, $seo_title, $seo_desc, $canon, $noindex, $nofollow, $word_count);
    }
} else {
    $ids = aegisseo_bulk_compute_ids($filter, $chosen_types, $search);
    $total = count($ids);
    $page_ids = array_slice($ids, $offset, $per_page);

    $posts = array();
    if (!empty($page_ids)) {
        $posts = get_posts(array(
            'post__in'       => array_map('intval', $page_ids),
            'orderby'        => 'post__in',
            'posts_per_page' => count($page_ids),
            'post_type'      => $chosen_types,
            'post_status'    => array('publish','draft','pending','future','private'),
        ));
    }

    foreach ($posts as $p) {
        $id = $p->ID;
        $seo_title = (string) get_post_meta($id, '_aegisseo_title', true);
        $seo_desc  = (string) get_post_meta($id, '_aegisseo_description', true);
        $canon     = (string) get_post_meta($id, '_aegisseo_canonical', true);
        $noindex   = (int) get_post_meta($id, '_aegisseo_noindex', true);
        $nofollow  = (int) get_post_meta($id, '_aegisseo_nofollow', true);

        $content_plain = wp_strip_all_tags((string) $p->post_content);
        $word_count = str_word_count($content_plain);

        $rows[] = array($p, $seo_title, $seo_desc, $canon, $noindex, $nofollow, $word_count);
    }
}

$total_pages = max(1, (int) ceil($total / $per_page));
if ($paged > $total_pages) { $paged = $total_pages; }

$nonce_ajax = wp_create_nonce('aegisseo_bulk_save_ajax');
?>
<div class="wrap">
    <h2><?php echo esc_html__('Bulk SEO Manager', 'aegisseo'); ?></h2>

    <div id="aegisseo-bulk-notice"></div>

    <form method="get" action="<?php echo esc_url(admin_url('admin.php')); ?>" style="margin: 12px 0;">
        <input type="hidden" name="page" value="aegisseo" />
        <input type="hidden" name="tab" value="bulk" />

        <label style="margin-right:12px;">
            <?php echo esc_html__('Post Type', 'aegisseo'); ?>
            <select name="post_type">
                <option value="any" <?php selected($post_type, 'any'); ?>><?php echo esc_html__('All', 'aegisseo'); ?></option>
                <?php foreach ($post_types as $k => $obj): ?>
                    <option value="<?php echo esc_attr($k); ?>" <?php selected($post_type, $k); ?>><?php echo esc_html($obj->labels->singular_name); ?></option>
                <?php endforeach; ?>
            </select>
        </label>

        <label style="margin-right:12px;">
            <?php echo esc_html__('Filter', 'aegisseo'); ?>
            <select name="filter">
                <option value="" <?php selected($filter, ''); ?>><?php echo esc_html__('None', 'aegisseo'); ?></option>
                <option value="no_meta" <?php selected($filter, 'no_meta'); ?>><?php echo esc_html__('No title & description', 'aegisseo'); ?></option>
                <option value="thin" <?php selected($filter, 'thin'); ?>><?php echo esc_html__('Thin content (<300 words)', 'aegisseo'); ?></option>
                <option value="noindex" <?php selected($filter, 'noindex'); ?>><?php echo esc_html__('Noindex enabled', 'aegisseo'); ?></option>
                <option value="dup_title" <?php selected($filter, 'dup_title'); ?>><?php echo esc_html__('Duplicate SEO title (site-wide)', 'aegisseo'); ?></option>
            </select>
        </label>

        <label style="margin-right:12px;">
            <?php echo esc_html__('Per page', 'aegisseo'); ?>
            <select name="per_page">
                <option value="25" <?php selected($per_page, 25); ?>>25</option>
                <option value="50" <?php selected($per_page, 50); ?>>50</option>
                <option value="100" <?php selected($per_page, 100); ?>>100</option>
            </select>
        </label>

        <label style="margin-right:12px;">
            <?php echo esc_html__('Search', 'aegisseo'); ?>
            <input type="text" name="s" value="<?php echo esc_attr($search); ?>" placeholder="<?php echo esc_attr__('Title or content…', 'aegisseo'); ?>" />
        </label>

        <button class="button"><?php echo esc_html__('Apply', 'aegisseo'); ?></button>
    </form>

    <div class="aegisseo-card" style="padding:12px; margin-bottom:12px;">
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline-block; margin-right:8px;">
            <?php wp_nonce_field('aegisseo_bulk_export'); ?>
            <input type="hidden" name="action" value="aegisseo_bulk_export" />
            <input type="hidden" name="post_type" value="<?php echo esc_attr($post_type); ?>" />
            <button class="button"><?php echo esc_html__('Export CSV', 'aegisseo'); ?></button>
        </form>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" enctype="multipart/form-data" style="display:inline-block;">
            <?php wp_nonce_field('aegisseo_bulk_import'); ?>
            <input type="hidden" name="action" value="aegisseo_bulk_import" />
            <input type="file" name="csv" accept=".csv,text/csv" required />
            <button class="button"><?php echo esc_html__('Import CSV', 'aegisseo'); ?></button>
        </form>

        <span style="margin-left:10px; color:#666;">
            <?php echo esc_html(sprintf(__('Matches: %d', 'aegisseo'), (int) $total)); ?>
        </span>
    </div>

    <form id="aegisseo-bulk-form" method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" data-aegisseo-ajax-nonce="<?php echo esc_attr($nonce_ajax); ?>">
        <?php wp_nonce_field('aegisseo_bulk_save'); ?>
        <input type="hidden" name="action" value="aegisseo_bulk_save" />

        <table class="widefat striped">
            <thead>
                <tr>
                    <th><?php echo esc_html__('Post', 'aegisseo'); ?></th>
                    <th><?php echo esc_html__('SEO Title', 'aegisseo'); ?></th>
                    <th><?php echo esc_html__('Meta Description', 'aegisseo'); ?></th>
                    <th><?php echo esc_html__('Canonical', 'aegisseo'); ?></th>
                    <th><?php echo esc_html__('Noindex', 'aegisseo'); ?></th>
                    <th><?php echo esc_html__('Nofollow', 'aegisseo'); ?></th>
                    <th><?php echo esc_html__('Words', 'aegisseo'); ?></th>
                </tr>
            </thead>
            <tbody>
            <?php if (empty($rows)): ?>
                <tr><td colspan="7"><?php echo esc_html__('No rows found for this view.', 'aegisseo'); ?></td></tr>
            <?php else: foreach ($rows as $r): list($p,$seo_title,$seo_desc,$canon,$noindex,$nofollow,$wc) = $r; ?>
                <tr data-post-id="<?php echo (int) $p->ID; ?>">
                    <td>
                        <strong><?php echo esc_html($p->post_title); ?></strong><br/>
                        <a href="<?php echo esc_url(get_edit_post_link($p->ID)); ?>"><?php echo esc_html__('Edit', 'aegisseo'); ?></a>
                        <span style="color:#666;">(#<?php echo (int)$p->ID; ?>)</span>
                    </td>
                    <td><input class="widefat" type="text" name="rows[<?php echo (int)$p->ID; ?>][title]" value="<?php echo esc_attr($seo_title); ?>" /></td>
                    <td><textarea class="widefat" name="rows[<?php echo (int)$p->ID; ?>][description]" rows="2"><?php echo esc_textarea($seo_desc); ?></textarea></td>
                    <td><input class="widefat" type="url" name="rows[<?php echo (int)$p->ID; ?>][canonical]" value="<?php echo esc_attr($canon); ?>" /></td>
                    <td style="text-align:center;"><input type="checkbox" name="rows[<?php echo (int)$p->ID; ?>][noindex]" value="1" <?php checked($noindex, 1); ?> /></td>
                    <td style="text-align:center;"><input type="checkbox" name="rows[<?php echo (int)$p->ID; ?>][nofollow]" value="1" <?php checked($nofollow, 1); ?> /></td>
                    <td><?php echo (int)$wc; ?></td>
                </tr>
            <?php endforeach; endif; ?>
            </tbody>
        </table>

        <p style="margin-top:12px; display:flex; gap:10px; align-items:center;">
            <button id="aegisseo-bulk-save-btn" class="button button-primary"><?php echo esc_html__('Save Changes', 'aegisseo'); ?></button>
            <span id="aegisseo-bulk-saving" style="display:none;"><?php echo esc_html__('Saving…', 'aegisseo'); ?></span>
        </p>
    </form>

    <?php if ($total_pages > 1): 
        $base_url = admin_url('admin.php?page=aegisseo&tab=bulk&post_type=' . urlencode($post_type) . '&filter=' . urlencode($filter) . '&per_page=' . urlencode((string)$per_page) . '&s=' . urlencode($search));
    ?>
        <div class="tablenav" style="margin-top:12px;">
            <div class="tablenav-pages">
                <span class="displaying-num"><?php echo esc_html(sprintf(__('%d items', 'aegisseo'), (int)$total)); ?></span>
                <span class="pagination-links">
                    <?php
                    $prev = max(1, $paged - 1);
                    $next = min($total_pages, $paged + 1);
                    ?>
                    <a class="button" href="<?php echo esc_url($base_url . '&paged=1'); ?>">&laquo;</a>
                    <a class="button" href="<?php echo esc_url($base_url . '&paged=' . $prev); ?>">&lsaquo;</a>
                    <span class="paging-input">
                        <?php echo esc_html($paged . ' / ' . $total_pages); ?>
                    </span>
                    <a class="button" href="<?php echo esc_url($base_url . '&paged=' . $next); ?>">&rsaquo;</a>
                    <a class="button" href="<?php echo esc_url($base_url . '&paged=' . $total_pages); ?>">&raquo;</a>
                </span>
            </div>
        </div>
    <?php endif; ?>
</div>
