<?php
/**
 * Plugin Name: AegisBackup
 * Description: Backup & migrate WordPress sites (files + database) with safe DB prefix migration.
 * Version: 1.1.17
 * Author: Aegisify
 * Text Domain: aegisbackup
 */

defined( 'ABSPATH' ) || exit;

define( 'AEGISBACKUP_VERSION', '1.1.17' );
define( 'AEGISBACKUP_FILE', __FILE__ );
define( 'AEGISBACKUP_DIR', plugin_dir_path( __FILE__ ) );
define( 'AEGISBACKUP_URL', plugin_dir_url( __FILE__ ) );

define( 'AEGISBACKUP_BASENAME', plugin_basename( __FILE__ ) );
define( 'AEGISBACKUP_SLUG', 'aegisbackup' );
define( 'AEGISBACKUP_CLM_BASE', 'https://aegisify.com' );
define( 'AEGISBACKUP_CLM_FREE_REG_ENDPOINT', AEGISBACKUP_CLM_BASE . '/wp-json/chue-license/v1/aegisbackup/free-register' );
define( 'AEGISBACKUP_CLM_UPDATE_ENDPOINT', AEGISBACKUP_CLM_BASE . '/wp-json/chue-license/v1/aegisbackup/update-check' );
define( 'AEGISBACKUP_CLM_ACTIVATE_ENDPOINT', AEGISBACKUP_CLM_BASE . '/wp-json/chue-license/v1/aegisbackup/activate' );
define( 'AEGISBACKUP_CLM_DEACTIVATE_ENDPOINT', AEGISBACKUP_CLM_BASE . '/wp-json/chue-license/v1/aegisbackup/deactivate' );
define( 'AEGISBACKUP_CLM_STATUS_ENDPOINT', AEGISBACKUP_CLM_BASE . '/wp-json/chue-license/v1/aegisbackup/status' );

require_once AEGISBACKUP_DIR . 'includes/class-ab-plugin.php';

require_once AEGISBACKUP_DIR . 'includes/utils/class-ab-license.php';

function aegisbackup() {
    return \AegisBackup\AB_Plugin::instance();
}

add_action( 'plugins_loaded', array( aegisbackup(), 'init' ) );

register_activation_hook( __FILE__, function() {
    if ( class_exists( '\\AegisBackup\\Utils\\License' ) ) {
        $lic = new \AegisBackup\Utils\License();
        $lic->maybe_send_free_register( AEGISBACKUP_VERSION );
    }
} );

function aegisbackup_clm_check_for_updates( $transient ) {
    if ( ! is_object( $transient ) || empty( $transient->checked ) || ! is_array( $transient->checked ) ) {
        return $transient;
    }

    if ( ! defined( 'AEGISBACKUP_BASENAME' ) || empty( $transient->checked[ AEGISBACKUP_BASENAME ] ) ) {
        return $transient;
    }

    $cache_key = 'aegisbackup_clm_update_cache';
    $cached    = get_site_transient( $cache_key );

    if ( is_object( $cached ) && ! empty( $cached->new_version ) ) {
        $installed = (string) $transient->checked[ AEGISBACKUP_BASENAME ];
        if ( $installed !== '' && version_compare( $installed, (string) $cached->new_version, '>=' ) ) {
            delete_site_transient( $cache_key );
        } else {
            $transient->response[ AEGISBACKUP_BASENAME ] = $cached;
            return $transient;
        }
    }

    $current_version = (string) $transient->checked[ AEGISBACKUP_BASENAME ];

    $payload = array(
        'product'     => defined( 'AEGISBACKUP_SLUG' ) ? AEGISBACKUP_SLUG : 'aegisbackup',
        'slug'        => defined( 'AEGISBACKUP_SLUG' ) ? AEGISBACKUP_SLUG : 'aegisbackup',
        'version'     => $current_version,
        'site_url'    => home_url(),
        'license_key' => (string) get_option( 'aegisbackup_license_key', '' ),
        'email'       => (string) get_option( 'aegisbackup_license_email', get_option( 'admin_email', '' ) ),
    );

    $resp = wp_remote_post( AEGISBACKUP_CLM_UPDATE_ENDPOINT, array(
        'timeout' => 20,
        'headers' => array( 'Content-Type' => 'application/json' ),
        'body'    => wp_json_encode( $payload ),
    ) );

    if ( is_wp_error( $resp ) ) {
        return $transient;
    }

    $code = (int) wp_remote_retrieve_response_code( $resp );
    if ( $code < 200 || $code >= 300 ) {
        return $transient;
    }

    $data = json_decode( (string) wp_remote_retrieve_body( $resp ), true );
    if ( ! is_array( $data ) ) {
        return $transient;
    }

    $d = $data;
    if ( ! empty( $data['data'] ) && is_array( $data['data'] ) ) {
        $d = $data['data'];
    }

    if ( empty( $d['success'] ) || empty( $d['update_available'] ) || empty( $d['version'] ) || empty( $d['download_url'] ) ) {
        return $transient;
    }

    $update              = new stdClass();
    $update->slug        = AEGISBACKUP_SLUG;
    $update->plugin      = AEGISBACKUP_BASENAME;
    $update->new_version = (string) $d['version'];
    $update->package     = (string) $d['download_url'];
    $update->url         = ! empty( $d['homepage'] ) ? (string) $d['homepage'] : 'https://aegisify.com';

    $transient->response[ AEGISBACKUP_BASENAME ] = $update;
    set_site_transient( $cache_key, $update, DAY_IN_SECONDS );

    return $transient;
}
add_filter( 'pre_set_site_transient_update_plugins', 'aegisbackup_clm_check_for_updates', 25, 1 );

add_action( 'upgrader_process_complete', function( $upgrader, $hook_extra ) {
    if ( ! empty( $hook_extra['action'] ) && 'update' === $hook_extra['action']
        && ! empty( $hook_extra['type'] ) && 'plugin' === $hook_extra['type']
        && ! empty( $hook_extra['plugins'] ) && is_array( $hook_extra['plugins'] )
        && in_array( AEGISBACKUP_BASENAME, $hook_extra['plugins'], true )
    ) {
        delete_site_transient( 'aegisbackup_clm_update_cache' );
    }
}, 10, 2 );

add_action( 'aegisbackup_clm_daily_check', function () {
    if ( class_exists( '\\AegisBackup\\Utils\\License' ) ) {
        $lic = new \AegisBackup\Utils\License();
        $lic->maybe_daily_check( AEGISBACKUP_VERSION );
    }
    delete_site_transient( 'aegisbackup_clm_update_cache' );
    delete_transient( 'aegisbackup_update_check' );
} );

register_activation_hook( __FILE__, function () {
    if ( ! function_exists( 'wp_next_scheduled' ) ) {
        return;
    }
    if ( ! wp_next_scheduled( 'aegisbackup_clm_daily_check' ) ) {
        wp_schedule_event( time() + HOUR_IN_SECONDS, 'daily', 'aegisbackup_clm_daily_check' );
    }
} );

register_deactivation_hook( __FILE__, function () {
    if ( ! function_exists( 'wp_next_scheduled' ) || ! function_exists( 'wp_unschedule_event' ) ) {
        return;
    }
    $ts = wp_next_scheduled( 'aegisbackup_clm_daily_check' );
    if ( $ts ) {
        wp_unschedule_event( $ts, 'aegisbackup_clm_daily_check' );
    }
} );

register_activation_hook( __FILE__, function() {
	$admin = get_role( 'administrator' );
	if ( $admin && ! $admin->has_cap( 'aegisbackup_manage_backups' ) ) {
		$admin->add_cap( 'aegisbackup_manage_backups' );
	}

	if ( ! get_role( 'aegisbackup_backup_manager' ) ) {
		add_role(
			'aegisbackup_backup_manager',
			'AegisBackup Backup Manager',
			array(
				'read' => true,
				'aegisbackup_manage_backups' => true,
			)
		);
	}
} );

register_activation_hook( __FILE__, function() {
    add_rewrite_rule( '^aegisbackup_connect/?$', 'index.php?aegisbackup_connect=1', 'top' );
    flush_rewrite_rules();
} );

register_activation_hook( __FILE__, function() {
	$admin = get_role( 'administrator' );
	if ( $admin && ! $admin->has_cap( 'aegisbackup_manage_backups' ) ) {
		$admin->add_cap( 'aegisbackup_manage_backups' );
	}
	if ( ! get_role( 'aegisbackup_backup_manager' ) ) {
		add_role(
			'aegisbackup_backup_manager',
			'AegisBackup Backup Manager',
			array(
				'read' => true,
				'aegisbackup_manage_backups' => true,
			)
		);
	}
} );
