/* global AegisBackupDBOverview */
(function(){
  'use strict';

  function byId(id){ return document.getElementById(id); }

  function destroyIfExists(canvas){
    if(!canvas || !window.Chart || !window.Chart.getChart) return;
    try {
      var existing = window.Chart.getChart(canvas);
      if(existing){ existing.destroy(); }
    } catch(e){}
  }

  function bytesToMB(bytes){
    bytes = Number(bytes||0);
    return Math.round((bytes/1024/1024) * 100) / 100;
  }

  function topNFromMap(mapObj, n){
    var arr=[];
    if(mapObj && typeof mapObj==='object'){
      Object.keys(mapObj).forEach(function(k){
        arr.push({k:k, v:Number(mapObj[k]||0)});
      });
    }
    arr.sort(function(a,b){ return b.v - a.v; });
    return arr.slice(0,n||10);
  }

  function ready(fn){
    if(document.readyState === 'loading'){
      document.addEventListener('DOMContentLoaded', fn);
    } else {
      fn();
    }
  }

  ready(function(){
    if(!window.Chart){ return; }
    var p = window.AegisBackupDBOverview || {};
    var topSizes = Array.isArray(p.top_sizes) ? p.top_sizes : [];
    var topRows  = Array.isArray(p.top_rows) ? p.top_rows : [];
    var storage  = (p.storage && typeof p.storage === 'object') ? p.storage : {};
    var cats     = (p.categories && typeof p.categories === 'object') ? p.categories : {};
    var engines  = (p.engine_bytes && typeof p.engine_bytes === 'object') ? p.engine_bytes : {};
    var bubbles  = Array.isArray(p.bubble_tables) ? p.bubble_tables : [];

    var c1 = byId('ab_chart_top_sizes_bar');
    if(c1){
      destroyIfExists(c1);
      var labels1 = topSizes.map(function(x){ return x.name; });
      var data1   = topSizes.map(function(x){ return bytesToMB(x.bytes); });
      new window.Chart(c1, {
        type: 'bar',
        data: {
          labels: labels1,
          datasets: [{ label: 'MB', data: data1 }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          plugins: { legend: { display: false } },
          scales: {
            x: { ticks: { autoSkip: true, maxRotation: 0 } },
            y: { beginAtZero: true, title: { display: true, text: 'MB' } }
          }
        }
      });
    }

    var c2 = byId('ab_chart_top_rows_line');
    if(c2){
      destroyIfExists(c2);
      var labels2 = topRows.map(function(x){ return x.name; });
      var data2   = topRows.map(function(x){ return Number(x.rows||0); });
      new window.Chart(c2, {
        type: 'line',
        data: {
          labels: labels2,
          datasets: [{ label: 'Rows', data: data2, tension: 0.25 }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          plugins: { legend: { display: false } },
          scales: {
            x: { ticks: { autoSkip: true, maxRotation: 0 } },
            y: { beginAtZero: true }
          }
        }
      });
    }

    var c3 = byId('ab_chart_storage_doughnut');
    if(c3){
      destroyIfExists(c3);
      var dataBytes  = Number(storage.data_bytes||0);
      var indexBytes = Number(storage.index_bytes||0);
      new window.Chart(c3, {
        type: 'doughnut',
        data: {
          labels: ['Data (MB)', 'Index (MB)'],
          datasets: [{ data: [bytesToMB(dataBytes), bytesToMB(indexBytes)] }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false
        }
      });
    }

    var c4 = byId('ab_chart_categories_pie');
    if(c4){
      destroyIfExists(c4);
      var coreB   = Number(cats.core_bytes||0);
      var plugB   = Number(cats.plugin_bytes||0);
      var otherB  = Number(cats.other_bytes||0);
      new window.Chart(c4, {
        type: 'pie',
        data: {
          labels: ['WP Core (MB)', 'Plugins (MB)', 'Other (MB)'],
          datasets: [{ data: [bytesToMB(coreB), bytesToMB(plugB), bytesToMB(otherB)] }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false
        }
      });
    }

    var c5 = byId('ab_chart_engine_polar');
    if(c5){
      destroyIfExists(c5);
      var topE = topNFromMap(engines, 8);
      new window.Chart(c5, {
        type: 'polarArea',
        data: {
          labels: topE.map(function(x){ return x.k; }),
          datasets: [{ data: topE.map(function(x){ return bytesToMB(x.v); }) }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false
        }
      });
    }

    var c6 = byId('ab_chart_bubble');
    if(c6){
      destroyIfExists(c6);
      var pts = bubbles.map(function(t){
        var rows = Number(t.rows||0);
        var mb   = bytesToMB(Number(t.bytes||0));
        var r    = Math.max(3, Math.min(18, Math.sqrt(Math.max(0.01, mb))));
        return { x: rows, y: mb, r: r, _name: String(t.name||'') };
      });

      new window.Chart(c6, {
        type: 'bubble',
        data: {
          datasets: [{
            label: 'Tables',
            data: pts
          }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          plugins: {
            tooltip: {
              callbacks: {
                label: function(ctx){
                  var raw = ctx.raw || {};
                  var nm = raw._name ? raw._name + ': ' : '';
                  return nm + 'Rows ' + (raw.x||0) + ' / MB ' + (raw.y||0);
                }
              }
            }
          },
          scales: {
            x: { beginAtZero: true, title: { display: true, text: 'Rows' } },
            y: { beginAtZero: true, title: { display: true, text: 'Size (MB)' } }
          }
        }
      });
    }
  });
})();
