<?php
namespace AegisBackup\Admin;

use AegisBackup\AB_Plugin;

defined( 'ABSPATH' ) || exit;

class AB_REST_API {

    protected $plugin;

    const TOKEN_TRANSIENT_PREFIX = 'aegisbackup_token_';

    const INCOMING_OPTION = 'aegisbackup_pp_incoming_connections';

    public function __construct( AB_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function register_routes() {
		register_rest_route(
			'aegisbackup/v1',
			'/ping',
			array(
				'methods'  => 'GET',
				'permission_callback' => '__return_true',
				'callback' => array( $this, 'ping' ),
			)
		);
		
        register_rest_route(
            'aegisbackup/v1',
            '/token',
            array(
                'methods'  => 'POST',
                'permission_callback' => array( $this, 'can_manage' ),
                'callback' => array( $this, 'token_route' ),
            )
        );

        register_rest_route(
            'aegisbackup/v1',
            '/receive-chunk',
            array(
                'methods'  => 'POST',
                'permission_callback' => '__return_true',
                'callback' => array( $this, 'receive_chunk' ),
            )
        );

        register_rest_route(
            'aegisbackup/v1',
            '/commit',
            array(
                'methods'  => 'POST',
                'permission_callback' => '__return_true',
                'callback' => array( $this, 'commit' ),
            )
        );

        register_rest_route(
            'aegisbackup/v1',
            '/file-put',
            array(
                'methods'  => 'POST',
                'permission_callback' => '__return_true',
                'callback' => array( $this, 'file_put' ),
            )
        );

        register_rest_route(
            'aegisbackup/v1',
            '/register-connection',
            array(
                'methods'  => 'POST',
                'permission_callback' => '__return_true',
                'callback' => array( $this, 'register_connection' ),
            )
        );

        register_rest_route(
            'aegisbackup/v1',
            '/status',
            array(
                'methods' => 'GET',
                'permission_callback' => array( $this, 'can_manage' ),
                'callback' => array( $this, 'status' ),
            )
        );

        register_rest_route(
            'aegisbackup/v1',
            '/pp-destlog',
            array(
                'methods'  => 'GET',
                'permission_callback' => array( $this, 'can_manage' ),
                'callback' => array( $this, 'pp_destlog' ),
            )
        );
    }

	
	public function ping( \WP_REST_Request $request ) {
		return rest_ensure_response(
			array(
				'success' => true,
				'plugin'  => 'aegisbackup',
				'version' => defined( 'AEGISBACKUP_VERSION' ) ? AEGISBACKUP_VERSION : '',
				'home_url' => home_url(),
				'ssl'     => is_ssl() ? 1 : 0,
				'rest_base' => home_url( '/wp-json/aegisbackup/v1/' ),
			)
		);
	}
	
    public function can_manage() {
        return current_user_can( 'manage_options' );
    }

    public function generate_token() {
        $id  = 'abt_' . wp_generate_password( 10, false, false );
        $exp = time() + ( 60 * 60 ); // 60 minutes

        $payload = $id . '|' . $exp;
        $sig = hash_hmac( 'sha256', $payload, wp_salt( 'auth' ) );

        $endpoint = home_url();
        $token = array(
            'endpoint' => $endpoint,
            'id' => $id,
            'exp' => $exp,
            'sig' => $sig,
            'v' => '1',
        );

        set_transient( self::TOKEN_TRANSIENT_PREFIX . $id, array( 'exp' => $exp, 'sig' => $sig ), 60 * 70 );

        return $token;
    }

    public function token_route( \WP_REST_Request $request ) {
        $token = $this->generate_token();

        $warnings = array();
        if ( ! is_ssl() ) {
            if ( ! defined( 'AEGISBACKUP_ALLOW_INSECURE_PUSH' ) || ! AEGISBACKUP_ALLOW_INSECURE_PUSH ) {
                return new \WP_Error( 'ab_https_required', 'Push/Pull requires HTTPS. Enable SSL or define AEGISBACKUP_ALLOW_INSECURE_PUSH for development.', array( 'status' => 400 ) );
            }
            $warnings[] = 'Destination is not using HTTPS. Insecure mode is enabled.';
        }

        return rest_ensure_response(
            array(
                'success' => true,
                'token' => $token,
                'warnings' => $warnings,
            )
        );
    }

    private function validate_token_from_headers() {
        $id  = isset( $_SERVER['HTTP_X_AB_TOKEN_ID'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_AB_TOKEN_ID'] ) ) : '';
        $exp = isset( $_SERVER['HTTP_X_AB_TOKEN_EXP'] ) ? (int) sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_AB_TOKEN_EXP'] ) ) : 0;
        $sig = isset( $_SERVER['HTTP_X_AB_TOKEN_SIG'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_AB_TOKEN_SIG'] ) ) : '';

        if ( empty( $id ) || empty( $exp ) || empty( $sig ) ) {
            return new \WP_Error( 'ab_token_missing', 'Missing token headers.', array( 'status' => 401 ) );
        }
        if ( time() > $exp ) {
            return new \WP_Error( 'ab_token_expired', 'Token expired.', array( 'status' => 401 ) );
        }

        if ( ! is_ssl() && ( ! defined( 'AEGISBACKUP_ALLOW_INSECURE_PUSH' ) || ! AEGISBACKUP_ALLOW_INSECURE_PUSH ) ) {
            return new \WP_Error( 'ab_https_required', 'HTTPS is required for Push/Pull.', array( 'status' => 400 ) );
        }

        $saved = get_transient( self::TOKEN_TRANSIENT_PREFIX . $id );
        if ( ! is_array( $saved ) ) {
            return new \WP_Error( 'ab_token_invalid', 'Token not found.', array( 'status' => 401 ) );
        }
        if ( (int) $saved['exp'] !== (int) $exp ) {
            return new \WP_Error( 'ab_token_invalid', 'Token mismatch.', array( 'status' => 401 ) );
        }
        if ( ! hash_equals( (string) $saved['sig'], (string) $sig ) ) {
            return new \WP_Error( 'ab_token_invalid', 'Token signature invalid.', array( 'status' => 401 ) );
        }

        return array( 'id' => $id, 'exp' => $exp, 'sig' => $sig );
    }

    public function register_connection( \WP_REST_Request $request ) {
        $tok = $this->validate_token_from_headers();
        if ( is_wp_error( $tok ) ) {
            return $tok;
        }

        $payload = json_decode( (string) $request->get_body(), true );
        if ( ! is_array( $payload ) ) {
            $payload = (array) $request->get_json_params();
        }

        $source_url  = isset( $payload['source_url'] ) ? esc_url_raw( (string) $payload['source_url'] ) : '';
        $source_name = isset( $payload['source_name'] ) ? sanitize_text_field( (string) $payload['source_name'] ) : '';
        $connect_name = isset( $payload['connect_name'] ) ? sanitize_text_field( (string) $payload['connect_name'] ) : '';

        if ( empty( $source_url ) ) {
            return new \WP_Error( 'ab_bad_request', 'Missing source_url.', array( 'status' => 400 ) );
        }

        $now = time();
        $incoming = get_option( self::INCOMING_OPTION, array() );
        if ( ! is_array( $incoming ) ) {
            $incoming = array();
        }

        $key = md5( strtolower( rtrim( (string) $source_url, '/' ) ) );
        $incoming[ $key ] = array(
            'source_url'    => $source_url,
            'source_name'   => $source_name,
            'connect_name'  => $connect_name,
            'token_id'      => (string) $tok['id'],
            'connected'     => 1,
            'created'       => isset( $incoming[ $key ]['created'] ) ? (int) $incoming[ $key ]['created'] : $now,
            'last_activity' => $now,
        );

        update_option( self::INCOMING_OPTION, $incoming, false );

        return rest_ensure_response(
            array(
                'success' => true,
                'message' => 'Registered.',
            )
        );
    }

    private function rate_limit_ok() {
        $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? (string) $_SERVER['REMOTE_ADDR'] : 'unknown';
        $key = 'aegisbackup_rl_' . md5( $ip );
        $count = (int) get_transient( $key );
        $count++;
        set_transient( $key, $count, 60 );

        return $count <= 120;
    }

    private function touch_incoming_by_token( $token_id ) {
        $token_id = (string) $token_id;
        if ( '' === $token_id ) {
            return;
        }

        $incoming = get_option( self::INCOMING_OPTION, array() );
        if ( ! is_array( $incoming ) || empty( $incoming ) ) {
            return;
        }

        $now = time();
        foreach ( $incoming as $k => $row ) {
            if ( ! is_array( $row ) ) {
                continue;
            }
            if ( isset( $row['token_id'] ) && (string) $row['token_id'] === $token_id ) {
                $incoming[ $k ]['connected']     = 1;
                $incoming[ $k ]['last_activity'] = $now;
            }
        }

        update_option( self::INCOMING_OPTION, $incoming, false );
    }

    public function receive_chunk( \WP_REST_Request $request ) {
        $tok = $this->validate_token_from_headers();
        if ( is_wp_error( $tok ) ) {
            return $tok;
        }
        $this->touch_incoming_by_token( $tok['id'] );
        if ( ! $this->rate_limit_ok() ) {
            return new \WP_Error( 'ab_rate_limited', 'Rate limited.', array( 'status' => 429 ) );
        }

        $file_id = isset( $_SERVER['HTTP_X_AB_FILE_ID'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_AB_FILE_ID'] ) ) : '';
        $chunk_index = isset( $_SERVER['HTTP_X_AB_CHUNK_INDEX'] ) ? (int) sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_AB_CHUNK_INDEX'] ) ) : -1;
        $total_chunks = isset( $_SERVER['HTTP_X_AB_TOTAL_CHUNKS'] ) ? (int) sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_AB_TOTAL_CHUNKS'] ) ) : 0;
        $total_size = isset( $_SERVER['HTTP_X_AB_TOTAL_SIZE'] ) ? (int) sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_AB_TOTAL_SIZE'] ) ) : 0;

        if ( empty( $file_id ) || $chunk_index < 0 || $total_chunks <= 0 ) {
            return new \WP_Error( 'ab_bad_request', 'Missing file/chunk metadata.', array( 'status' => 400 ) );
        }

        $upload = wp_upload_dir();
        $base = trailingslashit( $upload['basedir'] ) . 'aegisbackup/inbox/' . $file_id;
        $chunks_dir = trailingslashit( $base ) . 'chunks';
        wp_mkdir_p( $chunks_dir );

        $data = $request->get_body();
        if ( '' === $data ) {
            return new \WP_Error( 'ab_bad_request', 'Empty body.', array( 'status' => 400 ) );
        }

        $chunk_file = trailingslashit( $chunks_dir ) . sprintf( 'part-%06d.bin', (int) $chunk_index );
        file_put_contents( $chunk_file, $data );

        try {
            $src_ip = isset( $_SERVER['REMOTE_ADDR'] ) ? (string) $_SERVER['REMOTE_ADDR'] : '';
            $purpose_h = isset( $_SERVER['HTTP_X_AB_PURPOSE'] ) ? sanitize_key( (string) wp_unslash( $_SERVER['HTTP_X_AB_PURPOSE'] ) ) : '';
            $ci = (int) $chunk_index;
            $tc = (int) $total_chunks;
            if ( 0 === $ci || ( $tc > 0 && $ci === ( $tc - 1 ) ) || ( $ci % 10 ) === 0 ) {
                $this->append_destination_live_log(
                    sprintf(
                        '[%s] CHUNK OK token=%s ip=%s purpose=%s file_id=%s chunk=%d/%d bytes=%d',
                        gmdate( 'Y-m-d H:i:s' ),
                        (string) $tok['id'],
                        (string) $src_ip,
                        (string) $purpose_h,
                        (string) $file_id,
                        (int) $ci,
                        (int) $tc,
                        (int) strlen( $data )
                    )
                );
            }
        } catch ( \Throwable $e ) {
        }

        $meta = array(
            'file_id' => $file_id,
            'total_chunks' => $total_chunks,
            'total_size' => $total_size,
            'received_at' => time(),
        );
        file_put_contents( trailingslashit( $base ) . 'meta.json', wp_json_encode( $meta ) );

        $incoming = get_option( self::INCOMING_OPTION, array() );
        if ( is_array( $incoming ) ) {
            foreach ( $incoming as $k => $c ) {
                if ( is_array( $c ) && ! empty( $c['token_id'] ) && (string) $c['token_id'] === (string) $tok['id'] ) {
                    $incoming[ $k ]['connected'] = 1;
                    $incoming[ $k ]['last_activity'] = time();
                    update_option( self::INCOMING_OPTION, $incoming, false );
                    break;
                }
            }
        }

        return rest_ensure_response(
            array(
                'success' => true,
                'received' => $chunk_index,
            )
        );
    }

    public function commit( \WP_REST_Request $request ) {
        $tok = $this->validate_token_from_headers();
        if ( is_wp_error( $tok ) ) {
            return $tok;
        }
        $this->touch_incoming_by_token( $tok['id'] );

        $payload = json_decode( (string) $request->get_body(), true );
        if ( ! is_array( $payload ) ) {
            $payload = $request->get_json_params();
        }
        $file_id = isset( $payload['file_id'] ) ? sanitize_text_field( (string) $payload['file_id'] ) : '';
        $total_chunks = isset( $payload['total_chunks'] ) ? (int) $payload['total_chunks'] : 0;
		$src_root = isset( $payload['src_root'] ) ? sanitize_text_field( (string) $payload['src_root'] ) : '';
		$dst_root = isset( $payload['dst_root'] ) ? sanitize_text_field( (string) $payload['dst_root'] ) : '';
		$paths    = isset( $payload['paths'] ) && is_array( $payload['paths'] ) ? (array) $payload['paths'] : array();
		$purpose  = isset( $payload['purpose'] ) ? sanitize_key( (string) $payload['purpose'] ) : 'package';

        if ( empty( $file_id ) || $total_chunks <= 0 ) {
            return new \WP_Error( 'ab_bad_request', 'Missing commit payload.', array( 'status' => 400 ) );
        }

        $upload = wp_upload_dir();
        $inbox = trailingslashit( $upload['basedir'] ) . 'aegisbackup/inbox/' . $file_id;
        $chunks_dir = trailingslashit( $inbox ) . 'chunks';
        if ( ! is_dir( $chunks_dir ) ) {
            return new \WP_Error( 'ab_not_found', 'Inbox not found.', array( 'status' => 404 ) );
        }

        $assembled = trailingslashit( $inbox ) . 'package.zip';
        $out = fopen( $assembled, 'wb' );
        if ( ! $out ) {
            return new \WP_Error( 'ab_io', 'Failed to create assembled file.', array( 'status' => 500 ) );
        }

        for ( $i = 0; $i < $total_chunks; $i++ ) {
            $part = trailingslashit( $chunks_dir ) . sprintf( 'part-%06d.bin', $i );
            if ( ! is_file( $part ) ) {
                fclose( $out );
                return new \WP_Error( 'ab_missing_chunk', 'Missing chunk ' . $i, array( 'status' => 400 ) );
            }
            $in = fopen( $part, 'rb' );
            while ( ! feof( $in ) ) {
                $buf = fread( $in, 1024 * 1024 );
                if ( false === $buf ) { break; }
                fwrite( $out, $buf );
            }
            fclose( $in );
        }
        fclose( $out );

        $base = trailingslashit( $upload['basedir'] ) . 'aegisbackup';
        wp_mkdir_p( $base );

        $pkg_dir = trailingslashit( $base ) . ( ( 'db' === $purpose ) ? 'incoming-db-' : 'package-' ) . gmdate( 'Ymd_His' ) . '-' . substr( $file_id, -6 );
        wp_mkdir_p( $pkg_dir );

        $dest_zip = trailingslashit( $pkg_dir ) . 'AegisBackup-' . basename( $pkg_dir ) . '.zip';
        @rename( $assembled, $dest_zip );

		if ( 'db' === $purpose || 'files' === $purpose ) {
			try {
				$za = new \ZipArchive();
				if ( true === $za->open( $dest_zip ) ) {
					$za->extractTo( $pkg_dir );
					$za->close();
				}
			} catch ( \Throwable $e ) {

			}
		}

		if ( 'db' === $purpose ) {
            $db_list = get_option( 'aegisbackup_pp_received_db_backups', array() );
            if ( ! is_array( $db_list ) ) {
                $db_list = array();
            }
            $key = md5( $dest_zip );
            $db_list[ $key ] = array(
                'id' => $key,
				'zip' => $dest_zip,
				'package_path' => $pkg_dir,
                'received' => time(),
                'token_id' => (string) $tok['id'],
                'connected' => 1,
                'last_activity' => time(),
            );
            update_option( 'aegisbackup_pp_received_db_backups', $db_list, false );

try {
    $src_ip = isset( $_SERVER['REMOTE_ADDR'] ) ? (string) $_SERVER['REMOTE_ADDR'] : '';
    $this->append_destination_live_log(
        sprintf(
            '[%s] COMMIT OK token=%s ip=%s purpose=db file_id=%s stored=%s',
            gmdate( 'Y-m-d H:i:s' ),
            (string) $tok['id'],
            (string) $src_ip,
            (string) $file_id,
            (string) $pkg_dir
        )
    );
    $this->append_destination_live_log(
        sprintf(
            '[%s] DB STORED token=%s ip=%s %s',
            gmdate( 'Y-m-d H:i:s' ),
            (string) $tok['id'],
            (string) $src_ip,
            (string) $pkg_dir
        )
    );
} catch ( \Throwable $e ) {

}



            return rest_ensure_response(
                array(
                    'success' => true,
                    'stored' => true,
                    'purpose' => 'db',
					'package_dir' => $pkg_dir,
                )
            );
        }

		if ( 'files' === $purpose ) {
			$list = get_option( 'aegisbackup_pp_received_files_migrations', array() );
			if ( ! is_array( $list ) ) { $list = array(); }
			$key = md5( $dest_zip );
			$list[ $key ] = array(
				'id' => $key,
				'zip' => $dest_zip,
				'package_path' => $pkg_dir,
				'received' => time(),
				'token_id' => (string) $tok['id'],
				'connected' => 1,
				'last_activity' => time(),
			);
			update_option( 'aegisbackup_pp_received_files_migrations', $list, false );
			return rest_ensure_response(
				array(
					'success' => true,
					'stored' => true,
					'purpose' => 'files',
					'package_dir' => $pkg_dir,
				)
			);
		}

		if ( 'filebackup' === $purpose ) {
			$uploads = wp_upload_dir();
			$fb_dir = trailingslashit( (string) $uploads['basedir'] ) . 'aegisbackup/file-backups/';
			if ( ! is_dir( $fb_dir ) ) {
				wp_mkdir_p( $fb_dir );
			}

			$backup_id = 'filebackup_migrated_' . gmdate( 'Ymd_His' ) . '_' . substr( (string) $file_id, -6 );
			$dest_fb_zip = $fb_dir . $backup_id . '.zip';
			$dest_fb_manifest = $fb_dir . $backup_id . '.json';

			@copy( $dest_zip, $dest_fb_zip );

			$manifest = array(
				'id' => $backup_id,
				'name' => basename( $dest_fb_zip ),
				'created' => gmdate( 'Y-m-d H:i:s' ) . ' UTC',
				'size' => is_file( $dest_fb_zip ) ? size_format( (int) filesize( $dest_fb_zip ) ) : '',
				'migrated' => true,
				'migrated_from_token' => (string) $tok['id'],
			);
			@file_put_contents( $dest_fb_manifest, wp_json_encode( $manifest ) );

			$list = get_option( 'aegisbackup_pp_received_file_backups', array() );
			if ( ! is_array( $list ) ) { $list = array(); }
			$key = md5( $dest_fb_zip );
			$list[ $key ] = array(
				'id' => $key,
				'backup_id' => $backup_id,
				'name' => basename( $dest_fb_zip ),
				'zip' => $dest_fb_zip,
				'package_path' => $pkg_dir,
				'source_zip' => $dest_zip,
				'received_at' => time(),
				'token_id' => (string) $tok['id'],
				'connected' => 1,
				'last_activity' => time(),
			);
			update_option( 'aegisbackup_pp_received_file_backups', $list, false );

			try {
				$this->append_destination_live_log(
					sprintf(
						'[%s] FILEBACKUP STORED token=%s ip=%s backup_id=%s zip=%s',
						gmdate( 'Y-m-d H:i:s' ),
						(string) $tok['id'],
						(string) $src_ip,
						(string) $backup_id,
						(string) $dest_fb_zip
					)
				);
			} catch ( \Throwable $e ) {

			}

			return rest_ensure_response(
				array(
					'success' => true,
					'stored' => true,
					'purpose' => 'filebackup',
					'backup_id' => $backup_id,
					'zip' => $dest_fb_zip,
					'package_dir' => $pkg_dir,
				)
			);
		}

		$src_root = str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), (string) $src_root );
		$dst_root = str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), (string) $dst_root );
		if ( '' !== $src_root ) {
			$src_root = '/' . ltrim( $src_root, '/' );
			if ( '/' !== substr( $src_root, -1 ) ) { $src_root .= '/'; }
		}
		if ( '' !== $dst_root ) {
			$dst_root = '/' . ltrim( $dst_root, '/' );
			if ( '/' !== substr( $dst_root, -1 ) ) { $dst_root .= '/'; }
		}
		$clean_paths = array();
		foreach ( $paths as $p ) {
			$p = sanitize_text_field( (string) $p );
			$p = str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), $p );
			$p = ltrim( $p, '/' );
			if ( '' === $p ) { continue; }
			if ( false !== strpos( $p, '..' ) ) { continue; }
			$clean_paths[ $p ] = true;
		}

		$job = $this->plugin->restore->start_restore_job(
			array(
				'package_path' => $pkg_dir,
				'mode' => 'migrate',
				'new_prefix' => '',
				'old_domain' => '',
				'new_domain' => '',
				'src_root' => $src_root,
				'dst_root' => $dst_root,
				'paths'    => array_keys( $clean_paths ),
			)
		);

        $incoming = get_option( self::INCOMING_OPTION, array() );
        if ( is_array( $incoming ) ) {
            foreach ( $incoming as $k => $c ) {
                if ( is_array( $c ) && ! empty( $c['token_id'] ) && (string) $c['token_id'] === (string) $tok['id'] ) {
                    $incoming[ $k ]['connected'] = 1;
                    $incoming[ $k ]['last_activity'] = time();
                    update_option( self::INCOMING_OPTION, $incoming, false );
                    break;
                }
            }
        }

        return rest_ensure_response(
            array(
                'success' => true,
                'package_dir' => $pkg_dir,
                'restore_job' => $job,
            )
        );
    }

    public function file_put( \WP_REST_Request $request ) {
        $tok = $this->validate_token_from_headers();
        if ( is_wp_error( $tok ) ) {
            return $tok;
        }
        $this->touch_incoming_by_token( $tok['id'] );
        if ( ! $this->rate_limit_ok() ) {
            return new \WP_Error( 'ab_rate_limited', 'Rate limited.', array( 'status' => 429 ) );
        }

        $payload = json_decode( (string) $request->get_body(), true );
        if ( ! is_array( $payload ) ) {
            $payload = (array) $request->get_json_params();
        }

		$dst_root = isset( $payload['dst_root'] ) ? sanitize_text_field( (string) $payload['dst_root'] ) : '/public_html/';
        $rel_path = isset( $payload['rel_path'] ) ? sanitize_text_field( (string) $payload['rel_path'] ) : '';
        $content_b64 = isset( $payload['content_b64'] ) ? (string) $payload['content_b64'] : '';
        $is_dir = ! empty( $payload['is_dir'] ) ? 1 : 0;

		$src_ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( (string) wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';

        $dst_root = str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), (string) $dst_root );
        $rel_path = str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), (string) $rel_path );
        $rel_path = ltrim( $rel_path, '/' );
        if ( '' === $rel_path ) {
			$this->append_destination_live_log( sprintf( '[%s] FAIL token=%s ip=%s missing rel_path', gmdate( 'Y-m-d H:i:s' ), (string) $tok['id'], (string) $src_ip ) );
            return new \WP_Error( 'ab_bad_request', 'Missing rel_path.', array( 'status' => 400 ) );
        }
        if ( false !== strpos( $rel_path, '..' ) ) {
			$this->append_destination_live_log( sprintf( '[%s] FAIL token=%s ip=%s invalid path: %s', gmdate( 'Y-m-d H:i:s' ), (string) $tok['id'], (string) $src_ip, (string) $rel_path ) );
            return new \WP_Error( 'ab_bad_request', 'Invalid path.', array( 'status' => 400 ) );
        }

		$dst_root_norm = '/' . trim( $dst_root, '/' ) . '/';

		$abs_base = realpath( ABSPATH );
		if ( false === $abs_base ) {
			$abs_base = ABSPATH;
		}
		$abs_base = rtrim( str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), (string) $abs_base ), '/' ) . '/';

		$base = $abs_base;
		$dst_label = trim( $dst_root_norm, '/' );
		$abs_label = basename( rtrim( $abs_base, '/' ) );

		if ( $dst_label && $dst_label !== $abs_label ) {
			$parent = dirname( rtrim( $abs_base, '/' ) );
			$cand = rtrim( str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), (string) $parent ), '/' ) . '/' . $dst_label . '/';
			if ( is_dir( $cand ) ) {
				$cand_real = realpath( $cand );
				if ( false !== $cand_real ) {
					$cand_real = rtrim( str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), (string) $cand_real ), '/' ) . '/';
					if ( 0 === strpos( $cand_real, rtrim( $parent, '/' ) . '/' ) ) {
						$base = $cand_real;
					}
				}
			}
		}

        $target = trailingslashit( $base ) . $rel_path;
        $target_dir = $is_dir ? $target : dirname( $target );
        if ( ! is_dir( $target_dir ) ) {
            wp_mkdir_p( $target_dir );
        }

        if ( $is_dir ) {
            if ( ! is_dir( $target ) ) {
                wp_mkdir_p( $target );
            }
			$this->append_destination_live_log( sprintf( '[%s] DIR  OK token=%s ip=%s %s', gmdate( 'Y-m-d H:i:s' ), (string) $tok['id'], (string) $src_ip, (string) $rel_path ) );
            return rest_ensure_response( array( 'success' => true, 'dir' => true, 'path' => $rel_path ) );
        }

        if ( '' === $content_b64 ) {
			$this->append_destination_live_log( sprintf( '[%s] FILE FAIL token=%s ip=%s missing content: %s', gmdate( 'Y-m-d H:i:s' ), (string) $tok['id'], (string) $src_ip, (string) $rel_path ) );
            return new \WP_Error( 'ab_bad_request', 'Missing content.', array( 'status' => 400 ) );
        }
        $bin = base64_decode( $content_b64, true );
        if ( false === $bin ) {
			$this->append_destination_live_log( sprintf( '[%s] FILE FAIL token=%s ip=%s invalid base64: %s', gmdate( 'Y-m-d H:i:s' ), (string) $tok['id'], (string) $src_ip, (string) $rel_path ) );
            return new \WP_Error( 'ab_bad_request', 'Invalid base64.', array( 'status' => 400 ) );
        }

        $ok = @file_put_contents( $target, $bin );
        if ( false === $ok ) {
			$this->append_destination_live_log( sprintf( '[%s] FILE FAIL token=%s ip=%s write failed: %s', gmdate( 'Y-m-d H:i:s' ), (string) $tok['id'], (string) $src_ip, (string) $rel_path ) );
            return new \WP_Error( 'ab_io', 'Failed to write file.', array( 'status' => 500 ) );
        }

		$this->append_destination_live_log( sprintf( '[%s] FILE OK token=%s ip=%s %s [%d bytes]', gmdate( 'Y-m-d H:i:s' ), (string) $tok['id'], (string) $src_ip, (string) $rel_path, (int) $ok ) );

        return rest_ensure_response( array( 'success' => true, 'path' => $rel_path, 'bytes' => (int) $ok ) );
    }

	private function append_destination_live_log( $line ) {
		$line = (string) $line;
		if ( '' === $line ) {
			return;
		}
		$list = get_option( 'aegisbackup_pp_destination_live_log', array() );
		if ( ! is_array( $list ) ) {
			$list = array();
		}
		$list[] = $line;
		if ( count( $list ) > 300 ) {
			$list = array_slice( $list, -300 );
		}
		update_option( 'aegisbackup_pp_destination_live_log', $list, false );
	}

    public function pp_destlog( \WP_REST_Request $request ) {
        if ( ! $this->can_manage() ) {
            return new \WP_Error( 'ab_forbidden', 'Forbidden', array( 'status' => 403 ) );
        }

        $list = get_option( 'aegisbackup_pp_destination_live_log', array() );
        if ( ! is_array( $list ) ) {
            $list = array();
        }

        $max = 200;
        if ( count( $list ) > $max ) {
            $list = array_slice( $list, -$max );
        }

        $lines = array();
        foreach ( $list as $row ) {
            if ( is_string( $row ) ) {
                $lines[] = $row;
            }
        }

        return rest_ensure_response( array( 'success' => true, 'lines' => $lines ) );
    }

    public function status( \WP_REST_Request $request ) {
        $job_id = sanitize_text_field( (string) $request->get_param( 'job_id' ) );
        if ( empty( $job_id ) ) {
            return rest_ensure_response( array( 'success' => false, 'message' => 'Missing job_id.' ) );
        }

        $state = get_option( \AegisBackup\Restore\AB_Restore_Manager::JOB_OPTION_PREFIX . $job_id, array() );
        if ( empty( $state['job_id'] ) ) {
            return rest_ensure_response( array( 'success' => false, 'message' => 'Job not found.' ) );
        }

        return rest_ensure_response(
            array(
                'success' => true,
                'job_id' => $job_id,
                'status' => $state['status'],
                'phase' => $state['phase'],
                'progress' => (int) $state['progress'],
                'last_log' => isset( $state['last_log'] ) ? (string) $state['last_log'] : '',
            )
        );
    }
}
