<?php
namespace AegisBackup\Admin\Pages;

use AegisBackup\AB_Plugin;

defined( 'ABSPATH' ) || exit;

require_once AEGISBACKUP_DIR . 'includes/libs/class-ab-activity-log.php';

class AB_Page_Logs {

    protected $plugin;

    public function __construct( AB_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function render() {
        if ( ! current_user_can( 'manage_options' ) && ! current_user_can( 'aegisbackup_manage_backups' ) ) {
            wp_die( esc_html__( 'Insufficient permissions.', 'aegisbackup' ), 403 );
        }

        $tab = isset( $_GET['tab'] ) ? sanitize_key( (string) wp_unslash( $_GET['tab'] ) ) : 'logs';
        if ( ! in_array( $tab, array( 'logs', 'remote_storage', 'smtp', 'preupdate' ), true ) ) {
            $tab = 'logs';
        }

        $ab_ok  = isset( $_GET['ab_ok'] ) ? (int) $_GET['ab_ok'] : null;
        $ab_msg = isset( $_GET['ab_msg'] ) ? (string) wp_unslash( $_GET['ab_msg'] ) : '';
        $base_url = admin_url( 'admin.php?page=aegisbackup-logs' );
        $is_pro = false;
        if ( isset( $this->plugin ) && isset( $this->plugin->license ) && is_object( $this->plugin->license ) && method_exists( $this->plugin->license, 'is_pro_active' ) ) {
            $is_pro = (bool) $this->plugin->license->is_pro_active();
        }
        $license_url = admin_url( 'admin.php?page=aegisbackup-license' );
        $locked_tabs = array( 'remote_storage', 'smtp', 'preupdate' );
        $needs_lock  = ( ! $is_pro && in_array( $tab, $locked_tabs, true ) );
        ?>
        <div class="wrap aegisbackup-wrap">
            <h1><?php esc_html_e( 'AegisBackup Settings', 'aegisbackup' ); ?></h1>

            <?php if ( null !== $ab_ok && '' !== $ab_msg ) : ?>
                <div class="notice <?php echo $ab_ok ? 'notice-success' : 'notice-error'; ?> is-dismissible">
                    <p><?php echo esc_html( $ab_msg ); ?></p>
                </div>
            <?php endif; ?>

            <h2 class="nav-tab-wrapper">
                <a class="nav-tab <?php echo ( 'logs' === $tab ) ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( add_query_arg( array( 'tab' => 'logs' ), $base_url ) ); ?>"><?php esc_html_e( 'Logs', 'aegisbackup' ); ?></a>
                <a class="nav-tab <?php echo ( 'remote_storage' === $tab ) ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( add_query_arg( array( 'tab' => 'remote_storage' ), $base_url ) ); ?>"><?php esc_html_e( 'Remote Storage', 'aegisbackup' ); ?></a>
                <a class="nav-tab <?php echo ( 'smtp' === $tab ) ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( add_query_arg( array( 'tab' => 'smtp' ), $base_url ) ); ?>"><?php esc_html_e( 'SMTP & Notifications', 'aegisbackup' ); ?></a>
				<a class="nav-tab <?php echo ( 'preupdate' === $tab ) ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( add_query_arg( 'tab', 'preupdate', $base_url ) ); ?>"><?php esc_html_e( 'Before Updates', 'aegisbackup' ); ?></a>
            </h2>

            <?php if ( $needs_lock ) : ?>
                <div class="notice notice-warning" style="margin:12px 0 16px;">
                    <p style="margin:0;">
                        <strong><?php esc_html_e( 'PRO Feature:', 'aegisbackup' ); ?></strong>
                        <?php esc_html_e( 'This settings tab is available in AegisBackup Pro. Upgrade to unlock Remote Storage, SMTP & Notifications, and Before Updates automation.', 'aegisbackup' ); ?>
                        <a href="<?php echo esc_url( $license_url ); ?>" class="button button-primary" style="margin-left:10px;">
                            <?php esc_html_e( 'Upgrade / Activate License', 'aegisbackup' ); ?>
                        </a>
                    </p>
                </div>

                <style>
                    .aegisbackup-pro-locked{
                        opacity:0.45 !important;
                        pointer-events:none !important;
                        filter:grayscale(1) !important;
                        user-select:none !important;
                    }
                </style>
                <div class="aegisbackup-pro-locked" aria-disabled="true">
            <?php endif; ?>

            <?php
            if ( 'remote_storage' === $tab ) {
                $this->render_remote_storage_tab( $base_url );
			} elseif ( 'smtp' === $tab ) {
				$this->render_smtp_tab( $base_url );
			} elseif ( 'preupdate' === $tab ) {
				$this->render_preupdate_tab( $base_url );
			} else {
				$this->render_logs_tab();
			}
            ?>

            <?php if ( $needs_lock ) : ?>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    private function render_logs_tab() {
        $logs = get_option( 'aegisbackup_logs', array() );
        if ( ! is_array( $logs ) ) {
            $logs = array();
        }
        $logs = array_slice( array_reverse( $logs ), 0, 400 );
        ?>
        <div class="aegisbackup-card">
            <p class="description"><?php esc_html_e( 'Shows recent backup/restore activity and system messages. This remains lightweight (rolling log).', 'aegisbackup' ); ?></p>
            <pre class="aegisbackup-log" style="max-height:520px; overflow:auto; white-space:pre-wrap;"><?php echo esc_html( implode( "\n", $logs ) ); ?>
</div>

<?php
$activity = \AegisBackup\Libs\AB_Activity_Log::get( 120 );
$reports  = get_option( 'aegisbackup_restore_reports', array() );
if ( ! is_array( $reports ) ) {
    $reports = array();
}
$reports = array_slice( $reports, 0, 40 );
?>
<div class="aegisbackup-card">
    <h3 style="margin-top:0;"><?php esc_html_e( 'Activity Log', 'aegisbackup' ); ?></h3>
    <p class="description"><?php esc_html_e( 'Structured activity events (success/failure) for backups, restores, and webhooks.', 'aegisbackup' ); ?></p>
    <table class="widefat striped">
        <thead>
            <tr>
                <th style="width:160px;"><?php esc_html_e( 'Time', 'aegisbackup' ); ?></th>
                <th style="width:90px;"><?php esc_html_e( 'Type', 'aegisbackup' ); ?></th>
                <th style="width:90px;"><?php esc_html_e( 'Status', 'aegisbackup' ); ?></th>
                <th><?php esc_html_e( 'Message', 'aegisbackup' ); ?></th>
            </tr>
        </thead>
        <tbody>
        <?php if ( empty( $activity ) ) : ?>
            <tr><td colspan="4"><?php esc_html_e( 'No activity yet.', 'aegisbackup' ); ?></td></tr>
        <?php else : ?>
            <?php foreach ( $activity as $row ) : ?>
                <tr>
                    <td><?php echo esc_html( date_i18n( 'Y-m-d H:i:s', isset( $row['ts'] ) ? (int) $row['ts'] : time() ) ); ?></td>
                    <td><?php echo esc_html( isset( $row['type'] ) ? (string) $row['type'] : '' ); ?></td>
                    <td><?php echo esc_html( isset( $row['status'] ) ? (string) $row['status'] : '' ); ?></td>
                    <td><?php echo esc_html( isset( $row['message'] ) ? (string) $row['message'] : '' ); ?></td>
                </tr>
            <?php endforeach; ?>
        <?php endif; ?>
        </tbody>
    </table>
</div>

<div class="aegisbackup-card">
    <h3 style="margin-top:0;"><?php esc_html_e( 'Restore Reports', 'aegisbackup' ); ?></h3>
    <p class="description"><?php esc_html_e( 'Latest restore outcomes with snapshot state for auditing and troubleshooting.', 'aegisbackup' ); ?></p>

    <?php if ( empty( $reports ) ) : ?>
        <p><?php esc_html_e( 'No restore reports yet.', 'aegisbackup' ); ?></p>
    <?php else : ?>
        <?php foreach ( $reports as $rep ) : ?>
            <?php
                $ts = isset( $rep['ts'] ) ? (int) $rep['ts'] : 0;
                $jid = isset( $rep['job_id'] ) ? (string) $rep['job_id'] : '';
                $st  = isset( $rep['status'] ) ? (string) $rep['status'] : '';
                $msg = isset( $rep['message'] ) ? (string) $rep['message'] : '';
                $json = wp_json_encode( $rep );
            ?>
            <details style="margin:8px 0; padding:8px; border:1px solid #dcdcde; background:#fff;">
                <summary>
                    <strong><?php echo esc_html( strtoupper( $st ) ); ?></strong>
                    &nbsp;—&nbsp;<?php echo esc_html( $jid ); ?>
                    &nbsp;—&nbsp;<?php echo esc_html( $ts ? date_i18n( 'Y-m-d H:i:s', $ts ) : '' ); ?>
                    <?php if ( $msg ) : ?> — <?php echo esc_html( $msg ); ?><?php endif; ?>
                </summary>
                <textarea readonly="readonly" style="width:100%; min-height:140px; margin-top:8px; font-family:monospace;"><?php echo esc_textarea( $json ); ?></textarea>
            </details>
        <?php endforeach; ?>
    <?php endif; ?>
</div>
<?php

    }

    private function render_remote_storage_tab( $base_url ) {
        $remote = get_option( 'aegisbackup_remote_storage', array() );
        if ( ! is_array( $remote ) ) {
            $remote = array();
        }

        $enabled        = ! empty( $remote['enabled'] ) ? 1 : 0;
        $provider       = isset( $remote['provider'] ) ? (string) $remote['provider'] : 'generic';
        $location       = isset( $remote['location'] ) ? (string) $remote['location'] : '';
        $port           = isset( $remote['port'] ) ? (int) $remote['port'] : 0;
        $username       = isset( $remote['username'] ) ? (string) $remote['username'] : '';
        $password       = isset( $remote['password'] ) ? (string) $remote['password'] : '';
        $s3_bucket_url  = isset( $remote['s3_bucket_url'] ) ? (string) $remote['s3_bucket_url'] : '';
        $s3_region      = isset( $remote['s3_region'] ) ? (string) $remote['s3_region'] : '';
        $s3_access_key  = isset( $remote['s3_access_key'] ) ? (string) $remote['s3_access_key'] : '';
        $s3_secret_key  = isset( $remote['s3_secret_key'] ) ? (string) $remote['s3_secret_key'] : '';
        $s3_json        = isset( $remote['s3_json'] ) ? (string) $remote['s3_json'] : '';
        $s3_object_path = isset( $remote['s3_object_path'] ) ? (string) $remote['s3_object_path'] : '';
        ?>
        <div class="aegisbackup-card">
            <h2 style="margin-top:0;"><?php esc_html_e( 'Remote Storage (Generic)', 'aegisbackup' ); ?></h2>
            <p class="description">
                <?php esc_html_e( 'This is intentionally simple (generic). It stores credentials and will attempt to upload the final backup package when a backup completes.', 'aegisbackup' ); ?>
                <br /><?php esc_html_e( 'No SDKs, no deep integrations—just basic settings and best-effort uploads.', 'aegisbackup' ); ?>
            </p>

            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <?php wp_nonce_field( 'aegisbackup_save_settings', 'aegisbackup_settings_nonce' ); ?>
                <input type="hidden" name="action" value="aegisbackup_save_settings" />
                <input type="hidden" name="settings_section" value="remote_storage" />

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Enable Remote Upload', 'aegisbackup' ); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="remote_enabled" value="1" <?php checked( 1, $enabled ); ?> />
                                <?php esc_html_e( 'Upload backup packages offsite after completion', 'aegisbackup' ); ?>
                            </label>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><?php esc_html_e( 'Provider', 'aegisbackup' ); ?></th>
                        <td>
                            <select name="remote_provider">
                                <option value="generic" <?php selected( 'generic', $provider ); ?>><?php esc_html_e( 'Generic (HTTP PUT or SFTP)', 'aegisbackup' ); ?></option>
                                <option value="s3" <?php selected( 's3', $provider ); ?>><?php esc_html_e( 'S3 (Simple PUT)', 'aegisbackup' ); ?></option>
                            </select>
                            <p class="description"><?php esc_html_e( 'Generic: HTTP(S) PUT to a URL, or SFTP if the server has the ssh2 extension.', 'aegisbackup' ); ?></p>
<script type="text/javascript">
(function(){
    function ab_remote_provider_toggle(){
        var sel = document.querySelector('select[name="remote_provider"]');
        if(!sel){ return; }
        var generic = document.getElementById('ab-remote-generic-fields');
        var s3 = document.getElementById('ab-remote-s3-fields');
        var v = (sel.value || 'generic');
        if(generic){ generic.style.display = (v === 'generic') ? 'block' : 'none'; }
        if(s3){ s3.style.display = (v === 's3') ? 'block' : 'none'; }
    }
    document.addEventListener('DOMContentLoaded', function(){
        var sel = document.querySelector('select[name="remote_provider"]');
        if(sel){
            sel.addEventListener('change', ab_remote_provider_toggle);
            ab_remote_provider_toggle();
        }
    });

    ab_remote_provider_toggle();
})();
</script>

                        </td>
                    </tr>
                </table>


                <?php submit_button( __( 'Save Remote Storage Settings', 'aegisbackup' ) ); ?>

                <div id="ab-remote-generic-fields">
<h3><?php esc_html_e( 'Generic (HTTP/SFTP)', 'aegisbackup' ); ?></h3>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Location', 'aegisbackup' ); ?></th>
                        <td>
                            <input type="text" class="regular-text" name="remote_location" value="<?php echo esc_attr( $location ); ?>" placeholder="https://domain.com/wordpress/db_backup/" />
                            <p class="description"><?php esc_html_e( 'If this is an HTTPS URL, AegisBackup will attempt an HTTP PUT upload.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Port', 'aegisbackup' ); ?></th>
                        <td>
                            <input type="number" name="remote_port" value="<?php echo esc_attr( $port ); ?>" placeholder="22" />
                            <p class="description"><?php esc_html_e( 'Used for SFTP if available. For HTTP uploads, this is not used.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Username', 'aegisbackup' ); ?></th>
                        <td><input type="text" class="regular-text" name="remote_username" value="<?php echo esc_attr( $username ); ?>" /></td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Password', 'aegisbackup' ); ?></th>
                        <td><input type="password" class="regular-text" name="remote_password" value="<?php echo esc_attr( $password ); ?>" autocomplete="new-password" /></td>
                    </tr>
                </table>

                </div>
<div id="ab-remote-s3-fields">
<h3><?php esc_html_e( 'S3 (Simple PUT)', 'aegisbackup' ); ?></h3>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Bucket URL', 'aegisbackup' ); ?></th>
                        <td>
                            <input type="text" class="regular-text" name="s3_bucket_url" value="<?php echo esc_attr( $s3_bucket_url ); ?>" placeholder="https://bucket.s3.amazonaws.com" />
                            <p class="description"><?php esc_html_e( 'You can paste a full bucket URL. AegisBackup will upload as a single object (no multipart).', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Region', 'aegisbackup' ); ?></th>
                        <td><input type="text" class="regular-text" name="s3_region" value="<?php echo esc_attr( $s3_region ); ?>" placeholder="us-east-1" /></td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Access Key', 'aegisbackup' ); ?></th>
                        <td><input type="text" class="regular-text" name="s3_access_key" value="<?php echo esc_attr( $s3_access_key ); ?>" /></td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Secret Key', 'aegisbackup' ); ?></th>
                        <td><input type="password" class="regular-text" name="s3_secret_key" value="<?php echo esc_attr( $s3_secret_key ); ?>" autocomplete="new-password" /></td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Service Account / JSON', 'aegisbackup' ); ?></th>
                        <td>
                            <textarea name="s3_json" rows="6" class="large-text code" placeholder="{...}"><?php echo esc_textarea( $s3_json ); ?></textarea>
                            <p class="description"><?php esc_html_e( 'Optional. If provided, AegisBackup will try to extract access key fields from common JSON formats.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Object Path Prefix', 'aegisbackup' ); ?></th>
                        <td>
                            <input type="text" class="regular-text" name="s3_object_path" value="<?php echo esc_attr( $s3_object_path ); ?>" placeholder="aegisbackup/" />
                            <p class="description"><?php esc_html_e( 'Optional prefix. Example: aegisbackup/ (the filename will be appended).', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                </table>

            </form>
        </div>
        <?php
    }

    private function render_smtp_tab( $base_url ) {
        $smtp = get_option( 'aegisbackup_smtp_settings', array() );
        if ( ! is_array( $smtp ) ) {
            $smtp = array();
        }
        $notify = get_option( 'aegisbackup_notification_settings', array() );
        if ( ! is_array( $notify ) ) {
            $notify = array();
        }

        $smtp_enabled = ! empty( $smtp['enabled'] ) ? 1 : 0;
        $host = isset( $smtp['host'] ) ? (string) $smtp['host'] : '';
        $port = isset( $smtp['port'] ) ? (int) $smtp['port'] : 0;
        $enc  = isset( $smtp['encryption'] ) ? (string) $smtp['encryption'] : 'none';
        $user = isset( $smtp['username'] ) ? (string) $smtp['username'] : '';
        $pass = isset( $smtp['password'] ) ? (string) $smtp['password'] : '';
        $from_name  = isset( $smtp['from_name'] ) ? (string) $smtp['from_name'] : '';
        $from_email = isset( $smtp['from_email'] ) ? (string) $smtp['from_email'] : '';

        $recipient = isset( $notify['recipient'] ) ? (string) $notify['recipient'] : '';
        $on_backup = ! empty( $notify['on_backup_complete'] ) ? 1 : 0;
        $on_restore = ! empty( $notify['on_restore_complete'] ) ? 1 : 0;
        $on_offsite = ! empty( $notify['on_offsite_failure'] ) ? 1 : 0;
        $on_backup_failed  = ! empty( $notify['on_backup_failed'] ) ? 1 : 0;
        $on_restore_failed = ! empty( $notify['on_restore_failed'] ) ? 1 : 0;

        $webhook_url    = isset( $notify['webhook_url'] ) ? (string) $notify['webhook_url'] : '';
        $webhook_secret = isset( $notify['webhook_secret'] ) ? (string) $notify['webhook_secret'] : '';
        $wh_backup_complete = ! empty( $notify['webhook_on_backup_complete'] ) ? 1 : 0;
        $wh_backup_failed   = ! empty( $notify['webhook_on_backup_failed'] ) ? 1 : 0;
        $wh_restore_complete = ! empty( $notify['webhook_on_restore_complete'] ) ? 1 : 0;
        $wh_restore_failed   = ! empty( $notify['webhook_on_restore_failed'] ) ? 1 : 0;
        $wh_offsite_failed   = ! empty( $notify['webhook_on_offsite_failure'] ) ? 1 : 0;
        ?>
        <div class="aegisbackup-card">
            <h2 style="margin-top:0;"><?php esc_html_e( 'SMTP & Notifications', 'aegisbackup' ); ?></h2>
            <p class="description"><?php esc_html_e( 'Configure SMTP (optional) and notification emails for backups, restores, and offsite upload failures.', 'aegisbackup' ); ?></p>

            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <?php wp_nonce_field( 'aegisbackup_save_settings', 'aegisbackup_settings_nonce' ); ?>
                <input type="hidden" name="action" value="aegisbackup_save_settings" />
                <input type="hidden" name="settings_section" value="smtp" />

                <h3><?php esc_html_e( 'SMTP Settings', 'aegisbackup' ); ?></h3>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Enable SMTP', 'aegisbackup' ); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="smtp_enabled" value="1" <?php checked( 1, $smtp_enabled ); ?> />
                                <?php esc_html_e( 'Use SMTP for wp_mail()', 'aegisbackup' ); ?>
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Host', 'aegisbackup' ); ?></th>
                        <td><input type="text" class="regular-text" name="smtp_host" value="<?php echo esc_attr( $host ); ?>" /></td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Port', 'aegisbackup' ); ?></th>
                        <td><input type="number" name="smtp_port" value="<?php echo esc_attr( $port ); ?>" placeholder="587" /></td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Encryption', 'aegisbackup' ); ?></th>
                        <td>
                            <select name="smtp_encryption">
                                <option value="none" <?php selected( 'none', $enc ); ?>><?php esc_html_e( 'None', 'aegisbackup' ); ?></option>
                                <option value="tls" <?php selected( 'tls', $enc ); ?>><?php esc_html_e( 'TLS', 'aegisbackup' ); ?></option>
                                <option value="ssl" <?php selected( 'ssl', $enc ); ?>><?php esc_html_e( 'SSL', 'aegisbackup' ); ?></option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Username', 'aegisbackup' ); ?></th>
                        <td><input type="text" class="regular-text" name="smtp_username" value="<?php echo esc_attr( $user ); ?>" /></td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Password', 'aegisbackup' ); ?></th>
                        <td><input type="password" class="regular-text" name="smtp_password" value="<?php echo esc_attr( $pass ); ?>" autocomplete="new-password" /></td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'From Name', 'aegisbackup' ); ?></th>
                        <td><input type="text" class="regular-text" name="smtp_from_name" value="<?php echo esc_attr( $from_name ); ?>" /></td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'From Email', 'aegisbackup' ); ?></th>
                        <td><input type="email" class="regular-text" name="smtp_from_email" value="<?php echo esc_attr( $from_email ); ?>" /></td>
                    </tr>
                </table>

                <h3><?php esc_html_e( 'Notification Settings', 'aegisbackup' ); ?></h3>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Recipient Email', 'aegisbackup' ); ?></th>
                        <td>
                            <input type="email" class="regular-text" name="notify_recipient" value="<?php echo esc_attr( $recipient ); ?>" placeholder="admin@domain.com" />
                            <p class="description"><?php esc_html_e( 'If blank, AegisBackup will use the WordPress admin email.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Notify on Backup Complete', 'aegisbackup' ); ?></th>
                        <td><label><input type="checkbox" name="notify_on_backup_complete" value="1" <?php checked( 1, $on_backup ); ?> /> <?php esc_html_e( 'Send an email when a backup finishes', 'aegisbackup' ); ?></label></td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Notify on Restore Complete', 'aegisbackup' ); ?></th>
                        <td><label><input type="checkbox" name="notify_on_restore_complete" value="1" <?php checked( 1, $on_restore ); ?> /> <?php esc_html_e( 'Send an email when a restore finishes', 'aegisbackup' ); ?></label></td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Notify on Backup Failed', 'aegisbackup' ); ?></th>
                        <td><label><input type="checkbox" name="notify_on_backup_failed" value="1" <?php checked( 1, $on_backup_failed ); ?> /> <?php esc_html_e( 'Send an email when a backup fails', 'aegisbackup' ); ?></label></td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Notify on Restore Failed', 'aegisbackup' ); ?></th>
                        <td><label><input type="checkbox" name="notify_on_restore_failed" value="1" <?php checked( 1, $on_restore_failed ); ?> /> <?php esc_html_e( 'Send an email when a restore fails', 'aegisbackup' ); ?></label></td>
                    </tr>

                    <tr>
                        <th scope="row"><?php esc_html_e( 'Notify on Offsite Upload Failure', 'aegisbackup' ); ?></th>
                        <td><label><input type="checkbox" name="notify_on_offsite_failure" value="1" <?php checked( 1, $on_offsite ); ?> /> <?php esc_html_e( 'Send an email if remote upload fails', 'aegisbackup' ); ?></label></td>
                    </tr>
                </table>

                <h3><?php esc_html_e( 'Webhook Alerts', 'aegisbackup' ); ?></h3>
                <p class="description"><?php esc_html_e( 'Optional: send a JSON webhook on backup/restore success/failure. If you set a secret, an HMAC SHA-256 signature is added as X-AegisBackup-Signature.', 'aegisbackup' ); ?></p>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Webhook URL', 'aegisbackup' ); ?></th>
                        <td>
                            <input type="url" class="regular-text" name="notify_webhook_url" value="<?php echo esc_attr( $webhook_url ); ?>" placeholder="https://example.com/webhook" />
                            <p class="description"><?php esc_html_e( 'Leave blank to disable webhooks.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Webhook Secret', 'aegisbackup' ); ?></th>
                        <td>
                            <input type="text" class="regular-text" name="notify_webhook_secret" value="<?php echo esc_attr( $webhook_secret ); ?>" autocomplete="new-password" />
                            <p class="description"><?php esc_html_e( 'Optional shared secret used to sign the JSON payload.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Webhook Events', 'aegisbackup' ); ?></th>
                        <td>
                            <fieldset>
                                <label style="display:block;margin:3px 0;"><input type="checkbox" name="notify_webhook_on_backup_complete" value="1" <?php checked( 1, $wh_backup_complete ); ?> /> <?php esc_html_e( 'Backup Completed', 'aegisbackup' ); ?></label>
                                <label style="display:block;margin:3px 0;"><input type="checkbox" name="notify_webhook_on_backup_failed" value="1" <?php checked( 1, $wh_backup_failed ); ?> /> <?php esc_html_e( 'Backup Failed', 'aegisbackup' ); ?></label>
                                <label style="display:block;margin:3px 0;"><input type="checkbox" name="notify_webhook_on_restore_complete" value="1" <?php checked( 1, $wh_restore_complete ); ?> /> <?php esc_html_e( 'Restore Completed', 'aegisbackup' ); ?></label>
                                <label style="display:block;margin:3px 0;"><input type="checkbox" name="notify_webhook_on_restore_failed" value="1" <?php checked( 1, $wh_restore_failed ); ?> /> <?php esc_html_e( 'Restore Failed', 'aegisbackup' ); ?></label>
                                <label style="display:block;margin:3px 0;"><input type="checkbox" name="notify_webhook_on_offsite_failure" value="1" <?php checked( 1, $wh_offsite_failed ); ?> /> <?php esc_html_e( 'Offsite Upload Failed', 'aegisbackup' ); ?></label>
                            </fieldset>
                        </td>
                    </tr>
                </table>

                <?php submit_button( __( 'Save SMTP & Notifications', 'aegisbackup' ) ); ?>
            </form>

            <hr />

            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <?php wp_nonce_field( 'aegisbackup_send_test_email', 'aegisbackup_test_email_nonce' ); ?>
                <input type="hidden" name="action" value="aegisbackup_send_test_email" />
                <?php submit_button( __( 'Send Test Email', 'aegisbackup' ), 'secondary' ); ?>
            </form>
        </div>
        <?php
	}
	
	private function render_preupdate_tab( $base_url ) {
		$s = get_option( 'aegisbackup_preupdate_settings', array() );
		if ( ! is_array( $s ) ) {
			$s = array();
		}
		$enabled = ! empty( $s['enabled'] ) ? 1 : 0;

		$last = get_option( 'aegisbackup_preupdate_last_snapshot', array() );
		if ( ! is_array( $last ) ) {
			$last = array();
		}

		$restore_url = admin_url( 'admin.php?page=aegisbackup-restore&ab_preupdate=1' );
		?>
		<div class="aegisbackup-card">
			<h2 style="margin-top:0;"><?php esc_html_e( 'Automatic Backups Before Updates', 'aegisbackup' ); ?></h2>
			<p class="description">
				<?php esc_html_e( 'When enabled, AegisBackup will create a fast "pre-update snapshot" right before WordPress installs a plugin, theme, or core update. This gives you a quick rollback point if something breaks.', 'aegisbackup' ); ?>
			</p>

			<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
				<?php wp_nonce_field( 'aegisbackup_save_settings', 'aegisbackup_settings_nonce' ); ?>
				<input type="hidden" name="action" value="aegisbackup_save_settings" />
				<input type="hidden" name="settings_section" value="preupdate" />

				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><?php esc_html_e( 'Backup before updates', 'aegisbackup' ); ?></th>
						<td>
							<label>
								<input type="checkbox" name="preupdate_enabled" value="1" <?php checked( 1, $enabled ); ?> />
								<?php esc_html_e( 'Automatically create a rollback snapshot before plugin/theme/core updates', 'aegisbackup' ); ?>
							</label>
							<p class="description"><?php esc_html_e( 'Tip: For bulk updates, WordPress may run multiple upgrades. AegisBackup will keep the most recent pre-update snapshot as the rollback point.', 'aegisbackup' ); ?></p>
						</td>
					</tr>
				</table>

				<?php submit_button( __( 'Save Pre-Update Settings', 'aegisbackup' ) ); ?>
			</form>
		</div>

		<div class="aegisbackup-card" style="margin-top:16px;">
			<h2 style="margin-top:0;"><?php esc_html_e( 'Fast Rollback Point', 'aegisbackup' ); ?></h2>

			<?php if ( ! empty( $last['package'] ) ) : ?>
				<p><strong><?php esc_html_e( 'Last pre-update snapshot:', 'aegisbackup' ); ?></strong></p>
				<ul style="margin-left:18px; list-style:disc;">
					<li><?php esc_html_e( 'Created:', 'aegisbackup' ); ?> <?php echo esc_html( isset( $last['created_at'] ) ? (string) $last['created_at'] : '' ); ?></li>
					<li><?php esc_html_e( 'Update type:', 'aegisbackup' ); ?> <?php echo esc_html( isset( $last['update_type'] ) ? (string) $last['update_type'] : '' ); ?></li>
					<li><?php esc_html_e( 'Package:', 'aegisbackup' ); ?> <?php echo esc_html( basename( (string) $last['package'] ) ); ?></li>
				</ul>

				<p>
					<a class="button button-primary" href="<?php echo esc_url( $restore_url ); ?>">
						<?php esc_html_e( 'Restore last pre-update snapshot', 'aegisbackup' ); ?>
					</a>
					<span class="description" style="margin-left:8px;"><?php esc_html_e( 'Takes you to Restore & Migrate with the snapshot pre-selected.', 'aegisbackup' ); ?></span>
				</p>
			<?php else : ?>
				<div class="notice notice-info inline">
					<p><?php esc_html_e( 'No pre-update snapshot found yet. Turn on "Backup before updates", then run a plugin/theme/core update to generate your first rollback point.', 'aegisbackup' ); ?></p>
				</div>
			<?php endif; ?>
		</div>
		<?php
	}

    
}
