<?php
namespace AegisBackup\Admin\Pages;

use AegisBackup\AB_Plugin;

defined( 'ABSPATH' ) || exit;

class AB_Page_PushPull {

    protected $plugin;

    public function __construct( AB_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

	public function render() {
		$packages    = $this->plugin->backup->list_packages();

		$table_backups = $this->plugin->table_backup->get_backups();
		if ( ! is_array( $table_backups ) ) {
			$table_backups = array();
		}

		uasort(
			$table_backups,
			static function ( $a, $b ) {
				$ta = isset( $a['created'] ) ? (int) $a['created'] : 0;
				$tb = isset( $b['created'] ) ? (int) $b['created'] : 0;
				return $tb <=> $ta;
			}
		);
        $connect_url = trailingslashit( home_url() ) . 'aegisbackup_connect/';

        $ab_msg = isset( $_GET['ab_msg'] ) ? sanitize_text_field( (string) wp_unslash( $_GET['ab_msg'] ) ) : '';
        $ab_ok  = isset( $_GET['ab_ok'] ) ? (int) $_GET['ab_ok'] : 0;
		$ab_listen = isset( $_GET['ab_listen'] ) ? (int) $_GET['ab_listen'] : 0;
		$pp_tab  = isset( $_GET['pp_tab'] ) ? sanitize_key( (string) wp_unslash( $_GET['pp_tab'] ) ) : 'source';
		$pp_view = isset( $_GET['pp_view'] ) ? sanitize_key( (string) wp_unslash( $_GET['pp_view'] ) ) : '';
		$db_key  = isset( $_GET['db_key'] ) ? sanitize_text_field( (string) wp_unslash( $_GET['db_key'] ) ) : '';
		$fb_key  = isset( $_GET['fb_key'] ) ? sanitize_text_field( (string) wp_unslash( $_GET['fb_key'] ) ) : '';
		if ( ! in_array( $pp_tab, array( 'source', 'destination' ), true ) ) {
			$pp_tab = 'source';
		}

		$pp_tab_url_source      = add_query_arg( 'pp_tab', 'source' );
		$pp_tab_url_destination = add_query_arg( 'pp_tab', 'destination' );
		$file_backups = $this->plugin->file_backup->list_backups();
		if ( ! is_array( $file_backups ) ) {
			$file_backups = array();
		}

		$user_id = get_current_user_id();
		$saved_token_json = (string) get_user_meta( $user_id, 'aegisbackup_pp_token_json', true );
		$saved_dest_url   = (string) get_user_meta( $user_id, 'aegisbackup_pp_dest_url', true );
		$saved_package_path = (string) get_user_meta( $user_id, 'aegisbackup_pp_package_path', true );
		$saved_connect_name = (string) get_user_meta( $user_id, 'aegisbackup_pp_connect_name', true );

		$saved_src_root = (string) get_user_meta( $user_id, 'aegisbackup_pp_src_root', true );
		$saved_dst_root = (string) get_user_meta( $user_id, 'aegisbackup_pp_dst_root', true );
		$saved_pp_paths = get_user_meta( $user_id, 'aegisbackup_pp_paths', true );
		if ( ! is_array( $saved_pp_paths ) ) {
			$saved_pp_paths = array();
		}
		if ( '' === $saved_src_root ) {
			$saved_src_root = '/public_html/';
		}
		if ( '' === $saved_dst_root ) {
			$saved_dst_root = '/public_html/';
		}

		$connections = get_user_meta( $user_id, 'aegisbackup_pp_connections', true );
		if ( ! is_array( $connections ) ) {
			$connections = array();
		}

		$now = time();
		$changed = false;
		foreach ( $connections as $k => $c ) {
			if ( ! is_array( $c ) ) {
				continue;
			}
			$is_connected  = ! empty( $c['connected'] );
			$last_activity = isset( $c['last_activity'] ) ? (int) $c['last_activity'] : 0;

			if ( $is_connected && $last_activity > 0 && ( $now - $last_activity ) > 1800 ) {
				$connections[ $k ]['connected'] = 0;
				$changed = true;
			} elseif ( $is_connected ) {
				$connections[ $k ]['last_activity'] = $now;
				$changed = true;
			}
		}
		if ( $changed ) {
			update_user_meta( $user_id, 'aegisbackup_pp_connections', $connections );
		}

		$incoming = get_option( 'aegisbackup_pp_incoming_connections', array() );
		if ( ! is_array( $incoming ) ) {
			$incoming = array();
		}
		$incoming_changed = false;
		foreach ( $incoming as $k => $row ) {
			if ( ! is_array( $row ) ) {
				continue;
			}
			$is_connected  = ! empty( $row['connected'] );
			$last_activity = isset( $row['last_activity'] ) ? (int) $row['last_activity'] : 0;
			if ( $is_connected && $last_activity > 0 && ( $now - $last_activity ) > 1800 ) {
				$incoming[ $k ]['connected'] = 0;
				$incoming_changed = true;
			}
		}
		if ( $incoming_changed ) {
			update_option( 'aegisbackup_pp_incoming_connections', $incoming, false );
		}

		$edit_conn_id = isset( $_GET['ab_conn'] ) ? sanitize_text_field( (string) wp_unslash( $_GET['ab_conn'] ) ) : '';
		$edit_conn = array();
		if ( $edit_conn_id ) {
			foreach ( $connections as $c ) {
				if ( is_array( $c ) && ! empty( $c['id'] ) && (string) $c['id'] === (string) $edit_conn_id ) {
					$edit_conn = $c;
					break;
				}
			}
			if ( ! empty( $edit_conn ) ) {
				$saved_connect_name = isset( $edit_conn['name'] ) ? (string) $edit_conn['name'] : $saved_connect_name;
				$saved_dest_url     = isset( $edit_conn['dest_url'] ) ? (string) $edit_conn['dest_url'] : $saved_dest_url;
			}
		}

        $job_id = isset( $_GET['ab_job'] ) ? sanitize_text_field( (string) wp_unslash( $_GET['ab_job'] ) ) : '';
        $job_state = array();
        $job_log   = '';
        if ( $job_id ) {
            $job_state = get_option( \AegisBackup\Push\AB_Push_Manager::JOB_OPTION_PREFIX . $job_id, array() );
            $job_log   = (string) get_user_meta( $user_id, 'aegisbackup_pp_pushlog_' . $job_id, true );
        }

        ?>
        <div class="wrap aegisbackup-wrap">
            <h1><?php esc_html_e( 'Push / Pull Migration', 'aegisbackup' ); ?></h1>

            <?php if ( $ab_msg ) : ?>
                <div class="notice <?php echo $ab_ok ? 'notice-success' : 'notice-error'; ?> is-dismissible">
                    <p><?php echo esc_html( $ab_msg ); ?></p>
                </div>
            <?php endif; ?>

            <div class="nav-tab-wrapper" style="margin-top: 10px; margin-bottom: 16px;">
                <a href="<?php echo esc_url( $pp_tab_url_source ); ?>" class="nav-tab <?php echo ( 'source' === $pp_tab ) ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Source', 'aegisbackup' ); ?></a>
                <a href="<?php echo esc_url( $pp_tab_url_destination ); ?>" class="nav-tab <?php echo ( 'destination' === $pp_tab ) ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Destination', 'aegisbackup' ); ?></a>
            </div>

            <?php if ( 'source' === $pp_tab ) : ?>

                <div class="aegisbackup-card">
                    <h2><?php esc_html_e( 'Source: Push Package to Destination', 'aegisbackup' ); ?></h2>
                    <p class="description"><?php esc_html_e( 'On the source site (this site), select a package and paste the Destination Token JSON generated on the destination WordPress.', 'aegisbackup' ); ?></p>

					<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
						<input type="hidden" name="action" value="aegisbackup_pp_connect_post" />
						<?php wp_nonce_field( 'aegisbackup_pp_connect_post', 'aegisbackup_pp_nonce' ); ?>

						<?php if ( $edit_conn_id ) : ?>
							<input type="hidden" name="connection_id" value="<?php echo esc_attr( $edit_conn_id ); ?>" />
						<?php endif; ?>

						<input type="hidden" name="package_path" value="<?php echo esc_attr( $saved_package_path ); ?>" />

                        <table class="form-table" role="presentation">
                            <tbody>
							<tr>
								<th scope="row"><?php esc_html_e( 'Connect Name', 'aegisbackup' ); ?></th>
								<td>
									<input type="text" class="regular-text" name="connect_name" id="ab_connect_name" placeholder="My Destination" value="<?php echo esc_attr( $saved_connect_name ); ?>" />
									<p class="description"><?php esc_html_e( 'Give this connection a friendly name so you can identify it later.', 'aegisbackup' ); ?></p>
								</td>
							</tr>
                            <tr>
                                <th scope="row"><?php esc_html_e( 'Destination URL', 'aegisbackup' ); ?></th>
                                <td>
                                    <input type="url" class="regular-text" name="destination_url" id="ab_destination_url" placeholder="https://domain.com" value="<?php echo esc_attr( $saved_dest_url ); ?>" />
                                    <p class="description"><?php esc_html_e( 'This is the destination WordPress domain (HTTPS recommended). The token JSON also contains an endpoint; we use the URL as a sanity-check and for user clarity.', 'aegisbackup' ); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php esc_html_e( 'Destination Token JSON', 'aegisbackup' ); ?></th>
                                <td>
                                    <textarea class="large-text code" rows="6" name="token_json" id="ab_destination_token" placeholder="{ token json... }"></textarea>
                                    <p class="description"><?php esc_html_e( 'Generate this on the destination site (right panel).', 'aegisbackup' ); ?></p>
                                </td>
                            </tr>
                            </tbody>
                        </table>

                        <p>
                            <button type="submit" class="button button-primary" id="ab_connect" data-ab-post="1">
								<?php echo $edit_conn_id ? esc_html__( 'Save Connection', 'aegisbackup' ) : esc_html__( 'Connect', 'aegisbackup' ); ?>
							</button>
                        </p>
                    </form>
					<?php if ( ! empty( $connections ) ) : ?>
					<hr />
					<h3 style="margin-top: 6px;"><?php esc_html_e( 'Saved Connections', 'aegisbackup' ); ?></h3>

					<div style="overflow-x:auto; max-width:100%;">
					<table class="widefat striped" style="margin-top:10px; width:100%;">
						<thead>
							<tr>
								<th><?php esc_html_e( 'Name', 'aegisbackup' ); ?></th>
								<th><?php esc_html_e( 'Destination', 'aegisbackup' ); ?></th>
								<th><?php esc_html_e( 'Status', 'aegisbackup' ); ?></th>
								<th><?php esc_html_e( 'Last Activity', 'aegisbackup' ); ?></th>
								<th><?php esc_html_e( 'Action', 'aegisbackup' ); ?></th>
							</tr>
						</thead>
						<tbody>
							<?php foreach ( $connections as $c ) : ?>
								<?php if ( ! is_array( $c ) || empty( $c['id'] ) ) { continue; } ?>
								<?php
									$c_id = (string) $c['id'];
									$c_name = isset( $c['name'] ) ? (string) $c['name'] : '';
									$c_url = isset( $c['dest_url'] ) ? (string) $c['dest_url'] : '';
									$c_connected = ! empty( $c['connected'] );
									$c_last = isset( $c['last_activity'] ) ? (int) $c['last_activity'] : 0;
								?>
								<tr>
									<td><?php echo esc_html( $c_name ); ?></td>
									<td><code><?php echo esc_html( $c_url ); ?></code></td>
									<td><?php echo $c_connected ? esc_html__( 'Connected', 'aegisbackup' ) : esc_html__( 'Disconnected', 'aegisbackup' ); ?></td>
									<td><?php echo $c_last ? esc_html( gmdate( 'Y-m-d H:i:s', $c_last ) . ' UTC' ) : esc_html__( '—', 'aegisbackup' ); ?></td>
									<td>
										<a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisbackup-pushpull&ab_conn=' . rawurlencode( $c_id ) ) ); ?>"><?php esc_html_e( 'Edit', 'aegisbackup' ); ?></a>

										<?php if ( $c_connected ) : ?>
											<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="display:inline-block;margin-left:6px;">
												<input type="hidden" name="action" value="aegisbackup_pp_disconnect_post" />
												<input type="hidden" name="connection_id" value="<?php echo esc_attr( $c_id ); ?>" />
												<?php wp_nonce_field( 'aegisbackup_pp_disconnect_post', 'aegisbackup_pp_nonce' ); ?>
												<button type="submit" class="button button-small" data-ab-post="1"><?php esc_html_e( 'Disconnect', 'aegisbackup' ); ?></button>
											</form>
										<?php endif; ?>

										<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="display:inline-block;margin-left:6px;">
											<input type="hidden" name="action" value="aegisbackup_pp_delete_connection_post" />
											<input type="hidden" name="connection_id" value="<?php echo esc_attr( $c_id ); ?>" />
											<?php wp_nonce_field( 'aegisbackup_pp_delete_connection_post', 'aegisbackup_pp_nonce' ); ?>
											<button type="submit" class="button button-small" data-ab-post="1" onclick="return confirm('Delete this connection?');"><?php esc_html_e( 'Delete', 'aegisbackup' ); ?></button>
										</form>
									</td>
								</tr>
							<?php endforeach; ?>
						</tbody>
					</table>
											</div>

								<?php endif; ?>

					<hr />
					<h3 style="margin-top: 6px;"><?php esc_html_e( 'Database Migration', 'aegisbackup' ); ?></h3>
					<p class="description"><?php esc_html_e( 'Select a database table backup, then click “Migrate DB” to copy the Database backup over the secured connection to the Destination.', 'aegisbackup' ); ?></p>

					<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin-top: 10px;">
						<input type="hidden" name="action" value="aegisbackup_pp_migrate_db_post" />
						<?php wp_nonce_field( 'aegisbackup_pp_migrate_db_post', 'aegisbackup_pp_nonce' ); ?>

						<select name="package_path" id="ab_push_package">
						<option value="" selected="selected"><?php esc_html_e( '— Select a Backup —', 'aegisbackup' ); ?></option>

						<?php if ( ! empty( $packages ) ) : ?>
							<optgroup label="<?php echo esc_attr( 'Migration Packages' ); ?>">
								<?php foreach ( (array) $packages as $p ) : ?>
									<?php if ( empty( $p['package'] ) ) { continue; } ?>
									<option value="<?php echo esc_attr( $p['package'] ); ?>">
										<?php echo esc_html( $p['name'] . ' — ' . $p['created'] . ' — ' . $p['size'] ); ?>
									</option>
								<?php endforeach; ?>
							</optgroup>
						<?php endif; ?>

						<?php if ( ! empty( $table_backups ) ) : ?>
							<optgroup label="<?php echo esc_attr( 'Table Backups' ); ?>">
								<?php foreach ( (array) $table_backups as $tb ) : ?>
									<?php
										$tb_zip = isset( $tb['zip'] ) ? (string) $tb['zip'] : '';
										if ( empty( $tb_zip ) || ! is_file( $tb_zip ) ) { continue; }
										$tb_name = isset( $tb['name'] ) ? (string) $tb['name'] : 'Table Backup';
										$tb_created = isset( $tb['created'] ) ? (int) $tb['created'] : 0;
										$tb_db = isset( $tb['db'] ) ? (string) $tb['db'] : '';
										$tb_tables = isset( $tb['tables'] ) && is_array( $tb['tables'] ) ? $tb['tables'] : array();
										$tb_size = isset( $tb['size'] ) ? (int) $tb['size'] : 0;
										$tb_label = $tb_name;
										if ( $tb_created ) { $tb_label .= ' — ' . gmdate( 'Y-m-d H:i', $tb_created ) . ' UTC'; }
										if ( $tb_db ) { $tb_label .= ' — ' . $tb_db; }
										if ( ! empty( $tb_tables ) ) { $tb_label .= ' — ' . count( $tb_tables ) . ' tables'; }
										if ( $tb_size ) { $tb_label .= ' — ' . size_format( $tb_size ); }
									?>
									<option value="<?php echo esc_attr( $tb_zip ); ?>">
										<?php echo esc_html( $tb_label ); ?>
									</option>
								<?php endforeach; ?>
							</optgroup>
						<?php endif; ?>
						</select>

						<button type="submit" class="button button-primary" style="margin-left:8px;" data-ab-post="1"><?php esc_html_e( 'Migrate DB', 'aegisbackup' ); ?></button>

						<p class="description" style="margin-top: 8px;">
							<?php esc_html_e( 'Tip: Take a table backup first, then select it from the list.', 'aegisbackup' ); ?>
						</p>
					</form>

					<hr />
					<h3 style="margin-top: 6px;" id="ab_pp_paths_section"><?php esc_html_e( 'Directories / Files', 'aegisbackup' ); ?></h3>
					<p class="description"><?php esc_html_e( 'Optional: choose specific directories/files (relative to /public_html/) to include as part of the push. This mirrors the File Backups tree selector.', 'aegisbackup' ); ?></p>

					<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin-top: 10px;">
						<input type="hidden" name="action" value="aegisbackup_pp_migrate_file_backup_post" />
						<?php wp_nonce_field( 'aegisbackup_pp_migrate_file_backup_post', 'aegisbackup_pp_nonce' ); ?>

						<select name="file_backup_zip" id="ab_pp_file_backup_zip">
							<option value="" selected="selected"><?php esc_html_e( '— Select a File Backup ZIP —', 'aegisbackup' ); ?></option>
							<?php if ( ! empty( $file_backups ) ) : ?>
								<?php foreach ( (array) $file_backups as $fb ) : ?>
									<?php
										$fb_zip = isset( $fb['file'] ) ? (string) $fb['file'] : '';
										if ( '' === $fb_zip || ! is_file( $fb_zip ) ) { continue; }
										$fb_id = isset( $fb['id'] ) ? (string) $fb['id'] : basename( $fb_zip, '.zip' );
										$fb_created = isset( $fb['created'] ) ? (string) $fb['created'] : '';
										$fb_size = isset( $fb['size'] ) ? (string) $fb['size'] : '';
										$label = $fb_id;
										if ( $fb_created ) { $label .= ' — ' . $fb_created; }
										if ( $fb_size ) { $label .= ' — ' . $fb_size; }
									?>
									<option value="<?php echo esc_attr( $fb_zip ); ?>"><?php echo esc_html( $label ); ?></option>
								<?php endforeach; ?>
							<?php endif; ?>
						</select>

						<button type="submit" class="button button-primary" style="margin-left:8px;" data-ab-post="1"><?php esc_html_e( 'Migrate File Backup', 'aegisbackup' ); ?></button>

						<p class="description" style="margin-top: 8px;">
							<?php esc_html_e( 'Tip: Create a File Backup first (Backups → File Backups), then select its ZIP here to copy it to the Destination.', 'aegisbackup' ); ?>
						</p>
					</form>


					<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin-top: 10px;">
						<input type="hidden" name="action" value="aegisbackup_pp_migrate_files_post" />
						<?php wp_nonce_field( 'aegisbackup_pp_migrate_files_post', 'aegisbackup_pp_nonce' ); ?>

						<table class="form-table" role="presentation" style="margin-top:0;">
							<tbody>
							<tr>
								<th scope="row"><?php esc_html_e( 'Source Base Path', 'aegisbackup' ); ?></th>
								<td>
									<input type="text" class="regular-text" name="src_root" value="<?php echo esc_attr( $saved_src_root ); ?>" />
									<p class="description"><?php esc_html_e( 'Defaults to /public_html/. Selected paths below are relative to this base.', 'aegisbackup' ); ?></p>
								</td>
							</tr>
							<tr>
								<th scope="row"><?php esc_html_e( 'Destination Base Path', 'aegisbackup' ); ?></th>
								<td>
									<input type="text" class="regular-text" name="dst_root" value="<?php echo esc_attr( $saved_dst_root ); ?>" />
									<p class="description"><?php esc_html_e( 'Defaults to /public_html/. When restoring files on the Destination, AegisBackup will target this base path.', 'aegisbackup' ); ?></p>
								</td>
							</tr>
							<tr>
								<th scope="row"><?php esc_html_e( 'Directories / Files', 'aegisbackup' ); ?></th>
								<td>
									<style id="ab-pp-filetree-inline-css">
										.aegisbackup-wrap .ab-filetree{border:1px solid #dcdcde;background:#fff;padding:6px;max-width:100%;max-height:420px;overflow:auto;border-radius:6px;}
										.aegisbackup-wrap .ab-filetree .ab-ft-tree{list-style:none;margin:0;padding:0;}
										.aegisbackup-wrap .ab-filetree .ab-ft-details > summary{list-style:none;}
										.aegisbackup-wrap .ab-filetree .ab-ft-details > summary::-webkit-details-marker{display:none;}
										.aegisbackup-wrap .ab-filetree .ab-ft-row{display:flex;align-items:flex-start;gap:8px;padding:5px 6px;border-radius:4px;}
										.aegisbackup-wrap .ab-filetree .ab-ft-row:hover{background:#f6f7f7;}
										.aegisbackup-wrap .ab-filetree .ab-ft-row::before{content:'';width:14px;flex:0 0 14px;display:inline-block;margin-top:1px;}
										.aegisbackup-wrap .ab-filetree .ab-ft-folder > .ab-ft-details > summary.ab-ft-row::before{content:'▸';color:#646970;}
										.aegisbackup-wrap .ab-filetree .ab-ft-folder > .ab-ft-details[open] > summary.ab-ft-row::before{content:'▾';color:#646970;}
										.aegisbackup-wrap .ab-filetree .ab-ft-check{margin:2px 0 0 0;}
										.aegisbackup-wrap .ab-filetree .ab-ft-text{display:flex;flex-direction:column;gap:2px;}
										.aegisbackup-wrap .ab-filetree .ab-ft-name{font-size:13px;line-height:1.15;}
										.aegisbackup-wrap .ab-filetree .ab-ft-subpath code{color:#646970;font-size:11px;background:transparent;padding:0;}
										.aegisbackup-wrap .ab-filetree .ab-ft-children{margin:2px 0 0 0;}
									</style>

									<p class="description" style="margin-top:0;">
										<?php esc_html_e( 'Select folders and/or files. Click folders to expand like a tree.', 'aegisbackup' ); ?>
									</p>

									<div class="ab-filetree">
										<?php $this->ab_render_tree_pp( '', 0, 19, (array) $saved_pp_paths ); ?>
									</div>
									<p class="description" style="margin-top:8px;">
										<?php esc_html_e( 'Tip: Selecting a folder includes everything inside that folder. Selecting a file includes only that file.', 'aegisbackup' ); ?>
									</p>
								</td>
							</tr>
							</tbody>
						</table>

					<button type="submit" class="button button-primary" data-ab-post="1"><?php esc_html_e( 'Migrate Files', 'aegisbackup' ); ?></button>

					<div id="ab_pp_files_progress_wrap" style="display:none; margin-top:10px;">
						<div class="description" style="margin:0 0 6px 0;">
							<strong><?php esc_html_e( 'File Transfer Progress', 'aegisbackup' ); ?></strong>
							<span id="ab_pp_files_progress_label" style="margin-left:6px;"></span>
						</div>
						<textarea class="large-text code" rows="6" id="ab_pp_files_progress" readonly></textarea>
						<p class="description" style="margin:6px 0 0 0;">
							<?php esc_html_e( 'This live progress log requires admin-ajax to be available. If your host blocks admin-ajax, AegisBackup will use the admin-post fallback and show a summary after redirect.', 'aegisbackup' ); ?>
						</p>
					</div>
					</form>
					<?php

					?>

                    <?php if ( ! empty( $job_state['job_id'] ) ) : ?>
                        <?php
                        $progress = isset( $job_state['progress'] ) ? (int) $job_state['progress'] : 0;
                        $phase    = isset( $job_state['phase'] ) ? (string) $job_state['phase'] : '';
                        $status   = isset( $job_state['status'] ) ? (string) $job_state['status'] : '';
                        ?>
                        <hr />
                        <h3><?php esc_html_e( 'Push Progress', 'aegisbackup' ); ?></h3>
                        <p class="description">
                            <?php echo esc_html( sprintf( 'Job: %s | Status: %s | Phase: %s | Progress: %d%%', $job_id, $status, $phase, $progress ) ); ?>
                        </p>
                        <div class="aegisbackup-progress">
                            <div class="aegisbackup-progress-bar" style="width: <?php echo (int) $progress; ?>%;"></div>
                        </div>
                        <textarea class="large-text code" rows="10" readonly><?php echo esc_textarea( $job_log ); ?></textarea>

                        <?php if ( 'done' !== $phase ) : ?>
                            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin-top:12px;">
                                <input type="hidden" name="action" value="aegisbackup_process_push_post" />
                                <input type="hidden" name="job_id" value="<?php echo esc_attr( $job_id ); ?>" />
                                <?php wp_nonce_field( 'aegisbackup_process_push_post', 'aegisbackup_pp_nonce' ); ?>
                                <button type="submit" class="button button-secondary" data-ab-post="1"><?php esc_html_e( 'Continue Push (admin-post)', 'aegisbackup' ); ?></button>
                                <span class="description" style="margin-left:8px;">
                                    <?php esc_html_e( 'Runs a few chunks per request (shared hosting friendly). Click again if needed until done.', 'aegisbackup' ); ?>
                                </span>
                            </form>
                        <?php endif; ?>
                    <?php endif; ?>
                </div>

                <?php else : ?>
				<div class="aegisbackup-card">
                    <h2><?php esc_html_e( 'Destination: Connect URL + Generate Token', 'aegisbackup' ); ?></h2>
                    <p class="description"><?php esc_html_e( 'On the destination site, generate a short-lived token. Then paste that token JSON into the source site (left panel).', 'aegisbackup' ); ?></p>

                    <p class="description">
                        <strong><?php esc_html_e( 'Destination Connect URL:', 'aegisbackup' ); ?></strong>
                        <code><?php echo esc_html( $connect_url ); ?></code>
                    </p>
                    <p class="description"><?php esc_html_e( 'This URL is a simple “is AegisBackup installed?” endpoint used for connection tests. Actual transfer uses WordPress REST endpoints under /wp-json/aegisbackup/v1/.', 'aegisbackup' ); ?></p>

					<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
						<input type="hidden" name="action" value="aegisbackup_generate_token_post" />
						<?php wp_nonce_field( 'aegisbackup_generate_token_post', 'aegisbackup_pp_nonce' ); ?>
						<p>
							<?php if ( empty( $saved_token_json ) ) : ?>
								<button type="submit" class="button button-primary" id="ab_generate_token_post"><?php esc_html_e( 'Generate Token', 'aegisbackup' ); ?></button>
							<?php else : ?>
								<button type="submit" class="button button-primary" id="ab_generate_token_post"><?php esc_html_e( 'Re-generate Token', 'aegisbackup' ); ?></button>
								<button type="button" class="button button-secondary" id="ab_copy_token" style="margin-left:8px;" data-copy-target="#ab_token_out"><?php esc_html_e( 'Copy', 'aegisbackup' ); ?></button>
							<?php endif; ?>
						</p>
						<p class="description" id="ab_token_notice"></p>
					</form>

                    <textarea class="large-text code" rows="6" id="ab_token_out" placeholder="{ token json... }" readonly><?php echo esc_textarea( $saved_token_json ); ?></textarea>
                    <p class="description"><?php esc_html_e( 'Security: token is HMAC-signed and expires automatically. Use HTTPS for best safety.', 'aegisbackup' ); ?></p>
							<?php if ( ! empty( $saved_token_json ) && $ab_listen && empty( $incoming ) ) : ?>
								<script>
								(function(){
									try{
										var key='ab_pp_listen_until';
										var now=Date.now();
										var until=parseInt(sessionStorage.getItem(key)||'0',10);
										if(!until || until<now){ until = now + 120000; sessionStorage.setItem(key, String(until)); }
										if(now < until){ setTimeout(function(){ window.location.reload(); }, 6000); }
									}catch(e){}
								})();
								</script>
							<?php endif; ?>
							<hr />
							<h3 style="margin-top: 6px;"><?php esc_html_e( 'Incoming Connections (Destination)', 'aegisbackup' ); ?></h3>
							<p class="description"><?php esc_html_e( 'This destination is listening for a Source connection. When a source saves a connection using this token, it will appear here as "Connected".', 'aegisbackup' ); ?></p>
						<?php if ( empty( $incoming ) ) : ?>
						<div style="border:1px solid #ccd0d4;background:#fff;padding:18px;border-radius:8px;min-height:180px;">
							<h4 style="margin-top:0;"><?php esc_html_e( 'Listening for connections…', 'aegisbackup' ); ?></h4>
							<p class="description"><?php esc_html_e( 'Generate a token above, then paste it into the Source site (left panel) and click “Save Connection”. Once the source registers successfully, it will show up here.', 'aegisbackup' ); ?></p>
						</div>
					<?php else : ?>
						<div style="overflow-x:auto; max-width:100%;">
					<table class="widefat striped" style="margin-top:10px; width:100%;">
							<thead>
								<tr>
									<th><?php esc_html_e( 'Source Name', 'aegisbackup' ); ?></th>
									<th><?php esc_html_e( 'Source URL', 'aegisbackup' ); ?></th>
									<th><?php esc_html_e( 'Status', 'aegisbackup' ); ?></th>
									<th><?php esc_html_e( 'Last Activity', 'aegisbackup' ); ?></th>
									<th><?php esc_html_e( 'Action', 'aegisbackup' ); ?></th>
								</tr>
							</thead>
							<tbody>
								<?php foreach ( $incoming as $k => $row ) : ?>
									<?php if ( ! is_array( $row ) ) { continue; } ?>
									<?php
										$s_name = isset( $row['source_name'] ) ? (string) $row['source_name'] : '';
										$s_url  = isset( $row['source_url'] ) ? (string) $row['source_url'] : '';
										$s_connected = ! empty( $row['connected'] );
										$s_last = isset( $row['last_activity'] ) ? (int) $row['last_activity'] : 0;
									?>
									<tr>
										<td><?php echo esc_html( $s_name ? $s_name : '—' ); ?></td>
										<td><code><?php echo esc_html( $s_url ); ?></code></td>
										<td><?php echo $s_connected ? esc_html__( 'Connected', 'aegisbackup' ) : esc_html__( 'Disconnected', 'aegisbackup' ); ?></td>
										<td><?php echo $s_last ? esc_html( gmdate( 'Y-m-d H:i:s', $s_last ) . ' UTC' ) : esc_html__( '—', 'aegisbackup' ); ?></td>
										<td>
											<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="display:inline-block;">
												<input type="hidden" name="action" value="aegisbackup_pp_delete_incoming_post" />
												<input type="hidden" name="incoming_key" value="<?php echo esc_attr( (string) $k ); ?>" />
												<?php wp_nonce_field( 'aegisbackup_pp_delete_incoming_post', 'aegisbackup_pp_nonce' ); ?>
												<button type="submit" class="button button-small" data-ab-post="1" onclick="return confirm('Remove this incoming connection?');"><?php esc_html_e( 'Forget', 'aegisbackup' ); ?></button>
											</form>
										</td>
									</tr>
								<?php endforeach; ?>
							</tbody>
						</table>
							</div>
					<?php endif; ?>

					<hr />
					<h3 style="margin-top: 6px;" id="ab_pp_dest_log_section"><?php esc_html_e( 'Live Transfer Log (Destination)', 'aegisbackup' ); ?></h3>
					<p class="description"><?php esc_html_e( 'Shows file/folder writes received from the Source in real-time (success + failures).', 'aegisbackup' ); ?></p>
					<?php

					$__ab_dest_log = get_option( 'aegisbackup_pp_destination_live_log', array() );
					if ( ! is_array( $__ab_dest_log ) ) {
						$__ab_dest_log = array();
					}
					$__ab_dest_log = array_slice( array_values( array_filter( $__ab_dest_log, 'is_string' ) ), -200 );
					$__ab_dest_log_text = implode( "\n", $__ab_dest_log );
					?>
					<textarea class="large-text code" rows="10" id="ab_pp_dest_live_log" readonly placeholder="Waiting for transfers..."><?php echo esc_textarea( $__ab_dest_log_text ); ?></textarea>
					<p class="description" style="margin-top:6px;"><?php esc_html_e( 'This log auto-refreshes while this page is open.', 'aegisbackup' ); ?></p>

					<?php
					$fb_backups = get_option( 'aegisbackup_pp_received_file_backups', array() );
					if ( ! is_array( $fb_backups ) ) {
						$fb_backups = array();
					}
					?>
					<hr />
					<h3 style="margin-top: 6px;"><?php esc_html_e( 'Folders & Files Backup', 'aegisbackup' ); ?></h3>
					<p class="description"><?php esc_html_e( 'File Backup ZIPs received from a Source site will appear here. click Restore to start the Restore Wizard.', 'aegisbackup' ); ?></p>

					<?php

					if ( 'destination' === $pp_tab && 'file_restore' === $pp_view && $fb_key && isset( $fb_backups[ $fb_key ] ) ) :
						$w = $fb_backups[ $fb_key ];
						$w_name     = isset( $w['name'] ) ? (string) $w['name'] : '';
						$w_received = isset( $w['received_at'] ) ? (int) $w['received_at'] : 0;
						$w_backup_id = isset( $w['backup_id'] ) ? (string) $w['backup_id'] : '';
						$w_zip      = isset( $w['zip'] ) ? (string) $w['zip'] : '';
						?>
						<div style="border:1px solid #ccd0d4;background:#fff;padding:16px;border-radius:8px;margin:12px 0;">
							<h3 style="margin-top:0;"><?php esc_html_e( 'Restore Wizard — Files & Folders', 'aegisbackup' ); ?></h3>
							<p class="description" style="margin-top:0;"><?php esc_html_e( 'This guided restore is designed for non-technical admins. It uses safe defaults and prevents accidental restores.', 'aegisbackup' ); ?></p>

							<table class="widefat striped" style="margin:10px 0;">
								<tbody>
									<tr>
										<th style="width:120px;"><?php esc_html_e( 'Backup', 'aegisbackup' ); ?></th>
										<td><?php echo esc_html( $w_name ); ?></td>
									</tr>
									<tr>
										<th><?php esc_html_e( 'Date', 'aegisbackup' ); ?></th>
										<td><?php echo $w_received ? esc_html( gmdate( 'Y-m-d H:i:s', $w_received ) ) : ''; ?></td>
									</tr>
									<tr>
										<th><?php esc_html_e( 'ZIP', 'aegisbackup' ); ?></th>
										<td><code style="word-break:break-all;"><?php echo esc_html( $w_zip ); ?></code></td>
									</tr>
								</tbody>
							</table>

							<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
								<input type="hidden" name="action" value="aegisbackup_pp_file_restore_confirm_post" />
								<?php wp_nonce_field( 'aegisbackup_pp_file_restore_confirm_post', 'aegisbackup_pp_nonce' ); ?>
								<input type="hidden" name="fb_key" value="<?php echo esc_attr( $fb_key ); ?>" />

								<div style="margin-top:10px;">
									<label>
										<input type="checkbox" name="confirm_drop" value="1" />
										<?php esc_html_e( 'Yes, overwrite existing files using this backup', 'aegisbackup' ); ?>
									</label>
								</div>

								<div style="margin-top:10px;">
									<label><?php esc_html_e( 'Type RESTORE to confirm', 'aegisbackup' ); ?></label><br/>
									<input type="text" name="confirm_text" value="" class="regular-text" />
									<span class="description"><?php esc_html_e( 'This prevents accidental restores.', 'aegisbackup' ); ?></span>
								</div>

								<div style="margin-top:12px; display:flex; gap:8px; align-items:center;">
									<button type="submit" class="button button-primary"><?php esc_html_e( 'Start File Restore', 'aegisbackup' ); ?></button>
									<a class="button" href="<?php echo esc_url( remove_query_arg( array( 'pp_view', 'fb_key' ) ) ); ?>"><?php esc_html_e( 'Cancel', 'aegisbackup' ); ?></a>
								</div>
							</form>
						</div>
					<?php endif; ?>

					<?php if ( empty( $fb_backups ) ) : ?>
						<p class="description"><?php esc_html_e( 'No File Backup ZIPs received yet.', 'aegisbackup' ); ?></p>
					<?php else : ?>
						<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
							<input type="hidden" name="action" value="aegisbackup_pp_file_restore_bulk_post" />
							<?php wp_nonce_field( 'aegisbackup_pp_file_restore_bulk_post', 'aegisbackup_pp_nonce' ); ?>

							<table class="widefat striped">
								<colgroup>
									<col style="width:3%;" />   
									<col />                   
									<col />                    
									<col />                    
									<col style="width:15%;" />  
								</colgroup>
								<thead>
									<tr>
										<th><?php esc_html_e( 'Use', 'aegisbackup' ); ?></th>
										<th><?php esc_html_e( 'Backup', 'aegisbackup' ); ?></th>
										<th><?php esc_html_e( 'Received', 'aegisbackup' ); ?></th>
										<th><?php esc_html_e( 'Package Folder', 'aegisbackup' ); ?></th>
										<th><?php esc_html_e( 'Action', 'aegisbackup' ); ?></th>
									</tr>
								</thead>
								<tbody>
								<?php foreach ( (array) $fb_backups as $k => $w ) : ?>
									<?php
									$name        = isset( $w['name'] ) ? (string) $w['name'] : (string) $k;
									$received_at = isset( $w['received_at'] ) ? (int) $w['received_at'] : 0;
									$zip         = isset( $w['zip'] ) ? (string) $w['zip'] : '';
									$pkg_folder  = $zip;
									?>
									<tr>
										<td><input type="checkbox" name="fb_keys[]" value="<?php echo esc_attr( $k ); ?>" /></td>

										<td><?php echo esc_html( $name ); ?></td>

										<td><?php echo $received_at ? esc_html( gmdate( 'Y-m-d H:i:s', $received_at ) . ' UTC' ) : ''; ?></td>

										<td><code style="display:block;word-break:break-all;white-space:normal;"><?php echo esc_html( $pkg_folder ); ?></code></td>

										<td>
											<a class="button button-secondary" href="<?php echo esc_url( add_query_arg( array( 'pp_tab' => 'destination', 'pp_view' => 'file_restore', 'fb_key' => $k ) ) ); ?>">
												<?php esc_html_e( 'Restore', 'aegisbackup' ); ?>
											</a>

											<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="display:inline;">
												<input type="hidden" name="action" value="aegisbackup_pp_file_delete_post" />
												<?php wp_nonce_field( 'aegisbackup_pp_file_delete_post', 'aegisbackup_pp_nonce' ); ?>
												<input type="hidden" name="fb_key" value="<?php echo esc_attr( $k ); ?>" />
												<button type="submit" class="button button-link-delete" onclick="return confirm('Delete this received file backup?');">
													<?php esc_html_e( 'Delete', 'aegisbackup' ); ?>
												</button>
											</form>
										</td>
									</tr>
								<?php endforeach; ?>
								</tbody>
							</table>
						</form>
					<?php endif; ?>

					<?php
					$db_backups = get_option( 'aegisbackup_pp_received_db_backups', array() );
					if ( ! is_array( $db_backups ) ) {
						$db_backups = array();
					}
					?>

					<hr />
					<h3 style="margin-top: 6px;"><?php esc_html_e( 'Database Backup', 'aegisbackup' ); ?></h3>
					<p class="description"><?php esc_html_e( 'DB backups received from a Source site will appear here. Use Restore to apply the DB on this Destination site, or Delete to remove the received backup.', 'aegisbackup' ); ?></p>

					<?php
					if ( 'destination' === $pp_tab && 'db_restore' === $pp_view && $db_key && isset( $db_backups[ $db_key ] ) ) :
						$w = $db_backups[ $db_key ];
						$w_name = isset( $w['name'] ) ? (string) $w['name'] : ( isset( $w['backup_name'] ) ? (string) $w['backup_name'] : '' );
						$w_received = isset( $w['received_at'] ) ? (int) $w['received_at'] : 0;
						$w_pkg = isset( $w['package_path'] ) ? (string) $w['package_path'] : '';
						$w_tables = isset( $w['tables_count'] ) ? (int) $w['tables_count'] : 0;
						$w_db = isset( $w['db_name'] ) ? (string) $w['db_name'] : '';
					?>
						<div style="border:1px solid #ccd0d4;background:#fff;padding:16px;border-radius:8px;margin:12px 0;">
							<h3 style="margin-top:0;">Restore Wizard — Table Backup</h3>
							<p class="description" style="margin-top:0;">Table Backups restore selected database tables into this site. Confirm below to start the restore job.</p>

							<?php if ( '' !== $ab_ok && '' !== $ab_msg ) : ?>
								<div class="notice <?php echo ( '1' === $ab_ok ) ? 'notice-success' : 'notice-error'; ?> is-dismissible" style="margin:12px 0 10px 0;">
									<p><?php echo esc_html( $ab_msg ); ?></p>
								</div>
							<?php endif; ?>

							<div style="border-left:4px solid #dba617;padding:10px 12px;background:#fff;">
								<strong><?php esc_html_e( 'Important:', 'aegisbackup' ); ?></strong>
								<?php esc_html_e( 'Restoring a table backup will overwrite the selected tables on this site.', 'aegisbackup' ); ?>
							</div>

							<table class="widefat striped" style="margin-top:12px;">
								<tbody>
									<tr>
										<th style="width:180px;"><?php esc_html_e( 'Backup', 'aegisbackup' ); ?></th>
										<td><?php echo esc_html( $w_name ? $w_name : $db_key ); ?></td>
									</tr>
									<tr>
										<th><?php esc_html_e( 'Received', 'aegisbackup' ); ?></th>
										<td><?php echo $w_received ? esc_html( gmdate( 'Y-m-d H:i:s', $w_received ) . ' UTC' ) : esc_html__( '—', 'aegisbackup' ); ?></td>
									</tr>
									<?php if ( $w_db ) : ?>
										<tr>
											<th><?php esc_html_e( 'Database', 'aegisbackup' ); ?></th>
											<td><?php echo esc_html( $w_db ); ?></td>
										</tr>
									<?php endif; ?>
									<tr>
										<th><?php esc_html_e( 'Tables', 'aegisbackup' ); ?></th>
										<td><?php echo esc_html( (string) $w_tables ); ?></td>
									</tr>
								</tbody>
							</table>

							<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin-top:12px;">
								<input type="hidden" name="action" value="aegisbackup_pp_restore_db_post" />
								<?php wp_nonce_field( 'aegisbackup_pp_restore_db_post', 'aegisbackup_pp_nonce' ); ?>
								<input type="hidden" name="package_path" value="<?php echo esc_attr( $w_pkg ); ?>" />
								<input type="hidden" name="return_tab" value="destination" />
								<input type="hidden" name="return_view" value="db_restore" />
								<input type="hidden" name="return_key" value="<?php echo esc_attr( (string) $db_key ); ?>" />

								<label style="display:block;margin:10px 0;">
									<input type="checkbox" name="confirm_overwrite" value="1" />
									<?php esc_html_e( 'Yes, restore (overwrite) database tables using this table backup', 'aegisbackup' ); ?>
								</label>

								<div style="display:flex;gap:12px;align-items:center;">
									<div>
										<label for="ab_pp_confirm_text" style="font-weight:600;display:block;margin-bottom:4px;">Type RESTORE to confirm</label>
										<input type="text" id="ab_pp_confirm_text" name="confirm_text" class="regular-text" placeholder="RESTORE" />
										<p class="description" style="margin-top:6px;">This prevents accidental restores.</p>
									</div>
								</div>

								<div style="margin-top:12px;display:flex;gap:8px;align-items:center;">
									<button type="submit" class="button button-primary"><?php esc_html_e( 'Restore', 'aegisbackup' ); ?></button>
									<a class="button" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisbackup-pushpull&pp_tab=destination#ab_pp_dest_db_section' ) ); ?>"><?php esc_html_e( 'Back', 'aegisbackup' ); ?></a>
									<span class="description"><?php esc_html_e( 'Click Restore to start the database restore job.', 'aegisbackup' ); ?></span>
								</div>
							</form>
						</div>
					<?php endif; ?>

					<?php if ( empty( $db_backups ) ) : ?>
						<div style="border:1px solid #ccd0d4;background:#fff;padding:18px;border-radius:8px;min-height:90px;">
							<p class="description" style="margin:0;"><?php esc_html_e( 'No DB backups received yet.', 'aegisbackup' ); ?></p>
						</div>
					<?php else : ?>
						<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin-top:10px;">
							<input type="hidden" name="action" value="aegisbackup_pp_restore_db_bulk_post" />
							<?php wp_nonce_field( 'aegisbackup_pp_restore_db_bulk_post', 'aegisbackup_pp_nonce' ); ?>

							<div style="display:flex;gap:8px;align-items:center;justify-content:flex-start;margin:10px 0;">
							<span class="description"><?php esc_html_e( 'Select one received DB backup below, then click Restore to start the Retore Wizard.', 'aegisbackup' ); ?></span>
							</div>

							<table class="widefat striped">
								<thead>
									<tr>
										<th style="width:38px;"><?php esc_html_e( 'Use', 'aegisbackup' ); ?></th>
										<th><?php esc_html_e( 'Backup', 'aegisbackup' ); ?></th>
										<th><?php esc_html_e( 'Received', 'aegisbackup' ); ?></th>
										<th><?php esc_html_e( 'Package Folder', 'aegisbackup' ); ?></th>
										<th style="width:180px;"><?php esc_html_e( 'Action', 'aegisbackup' ); ?></th>
									</tr>
								</thead>
								<tbody>
									<?php foreach ( $db_backups as $k => $row ) : ?>
										<?php
											$pkg_path = ! empty( $row['package_path'] ) ? (string) $row['package_path'] : '';
											$label = 'DB Backup';
											$meta_file = $pkg_path ? trailingslashit( $pkg_path ) . 'meta.json' : '';
											if ( $meta_file && is_file( $meta_file ) ) {
												$raw = (string) @file_get_contents( $meta_file );
												$m = json_decode( $raw, true );
												if ( is_array( $m ) ) {
													if ( ! empty( $m['name'] ) ) { $label = (string) $m['name']; }
													if ( ! empty( $m['db'] ) ) { $label .= ' — ' . (string) $m['db']; }
													if ( ! empty( $m['tables'] ) && is_array( $m['tables'] ) ) { $label .= ' — ' . count( $m['tables'] ) . ' tables'; }
												}
											}
										?>
										<tr>
											<td>
												<label class="screen-reader-text" for="ab_db_sel_<?php echo esc_attr( (string) $k ); ?>"><?php esc_html_e( 'Select', 'aegisbackup' ); ?></label>
												<input id="ab_db_sel_<?php echo esc_attr( (string) $k ); ?>" type="checkbox" name="db_keys[]" value="<?php echo esc_attr( (string) $k ); ?>" />
											</td>
											<td><?php echo esc_html( $label ); ?></td>
											<td><?php echo ! empty( $row['received'] ) ? esc_html( gmdate( 'Y-m-d H:i:s', (int) $row['received'] ) . ' UTC' ) : ''; ?></td>
											<td><code><?php echo esc_html( $pkg_path ); ?></code></td>
							<td>
								<a class="button button-secondary" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisbackup-pushpull&pp_tab=destination&pp_view=db_restore&db_key=' . rawurlencode( (string) $k ) . '#ab_pp_dest_db_section' ) ); ?>"><?php esc_html_e( 'Restore', 'aegisbackup' ); ?></a>

												<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="display:inline-block;margin-left:6px;">
													<input type="hidden" name="action" value="aegisbackup_pp_delete_db_post" />
													<input type="hidden" name="db_key" value="<?php echo esc_attr( (string) $k ); ?>" />
													<?php wp_nonce_field( 'aegisbackup_pp_delete_db_post', 'aegisbackup_pp_nonce' ); ?>
													<button type="submit" class="button button-link-delete" onclick="return confirm('Delete this received DB backup?');"><?php esc_html_e( 'Delete', 'aegisbackup' ); ?></button>
												</form>
											</td>
										</tr>
									<?php endforeach; ?>
								</tbody>
							</table>
						</form>
					<?php endif; ?>
                </div>

            AegisBackup Push/Pull Instructions and Details', 'aegisbackup' ); ?></h2>
                <ul class="ul-disc">
                    <strong><span style="font-size: 14px;"><?php esc_html_e( '**** On Destination site (the site you’re restoring INTO)' ); ?></strong></span>
                    <li><?php esc_html_e( '1. Go to AegisBackup → Migration Wizard' ); ?></li>
                    <li><?php esc_html_e( '2. You will see:' ); ?></li>
                    <li><?php esc_html_e( '- Destination Connect URL (copy it if needed)' ); ?></li>
                    <li><?php esc_html_e( '- Click Generate Token' ); ?></li>
                    <li><?php esc_html_e( '3. Copy the Token JSON from the textarea' ); ?></li>
                    <hr>
                    <strong><span style="font-size: 14px;"><?php esc_html_e( '**** On Source site (the site you’re migrating FROM)' ); ?></strong></span>
                    <li><?php esc_html_e( '1. Go to AegisBackup → Migration Wizard' ); ?></li>
                    <li><?php esc_html_e( '2. Choose a name for your Destination' ); ?></li>
                    <li><?php esc_html_e( '3. Paste the Destination Token JSON' ); ?></li>
                    <li><?php esc_html_e( '4. Click Connect' ); ?></li>
                    <hr>
                    <h2><?php esc_html_e( 'Notes', 'aegisbackup' ); ?></h2>
                    <li><?php esc_html_e( 'Push/Pull is designed for shared hosting: small chunks, short requests, resume-friendly.', 'aegisbackup' ); ?></li>
                    <li><?php esc_html_e( 'Destination assembles the uploaded ZIP, then automatically starts a restore job.', 'aegisbackup' ); ?></li>
                    <li><?php esc_html_e( 'For best results, disable aggressive security/caching temporarily during migration.', 'aegisbackup' ); ?></li>
                    <li><?php esc_html_e( 'For a full “file restore”, the migration package should include wp-content (plugins/themes/uploads) and the WordPress root config files (wp-config.php, .htaccess) as applicable.', 'aegisbackup' ); ?></li>
                </ul>

            <?php endif; ?>

        </div>
        <?php
    }
	private function ab_render_tree_pp( $rel, $depth, $max_depth, array $selected ) {
		$depth = (int) $depth;
		$pad_left = 10 + ( $depth * 18 );
		$rel = (string) $rel;
		$rel = str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), $rel );
		$rel = ltrim( $rel, '/' );

		$root_real = realpath( (string) ABSPATH );
		if ( false === $root_real ) {
			echo '<div class="ab-ft-error">' . esc_html__( 'ABSPATH not accessible.', 'aegisbackup' ) . '</div>';
			return;
		}
		$root_real = rtrim( $root_real, "/\\" );

		$target = $root_real . ( $rel ? ( DIRECTORY_SEPARATOR . str_replace( '/', DIRECTORY_SEPARATOR, $rel ) ) : '' );
		$target_real = realpath( $target );
		if ( false === $target_real ) {
			echo '<div class="ab-ft-error">' . esc_html__( 'Path not found.', 'aegisbackup' ) . '</div>';
			return;
		}
		$target_real = rtrim( $target_real, "/\\" );

		if ( 0 !== strpos( $target_real, $root_real ) ) {
			echo '<div class="ab-ft-error">' . esc_html__( 'Invalid path.', 'aegisbackup' ) . '</div>';
			return;
		}

		if ( ! is_dir( $target_real ) || ! is_readable( $target_real ) ) {
			echo '<div class="ab-ft-error">' . esc_html__( 'Directory not readable.', 'aegisbackup' ) . '</div>';
			return;
		}

		$items = @scandir( $target_real );
		if ( ! is_array( $items ) ) {
			echo '<div class="ab-ft-error">' . esc_html__( 'Failed to scan directory.', 'aegisbackup' ) . '</div>';
			return;
		}

		$sel_map = array();
		foreach ( $selected as $p ) {
			$p = (string) $p;
			$p = str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), $p );
			$p = ltrim( $p, '/' );
			if ( '' !== $p ) {
				$sel_map[ $p ] = true;
			}
		}

		$dirs  = array();
		$files = array();
		foreach ( $items as $name ) {
			if ( '.' === $name || '..' === $name ) {
				continue;
			}
			$abs = $target_real . DIRECTORY_SEPARATOR . $name;
			$child_rel = $rel ? ( $rel . '/' . $name ) : $name;
			$child_rel = str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), (string) $child_rel );
			$child_rel = ltrim( $child_rel, '/' );
			if ( @is_dir( $abs ) ) {
				$dirs[] = array( 'name' => $name, 'rel' => $child_rel );
			} else {
				$files[] = array( 'name' => $name, 'rel' => $child_rel );
			}
		}
		usort( $dirs, static function( $a, $b ) { return strcasecmp( (string) $a['name'], (string) $b['name'] ); } );
		usort( $files, static function( $a, $b ) { return strcasecmp( (string) $a['name'], (string) $b['name'] ); } );

		echo '<ul class="ab-ft-tree" data-depth="' . esc_attr( $depth ) . '">';

		foreach ( $dirs as $d ) {
			$folder_rel  = (string) $d['rel'];
			$folder_name = (string) $d['name'];
			$disabled    = ( $depth >= $max_depth );
			$checked     = ! empty( $sel_map[ $folder_rel ] );

			echo '<li class="ab-ft-node ab-ft-folder">';
			if ( $disabled ) {
				echo '<div class="ab-ft-row" style="padding-left:' . esc_attr( (int) $pad_left ) . 'px;">';
				echo '<input class="ab-ft-check" type="checkbox" name="pp_paths[]" value="' . esc_attr( $folder_rel ) . '" ' . checked( $checked, true, false ) . ' onclick="event.stopPropagation();" />';
				echo '<div class="ab-ft-text">';
				echo '<div class="ab-ft-name">📁 ' . esc_html( $folder_name ) . ' <span class="ab-ft-muted">' . esc_html__( '(max depth)', 'aegisbackup' ) . '</span></div>';
				echo '<div class="ab-ft-subpath"><code>' . esc_html( $folder_rel ) . '</code></div>';
				echo '</div>';
				echo '</div>';
			} else {
				echo '<details class="ab-ft-details">';
				echo '<summary class="ab-ft-row" style="padding-left:' . esc_attr( (int) $pad_left ) . 'px;">';
				echo '<input class="ab-ft-check" type="checkbox" name="pp_paths[]" value="' . esc_attr( $folder_rel ) . '" ' . checked( $checked, true, false ) . ' onclick="event.stopPropagation();" />';
				echo '<div class="ab-ft-text">';
				echo '<div class="ab-ft-name">📁 ' . esc_html( $folder_name ) . '</div>';
				echo '<div class="ab-ft-subpath"><code>' . esc_html( $folder_rel ) . '</code></div>';
				echo '</div>';
				echo '</summary>';
				echo '<div class="ab-ft-children">';
				$this->ab_render_tree_pp( $folder_rel, $depth + 1, $max_depth, $selected );
				echo '</div>';
				echo '</details>';
			}
			echo '</li>';
		}

		foreach ( $files as $f ) {
			$file_rel  = (string) $f['rel'];
			$file_name = (string) $f['name'];
			$checked   = ! empty( $sel_map[ $file_rel ] );
			echo '<li class="ab-ft-node ab-ft-file">';
			echo '<div class="ab-ft-row" style="padding-left:' . esc_attr( (int) $pad_left ) . 'px;">';
			echo '<input class="ab-ft-check" type="checkbox" name="pp_paths[]" value="' . esc_attr( $file_rel ) . '" ' . checked( $checked, true, false ) . ' onclick="event.stopPropagation();" />';
			echo '<div class="ab-ft-text">';
			echo '<div class="ab-ft-name">📄 ' . esc_html( $file_name ) . '</div>';
			echo '<div class="ab-ft-subpath"><code>' . esc_html( $file_rel ) . '</code></div>';
			echo '</div>';
			echo '</div>';
			echo '</li>';
		}

		echo '</ul>';
	}
}
