<?php
namespace AegisBackup\Admin\Panels;

use AegisBackup\AB_Plugin;
use AegisBackup\Modules\AB_Module_DB_Tools;

defined( 'ABSPATH' ) || exit;

require_once AEGISBACKUP_DIR . 'includes/modules/class-ab-module-db-tools.php';
require_once AEGISBACKUP_DIR . 'includes/admin/panels/db-tools/class-ab-dbtools-tab-table-backups.php';

class AB_Panel_DB_Tools {

    protected $plugin;

    public function __construct( AB_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function render() {
        $current_tab = isset( $_GET['dbtab'] ) ? sanitize_key( wp_unslash( $_GET['dbtab'] ) ) : 'overview';
        $tabs = array(
            'overview'  => __( 'Overview', 'aegisbackup' ),
            'table_backups' => __( 'Table Backups', 'aegisbackup' ),
            'optimize'  => __( 'Optimize', 'aegisbackup' ),
            'prefix'    => __( 'DB Prefix Manager', 'aegisbackup' ),
            'backups'   => __( 'DB Snapshots', 'aegisbackup' ),
        );
        if ( ! isset( $tabs[ $current_tab ] ) ) {
            $current_tab = 'overview';
        }

        $base_url = add_query_arg(
            array(
                'page' => 'aegisbackup',
                'tab'  => 'db',
            ),
            admin_url( 'admin.php' )
        );

        $module = new AB_Module_DB_Tools();

        ?>
        <div class="aegisbackup-card">
            <h2><?php esc_html_e( 'Database Backups & Tools', 'aegisbackup' ); ?></h2>
            <p class="description"><?php esc_html_e( 'Export reliable DB backups, optimize tables, and safely change your WordPress table prefix.', 'aegisbackup' ); ?></p>

            <?php

            ?>

            <h2 class="nav-tab-wrapper" style="margin-top:12px;">
                <?php foreach ( $tabs as $slug => $label ) : ?>
                    <?php
                    $url = add_query_arg( 'dbtab', $slug, $base_url );
                    $active = ( $slug === $current_tab ) ? 'nav-tab-active' : '';
                    ?>
                    <a href="<?php echo esc_url( $url ); ?>" class="nav-tab <?php echo esc_attr( $active ); ?>">
                        <?php echo esc_html( $label ); ?>
                    </a>
                <?php endforeach; ?>
            </h2>

            <?php
            $is_pro = false;
            if ( isset( $this->plugin ) && isset( $this->plugin->license ) && is_object( $this->plugin->license ) && method_exists( $this->plugin->license, 'is_pro_active' ) ) {
                $is_pro = (bool) $this->plugin->license->is_pro_active();
            }
            $license_url = admin_url( 'admin.php?page=aegisbackup-license' );
            $locked_tabs = array( 'optimize', 'prefix', 'backups' );
            $needs_lock  = ( ! $is_pro && in_array( $current_tab, $locked_tabs, true ) );

            if ( $needs_lock ) :
                ?>
                <div class="notice notice-warning" style="margin:12px 0 16px;">
                    <p style="margin:0;">
                        <strong><?php esc_html_e( 'PRO Feature:', 'aegisbackup' ); ?></strong>
                        <?php esc_html_e( 'This tool is available in AegisBackup Pro. Upgrade to unlock advanced database tools.', 'aegisbackup' ); ?>
                        <a href="<?php echo esc_url( $license_url ); ?>" class="button button-primary" style="margin-left:10px;">
                            <?php esc_html_e( 'Upgrade / Activate License', 'aegisbackup' ); ?>
                        </a>
                    </p>
                </div>

                <style>
                    .aegisbackup-pro-locked{
                        opacity:0.45 !important;
                        pointer-events:none !important;
                        filter:grayscale(1) !important;
                        user-select:none !important;
                    }
                </style>

                <div class="aegisbackup-pro-locked" aria-disabled="true">
                <?php
            endif;

            switch ( $current_tab ) {
                case 'table_backups':
                    ( new \AegisBackup\Admin\Panels\DB_Tools\AB_DBTools_Tab_Table_Backups( $this->plugin ) )->render();
                    break;
                case 'optimize':
                    $this->render_optimize( $module );
                    break;
                case 'prefix':
                    $this->render_prefix( $module );
                    break;
                case 'backups':
                    $this->render_backups( $module );
                    break;
                case 'health':
                    $this->render_health( $module );
                    break;
                case 'overview':
                default:
                    $this->render_overview( $module );
                    break;
            }
            if ( $needs_lock ) :
                ?>
                </div>
                <?php
            endif;
            ?>
        </div>
        <?php
    }

    protected function render_overview( AB_Module_DB_Tools $module ) {
        global $wpdb;

        $summary = $module->get_table_summary();
        $tables  = isset( $summary['tables'] ) && is_array( $summary['tables'] ) ? $summary['tables'] : $summary;
        $status_rows = $wpdb->get_results( 'SHOW TABLE STATUS', ARRAY_A );
        if ( ! is_array( $status_rows ) ) {
            $status_rows = array();
        }

        $prefix = isset( $wpdb->prefix ) ? (string) $wpdb->prefix : '';

        $top_sizes = array();
        $top_rows  = array();
        $bubble_tables = array();
        $engine_bytes = array();
        $collation_bytes = array();
        $core_bytes = 0;
        $plugin_bytes = 0;
        $other_bytes = 0;
        $sum_data_bytes  = 0;
        $sum_index_bytes = 0;

        foreach ( $status_rows as $r ) {
            $name = isset( $r['Name'] ) ? (string) $r['Name'] : '';
            if ( '' === $name ) {
                continue;
            }
            $data_len  = isset( $r['Data_length'] ) ? (int) $r['Data_length'] : 0;
            $index_len = isset( $r['Index_length'] ) ? (int) $r['Index_length'] : 0;
            $bytes = (int) ( $data_len + $index_len );
            $rows  = isset( $r['Rows'] ) ? (int) $r['Rows'] : 0;
            $engine = isset( $r['Engine'] ) ? (string) $r['Engine'] : '';
            $collation = isset( $r['Collation'] ) ? (string) $r['Collation'] : '';
            $sum_data_bytes  += $data_len;
            $sum_index_bytes += $index_len;
            $top_sizes[] = array( 'name' => $name, 'bytes' => $bytes );
            $top_rows[]  = array( 'name' => $name, 'rows' => $rows );
            $bubble_tables[] = array(
                'name'       => $name,
                'rows'       => $rows,
                'data_bytes' => $data_len,
                'index_bytes'=> $index_len,
                'bytes'      => $bytes,
            );

            if ( '' === $engine ) {
                $engine = 'Unknown';
            }
            if ( ! isset( $engine_bytes[ $engine ] ) ) {
                $engine_bytes[ $engine ] = 0;
            }
            $engine_bytes[ $engine ] += $bytes;

            if ( '' === $collation ) {
                $collation = 'Unknown';
            }
            if ( ! isset( $collation_bytes[ $collation ] ) ) {
                $collation_bytes[ $collation ] = 0;
            }
            $collation_bytes[ $collation ] += $bytes;

            $suffix = $name;
            if ( '' !== $prefix && 0 === strpos( $name, $prefix ) ) {
                $suffix = substr( $name, strlen( $prefix ) );
            }
            $suffix = strtolower( (string) $suffix );
            $core_tables = array(
                'posts','postmeta','users','usermeta','options','comments','commentmeta','links','terms','term_taxonomy','term_relationships',
                'termmeta','site','sitemeta','blogs','blog_versions','blogmeta','registration_log','signups',
            );
            if ( in_array( $suffix, $core_tables, true ) ) {
                $core_bytes += $bytes;
            } elseif ( false !== strpos( $suffix, 'woocommerce' ) || false !== strpos( $suffix, 'wc_' ) || false !== strpos( $suffix, 'yoast' ) || false !== strpos( $suffix, 'rankmath' ) ) {
                $plugin_bytes += $bytes;
            } elseif ( false !== strpos( $suffix, '_' ) ) {
                $plugin_bytes += $bytes;
            } else {
                $other_bytes += $bytes;
            }
        }

        usort( $top_sizes, function( $a, $b ) {
            return (int) $b['bytes'] - (int) $a['bytes'];
        } );
        usort( $top_rows, function( $a, $b ) {
            return (int) $b['rows'] - (int) $a['rows'];
        } );

        $top_sizes = array_slice( $top_sizes, 0, 10 );
        $top_rows  = array_slice( $top_rows, 0, 10 );

        usort( $bubble_tables, function( $a, $b ) {
            return (int) $b['bytes'] - (int) $a['bytes'];
        } );
        $bubble_tables = array_slice( $bubble_tables, 0, 20 );

        $settings = get_option( 'aegisbackup_dbtools_settings', array() );
        $last_total = isset( $settings['growth_last_total'] ) ? (int) $settings['growth_last_total'] : 0;
        $last_check = isset( $settings['growth_last_check'] ) ? (int) $settings['growth_last_check'] : 0;
        $current_total = isset( $summary['totals']['bytes'] ) ? (int) $summary['totals']['bytes'] : 0;
        $current_ts = (int) current_time( 'timestamp' );

        $overview_payload = array(
            'top_sizes' => $top_sizes,
            'top_rows'  => $top_rows,
            'bubble_tables' => $bubble_tables,
            'storage'   => array(
                'data_bytes'  => $sum_data_bytes,
                'index_bytes' => $sum_index_bytes,
            ),
            'engine_bytes'    => $engine_bytes,
            'collation_bytes' => $collation_bytes,
            'categories' => array(
                'core_bytes'   => $core_bytes,
                'plugin_bytes' => $plugin_bytes,
                'other_bytes'  => $other_bytes,
            ),
            'growth' => array(
                'last_total' => $last_total,
                'last_check' => $last_check,
                'current_total' => $current_total,
                'current_check' => $current_ts,
            ),
        );

		if ( ! wp_script_is( 'aegisbackup-chartjs', 'enqueued' ) ) {
			wp_enqueue_script(
				'aegisbackup-chartjs',
				AEGISBACKUP_URL . 'assets/chart.umd.min.js',
				array(),
				'4.4.1',
				true
			);
		}

		if ( ! wp_script_is( 'aegisbackup-dbtools-overview-charts', 'enqueued' ) ) {
			wp_register_script(
				'aegisbackup-dbtools-overview-charts',
				'',
				array( 'aegisbackup-chartjs' ),
				AEGISBACKUP_VERSION,
				true
			);
			wp_enqueue_script( 'aegisbackup-dbtools-overview-charts' );

			$inline_payload = 'window.AegisBackupDBOverview = ' . wp_json_encode( $overview_payload ) . ';';
			wp_add_inline_script( 'aegisbackup-dbtools-overview-charts', $inline_payload, 'before' );

			$inline_js = file_exists( AEGISBACKUP_DIR . 'assets/db-overview-charts.js' )
				? (string) file_get_contents( AEGISBACKUP_DIR . 'assets/db-overview-charts.js' )
				: '';
			if ( $inline_js ) {
				wp_add_inline_script( 'aegisbackup-dbtools-overview-charts', $inline_js, 'after' );
			}
		}

		?>
		<div style="margin-top:12px;">

			<div class="ab-card" style="margin-bottom:12px;">
				<h3 style="margin-top:0;"><?php esc_html_e( 'Database Intelligence', 'aegisbackup' ); ?></h3>
				<p class="description"><?php esc_html_e( 'Six at-a-glance charts powered by local Chart.js (no external calls).', 'aegisbackup' ); ?></p>

				<div style="display:grid; grid-template-columns: repeat(3, minmax(0, 1fr)); gap: 14px;">
					<div style="padding:12px; border:1px solid #e5e5e5; border-radius:10px; background:#fff;">
						<strong><?php esc_html_e( 'Top Tables by Size (Bar)', 'aegisbackup' ); ?></strong>
						<div style="height:260px;"><canvas id="ab_chart_top_sizes_bar"></canvas></div>
					</div>

					<div style="padding:12px; border:1px solid #e5e5e5; border-radius:10px; background:#fff;">
						<strong><?php esc_html_e( 'Top Tables by Rows (Line)', 'aegisbackup' ); ?></strong>
						<div style="height:260px;"><canvas id="ab_chart_top_rows_line"></canvas></div>
					</div>

					<div style="padding:12px; border:1px solid #e5e5e5; border-radius:10px; background:#fff;">
						<strong><?php esc_html_e( 'Data vs Index Storage (Doughnut)', 'aegisbackup' ); ?></strong>
						<div style="height:260px;"><canvas id="ab_chart_storage_doughnut"></canvas></div>
					</div>

					<div style="padding:12px; border:1px solid #e5e5e5; border-radius:10px; background:#fff;">
						<strong><?php esc_html_e( 'WP Core vs Plugins vs Other (Pie)', 'aegisbackup' ); ?></strong>
						<div style="height:260px;"><canvas id="ab_chart_categories_pie"></canvas></div>
					</div>

					<div style="padding:12px; border:1px solid #e5e5e5; border-radius:10px; background:#fff;">
						<strong><?php esc_html_e( 'Storage by Engine (Polar)', 'aegisbackup' ); ?></strong>
						<div style="height:260px;"><canvas id="ab_chart_engine_polar"></canvas></div>
					</div>

					<div style="padding:12px; border:1px solid #e5e5e5; border-radius:10px; background:#fff;">
						<strong><?php esc_html_e( 'Rows vs Size (Bubble)', 'aegisbackup' ); ?></strong>
						<div style="height:260px;"><canvas id="ab_chart_bubble"></canvas></div>
					</div>
				</div>

                <?php
                $save_url   = admin_url( 'admin-post.php?action=aegisbackup_dbtools_save_settings' );
                $opt_url    = admin_url( 'admin-post.php?action=aegisbackup_dbtools_run_optimization' );
                $growth_url = admin_url( 'admin-post.php?action=aegisbackup_dbtools_run_growth' );
                ?>
               
            </div>
            <div id="ab_dbtools_overview">
                <?php $this->render_table_summary_table( $tables, false, true ); ?>
            </div>

            <h4><?php esc_html_e( 'Activity Log', 'aegisbackup' ); ?></h4>
            <p class="description"><?php esc_html_e( 'Shows a history of manual runs (optimization, growth checks, snapshots).', 'aegisbackup' ); ?></p>
            
            <div id="ab_dbtools_activity_wrap">
                <?php
                $logs = $module->get_activity_logs( 50 );
                if ( empty( $logs ) ) :
                    ?>
                    <p><em><?php esc_html_e( 'No activity recorded yet.', 'aegisbackup' ); ?></em></p>
                    <?php
                else :
                    ?>
                    <table class="widefat striped">
                        <thead>
                            <tr>
                                <th style="width:110px;"><?php esc_html_e( 'Day', 'aegisbackup' ); ?></th>
                                <th style="width:90px;"><?php esc_html_e( 'Time', 'aegisbackup' ); ?></th>
                                <th style="width:120px;"><?php esc_html_e( 'Action', 'aegisbackup' ); ?></th>
                                <th><?php esc_html_e( 'Result', 'aegisbackup' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $logs as $row ) :
                                $ts = isset( $row['ts'] ) ? (int) $row['ts'] : 0;
                                $day = $ts ? wp_date( 'Y-m-d', $ts ) : '';
                                $time = $ts ? wp_date( 'H:i:s', $ts ) : '';
                                $action = isset( $row['action'] ) ? (string) $row['action'] : '';
                                $result = isset( $row['result'] ) ? (string) $row['result'] : '';
                                ?>
                                <tr>
                                    <td><?php echo esc_html( $day ); ?></td>
                                    <td><?php echo esc_html( $time ); ?></td>
                                    <td><code><?php echo esc_html( $action ); ?></code></td>
                                    <td><?php echo esc_html( $result ); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <?php
                endif;
                ?>
            </div>
			<div style="clear:both;"></div>
                <script>
                (function(){
                    function sync(){
                        var np = document.getElementById('ab_new_prefix');
                        var hp = document.getElementById('ab_new_prefix_apply');
                        var he = document.getElementById('ab_prefix_expected');
                        if(np && hp){ hp.value = np.value || ''; }
                        if(np && he){ he.value = np.value || ''; }
                        var cb = document.getElementById('ab_prefix_confirm_apply');
                        var snap = document.getElementById('ab_prefix_snapshot');
                        var btn = document.getElementById('ab_dbtools_apply_prefix');
                        if(btn){
                            var ok = (!!np && np.value && np.value.length>0) && (!!cb && cb.checked) && (!!snap && snap.value);
                            btn.disabled = !ok;
                        }
                    }
                    document.addEventListener('change', sync);
                    document.addEventListener('input', sync);
                    document.addEventListener('DOMContentLoaded', sync);
                    sync();
                })();
                </script>
            <script>
            (function(){
                function sync(){
                    var v = document.getElementById('ab_new_prefix') ? document.getElementById('ab_new_prefix').value : '';
                    var h1 = document.getElementById('ab_new_prefix_apply');
                    var h2 = document.getElementById('ab_prefix_expected');
                    if(h1) h1.value = v;
                    if(h2) h2.value = v;
                    var snap = document.getElementById('ab_prefix_snapshot');
                    var cb = document.getElementById('ab_prefix_confirm_apply');
                    var btn = document.getElementById('ab_dbtools_apply_prefix');
                    if(btn){
                        var ok = !!v && v.length>0 && (!snap || snap.value) && (!!cb && cb.checked);
                        btn.disabled = !ok;
                    }
                }
                document.addEventListener('change', sync);
                document.addEventListener('input', sync);
                document.addEventListener('DOMContentLoaded', sync);
                setTimeout(sync, 0);
            })();
            </script>
            <pre class="aegisbackup-log" id="ab_dbtools_log" style="margin-top:12px;"></pre>
        </div>
        <?php
    }

    protected function render_optimize( AB_Module_DB_Tools $module ) {
        $settings = get_option( 'aegisbackup_dbtools_settings', array() );
        $weekly   = ! empty( $settings['weekly_optimize_enabled'] );
        $growth   = ! empty( $settings['growth_monitor_enabled'] );
        $all_logs = $module->get_activity_logs( 200 );
        if ( ! is_array( $all_logs ) ) {
            $all_logs = array();
        }

        $summary = $module->get_table_summary();
        $tables  = isset( $summary['tables'] ) && is_array( $summary['tables'] ) ? $summary['tables'] : $summary;

		$key = 'ab_dbtools_tableop_' . (int) get_current_user_id();
		$last_res = get_transient( $key );
		if ( false !== $last_res ) {
			delete_transient( $key );
		}
        ?>
        <div style="margin-top:12px;">

            <div class="ab-card" style="margin-bottom:12px;">
                <h3 style="margin-top:0;"><?php esc_html_e( 'Weekly Optimization', 'aegisbackup' ); ?></h3>

                <?php if ( isset( $_GET['ab_msg'] ) && '' !== (string) $_GET['ab_msg'] ) :
                    $ok  = isset( $_GET['ab_ok'] ) && (int) $_GET['ab_ok'] === 1;
                    $cls = $ok ? 'notice notice-success' : 'notice notice-error';
                    ?>
                    <div class="<?php echo esc_attr( $cls ); ?> inline" style="margin-top:10px;">
                        <p><?php echo esc_html( sanitize_text_field( (string) wp_unslash( $_GET['ab_msg'] ) ) ); ?></p>
                    </div>
                <?php endif; ?>

                <?php

                $save_url   = admin_url( 'admin-post.php?action=aegisbackup_dbtools_save_settings' );
                $opt_url    = admin_url( 'admin-post.php?action=aegisbackup_dbtools_run_optimization' );
                $growth_url = admin_url( 'admin-post.php?action=aegisbackup_dbtools_run_growth' );
                ?>

                <form method="post" action="<?php echo esc_url( $save_url ); ?>" style="margin:0;">
                    <?php wp_nonce_field( 'aegisbackup_dbtools_save_settings' ); ?>

                    <table class="form-table" role="presentation">
                        <tr>
                            <th scope="row"><?php esc_html_e( 'Weekly Optimization', 'aegisbackup' ); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="weekly" value="1" id="ab_dbtools_weekly_opt" <?php checked( $weekly ); ?> />
                                    <?php esc_html_e( 'Enable weekly ANALYZE + OPTIMIZE for all tables', 'aegisbackup' ); ?>
                                </label>
                                <p class="description"><?php esc_html_e( 'Runs best-effort optimization once per week. You can also run it manually.', 'aegisbackup' ); ?></p>

                                <p>
                                    <button type="submit" class="button" id="ab_dbtools_run_optimize" form="ab_dbtools_run_optimize_form"><?php esc_html_e( 'Run Optimization Now', 'aegisbackup' ); ?></button>
                                    <span class="spinner" id="ab_dbtools_opt_spinner"></span>
                                </p>

                                <?php
                                $opt_rows = array();
                                foreach ( $all_logs as $row ) {
                                    if ( ! is_array( $row ) ) { continue; }
                                    $act = isset( $row['action'] ) ? (string) $row['action'] : '';
                                    if ( 'optimize' === $act ) { $opt_rows[] = $row; }
                                    if ( count( $opt_rows ) >= 10 ) { break; }
                                }
                                ?>
                                <div style="margin-top:8px;">
                                    <strong style="display:block; margin-left:10px; margin-bottom:6px;">
										<?php esc_html_e( 'Last 10 Optimization Runs', 'aegisbackup' ); ?>
									</strong>
                                    <?php if ( empty( $opt_rows ) ) : ?>
                                        <p><em><?php esc_html_e( 'No optimization runs recorded yet.', 'aegisbackup' ); ?></em></p>
                                    <?php else : ?>
                                        <table class="widefat striped" style="width:100%; max-width:none;">
                                            <thead>
                                                <tr>
                                                    <th style="width:110px;"><?php esc_html_e( 'Day', 'aegisbackup' ); ?></th>
                                                    <th style="width:90px;"><?php esc_html_e( 'Time', 'aegisbackup' ); ?></th>
                                                    <th><?php esc_html_e( 'Result', 'aegisbackup' ); ?></th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ( $opt_rows as $r ) :
                                                    $ts     = isset( $r['ts'] ) ? (int) $r['ts'] : 0;
                                                    $day    = $ts ? wp_date( 'Y-m-d', $ts ) : '';
                                                    $time   = $ts ? wp_date( 'H:i:s', $ts ) : '';
                                                    $result = isset( $r['result'] ) ? (string) $r['result'] : '';
                                                    ?>
                                                    <tr>
                                                        <td><?php echo esc_html( $day ); ?></td>
                                                        <td><?php echo esc_html( $time ); ?></td>
                                                        <td><?php echo esc_html( $result ); ?></td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row"><?php esc_html_e( 'Growth Monitor', 'aegisbackup' ); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="growth" value="1" id="ab_dbtools_growth_monitor" <?php checked( $growth ); ?> />
                                    <?php esc_html_e( 'Track DB size growth (stored as history)', 'aegisbackup' ); ?>
                                </label>
                                <p class="description"><?php esc_html_e( 'Records total DB size periodically so you can spot unusual growth. Email alerts can be added later.', 'aegisbackup' ); ?></p>

                                <p>
                                    <button type="submit" class="button" id="ab_dbtools_run_growth" form="ab_dbtools_run_growth_form"><?php esc_html_e( 'Run Growth Check Now', 'aegisbackup' ); ?></button>
                                    <span class="spinner" id="ab_dbtools_growth_spinner"></span>
                                </p>

                                <?php
                                $growth_rows = array();
                                foreach ( $all_logs as $row ) {
                                    if ( ! is_array( $row ) ) { continue; }
                                    $act = isset( $row['action'] ) ? (string) $row['action'] : '';
                                    if ( 'growth' === $act || 'growth_check' === $act ) { $growth_rows[] = $row; }
                                    if ( count( $growth_rows ) >= 10 ) { break; }
                                }
                                ?>
                                <div style="margin-top:8px;">
                                    <strong style="display:block; margin-left:10px; margin-bottom:6px;">
										<?php esc_html_e( 'Last 10 Growth Checks', 'aegisbackup' ); ?>
									</strong>
                                    <?php if ( empty( $growth_rows ) ) : ?>
                                        <p><em><?php esc_html_e( 'No growth checks recorded yet.', 'aegisbackup' ); ?></em></p>
                                    <?php else : ?>
                                        <table class="widefat striped" style="width:100%; max-width:none;">
                                            <thead>
                                                <tr>
                                                    <th style="width:110px;"><?php esc_html_e( 'Day', 'aegisbackup' ); ?></th>
                                                    <th style="width:90px;"><?php esc_html_e( 'Time', 'aegisbackup' ); ?></th>
                                                    <th><?php esc_html_e( 'Result', 'aegisbackup' ); ?></th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ( $growth_rows as $r ) :
                                                    $ts     = isset( $r['ts'] ) ? (int) $r['ts'] : 0;
                                                    $day    = $ts ? wp_date( 'Y-m-d', $ts ) : '';
                                                    $time   = $ts ? wp_date( 'H:i:s', $ts ) : '';
                                                    $result = isset( $r['result'] ) ? (string) $r['result'] : '';
                                                    ?>
                                                    <tr>
                                                        <td><?php echo esc_html( $day ); ?></td>
                                                        <td><?php echo esc_html( $time ); ?></td>
                                                        <td><?php echo esc_html( $result ); ?></td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                    </table>

                    <p>
                        <button type="submit" class="button button-primary" id="ab_dbtools_save_settings"><?php esc_html_e( 'Save Settings', 'aegisbackup' ); ?></button>
                        <span class="spinner" id="ab_dbtools_save_spinner"></span>
                    </p>
                </form>

                <form id="ab_dbtools_run_optimize_form" method="post" action="<?php echo esc_url( $opt_url ); ?>" style="display:none;">
                    <?php wp_nonce_field( 'aegisbackup_dbtools_health' ); ?>
                </form>
                <form id="ab_dbtools_run_growth_form" method="post" action="<?php echo esc_url( $growth_url ); ?>" style="display:none;">
                    <?php wp_nonce_field( 'aegisbackup_dbtools_health' ); ?>
                </form>
            </div>

            <p class="description"><?php esc_html_e( 'Optimize helps reclaim space and improves performance after heavy content edits. Run during low traffic.', 'aegisbackup' ); ?></p>

			<?php if ( is_array( $last_res ) && ! empty( $last_res ) ) : ?>
				<div class="notice <?php echo ! empty( $last_res['ok'] ) ? 'notice-success' : 'notice-error'; ?>" style="padding:10px 12px; margin: 10px 0 12px 0;">
					<strong><?php echo ! empty( $last_res['ok'] ) ? esc_html__( 'DB Tools:', 'aegisbackup' ) : esc_html__( 'DB Tools Error:', 'aegisbackup' ); ?></strong>
					<?php
					$messages = array();
					if ( isset( $last_res['messages'] ) && is_array( $last_res['messages'] ) ) {
						$messages = $last_res['messages'];
					} elseif ( isset( $last_res['message'] ) ) {
						$messages = array( (string) $last_res['message'] );
					}
					?>
					<?php if ( ! empty( $messages ) ) : ?>
						<ul style="margin:6px 0 0 18px;">
							<?php foreach ( array_slice( $messages, 0, 20 ) as $m ) : ?>
								<li><?php echo esc_html( (string) $m ); ?></li>
							<?php endforeach; ?>
						</ul>
					<?php endif; ?>
				</div>
			<?php endif; ?>

			<?php
			$action_url = admin_url( 'admin-post.php?action=aegisbackup_dbtools_table_op' );
			$pp = isset( $_GET['ab_db_pp'] ) ? (int) $_GET['ab_db_pp'] : 25;
			$p  = isset( $_GET['ab_db_p'] ) ? (int) $_GET['ab_db_p'] : 1;
			?>
			<form method="post" action="<?php echo esc_url( $action_url ); ?>" style="margin:0;">
				<?php wp_nonce_field( 'aegisbackup_dbtools_optimize' ); ?>
				<input type="hidden" name="ab_db_pp" value="<?php echo esc_attr( $pp ); ?>" />
				<input type="hidden" name="ab_db_p" value="<?php echo esc_attr( $p ); ?>" />
				<p>
					<button type="submit" class="button button-secondary" name="op" value="analyze"><?php esc_html_e( 'Analyze Tables', 'aegisbackup' ); ?></button>
					<button type="submit" class="button button-secondary" name="op" value="optimize"><?php esc_html_e( 'Optimize Tables', 'aegisbackup' ); ?></button>
					<button type="submit" class="button" name="op" value="repair"><?php esc_html_e( 'Repair Tables', 'aegisbackup' ); ?></button>
				</p>

				<?php $this->render_table_summary_table( $tables, true, true ); ?>
			</form>

            <pre class="aegisbackup-log" id="ab_dbtools_log" style="margin-top:12px;"></pre>
			<div style="clear:both;"></div>
        </div>
        <?php
    }

    protected function render_prefix( AB_Module_DB_Tools $module ) {
        global $wpdb;
        $current = (string) $wpdb->prefix;
        $uid = (int) get_current_user_id();
        $preview_key = 'ab_dbtools_prefix_preview_' . $uid;
        $apply_key   = 'ab_dbtools_prefix_apply_' . $uid;
        $verify_key  = 'ab_dbtools_prefix_verify_' . $uid;

        $prefix_preview = ( isset( $_GET['ab_prefix_previewed'] ) ) ? get_transient( $preview_key ) : false;
        $prefix_apply   = ( isset( $_GET['ab_prefix_applied'] ) ) ? get_transient( $apply_key ) : false;
        $prefix_verify  = ( isset( $_GET['ab_prefix_verified'] ) ) ? get_transient( $verify_key ) : false;

        if ( $prefix_preview !== false ) { delete_transient( $preview_key ); }
        if ( $prefix_apply   !== false ) { delete_transient( $apply_key ); }
        if ( $prefix_verify  !== false ) { delete_transient( $verify_key ); }
        $uid = (int) get_current_user_id();
        $t_preview = get_transient( 'ab_dbtools_prefix_preview_' . $uid );
        $t_apply   = get_transient( 'ab_dbtools_prefix_apply_' . $uid );
        $t_verify  = get_transient( 'ab_dbtools_prefix_verify_' . $uid );
        ?>
		<div style="margin-top:12px; max-width: 980px;">
			<p class="description"><?php esc_html_e( 'This is a high-risk operation. AegisBackup follows a strict flow: Preview (dry-run) → Select a snapshot (required) → Apply → Verification scan.', 'aegisbackup' ); ?></p>


                <?php if ( $prefix_apply !== false && is_array( $prefix_apply ) ) : ?>
                    <div class="notice <?php echo ! empty( $prefix_apply['ok'] ) ? 'notice-success' : 'notice-error'; ?>" style="padding:10px 12px; margin: 10px 0 12px 0;">
                        <strong><?php echo ! empty( $prefix_apply['ok'] ) ? esc_html__( 'DB Prefix:', 'aegisbackup' ) : esc_html__( 'DB Prefix Error:', 'aegisbackup' ); ?></strong>
                        <?php if ( ! empty( $prefix_apply['message'] ) ) : ?>
                            <?php echo ' ' . esc_html( (string) $prefix_apply['message'] ); ?>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <?php if ( $prefix_verify !== false && is_array( $prefix_verify ) ) : ?>
                    <div class="notice <?php echo ! empty( $prefix_verify['ok'] ) ? 'notice-success' : 'notice-error'; ?>" style="padding:10px 12px; margin: 10px 0 12px 0;">
                        <strong><?php echo ! empty( $prefix_verify['ok'] ) ? esc_html__( 'Verification:', 'aegisbackup' ) : esc_html__( 'Verification Error:', 'aegisbackup' ); ?></strong>
                        <?php if ( ! empty( $prefix_verify['message'] ) ) : ?>
                            <?php echo ' ' . esc_html( (string) $prefix_verify['message'] ); ?>
                        <?php endif; ?>
                        <?php if ( isset( $prefix_verify['issues'] ) && is_array( $prefix_verify['issues'] ) && ! empty( $prefix_verify['issues'] ) ) : ?>
                            <ul style="margin:6px 0 0 18px;">
                                <?php foreach ( array_slice( $prefix_verify['issues'], 0, 30 ) as $iss ) : ?>
                                    <li><?php echo esc_html( (string) $iss ); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

			<input type="hidden" id="aegisbackup_nonce" value="<?php echo esc_attr( wp_create_nonce( 'aegisbackup_nonce' ) ); ?>" />
			<input type="hidden" id="aegisbackup_ajaxurl" value="<?php echo esc_url( admin_url( 'admin-ajax.php' ) ); ?>" />
			<input type="hidden" id="ab_prefix_current" name="current_prefix" value="<?php echo esc_attr( $current ); ?>" />

            <div class="ab-card" style="margin-top:10px;">
                <h3 style="margin-top:0;"><?php esc_html_e( 'Phase 1 — Preview (dry-run, no writes)', 'aegisbackup' ); ?></h3>
                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin:0;">
                    <input type="hidden" name="action" value="aegisbackup_dbtools_prefix_preview" />
                    <?php wp_nonce_field( 'aegisbackup_nonce' ); ?>

                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin:0;">
                    <input type="hidden" name="action" value="aegisbackup_dbtools_prefix_preview" />
                    <?php wp_nonce_field( 'aegisbackup_nonce' ); ?>
                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th scope="row"><?php esc_html_e( 'Current Prefix', 'aegisbackup' ); ?></th>
                            <td><code><?php echo esc_html( $current ); ?></code></td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e( 'New Prefix', 'aegisbackup' ); ?></th>
                            <td>
                                <input type="text" id="ab_new_prefix" name="new_prefix" class="regular-text" placeholder="wpsecure_" />
                                <p class="description"><?php esc_html_e( 'Use only letters, numbers, and underscores. A trailing underscore is recommended (e.g. wpsecure_).', 'aegisbackup' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e( 'Scope', 'aegisbackup' ); ?></th>
                            <td>
                                <select id="ab_prefix_scope" disabled>
                                    <option value="all" selected><?php esc_html_e( 'All matching tables (recommended)', 'aegisbackup' ); ?></option>
                                </select>
                                <p class="description"><?php esc_html_e( 'AegisBackup only allows full-prefix migrations (safer).', 'aegisbackup' ); ?></p>
                            </td>
                        </tr>
                    </tbody>
                </table>

                <p>
                    <button type="submit" class="button" id="ab_dbtools_prefix_preview"><?php esc_html_e( 'Preview Changes', 'aegisbackup' ); ?></button>
                </p>


                    <?php if ( is_array( $t_preview ) && ! empty( $t_preview ) ) : ?>
                        <div class="notice <?php echo ! empty( $t_preview['ok'] ) ? 'notice-success' : 'notice-error'; ?>" style="padding:10px 12px; margin: 10px 0 12px 0;">
                            <strong><?php echo ! empty( $t_preview['ok'] ) ? esc_html__( 'DB Tools:', 'aegisbackup' ) : esc_html__( 'DB Tools Error:', 'aegisbackup' ); ?></strong>
                            <?php if ( isset( $t_preview['message'] ) ) : ?>
                                <span> <?php echo esc_html( (string) $t_preview['message'] ); ?></span>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                <div id="ab_prefix_preview_wrap" style="<?php echo ( $prefix_preview !== false && is_array( $prefix_preview ) ) ? 'display:block;' : 'display:none;'; ?>">
                    <h4><?php esc_html_e( 'Preview of Affected Tables', 'aegisbackup' ); ?></h4>
                    <div id="ab_prefix_preview_tables">
<?php if ( $prefix_preview !== false && is_array( $prefix_preview ) && isset( $prefix_preview['tables'] ) && is_array( $prefix_preview['tables'] ) ) : ?>
                        <table class="widefat striped" style="max-width:760px;">
                            <thead><tr><th><?php esc_html_e( 'Existing Table', 'aegisbackup' ); ?></th><th><?php esc_html_e( 'New Table', 'aegisbackup' ); ?></th></tr></thead>
                            <tbody>
                            <?php foreach ( array_slice( $prefix_preview['tables'], 0, 500 ) as $row ) : ?>
                                <tr><td><code><?php echo esc_html( (string) $row['from'] ); ?></code></td><td><code><?php echo esc_html( (string) $row['to'] ); ?></code></td></tr>
                            <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                    </div>

                    <h4 style="margin-top:16px;"><?php esc_html_e( 'Preview of Prefix-Bound Option Keys', 'aegisbackup' ); ?></h4>
                    <p class="description"><?php esc_html_e( 'These option_name values must be updated to match the new prefix (includes user roles and plugin options).', 'aegisbackup' ); ?></p>
                    <div id="ab_prefix_preview_options">
<?php if ( $prefix_preview !== false && is_array( $prefix_preview ) && isset( $prefix_preview['options'] ) && is_array( $prefix_preview['options'] ) ) : ?>
                        <table class="widefat striped" style="max-width:760px;">
                            <thead><tr><th><?php esc_html_e( 'Option Key', 'aegisbackup' ); ?></th></tr></thead>
                            <tbody>
                            <?php foreach ( array_slice( $prefix_preview['options'], 0, 500 ) as $k ) : ?>
                                <tr><td><code><?php echo esc_html( (string) $k ); ?></code></td></tr>
                            <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                    </div>

                    <h4 style="margin-top:16px;"><?php esc_html_e( 'Preview of Prefix-Bound Usermeta Keys', 'aegisbackup' ); ?></h4>
                    <p class="description"><?php esc_html_e( 'These meta_key values MUST be updated (capabilities/user_level). Missing this step can lock you out of wp-admin.', 'aegisbackup' ); ?></p>
                    <div id="ab_prefix_preview_usermeta">
<?php if ( $prefix_preview !== false && is_array( $prefix_preview ) && isset( $prefix_preview['usermeta'] ) && is_array( $prefix_preview['usermeta'] ) ) : ?>
                        <table class="widefat striped" style="max-width:760px;">
                            <thead><tr><th><?php esc_html_e( 'Usermeta Key', 'aegisbackup' ); ?></th></tr></thead>
                            <tbody>
                            <?php foreach ( array_slice( $prefix_preview['usermeta'], 0, 500 ) as $k ) : ?>
                                <tr><td><code><?php echo esc_html( (string) $k ); ?></code></td></tr>
                            <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                    </div>

                    <h4 style="margin-top:16px;"><?php esc_html_e( 'wp-config.php Readiness', 'aegisbackup' ); ?></h4>
                    <div id="ab_prefix_preview_wpconfig">
<?php if ( $prefix_preview !== false && is_array( $prefix_preview ) ) : ?>
                        <p class="description"><?php echo ! empty( $prefix_preview['wpconfig']['ok'] ) ? esc_html__( 'wp-config.php looks ready for $table_prefix update.', 'aegisbackup' ) : esc_html__( 'wp-config.php requires update after applying. Make sure you can edit it.', 'aegisbackup' ); ?></p>
                    <?php endif; ?>
                    </div>
                </div>
            </div>

                </form>
                </form>
                </form>
            <div class="ab-card" style="margin-top:12px;">
                <h3 style="margin-top:0;"><?php esc_html_e( 'Phase 2 — Apply (requires snapshot + confirmation)', 'aegisbackup' ); ?></h3>
                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin:0;">
                    <input type="hidden" name="action" value="aegisbackup_dbtools_prefix_change" />
                    <?php wp_nonce_field( 'aegisbackup_nonce' ); ?>
                    <input type="hidden" name="new_prefix" id="ab_new_prefix_apply" value="" />

                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin:0;">
                    <input type="hidden" name="action" value="aegisbackup_dbtools_prefix_change" />
                    <?php wp_nonce_field( 'aegisbackup_nonce' ); ?>
                    <input type="hidden" name="new_prefix" id="ab_new_prefix_apply" value="" />
                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th scope="row"><?php esc_html_e( 'Backup Snapshot (required)', 'aegisbackup' ); ?></th>
                            <td>
							<?php
							$items = $module->list_db_backups();
							usort($items, function($a, $b){
								$am = isset($a['mtime']) ? (int) $a['mtime'] : 0;
								$bm = isset($b['mtime']) ? (int) $b['mtime'] : 0;
								return $bm <=> $am; // newest first
							});
							?>
							<select id="ab_prefix_snapshot" name="snapshot_file" style="min-width:420px;">
								<option value=""><?php esc_html_e( '-- Select a DB snapshot --', 'aegisbackup' ); ?></option>
								<?php foreach ( $items as $it ) :
									$file  = isset( $it['file'] ) ? (string) $it['file'] : '';
									if ( '' === $file ) { continue; }
									$size  = isset( $it['size'] ) ? (int) $it['size'] : 0;
									$mtime = isset( $it['mtime'] ) ? (int) $it['mtime'] : 0;

									$label = $file;
									if ( $mtime ) {
										$label .= ' (' . wp_date( 'Y-m-d H:i:s', $mtime ) . ( $size ? (', ' . size_format( $size, 2 )) : '' ) . ')';
									}
								?>
									<option value="<?php echo esc_attr( $file ); ?>">
										<?php echo esc_html( $label ); ?>
									</option>
								<?php endforeach; ?>
							</select>
								<p style="margin-top:6px;">
									<?php
									$snapshots_tab_url = add_query_arg(
										array(
											'page'  => 'aegisbackup',
											'tab'   => 'db',
											'dbtab' => 'backups',
										),
										admin_url( 'admin.php' )
									);
									?>
									<a class="button" href="<?php echo esc_url( $snapshots_tab_url ); ?>">
										<?php esc_html_e( 'Create a snapshot → DB Snapshots tab', 'aegisbackup' ); ?>
									</a>
								</p>
                                <p class="description"><?php esc_html_e( 'Select a snapshot created before applying. If something goes wrong, this is your fastest recovery path.', 'aegisbackup' ); ?></p>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row"><?php esc_html_e( 'Confirmation', 'aegisbackup' ); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" id="ab_prefix_confirm_apply" name="confirm_apply" value="1" />
                                    <?php esc_html_e( 'I understand this can break my site if wp-config.php is not updated. I have selected a snapshot and I am ready to apply.', 'aegisbackup' ); ?>
                                </label>
                                <p class="description"><?php esc_html_e( 'After apply, WordPress will only work when wp-config.php $table_prefix matches the new prefix.', 'aegisbackup' ); ?></p>
                            </td>
                        </tr>
                    </tbody>
                </table>

                <p>
                    <button type="submit" class="button button-primary" id="ab_dbtools_apply_prefix" disabled><?php esc_html_e( 'Apply Prefix Change', 'aegisbackup' ); ?></button>

                    <?php if ( is_array( $t_apply ) && ! empty( $t_apply ) ) : ?>
                        <div class="notice <?php echo ! empty( $t_apply['ok'] ) ? 'notice-success' : 'notice-error'; ?>" style="padding:10px 12px; margin: 10px 0 12px 0;">
                            <strong><?php echo ! empty( $t_apply['ok'] ) ? esc_html__( 'DB Tools:', 'aegisbackup' ) : esc_html__( 'DB Tools Error:', 'aegisbackup' ); ?></strong>
                            <?php if ( isset( $t_apply['message'] ) ) : ?>
                                <span> <?php echo esc_html( (string) $t_apply['message'] ); ?></span>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                </p>

                <div class="notice notice-warning inline">
                    <p><strong><?php esc_html_e( 'High Risk Operation:', 'aegisbackup' ); ?></strong> <?php esc_html_e( 'Do not navigate away until you verify wp-config.php and can load wp-admin again.', 'aegisbackup' ); ?></p>
                </div>
            </div>

                </form>
            <div class="ab-card" style="margin-top:12px;">
                <h3 style="margin-top:0;"><?php esc_html_e( 'Phase 3 — Verification Scan (read-only)', 'aegisbackup' ); ?></h3>
                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin:0;">
                    <input type="hidden" name="action" value="aegisbackup_dbtools_prefix_verify" />
                    <?php wp_nonce_field( 'aegisbackup_nonce' ); ?>
                    <input type="hidden" name="prefix_expected" id="ab_prefix_expected" value="" />
                    <input type="hidden" name="prefix_old" value="<?php echo esc_attr( $current ); ?>" />

                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin:0;">
                    <input type="hidden" name="action" value="aegisbackup_dbtools_prefix_verify" />
                    <?php wp_nonce_field( 'aegisbackup_nonce' ); ?>
                    <input type="hidden" name="prefix_expected" id="ab_prefix_expected" value="" />
                    <input type="hidden" name="prefix_old" value="<?php echo esc_attr( $current ); ?>" />
                <p class="description"><?php esc_html_e( 'After applying, run a read-only scan to confirm: required core tables exist under the expected prefix, no critical tables remain on the old prefix, and no old prefix-bound option/usermeta keys remain.', 'aegisbackup' ); ?></p>
                <p>
                    <button type="submit" class="button" id="ab_dbtools_prefix_verify"><?php esc_html_e( 'Run Verification Scan', 'aegisbackup' ); ?></button>

                    <?php if ( is_array( $t_verify ) && ! empty( $t_verify ) ) : ?>
                        <div class="notice <?php echo ! empty( $t_verify['ok'] ) ? 'notice-success' : 'notice-error'; ?>" style="padding:10px 12px; margin: 10px 0 12px 0;">
                            <strong><?php echo ! empty( $t_verify['ok'] ) ? esc_html__( 'DB Tools:', 'aegisbackup' ) : esc_html__( 'DB Tools Error:', 'aegisbackup' ); ?></strong>
                            <?php if ( isset( $t_verify['message'] ) ) : ?>
                                <span> <?php echo esc_html( (string) $t_verify['message'] ); ?></span>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                </p>
                </form>
                <div id="ab_prefix_verify_wrap" style="display:none;"></div>
                </form>
            </div>

            <pre class="aegisbackup-log" id="ab_dbtools_log" style="margin-top:12px;"></pre>
        </div>
        <?php
    }

    protected function render_table_summary_table( array $summary, $with_checkboxes = false, $paginate = false ) {
        $rows_all = $summary;
        $total    = count( $rows_all );

        if ( ! $with_checkboxes ) {
            $allowed_per = array( 10, 25, 50 );
            $per_page    = 10;
        } else {
            $allowed_per = array( 25, 50, 100, 200 );
            $per_page    = 25;
        }
        $page     = 1;

        if ( $paginate ) {
            $pp = isset( $_GET['ab_db_pp'] ) ? (int) $_GET['ab_db_pp'] : (int) $per_page;
            if ( in_array( $pp, $allowed_per, true ) ) {
                $per_page = $pp;
            }
            $page = isset( $_GET['ab_db_p'] ) ? max( 1, (int) $_GET['ab_db_p'] ) : 1;
        }

        $pages = ( $paginate && $per_page > 0 ) ? (int) ceil( $total / $per_page ) : 1;
        if ( $page > $pages ) {
            $page = $pages;
        }

        if ( $paginate ) {
            $offset  = ( $page - 1 ) * $per_page;
            $summary = array_slice( $rows_all, $offset, $per_page );
        }

        $base_args = array();
        foreach ( array( 'page', 'tab', 'dbtab' ) as $k ) {
            if ( isset( $_GET[ $k ] ) ) {
                $base_args[ $k ] = sanitize_text_field( wp_unslash( $_GET[ $k ] ) );
            }
        }

        if ( $paginate ) {
            echo '<div style="display:flex;align-items:center;justify-content:space-between;margin-top:8px;">';
            echo '<div style="font-size:12px;color:#666;">' . esc_html( sprintf( __( 'Showing %1$d–%2$d of %3$d tables', 'aegisbackup' ), ( ( $page - 1 ) * $per_page ) + 1, min( $page * $per_page, $total ), $total ) ) . '</div>';

			$pp_base = add_query_arg( array_merge( $base_args, array( 'ab_db_p' => 1 ) ), admin_url( 'admin.php' ) );
			echo '<label style="font-size:12px;color:#666;">' . esc_html__( 'Per page', 'aegisbackup' ) . ': ';
			echo '<select id="ab_db_pp_select" data-base="' . esc_attr( esc_url( $pp_base ) ) . '">';
			foreach ( $allowed_per as $n ) {
				echo '<option value="' . esc_attr( $n ) . '" ' . selected( $per_page, $n, false ) . '>' . esc_html( $n ) . '</option>';
			}
			echo '</select></label>';
			echo '<script>(function(){var s=document.getElementById("ab_db_pp_select"); if(!s) return; s.addEventListener("change", function(){ try{ var u=new URL(s.getAttribute("data-base"), window.location.origin); u.searchParams.set("ab_db_pp", s.value); window.location = u.toString(); }catch(e){ var base=s.getAttribute("data-base"); window.location = base + (base.indexOf("?")===-1?"?":"&") + "ab_db_pp=" + encodeURIComponent(s.value); } });})();</script>';
            echo '</div>';
		echo '</div>';
        }
        ?>
        <table class="widefat striped" style="margin-top:12px;">
            <thead>
                <tr>
				<?php if ( $with_checkboxes ) : ?>
					<th style="width:28px;">
						<input
							type="checkbox"
							id="ab_dbtools_select_all"
							value="1"
							aria-label="<?php echo esc_attr__( 'Select all tables', 'aegisbackup' ); ?>"
							onclick="if(window.jQuery){jQuery('.ab_dbtools_table').prop('checked', this.checked);}"
						/>
					</th>
				<?php endif; ?>
                    <th><?php esc_html_e( 'Table', 'aegisbackup' ); ?></th>
                    <th><?php esc_html_e( 'Rows', 'aegisbackup' ); ?></th>
                    <th><?php esc_html_e( 'Size', 'aegisbackup' ); ?></th>
                    <th><?php esc_html_e( 'Engine', 'aegisbackup' ); ?></th>
                    <th><?php esc_html_e( 'Collation', 'aegisbackup' ); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ( $summary as $row ) : ?>
                    <tr>
                        <?php if ( $with_checkboxes ) : ?>
						<td><input type="checkbox" class="ab_dbtools_table" name="tables[]" value="<?php echo esc_attr( $row['name'] ); ?>"></td>
                        <?php endif; ?>
                        <td><code><?php echo esc_html( $row['name'] ); ?></code></td>
                        <td><?php echo esc_html( $row['rows'] ); ?></td>
                        <td><?php echo esc_html( $row['size'] ); ?></td>
                        <td><?php echo esc_html( $row['engine'] ); ?></td>
                        <td><?php echo esc_html( $row['collation'] ); ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php

        if ( $paginate && $pages > 1 ) {
            $make_url = function( $p ) use ( $base_args, $per_page ) {
                return add_query_arg( array_merge( $base_args, array( 'ab_db_p' => (int) $p, 'ab_db_pp' => (int) $per_page ) ), admin_url( 'admin.php' ) );
            };

            echo '<div class="tablenav" style="margin-top:10px;">';
            echo '<div class="tablenav-pages">';
            echo '<span class="pagination-links">';

            $prev = max( 1, $page - 1 );
            $next = min( $pages, $page + 1 );

            printf( '<a class="first-page button" href="%s" aria-label="%s">&laquo;</a> ', esc_url( $make_url( 1 ) ), esc_attr__( 'First page', 'aegisbackup' ) );
            printf( '<a class="prev-page button" href="%s" aria-label="%s">&lsaquo;</a> ', esc_url( $make_url( $prev ) ), esc_attr__( 'Previous page', 'aegisbackup' ) );

            echo '<span class="paging-input" style="margin:0 6px;">';
            echo '<span class="tablenav-paging-text">' . esc_html( sprintf( __( '%1$d of %2$d', 'aegisbackup' ), (int) $page, (int) $pages ) ) . '</span>';
            echo '</span>';

            printf( '<a class="next-page button" href="%s" aria-label="%s">&rsaquo;</a> ', esc_url( $make_url( $next ) ), esc_attr__( 'Next page', 'aegisbackup' ) );
            printf( '<a class="last-page button" href="%s" aria-label="%s">&raquo;</a>', esc_url( $make_url( $pages ) ), esc_attr__( 'Last page', 'aegisbackup' ) );

            echo '</span></div></div>';
        }
    }


    protected function render_backups( AB_Module_DB_Tools $module ) {
        ?>
        <div class="ab-card">
            <h3><?php esc_html_e( 'DB Snapshots', 'aegisbackup' ); ?></h3>
            <p class="description"><?php esc_html_e( 'Create a quick SQL snapshot before risky operations (prefix changes, restores, etc.). Stored in uploads/aegisbackup/db-tools/ and protected with .htaccess.', 'aegisbackup' ); ?></p>

            <?php
            $create_url = admin_url( 'admin-post.php?action=aegisbackup_dbtools_create_snapshot' );
            ?>
            <form method="post" action="<?php echo esc_url( $create_url ); ?>" style="margin:0 0 12px 0;">
                <?php wp_nonce_field( 'aegisbackup_dbtools_snapshot' ); ?>
                <button type="submit" class="button button-primary" id="ab_dbtools_create_snapshot">
                    <?php esc_html_e( 'Create DB Snapshot Now', 'aegisbackup' ); ?>
                </button>
                <span class="description" style="margin-left:8px;"><?php esc_html_e( 'Creates a .sql snapshot and lists it below.', 'aegisbackup' ); ?></span>
            </form>

            <?php
            $items = $module->list_db_backups();
            ?>
            <div id="ab_dbtools_snapshots_wrap">
                <?php if ( empty( $items ) ) : ?>
                    <p><em><?php esc_html_e( 'No snapshots found yet.', 'aegisbackup' ); ?></em></p>
                <?php else : ?>
                    <table class="widefat striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'File', 'aegisbackup' ); ?></th>
                                <th style="width:120px;"><?php esc_html_e( 'Size', 'aegisbackup' ); ?></th>
                                <th style="width:180px;"><?php esc_html_e( 'Modified', 'aegisbackup' ); ?></th>
                                <th style="width:200px;"><?php esc_html_e( 'Actions', 'aegisbackup' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $items as $it ) :
                                $file = isset( $it['file'] ) ? (string) $it['file'] : '';
                                $size = isset( $it['size'] ) ? (int) $it['size'] : 0;
                                $mtime = isset( $it['mtime'] ) ? (int) $it['mtime'] : 0;

                                $download = add_query_arg(
                                    array(
                                        'action' => 'aegisbackup_dbtools_download_snapshot',
                                        'file'   => basename( $file ),
                                    ),
                                    admin_url( 'admin-post.php' )
                                );
                                $download = wp_nonce_url( $download, 'aegisbackup_dbtools_snapshot' );

                                $delete = add_query_arg(
                                    array(
                                        'action' => 'aegisbackup_dbtools_delete_snapshot',
                                        'file'   => basename( $file ),
                                    ),
                                    admin_url( 'admin-post.php' )
                                );
                                $delete = wp_nonce_url( $delete, 'aegisbackup_dbtools_snapshot' );

                                ?>
                                <tr>
                                    <td><code><?php echo esc_html( basename( $file ) ); ?></code></td>
                                    <td><?php echo esc_html( size_format( $size, 2 ) ); ?></td>
                                    <td><?php echo esc_html( $mtime ? wp_date( 'Y-m-d H:i:s', $mtime ) : '' ); ?></td>
                                    <td>
                                        <a class="button button-small" href="<?php echo esc_url( $download ); ?>"><?php esc_html_e( 'Download', 'aegisbackup' ); ?></a>
                                        <a class="button button-small" href="<?php echo esc_url( $delete ); ?>" onclick="return confirm('<?php echo esc_js( __( 'Delete this snapshot?', 'aegisbackup' ) ); ?>');"><?php esc_html_e( 'Delete', 'aegisbackup' ); ?></a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>

            <p class="description" style="margin-top:10px;">
                <?php esc_html_e( 'If your host blocks AJAX, these actions still work because they run through admin-post.', 'aegisbackup' ); ?>
            </p>
        </div>
        
        <div class="aegisbackup-card" style="margin-top:16px;">
            <h2><?php esc_html_e( 'Local Backup Retention', 'aegisbackup' ); ?></h2>
            <p class="description"><?php esc_html_e( 'Automatically manage local DB snapshots after a number of days.', 'aegisbackup' ); ?></p>

            <?php
            $ret = get_option( 'aegisbackup_local_retention', array() );
            $days = isset( $ret['db_snapshots']['days'] ) ? absint( $ret['db_snapshots']['days'] ) : 30;
            $act  = isset( $ret['db_snapshots']['action'] ) ? sanitize_key( $ret['db_snapshots']['action'] ) : 'purge';
            if ( $days < 1 ) { $days = 1; }
            if ( $days > 90 ) { $days = 90; }
            ?>
            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <?php wp_nonce_field( 'aegisbackup_save_settings', 'aegisbackup_settings_nonce' ); ?>
                <input type="hidden" name="action" value="aegisbackup_save_settings" />
                <input type="hidden" name="settings_section" value="retention" />
                <input type="hidden" name="retention_context" value="db_snapshots" />
                <input type="hidden" name="retention_redirect" value="<?php echo esc_url( admin_url( 'admin.php?page=aegisbackup&tab=db&dbtab=backups' ) ); ?>" />

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Days', 'aegisbackup' ); ?></th>
                        <td>
                            <select name="retention_days">
                                <?php for ( $i = 1; $i <= 90; $i++ ) : ?>
                                    <option value="<?php echo esc_attr( $i ); ?>" <?php selected( $days, $i ); ?>><?php echo esc_html( $i ); ?></option>
                                <?php endfor; ?>
                            </select>
                            <p class="description"><?php esc_html_e( 'Keep local DB snapshots for this many days.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Action', 'aegisbackup' ); ?></th>
                        <td>
                            <select name="retention_action">
                                <option value="purge" <?php selected( 'purge', $act ); ?>><?php esc_html_e( 'Delete / Purge', 'aegisbackup' ); ?></option>
                                <option value="offsite" <?php selected( 'offsite', $act ); ?>><?php esc_html_e( 'Move to Offsite Remote Storage', 'aegisbackup' ); ?></option>
                            </select>
                            <p class="description"><?php esc_html_e( 'If “Move to Offsite” is selected, AegisBackup will attempt an offsite upload (if enabled) before deleting local copies.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="submit" class="button button-primary"><?php esc_html_e( 'Save Retention', 'aegisbackup' ); ?></button>
                </p>
            </form>
        </div>

        <?php
    }

    protected function render_health( AB_Module_DB_Tools $module ) {
        $settings = get_option( 'aegisbackup_dbtools_settings', array() );
        $weekly   = ! empty( $settings['weekly_optimize_enabled'] );
        $growth   = ! empty( $settings['growth_monitor_enabled'] );
        $all_logs = $module->get_activity_logs( 200 );
        if ( ! is_array( $all_logs ) ) {
            $all_logs = array();
        }
        ?>
        <?php
    }


}
