<?php
namespace AegisBackup\Backup;

defined( 'ABSPATH' ) || exit;

class AB_WP_Backup_Manager {
    const PLANS_OPTION = 'aegisbackup_wp_backup_plans';
    const CRON_HOOK    = 'aegisbackup_run_wp_backup_plan';

    protected $backup;

    public function __construct( AB_Backup_Manager $backup ) {
        $this->backup = $backup;
        add_action( self::CRON_HOOK, array( $this, 'run_plan_cron' ), 10, 1 );
        add_filter( 'cron_schedules', array( $this, 'add_custom_schedules' ) );
    }

    public function add_custom_schedules( $schedules ) {
        if ( ! is_array( $schedules ) ) {
            $schedules = array();
        }
        if ( ! isset( $schedules['aegisbackup_weekly'] ) ) {
            $schedules['aegisbackup_weekly'] = array(
                'interval' => 7 * DAY_IN_SECONDS,
                'display'  => 'Once Weekly (AegisBackup)',
            );
        }
        if ( ! isset( $schedules['aegisbackup_monthly'] ) ) {
            $schedules['aegisbackup_monthly'] = array(
                'interval' => 30 * DAY_IN_SECONDS,
                'display'  => 'Once Monthly (AegisBackup)',
            );
        }
        return $schedules;
    }

    public function get_plans() {
        $plans = get_option( self::PLANS_OPTION, array() );
        return is_array( $plans ) ? $plans : array();
    }

    public function get_plan( $plan_id ) {
        $plans = $this->get_plans();
        return isset( $plans[ $plan_id ] ) && is_array( $plans[ $plan_id ] ) ? $plans[ $plan_id ] : null;
    }

    public function save_plan( array $plan ) {
        $plans = $this->get_plans();
        if ( empty( $plan['id'] ) ) {
            $plan['id'] = 'wp_' . wp_generate_password( 10, false, false );
        }
        $id = (string) $plan['id'];

        $defaults = array(
            'id' => $id,
            'name' => 'WP Snapshot',
            'enabled' => 1,
            'frequency' => 'daily', // daily|weekly|monthly
            'time' => '02:00',
            'weekly_day' => 1, // Mon
            'monthly_day' => 1, // 1..28
            'include_core' => 1,
            'include_config' => 1,
            'include_htaccess' => 1,
            'exclude' => "wp-content/cache\nwp-content/uploads/aegisbackup\nwp-content/uploads/backup\nwp-content/plugins/aegisbackup/backups",
        );

        $merged = array_merge( $defaults, $plan );
        $plans[ $id ] = $merged;
        update_option( self::PLANS_OPTION, $plans, false );

        $this->schedule_plan( $id );
        return $merged;
    }

    public function delete_plan( $plan_id ) {
        $plans = $this->get_plans();
        if ( isset( $plans[ $plan_id ] ) ) {
            unset( $plans[ $plan_id ] );
            update_option( self::PLANS_OPTION, $plans, false );
        }
        $this->unschedule_plan( $plan_id );
        return true;
    }

    public function run_plan_now( $plan_id ) {
        $plan = $this->get_plan( $plan_id );
        if ( empty( $plan ) ) {
            return false;
        }
        $this->run_plan( $plan_id, $plan, true );
        return true;
    }

    public function schedule_plan( $plan_id ) {
        $plan = $this->get_plan( $plan_id );
        if ( empty( $plan ) ) {
            return;
        }
        $this->unschedule_plan( $plan_id );

        if ( empty( $plan['enabled'] ) ) {
            return;
        }

        $frequency = isset( $plan['frequency'] ) ? (string) $plan['frequency'] : 'daily';
        $time = isset( $plan['time'] ) ? (string) $plan['time'] : '02:00';

        $ts = $this->next_run_timestamp( $frequency, $time, $plan );
        if ( ! $ts ) {
            return;
        }

        $recurrence = 'daily';
        if ( 'weekly' === $frequency ) {
            $recurrence = 'aegisbackup_weekly';
        } elseif ( 'monthly' === $frequency ) {
            $recurrence = 'aegisbackup_monthly';
        }

        wp_schedule_event( $ts, $recurrence, self::CRON_HOOK, array( $plan_id ) );
    }

    public function unschedule_plan( $plan_id ) {
        $timestamp = wp_next_scheduled( self::CRON_HOOK, array( $plan_id ) );
        while ( $timestamp ) {
            wp_unschedule_event( $timestamp, self::CRON_HOOK, array( $plan_id ) );
            $timestamp = wp_next_scheduled( self::CRON_HOOK, array( $plan_id ) );
        }
    }

    protected function next_run_timestamp( $frequency, $time, array $plan ) {
        $tz = wp_timezone();
        $now = new \DateTimeImmutable( 'now', $tz );

        $parts = explode( ':', $time );
        $hh = isset( $parts[0] ) ? max( 0, min( 23, (int) $parts[0] ) ) : 2;
        $mm = isset( $parts[1] ) ? max( 0, min( 59, (int) $parts[1] ) ) : 0;

        if ( 'weekly' === $frequency ) {
            $weekday = isset( $plan['weekly_day'] ) ? (int) $plan['weekly_day'] : 1; // 0..6
            $weekday = max( 0, min( 6, $weekday ) );
            $candidate = $now->setTime( $hh, $mm, 0 );
            $cur_w = (int) $candidate->format( 'w' );
            $delta = ( $weekday - $cur_w + 7 ) % 7;
            if ( 0 === $delta && $candidate <= $now ) {
                $delta = 7;
            }
            $candidate = $candidate->modify( '+' . $delta . ' days' );
            return $candidate->getTimestamp();
        }

        if ( 'monthly' === $frequency ) {
            $day = isset( $plan['monthly_day'] ) ? (int) $plan['monthly_day'] : 1;
            $day = max( 1, min( 28, $day ) );

            $candidate = $now->setDate( (int)$now->format('Y'), (int)$now->format('m'), $day )->setTime( $hh, $mm, 0 );
            if ( $candidate <= $now ) {
                $candidate = $candidate->modify( 'first day of next month' );
                $candidate = $candidate->setDate( (int)$candidate->format('Y'), (int)$candidate->format('m'), $day )->setTime( $hh, $mm, 0 );
            }
            return $candidate->getTimestamp();
        }

        $candidate = $now->setTime( $hh, $mm, 0 );
        if ( $candidate <= $now ) {
            $candidate = $candidate->modify( '+1 day' );
        }
        return $candidate->getTimestamp();
    }

    public function run_plan_cron( $plan_id ) {
        $plan = $this->get_plan( $plan_id );
        if ( empty( $plan ) || empty( $plan['enabled'] ) ) {
            return;
        }
        $this->run_plan( $plan_id, $plan, false );

        if ( isset( $plan['frequency'] ) && 'monthly' === (string) $plan['frequency'] ) {
            $this->schedule_plan( $plan_id );
        }
    }

    protected function run_plan( $plan_id, array $plan, $manual ) {
        $args = array(
            'include_files'  => 1,
            'include_db'     => 1,
            'include_config' => ! empty( $plan['include_config'] ) ? 1 : 0,
            'include_core'   => 1,
            'excludes'       => isset( $plan['exclude'] ) ? (string) $plan['exclude'] : '',
            'db_export_mode' => 'auto',
            'backup_type'    => 'full',
            'package_purpose'=> 'wordpress',
            'snapshot'       => 1,
            'include_htaccess' => ! empty( $plan['include_htaccess'] ) ? 1 : 0,
        );

        $job = $this->backup->start_backup_job( $args );
        if ( empty( $job['job_id'] ) ) {
            return;
        }
        $job_id = (string) $job['job_id'];
        $start = time();
        $max_seconds = $manual ? 25 : 20;

        do {
            $step = $this->backup->process_backup_job( $job_id );
            if ( ! empty( $step['done'] ) ) {
                break;
            }
        } while ( ( time() - $start ) < $max_seconds );

        $plans = $this->get_plans();
        if ( isset( $plans[ $plan_id ] ) && is_array( $plans[ $plan_id ] ) ) {
            $plans[ $plan_id ]['last_run'] = current_time( 'mysql' );
            if ( ! empty( $step['package'] ) ) {
                $plans[ $plan_id ]['last_package'] = (string) $step['package'];
            }
            update_option( self::PLANS_OPTION, $plans, false );
        }
    }
}
